/**
 * OPEN-XCHANGE legal information
 *
 * All intellectual property rights in the Software are protected by
 * international copyright laws.
 *
 *
 * In some countries OX, OX Open-Xchange, open xchange and OXtender
 * as well as the corresponding Logos OX Open-Xchange and OX are registered
 * trademarks of the Open-Xchange, Inc. group of companies.
 * The use of the Logos is not covered by the GNU General Public License.
 * Instead, you are allowed to use these Logos according to the terms and
 * conditions of the Creative Commons License, Version 2.5, Attribution,
 * Non-commercial, ShareAlike, and the interpretation of the term
 * Non-commercial applicable to the aforementioned license is published
 * on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 * Please make sure that third-party modules and libraries are used
 * according to their respective licenses.
 *
 * Any modifications to this package must retain all copyright notices
 * of the original copyright holder(s) for the original code used.
 *
 * After any such modifications, the original and derivative code shall remain
 * under the copyright of the copyright holder(s) and/or original author(s)per
 * the Attribution and Assignment Agreement that can be located at
 * http://www.open-xchange.com/EN/developer/. The contributing author shall be
 * given Attribution for the derivative code and a license granting use.
 *
 *  Copyright (C) 2004-2014 Open-Xchange, Inc.
 *  Mail: info@open-xchange.com
 *
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License, Version 2 as published
 *  by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *  or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc., 59
 *  Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
package com.openxchange.office_communication.jms.core.camel;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.config.PropertyPlaceholderConfigurer;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.context.support.ClassPathXmlApplicationContext;

import com.openxchange.office_communication.tools.common.CollectionUtils;

//=============================================================================
/** provide an embedded camel/jms/spring context which can be configured
 *  and controlled within it's lifetime.
 */
public class EmbeddedCamelContext
{
	//-------------------------------------------------------------------------
	private static final Logger LOG = LoggerFactory.getLogger(EmbeddedCamelContext.class);
	
	//-------------------------------------------------------------------------
	/** force using of our static factory method.
	 */
	private EmbeddedCamelContext ()
		throws Exception
	{}
	
	//-------------------------------------------------------------------------
	/** create new ready-to-use instances of such embedded context objects.
	 * 
	 *  @return a new context always.
	 */
	public static synchronized EmbeddedCamelContext get ()
		throws Exception
	{
		LOG.info("create new context ...");
		final EmbeddedCamelContext aContext = new EmbeddedCamelContext ();
		LOG.info("... new context is : ["+aContext+"]");
		return aContext;
	}

	//-------------------------------------------------------------------------
	/** configure this context.
	 * 
	 *  You can pass relative paths to configuration files inside class path
	 *  to this method. They all will be loaded on calling start() then.
	 *  
	 *  Has to be called BEFORE start. Throws an exception otherwise.
	 * 
	 *	@param	lConfigs [IN]
	 *			the list of configuration files.
	 *			Has not to be null nor empty.
	 */
	public synchronized void configure (final String... lNewConfigs)
	    throws Exception
	{
		LOG.info("configure context : ["+lNewConfigs+"] ...");
		Validate.notEmpty(lNewConfigs, "Invalid argument 'configs'.");

		if (impl_isRunning ())
			throw new RuntimeException ("Method shouldnt be called. Context already running.");

		LOG.info("... take over new config");
		final List< String > lConfigs = mem_Configs ();
		for (final String sNewConfig : lNewConfigs)
		{
			if ( ! lConfigs.contains(sNewConfig))
				lConfigs.add(sNewConfig);
		}
	}
	
	//-------------------------------------------------------------------------
	public synchronized void configure (final ContextVariable... lNewVars)
	    throws Exception
	{
		LOG.info("configure variables : ["+lNewVars+"] ...");
		Validate.notEmpty(lNewVars, "Invalid argument 'vars'.");
		
		if (impl_isRunning ())
			throw new RuntimeException ("Method shouldnt be called. Context already running.");

		LOG.info("... take over new variables");
		final Map< String, ContextVariable > lVars = mem_Vars ();
		for (final ContextVariable aNewVar : lNewVars)
		{
			final String sKey = aNewVar.getKey();
			lVars.put(sKey, aNewVar);
		}
	}
	
	//-------------------------------------------------------------------------
	public synchronized void start ()
	    throws Exception
	{
		LOG.info("start context ...");
		if (impl_isRunning ())
			return;
	
		Validate.notEmpty(m_lConfigs, "Invalid configuration. Did you called configure() before?");

		LOG.info("... prepare context");
		final String[]                       lConfigs = CollectionUtils.listToArray(mem_Configs(), String.class);
		final Collection< ContextVariable >  lVars    = mem_Vars().values();
		final ConfigurableApplicationContext aContext = new ClassPathXmlApplicationContext (lConfigs, false);

		if ( ! CollectionUtils.isEmpty(lVars))
		{
			LOG.info("... found context variables");
			final PropertyPlaceholderConfigurer aConfig = new PropertyPlaceholderConfigurer();
		    final Properties                    lProps  = new Properties();
			LOG.info("... prepare context variable");
			for (final ContextVariable aVar : lVars)
			{
				LOG.info("... prepare context variable : "+aVar);
				final String sKey   = aVar.getKey  ();
				final String sValue = aVar.getValue();
			    lProps.setProperty(sKey, sValue);
			}
		    aConfig .setProperties(lProps);
		    
			LOG.info("... set context variables");
		    aContext.addBeanFactoryPostProcessor(aConfig);
		}
		
		aContext.refresh();
		m_aSpringContext = aContext;

		LOG.info("... context starts now");
		aContext.start();
		LOG.info("... context started");
	}

	//-------------------------------------------------------------------------
	/** shutdown and stop the context.
	 * 
	 *  Does nothing if context is not yet running.
	 */
	public synchronized void stop ()
	    throws Exception
	{
		LOG.info("stop context ...");
		if ( ! impl_isRunning ())
			return;
		
		final ConfigurableApplicationContext aContext         = m_aSpringContext;
		                                     m_aSpringContext = null;

 		LOG.info("... context stops now");
        aContext.stop  ();
        aContext.close ();
 		LOG.info("... context stopped");
	}

	//-------------------------------------------------------------------------
	/** provide read access to beans available inside this context.
	 * 
	 *  @param	sID [IN]
	 *  		the bean as its used within the corresponding XML.
	 *
	 *	@param	aType [IN]
	 *			the type the bean has to be cast to.
	 *
	 *	@return	the bean if it exists; null otherwise.
	 */
	@SuppressWarnings("unchecked")
	public synchronized < T > T getBean (final String     sID  ,
							             final Class< T > aType)
		throws Exception
	{
		LOG.info("asked for bean : [id='"+sID+"', type='"+aType+"'] ...");
		final Object aBean = m_aSpringContext.getBean(sID, aType);

		LOG.info("... found : [bean='"+aBean+"'].");
		return (T) aBean;
	}

	//-------------------------------------------------------------------------
	private boolean impl_isRunning ()
	     throws Exception
	{
		LOG.info("check if context is running ...");
		final boolean bRunning = (m_aSpringContext != null);
		LOG.info("... "+(bRunning ? "is running" : "is not running"));
		return bRunning;
	}
	
	//-------------------------------------------------------------------------
	private List< String > mem_Configs ()
		throws Exception
	{
		if (m_lConfigs == null)
			m_lConfigs = new ArrayList< String > ();
		return m_lConfigs;
	}
	
	//-------------------------------------------------------------------------
	private Map< String, ContextVariable > mem_Vars ()
		throws Exception
	{
		if (m_lVars == null)
			m_lVars = new HashMap< String, ContextVariable > ();
		return m_lVars;
	}

	//-------------------------------------------------------------------------
	private ConfigurableApplicationContext m_aSpringContext = null;

	//-------------------------------------------------------------------------
	private List< String > m_lConfigs = null;
	
	//-------------------------------------------------------------------------
	private Map< String, ContextVariable > m_lVars = null;
}
