/**
 * OPEN-XCHANGE legal information
 *
 * All intellectual property rights in the Software are protected by
 * international copyright laws.
 *
 *
 * In some countries OX, OX Open-Xchange, open xchange and OXtender
 * as well as the corresponding Logos OX Open-Xchange and OX are registered
 * trademarks of the Open-Xchange, Inc. group of companies.
 * The use of the Logos is not covered by the GNU General Public License.
 * Instead, you are allowed to use these Logos according to the terms and
 * conditions of the Creative Commons License, Version 2.5, Attribution,
 * Non-commercial, ShareAlike, and the interpretation of the term
 * Non-commercial applicable to the aforementioned license is published
 * on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 * Please make sure that third-party modules and libraries are used
 * according to their respective licenses.
 *
 * Any modifications to this package must retain all copyright notices
 * of the original copyright holder(s) for the original code used.
 *
 * After any such modifications, the original and derivative code shall remain
 * under the copyright of the copyright holder(s) and/or original author(s)per
 * the Attribution and Assignment Agreement that can be located at
 * http://www.open-xchange.com/EN/developer/. The contributing author shall be
 * given Attribution for the derivative code and a license granting use.
 *
 *  Copyright (C) 2004-2014 Open-Xchange, Inc.
 *  Mail: info@open-xchange.com
 *
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License, Version 2 as published
 *  by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *  or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc., 59
 *  Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
package com.openxchange.office_communication.cluster_management;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.openxchange.office_communication.configuration.configitems.cluster_management.AppDescriptor;
import com.openxchange.office_communication.configuration.configitems.cluster_management.ClusterNodeConfig;

//=============================================================================
public class ClusterNodeManager
{
	//-------------------------------------------------------------------------
	private static final Logger LOG = LoggerFactory.getLogger(ClusterNodeManager.class);
	
	//-------------------------------------------------------------------------
	private ClusterNodeManager ()
		throws Exception
	{
		impl_registerShutdownHook ();
	}
	
	//-------------------------------------------------------------------------
	public static synchronized ClusterNodeManager get ()
	    throws Exception
	{
		if (m_gSingleton == null)
			m_gSingleton = new ClusterNodeManager ();
		return m_gSingleton;
	}
	
	//-------------------------------------------------------------------------
	public void start ()
		throws Exception
	{
		LOG.info("start ...");
		if (m_bStarted)
		{
			LOG.info("... already started.");
			return;
		}

		impl_defineApps           ();
		impl_ensureAppsAreRunning ();
		
		m_bStarted = true;
		LOG.info("started.");
	}

	//-------------------------------------------------------------------------
	public void stop ()
		throws Exception
	{
		LOG.info("start ...");
		if ( ! m_bStarted)
		{
			LOG.info("... already stopped.");
			return;
		}

		impl_stopApps ();
		
		m_bStarted = false;
		LOG.info("stopped.");
	}

	//-------------------------------------------------------------------------
	private void impl_defineApps ()
		throws Exception
	{
		LOG.info("define apps ...");
		final ClusterNodeConfig    aConfig      = mem_Config ();
		final Set< AppDescriptor > lAppConfigs  = aConfig.getApps();
		final Map< String, App >   aAppRegistry = mem_AppRegistry ();

		for (final AppDescriptor aAppConfig : lAppConfigs)
		{
			final int nInstances = aAppConfig.getMaxInstances();
			for (int i=0; i<nInstances; ++i)
			{
				LOG.info("... define app '"+aAppConfig.getId()+"' [inst="+i+"]");
				final App aApp = new App ();
				aApp.setDescriptor(aAppConfig);
				aAppRegistry.put(aAppConfig.getId(), aApp);
			}
		}
		LOG.info("apps defined.");
	}

	//-------------------------------------------------------------------------
	private void impl_ensureAppsAreRunning ()
		throws Exception
	{
		LOG.info("ensure apps are running ...");
		final Iterator< App > rApps = mem_AppRegistry ().values().iterator();
		while (rApps.hasNext())
		{
			try
			{
				final App aApp = rApps.next();
				if ( ! aApp.isRunning())
				{
					LOG.info("... start app '"+aApp.m_aDescriptor.getId()+"'");
					aApp.start();
					LOG.info("... OK");
				}
				else
					LOG.info("... app '"+aApp.m_aDescriptor.getId()+"' is already running");
			}
			catch (final Throwable ex)
			{
				ex.printStackTrace();
				/// TODO handle errors
			}
		}
		LOG.info("apps are running.");
	}

	//-------------------------------------------------------------------------
	private void impl_stopApps ()
		throws Exception
	{
		LOG.info("stop apps ...");
		final Iterator< App > rApps = mem_AppRegistry ().values().iterator();

		while (rApps.hasNext())
		{
			try
			{
				final App aApp = rApps.next();
				if (aApp.isRunning())
				{
					LOG.info("... stop app '"+aApp.m_aDescriptor.getId()+"'");
					aApp.stop();
					LOG.info("... OK");
				}
				else
					LOG.info("... app '"+aApp.m_aDescriptor.getId()+"' is already stopped");
			}
			catch (final Throwable ex)
			{
				ex.printStackTrace();
				/// TODO handle errors
			}
		}
		LOG.info("apps stopped.");
	}

	//-------------------------------------------------------------------------
	private void impl_registerShutdownHook ()
	    throws Exception
	{
		LOG.info("register shutdown hook ...");
		Runtime.getRuntime().addShutdownHook(new Thread ()
		{
			@Override
			public void run ()
			{
				impl_shutdown ();
			}
		});
	}

	//-------------------------------------------------------------------------
	private void impl_shutdown ()
	{
		LOG.info("shutdown hook triggered ...");
		try
		{
			impl_stopApps ();
		}
		catch (Throwable ex)
		{
			// TODO find suitable error handling
		}
	}
	
	//-------------------------------------------------------------------------
	private ClusterNodeConfig mem_Config ()
		throws Exception
	{
		if (m_aConfig == null)
			m_aConfig = ClusterNodeConfig.access();
		return m_aConfig;
	}

	//-------------------------------------------------------------------------
	private Map< String, App > mem_AppRegistry ()
	    throws Exception
	{
		if (m_aAppRegistry == null)
			m_aAppRegistry = new HashMap< String, App > ();
		return m_aAppRegistry;
	}

	//-------------------------------------------------------------------------
	private static ClusterNodeManager m_gSingleton = null;
	
	//-------------------------------------------------------------------------
	private boolean m_bStarted = false;
	
	//-------------------------------------------------------------------------
	private ClusterNodeConfig m_aConfig = null;

	//-------------------------------------------------------------------------
	private Map< String, App > m_aAppRegistry = null;
}
