/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.tools.jobpool;


//=============================================================================
/** A job pool can be used to 'borrow' jobs of same type from it ...
 *
 *  Their environment will be checked and provided by the pool
 *  Their life time will be managed by the pool.
 *  
 *  Use the {@link JobPoolManager} to retrieve instances of (different) pools.
 *
 *  An example :
 *  <code>
 *  IJobPool       iPool   = JobPoolManager.accessPool(YourJob.class);
 *  JobPoolConfig  aConfig = new JobPoolConfig  ();
 *  JobEnvironment aEnv    = new JobEnvironment ();
 *  
 *	aConfig.set(JobPoolConfig.PROP_POOL_SIZE     , 10          );
 *	aConfig.set(JobPoolConfig.PROP_BLOCK_ON_LEASE, Boolean.TRUE);
 *
 *  aEnv.set (YourJob.ENV_EXECUTABLE , "foo.exe"   );
 *  aEnv.set (YourJob.ENV_COMMANDLINE, "-d -a doit");
 *
 *	iPool.configureJobPool        (aConfig);
 *  iPool.configureJobEnvironment (aEnv   );
 *
 *  // no the pool is ready to obtain jobs from it ...
 *  // all jobs will use the provided environment ...
 *  </code>
 */
public interface IJobPool
{
	//-------------------------------------------------------------------------
	/** configure the pool ...
	 * 
	 *  Every pool is configured by default within the code ...
	 *  but you can configure it from outside and override those code defaults.
	 * 
	 *  @param	aConfig [IN]
	 *  		contains the configuration to be applied to the pool.
	 */
	public void configureJobPool (final JobPoolConfig aConfig)
	    throws Exception;

	//-------------------------------------------------------------------------
	/** configure an environment for a specific job ...
	 * 
	 *  e.g. in case jobs will use external processes the environment of
	 *  machines, ports and other useful resources has to be defined here.
	 *  
	 *  Every job (instance) will get its own environment on it's de-/activation
	 *  routine so it can e.g. start/stop external processes.
	 *  We do not share environments ... because e.g. ports has to be
	 *  different for different jobs. If not ... it doesn't matter if all
	 *  job will be configured with the same environment.
	 *  
	 *  You decide .-)
	 * 
	 *  @param	nJobId [IN]
	 *  		the job id, where the environment is for.
	 *  		Needs to be in range [1...POOLSIZE]
	 * 
	 *	@param	aEnv [IN]
	 *			the environment.
	 */
	public void configureJobEnvironment (final int            nJobId,
										 final JobEnvironment aEnv  )
	    throws Exception;

	//-------------------------------------------------------------------------
	public void configureDefaultJobEnvironment (final JobEnvironment aEnv)
		throws Exception;

	//-------------------------------------------------------------------------
	/** 'lease' a job from the pool ...
	 * 
	 *  Depending from {@link JobPoolConfig.PROP_BLOCK_ON_LEASE} this method will :<br/>
	 *  <ul>
	 *  	<li>block - if no free jobs exists inside the pool</li>
	 *  	<li>throw an exception - if no free jobs exists inside the pool</li>
	 *  </ul>
	 * 
	 *  <ul>
	 *      <li>For the time of the lease you own that job and can use it as you like.</li>
	 *      <li>But you can call it's execute method one times only !
	 *          <br/>(see {@link IJob})</li>
	 *      <li>You have to ensure to put back the job into the pool by calling freeJob()
	 *          in pair to leaseJob ().</li>
	 *      <li>On the other side some background processes (try) to ensure it's deleted
	 *          automatically ... if needed.</li>
	 *  </ul>
	 * 
	 *  @return the job you can use from now on.
	 */
	public IJob leaseJob ()
	    throws Exception;

	public IJob leaseStickyJob (final Integer nJobId)
		throws Exception;

	//-------------------------------------------------------------------------
	/** mark a (leased) job as 'free for others' ...
	 *  as already mentioned in {@link leaseJob} this has to be called in pair.
	 * 
	 *  @param	iJob [IN]
	 *  		the job you used but want to put back into the pool.
	 */
	public void freeJob (final IJob iJob)
	    throws Exception;

	//-------------------------------------------------------------------------
	/**
	 */
	public void restartJob (final Integer nJobId)
	    throws Exception;
}
