package com.openexchange.office.tools;

import com.openexchange.office.tools.message.MessageHelper;
import com.openexchange.realtime.packet.ID;
import com.openexchange.tools.session.ServerSession;

/**
 * Helper class to extract or get information from a real-time stanza ID.
 *
 * {@link IDUtils}
 *
 * @author <a href="mailto:carsten.driesner@open-xchange.com">Carsten Driesner</a>
 * @since v7.6.0
 */
public class IDUtils {

    /**
     * Tries to extract the file id from a real-time packet ID.
     *
     * @param id
     *  A real-time packet ID to be checked for the file id.
     *
     * @return
     *  The file id or null if the file id couldn't be extracted.
     */
	public static String extractFileId(ID id) {
    	String resource = extractResourceString(id);

        if (null != resource) {
	        final int fileIdDelimiter = resource.indexOf('.');
	        if ((-1 < fileIdDelimiter) && (fileIdDelimiter < (resource.length() - 1))) {
                return resource.substring(fileIdDelimiter + 1);
            }
        }

        return null;
    }

    /**
     * Tries to extract the folder id from a real-time packet ID.
     *
     * @param id
     *  A real-time packet ID to be checked for the folder id.
     *
     * @return
     *  The folder id or null if the file id couldn't be extracted.
     */
    public static String extractFolderId(ID id) {
    	String resource = extractResourceString(id);

        if (null != resource) {
	        final int fileIdDelimiter = resource.indexOf('.');
	        if ((-1 < fileIdDelimiter) && (fileIdDelimiter < (resource.length() - 1))) {
	            return resource.substring(0, fileIdDelimiter);
	        }
        }
        return null;
    }

    /**
     * Tries to extract the folder & file id from a real-time packet ID.
     *
     * @param id
     *  A real-time packet ID to be checked for the folder id.
     *
     * @return
     *  <folder id>.<file id> as string or null if this information couldn't
     *  be extracted from the provided ID.
     */
    public static String extractFolderAndFileId(ID id) {
    	return extractResourceString(id);
    }

    /**
     * Tries to get the user id from a real-time id string.
     *
     * @param id
     *  The real-time ID as string where we want to retrieve the user id.
     *
     * @return
     *  The user id or -1 if the id could not be determined.
     *
     * @throws Exception
     */
    public static int getUserIdFromRealTimeId(String id) throws Exception {
        return getUserIdFromRealTimeId(new ID(id));
    }

    /**
     * Tries to get the user id from a real-time id.
     *
     * @param id
     *  The real-time ID where we want to retrieve the user id.
     *
     * @return
     *  The user id or -1 if the id could not be determined.
     *
     * @throws Exception
     */
    public static int getUserIdFromRealTimeId(ID id) throws Exception {
        int result = -1;

        // try to get session from ID
        if (null != id) {
            result = SessionUtils.getUserId(id.toSession());
        }

        return result;
    }

    /**
     * Tries to extract the resource data from a real-time packet ID.
     *
     * @param id
     *  A real-time packet ID to be checked for the resource data.
     *
     * @return
     *  <folder id>.<file id> as string or null if this information couldn't
     *  be extracted from the provided ID.
     */
    private static String extractResourceString(ID id) {
        // resource layout is: folderId.fileId
        final String resource = id.getResource();

        if (null != resource) {
            final int fileIdDelimiter = resource.indexOf('.');
            if ((-1 < fileIdDelimiter) && (fileIdDelimiter < (resource.length() - 1))) {
            	return resource;
            }
        }

        return null;
    }

}
