/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.tools;

import org.json.JSONException;
import org.json.JSONObject;

/**
 * An enumeration to provide specific error information from the server to
 * the client side. The enumeration provides method to convert to JSON object
 * or to be created from a JSON object.
 *
 * {@link ErrorCode}
 *
 * @author <a href="mailto:carsten.driesner@open-xchange.com">Carsten Driesner</a>
 */
public enum ErrorCode {

	NO_ERROR(0, "NO_ERROR", "No error"),
	GENERAL_UNKNOWN_ERROR(1, "GENERAL_UNKNOWN_ERROR", "An unknown error has occurred."),
    GENERAL_ARGUMENTS_ERROR(10, "GENERAL_ARGUMENTS_ERROR", "The request have been called with missing or wrong arguments"),
    GENERAL_FILE_NOT_FOUND_ERROR(11, "GENERAL_FILE_NOT_FOUND_ERROR", "The document file couldn't be found"),
    GENERAL_QUOTA_REACHED_ERROR(12, "GENERAL_QUOTA_REACHED_ERROR", "The quota for the user has been reached. Storing additonal data is not possible."),
    GENERAL_PERMISSION_CREATE_MISSING_ERROR(13, "GENERAL_PERMISSION_CREATE_MISSING_ERROR", "The permission to create a file is not available."),
    GENERAL_PERMISSION_READ_MISSING_ERROR(14, "GENERAL_PERMISSION_READ_MISSING_ERROR", "The permission to read a file is not available."),
    GENERAL_PERMISSION_WRITE_MISSING_ERROR(15, "GENERAL_PERMISSION_WRITE_MISSING_ERROR", "The permission to write a file is not available."),
    GENERAL_MEMORY_TOO_LOW_ERROR(16, "GENERAL_MEMORY_TOO_LOW_ERROR", "The memory of the system is not sufficient to process the request."),
    CREATEDOCUMENT_CONVERSION_FAILED_ERROR(200, "CREATEDOCUMENT_CONVERSION_FAILED_ERROR", "The document couldn't be created because of an conversion error!"),
    CREATEDOCUMENT_CANNOT_READ_TEMPLATEFILE_ERROR(201, "CREATEDOCUMENT_CANNOT_READ_TEMPLATEFILE_ERROR", "The document couldn't be created because the template file couldn't be read!"),
    CREATEDOCUMENT_CANNOT_READ_DEFAULTTEMPLATEFILE_ERROR(202, "CREATEDOCUMENT_CANNOT_READ_DEFAULTTEMPLATEFILE_ERROR", "The document coulnd't be created because the default template file couldn't be read!"),
    RENAMEDOCUMENT_FAILED_ERROR(300, "RENAMEDOCUMENT_FAILED_ERROR", "Renaming the document failed!"),
    RENAMEDOCUMENT_VALIDATION_FAILED_CHARACTERS_ERROR(301, "RENAMEDOCUMENT_VALIDATION_FAILED_CHARACTERS_ERROR", "The document couldn't be renamed as the validation of the new file name failed. File name contains invalid characters!"),
    LOADDOCUMENT_FAILED_ERROR(400, "LOADDOCUMENT_FAILED_ERROR", "Loading the document failed!"),
    LOADDOCUMENT_CANNOT_RETRIEVE_OPERATIONS_ERROR(401, "LOADDOCUMENT_CANNOT_RETRIEVE_OPERATIONS_ERROR", "Loading the document failed because operations couldn't be generated!"),
    LOADDOCUMENT_CANNOT_READ_PASSWORD_PROTECTED_ERROR(402, "LOADDOCUMENT_CANNOT_READ_PASSWORD_PROTECTED_ERROR", "Loading the document failed because it's password protected!"),
    LOADDOCUMENT_COMPLEXITY_TOO_HIGH_ERROR(403, "LOADDOCUMENT_COMPLEXITY_TOO_HIGH_ERROR", "Loading the document failed because the complexity of the content is too high!"),
    LOADDOCUMENT_CALCENGINE_NOT_WORKING_ERROR(405, "LOADDOCUMENT_CALCENGINE_NOT_WORKING_ERROR", "Loading the document failed because the server-side engine used for calculating formulas is not available."),
    HANGUP_INVALID_OPERATIONS_SENT_ERROR(500, "HANGUP_INVALID_OPERATIONS_SENT_ERROR", "Client sent invalid operations and therefore lost edit rights!"),
    HANGUP_NO_EDIT_RIGHTS_ERROR(501, "HANGUP_NO_EDIT_RIGHTS_ERROR", "Client sent operations without having edit rights. Operations are ignored and client must switch to read-only mode!"),
    HANGUP_INVALID_OSN_DETECTED_ERROR(502, "HANGUP_INVALID_OSN_DETECTED_ERROR", "Client has invalid OSN after losing edit rights. Must switch to read-only mode."),
    HANGUP_CALCENGINE_NOT_RESPONDING_ERROR(503, "HANGUP_CALCENGINE_NOT_RESPONDING_ERROR", "Calcengine is not responding. Clients must be set to read-only to resync again."),
    SAVEDOCUMENT_FAILED_ERROR(601, "SAVEDOCUMENT_FAILED_ERROR", "Saving document failed"),
    SAVEDOCUMENT_FAILED_NOBACKUP_ERROR(602, "SAVEDOCUMENT_FAILED_NOBACKUP_ERROR", "Saving document failed - no backup file could be written."),
    SAVEDOCUMENT_FAILED_OPERATIONS_ERROR(603, "SAVEDOCUMENT_FAILED_OPERATIONS_ERROR", "Saving document failed - the filter was not able to apply the operations to the document."),
    SAVEDOCUMENT_FAILED_NO_FILTER_ERROR(605, "SAVEDOCUMENT_FAILED_NO_FILTER_ERROR", "Saving document failed - document export filter not available."),
    SAVEDOCUMENT_FAILED_NO_RESOURCEMANAGER_ERROR(606, "SAVEDOCUMENT_FAILED_NO_RESOURCEMANAGER_ERROR", "Saving document failed - no resource manager available for filter."),
    SAVEDOCUMENT_FAILED_NO_DOCUMENTSTREAM_ERROR(607, "SAVEDOCUMENT_FAILED_NO_DOCUMENTSTREAM_ERROR", "Saving document failed - no document stream available."),
    SAVEDOCUMENT_FAILED_FILTER_OPERATION_ERROR(608, "SAVEDOCUMENT_FAILED_FILTER_OPERATION_ERROR", "Saving document failed - The document is inconsistent and is saved as _ox file!"),
    COPYDOCUMENT_FAILED_ERROR(700, "COPYDOCUMENT_FAILED_ERROR", "Copying the document failed due to unknown reasons."),
    COPYDOCUMENT_USERFOLDER_UNKOWN_ERROR(701, "COPYDOCUMENT_USERFOLDER_UNKOWN_ERROR", "Copying the document failed, because the user's folder is unknown."),
    COPYDOCUMENT_FILENAME_UNKNOWN_ERROR(702, "COPYDOCUMENT_FILENAME_UNKNOWN_ERROR", "Copying the document failed, because the file name is unknown"),
    COPYDOCUMENT_TEMPLATE_FORMAT_UNKOWN_ERROR(703, "COPYDOCUMENT_TEMPLATE_FORMAT_UNKOWN_ERROR", "Copying the document failed, because the template format couldn't be determined."),
    GETTEMPLATELIST_UNKNOWN_ERROR(800, "GETTEMPLATELIST_UNKNOWN_ERROR", "Retrieving the template list failed."),
    GETTEMPLATELIST_USERFOLDER_UNKNOWN_ERROR(801, "GETTEMPLATELIST_USERFOLDER_UNKNOWN_ERROR", "The user folder is unknown, therefore the user template list cannot be retrieved."),
    GETTEMPLATELIST_GLOBALFOLDER_IO_ERROR(802, "GETTEMPLATELIST_GLOBALFOLDER_IO_ERROR", "There was an IO exception accessing the local file system."),
    LOCALFILE_FILE_NOT_FOUND_ERROR(900, "LOCALFILE_FILE_NOT_FOUND_ERROR", "The local file cannot be found."),
    LOCALFILE_CANNOT_READ_FILE_ERROR(901, "LOCALFILE_CANNOT_READ_FILE_ERROR", "The local file cannot be read."),
    LOCALFILE_INVALID_ID_ERROR(902, "LOCALFILE_INVALID_ID_ERROR", "The file id provided is invalid."),
    GETUSERINFO_UNKNOWN_ERROR(1000, "GETUSERINFO_UNKNOWN_ERROR", "Retrieving the user info failed.");

    public final static int CODE_NO_ERROR        = 0;
    public final static String CONSTANT_NO_ERROR = "NO_ERROR";

    public final static int ERRORCLASS_NO_ERROR    = 0;
    public final static int ERRORCLASS_WARNING     = 1;
    public final static int ERRORCLASS_ERROR       = 2;
    public final static int ERRORCLASS_FATAL_ERROR = 3;
    // must be adapted
    private final static int ERRORCLASS_COUNT      = 4;

    private final int m_code;
    private final String m_codeAsStringConstant;
    private final String m_description;
    // can be changed by client code to change severity dependent on the context
    private int m_errorClass;

    /**
     * Creates a ErrorCode instance with the code, a unique constant string for the
     * code and a description.
     *
     * @param code
     *  The error code.
     *
     * @param codeAsString
     *  The code as a unique constant string.
     *
     * @param description
     *  The description of the error.
     */
    private ErrorCode(int code, final String codeAsString, final String description) {
        this.m_code = code;
        this.m_codeAsStringConstant = codeAsString;
        this.m_description = description;
        if (code == CODE_NO_ERROR) {
            this.m_errorClass = ERRORCLASS_NO_ERROR;
        } else {
            this.m_errorClass = ERRORCLASS_ERROR;
        }
    }

    /**
     * Determines if this instance represents an error or not.
     *
     * @return
     *  TRUE if the instance represents an error and FALSE if not.
     */
    public boolean isError() {
    	return m_code != CODE_NO_ERROR;
    }

    /**
     * Provides a description of the error.
     *
     * @return
     *  The description.
     */
    public String getDescription() {
       return m_description;
    }

    /**
     * Provides a unique number which describes the error case.
     *
     * @return
     *  The error code.
     */
    public int getCode() {
       return m_code;
    }

    /**
     * Provides a unique constant string for the error case.
     * @return
     */
    public String getCodeAsStringConstant() {
        return m_codeAsStringConstant;
    }

    /**
     * Provides the error class of the error.
     *
     * @return
     *  The error class of the error, which describes how severe the
     *  error condition must be treated by the code.
     */
    public int getErrorClass() {
        return m_errorClass;
    }

    /**
     * Sets the error class of the error. Sometimes an error has a
     * different impact and therefore the error class must be adapted.
     * Use this method with care as it can have a deep impact on the
     * error processing.
     *
     * @param errorClass
     */
    public void setErrorClass(int errorClass) {
        if ((errorClass >= 0) && (errorClass < ERRORCLASS_COUNT)) {
            m_errorClass = errorClass;
        }
    }

    /**
     * Provides a string representation of the error code instance.
     *
     * @return
     *  The string representation of the error code instance.
     */
    @Override
    public String toString() {
      return m_code + ": " + m_description;
    }

    /**
     * Provides a JSONObject which contains the state of the error
     * code instance.
     *
     * @return
     *  A JSONObject which contains the state of the error code
     *  instance or an empty object in case of an error.
     */
    public JSONObject getAsJSON() {
        JSONObject jsonObject = new JSONObject();
        try {
            jsonObject.put("errorClass", this.getErrorClass());
            jsonObject.put("errorCode", this.getCode());
            jsonObject.put("error", this.getCodeAsStringConstant());
            jsonObject.put("errorDescription", this.getDescription());
        } catch (final JSONException e) {
            //
        }
        return jsonObject;
    }

    /**
     * Creates a JSONObject that can be directly used as a result object
     * for server answers.
     *
     * @return
     *  A JSONObject which can be used as a server answer or an empty
     *  object in case of an error.
     */
    public JSONObject getAsJSONResultObject() {
    	JSONObject resultObject = new JSONObject();
    	JSONObject errorObject = this.getAsJSON();
    	try {
    		resultObject.put("error", errorObject);
    	} catch (final JSONException e) {
    		//
    	}
    	return resultObject;
    }

    /**
     * Tries to extract the error code from a result JSON object.
     *
     * @param jsonObject
     *  A JSON object that contains the result to be sent to a client, where
     *  the error code should be extracted.
     *
     * @return
     *  The error code extracted from the JSON object. Could be ErrorCode.NO_ERROR
     *  if no error could be extracted.
     */
    static public ErrorCode getErrorCodeFromJSON(JSONObject jsonObject) {
    	if (null != jsonObject) {
    	    JSONObject jsonError = jsonObject.optJSONObject("error");
    	    if (null != jsonError) {
    	    	final String errCodeConstant = getErrorConstantFromJSON(jsonError, ErrorCode.CONSTANT_NO_ERROR);
    	    	ErrorCode errorCode = ErrorCode.valueOf(errCodeConstant);
    	    	return (null != errorCode) ? errorCode : ErrorCode.NO_ERROR;
    	    }
    	}
    	return ErrorCode.NO_ERROR;
    }

    /**
     * Extracts the error code from a JSON object that is the full representation
     * of a ErrorCode instance (created by getAsJSON()).
     *
     * @param jsonObject
     *  The JSON object where the error code shall be extracted.
     *
     * @param defaultValue
     *  The default value to be used if the error code cannot be extracted.
     *
     * @return
     *  The error code or the default value if the code couldn't be extracted
     *  from the JSONObject.
     */
    static public int getErrorCodeFromJSON(JSONObject jsonObject, int defaultValue) {
        int result = defaultValue;
        if (null != jsonObject) {
            result = jsonObject.optInt("errorCode", defaultValue);
        }
        return result;
    }

    /**
     * Extracts the error constant string from a JSON object that contains the
     * full representation of a ErrorCode instance (created by getAsJSON()).
     *
     * @param jsonObject
     *  The JSON object where the error constant string shall be extracted.
     *
     * @param defaultValue
     *  The default value to be used if the error code cannot be extracted.
     *
     * @return
     *  The error constant string or the default value if the code couldn't
     *  be extracted from the JSONObject.
     */
    static public String getErrorConstantFromJSON(JSONObject jsonObject, String defaultValue) {
        String result = defaultValue;
        if (null != jsonObject) {
            result = jsonObject.optString("error", defaultValue);
        }
        return result;
    }

    /**
     * Extracts the error description from a JSON object that contains the
     * full representation of a ErrorCode instance (created by getAsJSON()).
     *
     * @param jsonObject
     *  The JSON object where the error description string shall be extracted.
     *
     * @param defaultValue
     *  The default value to be used if the error description cannot be extracted.
     *
     * @return
     *  The error description or the default value if the string couldn't be
     *  extracted from the JSONObject.
     */
    static public String getErrorDescriptionFromJSON(JSONObject jsonObject, String defaultValue) {
        String result = defaultValue;
        if (null != jsonObject) {
            result = jsonObject.optString("errorDescription", defaultValue);
        }
        return result;
    }
 }
