/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.realtime.impl.calc;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.atomic.AtomicBoolean;

import com.openexchange.config.ConfigurationService;
import com.openexchange.office.realtime.impl.Connection;
import com.openexchange.server.ServiceLookup;
import com.openexchange.threadpool.ThreadPoolService;

public class AsyncLoadRequestQueue implements Runnable {

    static protected final org.apache.commons.logging.Log LOG = com.openexchange.log.LogFactory.getLog(AsyncLoadRequestQueue.class);

	private ServiceLookup services = null;
    private Queue<Request> requestQueue = new LinkedList<Request>();
    private HashMap<String, Request> runningRequests = new HashMap<String, Request>();
    private AtomicBoolean bTerminate = new AtomicBoolean(false);
    private Boolean queueNotEmpty = false;
    private RequestExecutor requestExecutor = null;
    private AtomicBoolean bStarted = new AtomicBoolean(false);
    private Thread queueThread = null;

    public AsyncLoadRequestQueue(ServiceLookup services) {
		this.services = services;
		requestExecutor = new RequestExecutor(services);
	}

    /**
     * Enables the queue to process requests asynchronously via background
     * threads. This method does nothing if call more than once. A queue
     * destroyed via destroyQueue() cannot be restored using startQueue().
     */
    public synchronized void startQueue() {
    	boolean started = bStarted.getAndSet(true);
    	if (!started) {
    		requestExecutor = new RequestExecutor(services);
    		requestExecutor.start();
    		queueThread = new Thread(null, this, "CalcLoadRequestQueueRunner");
    		queueThread.start();
    	}
    }

	/**
	 * Destroys the queue and synchronize the shutdown of the queue thread and
	 * the Executor used to parallelize the Spreadsheet load requests.
	 */
    public void destroyQueue() {
	    boolean terminated = bTerminate.getAndSet(true);
	    boolean started = bStarted.get();
		
		try {
			if (started && !terminated) {
				synchronized(requestQueue) {
			        requestQueue.clear();
				}
			    requestExecutor.shutdown();
			    synchronized(queueNotEmpty) {
		            queueNotEmpty.notify();
			    }
			    queueThread.join(100);
			}
		} catch (InterruptedException e) {
            LOG.debug("InterruptedException preparing shutdown of the CalcLoadRequestQueue", e);
		} 
	}

	/**
	 * Put a new load/sync request into the request queue.
	 *
	 * @param request
	 *  A load or sync request to be processed in one of the background
	 *  threads.
	 */
    public void putRequest(final Request request) throws IllegalStateException {
    	
		if (bStarted.get() && !bTerminate.get()) {
			synchronized(requestQueue) {
			    requestQueue.add(request);
			}
			synchronized(queueNotEmpty) {
			    queueNotEmpty.notify();
			}
            LOG.info("RT connection: new async request for document: " + request.getDocumentId() + ", request: " + request.getType().toString());
		} else {
			throw new IllegalStateException("Cannot take over request, if queue was not started yet or has been terminated!");
		}
    }

    /**
     * Removes pending requests for a specific document from the request queue.
     *
     * @param documentId
     *  The document id for which the remaining requests should be removed
     *  from the request queue.
     */
    public void purgeDocumentRequests(final String documentId) {
    	synchronized(requestQueue){
    		ArrayList<Request> requestsToBeRemoved = new ArrayList<Request>();
    		Iterator<Request> iter = requestQueue.iterator();
    	    while (iter.hasNext()) {
    	    	Request request = iter.next();
                if (request.getDocumentId().equals(documentId)) {
    	    		requestsToBeRemoved.add(request);
    	    	}
    	    }
  	    	requestQueue.removeAll(requestsToBeRemoved);
    	}
    }
    
	/**
	 * Central run method to be executed by a special thread to
	 * process the Requests put into the CalcLoadRequestQueue.
	 * This method ensures that only one request for every
	 * document instance is processed by the RequestExecutor.
	 * Otherwise it can happen that additional clients requests,
	 * to load the same document, can overtake the first request.
	 * This must never happen!
	 */
	@Override
	public void run() {
		while (!bTerminate.get()) {
			Request requestToBeProcessed = null;

			try {
				synchronized(requestQueue) {
					if (!requestQueue.isEmpty()) {
						Request tmp = null;
						Iterator<Request> iterator = requestQueue.iterator();
						while (iterator.hasNext()) {
							tmp = iterator.next();
							final String requestDocId = tmp.getDocumentId();
							if (!isRequestForDocumentRunning(requestDocId)) {
								requestToBeProcessed = tmp;
								iterator.remove();
								break;
							}
						}
					}
				}
				
				if (null == requestToBeProcessed) {
					synchronized(queueNotEmpty) {
					    queueNotEmpty.wait(100);
					}
				} else {
					final Request processRequest = requestToBeProcessed;

					// set new request as running
					runningRequests.put(requestToBeProcessed.getDocumentId(), requestToBeProcessed);
					requestExecutor.execute(new Runnable() {
						public void run() {
							try {
								IRequestProcessor processor = processRequest.getProcessor();
								if (null != processor) {
								    processor.processRequest(processRequest);
								}
							} catch (Throwable e) {
					            LOG.error("Exception while processing spreadsheet document load request", e);
							}
						}
					});
				}
			} catch (Exception e) {
	            LOG.error("Exception while processing spreadsheet document load request", e);
	 		} catch (Throwable t) {
				LOG.error("Throwable while processing spreadsheet document load request", t);
			} finally {
				removeFinishedRequests();
			}
		}
	}

	/**
	 * Removes finished requests from the request queue. This method is NOT
	 * thread-safe and must only be called from the run() method!
	 */
	private void removeFinishedRequests() {
		ArrayList<String> requestsToBeRemoved = new ArrayList<String>();
		Iterator<Request> iter = runningRequests.values().iterator();
	    while (iter.hasNext()) {
	    	Request request = iter.next();
	    	if (request.isFinished()) {
	    		requestsToBeRemoved.add(request.getDocumentId());
	    	}
	    }
	    for (String id : requestsToBeRemoved) {
	    	runningRequests.remove(id);
	    }
	}
	
	/**
	 * Determines if a request for a certain document is currently running or
	 * not. This method is NOT thread-safe and must only be called from the run()
	 * method!
	 *
	 * @param docId
	 *  The unique document identifier, consists of the <folderid>.<fileid>
	 *
	 * @return
	 *  TRUE if there is a request running, otherwise FALSE.
	 */
	private boolean isRequestForDocumentRunning(String docId) {
		boolean result = false;

		Request request = runningRequests.get(docId);
		if ((null != request) && (!request.isFinished())) {
			result = true;
	    }

		return result;
	}
}
