/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.drawingml;

import java.util.List;
import java.util.logging.Logger;

import org.docx4j.dml.CTBlip;
import org.docx4j.dml.CTBlipFillProperties;
import org.docx4j.dml.CTHslColor;
import org.docx4j.dml.CTLineProperties;
import org.docx4j.dml.CTNoFillProperties;
import org.docx4j.dml.CTNonVisualDrawingProps;
import org.docx4j.dml.CTPoint2D;
import org.docx4j.dml.CTPositiveSize2D;
import org.docx4j.dml.CTPresetColor;
import org.docx4j.dml.CTPresetGeometry2D;
import org.docx4j.dml.CTRegularTextRun;
import org.docx4j.dml.CTRelativeRect;
import org.docx4j.dml.CTSRgbColor;
import org.docx4j.dml.CTScRgbColor;
import org.docx4j.dml.CTSchemeColor;
import org.docx4j.dml.CTShapeProperties;
import org.docx4j.dml.CTSolidColorFillProperties;
import org.docx4j.dml.CTSystemColor;
import org.docx4j.dml.CTTextBody;
import org.docx4j.dml.CTTextBodyProperties;
import org.docx4j.dml.CTTextCharacterProperties;
import org.docx4j.dml.CTTextField;
import org.docx4j.dml.CTTextLineBreak;
import org.docx4j.dml.CTTextListStyle;
import org.docx4j.dml.CTTextParagraph;
import org.docx4j.dml.CTTextParagraphProperties;
import org.docx4j.dml.CTTransform2D;
import org.docx4j.dml.STSchemeColorVal;
import org.docx4j.dml.STShapeType;
import org.docx4j.dml.STTextAnchoringType;
import org.docx4j.dml.STTextStrikeType;
import org.docx4j.dml.STTextUnderlineType;
import org.docx4j.dml.STTextVertOverflowType;
import org.docx4j.dml.STTextVerticalType;
import org.docx4j.dml.STTextWrappingType;
import org.docx4j.dml.TextFont;
import org.docx4j.dml.chart.CTBoolean;
import org.docx4j.dml.chart.CTCatAx;
import org.docx4j.dml.chart.CTChartLines;
import org.docx4j.dml.chart.CTLayout;
import org.docx4j.dml.chart.CTNumFmt;
import org.docx4j.dml.chart.CTPlotArea;
import org.docx4j.dml.chart.CTScaling;
import org.docx4j.dml.chart.CTTickLblPos;
import org.docx4j.dml.chart.CTTickMark;
import org.docx4j.dml.chart.CTTitle;
import org.docx4j.dml.chart.CTTx;
import org.docx4j.dml.chart.CTValAx;
import org.docx4j.dml.chart.IAxisDescription;
import org.docx4j.dml.chart.IListSer;
import org.docx4j.dml.chart.STTickLblPos;
import org.docx4j.dml.chart.STTickMark;
import org.docx4j.dml.chart.chartStyle.CTChartColor;
import org.docx4j.jaxb.Context;
import org.docx4j.openpackaging.parts.Part;
import org.docx4j.relationships.Relationship;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import com.openexchange.office.ooxml.OperationDocument;
import com.openexchange.office.ooxml.tools.Commons;
import com.openexchange.office.ooxml.tools.json.JSONHelper;

final public class DMLHelper {

	private static final CTTextCharacterProperties GERMAN = new CTTextCharacterProperties();
	static {
		GERMAN.setLang("de-DE");
	}

    public static void createJsonFromPositiveSize2D(JSONObject drawingProperties, CTPositiveSize2D positiveSize2D)
        throws JSONException {

        if(positiveSize2D!=null) {
            drawingProperties.put("width", Commons.coordinateTo100TH_MM(positiveSize2D.getCx()));
            drawingProperties.put("height", Commons.coordinateTo100TH_MM(positiveSize2D.getCy()));
        }
    }

    public static void createJsonFromBlipFillProperties(JSONObject drawingProperties, CTBlipFillProperties blipFillProperties)
        throws JSONException {

        if(blipFillProperties!=null) {
            final CTRelativeRect srcRect = blipFillProperties.getSrcRect();
            if (srcRect!=null) {
                final int l = srcRect.getL() / 1000;
                final int t = srcRect.getT() / 1000;
                final int r = srcRect.getR() / 1000;
                final int b = srcRect.getB() / 1000;

                // add possible crop properties
                if (l != 0)
                    drawingProperties.put("cropLeft", l);
                if (t != 0)
                    drawingProperties.put("cropTop", t);
                if (r != 0)
                    drawingProperties.put("cropRight", r);
                if (b != 0)
                    drawingProperties.put("cropBottom", b);
            }
        }
    }

    public static void createJsonFromShapeProperties(JSONObject drawingProperties, CTShapeProperties shapeProperties)
        throws JSONException {

        if(shapeProperties!=null) {
            final CTTransform2D xFrm = shapeProperties.getXfrm();
            if(xFrm!=null) {
                final CTPoint2D point2D = xFrm.getOff();
                if(point2D!=null) {
                    if(point2D.getX()!=0) {
                        if(!drawingProperties.hasAndNotNull("left")) {
                            drawingProperties.put("left", Commons.coordinateTo100TH_MM(point2D.getX()));
                        }
                    }
                    if(point2D.getY()!=0) {
                        if(!drawingProperties.hasAndNotNull("top")) {
                            drawingProperties.put("top", Commons.coordinateTo100TH_MM(point2D.getX()));
                        }
                    }
                }
                final CTPositiveSize2D positiveSize2D = xFrm.getExt();
                if(positiveSize2D!=null) {
                    if(!drawingProperties.hasAndNotNull("width")) {
                        drawingProperties.put("width", Commons.coordinateTo100TH_MM(positiveSize2D.getCx()));
                    }
                    if(!drawingProperties.hasAndNotNull("height")) {
                        drawingProperties.put("height", Commons.coordinateTo100TH_MM(positiveSize2D.getCy()));
                    }
                }
                if(xFrm.isFlipH()) {
                    if(!drawingProperties.hasAndNotNull("flipH")) {
                        drawingProperties.put("flipH", true );
                    }
                }
                if(xFrm.isFlipV()) {
                    if(!drawingProperties.hasAndNotNull("flipV")) {
                        drawingProperties.put("flipV", true );
                    }
                }
            }
            final CTLineProperties lineProperties = shapeProperties.getLn();
            if(lineProperties!=null) {
                final JSONObject jsonLineProperties = createJsonFromLineProperties(lineProperties);
                if(!jsonLineProperties.isEmpty()) {
                    drawingProperties.put("line", jsonLineProperties);
                }
            }
        }
    }

    public static void createJsonFromChartShapeProperties(JSONObject dest, CTShapeProperties shapePr)
        throws JSONException {

        if (shapePr!= null) {
            final CTLineProperties lineProperties = shapePr.getLn();
            if(lineProperties!=null) {
                final JSONObject jsonLineProperties = DMLHelper.createJsonFromLineProperties(lineProperties);
                if(jsonLineProperties!=null&&!jsonLineProperties.isEmpty()) {
                    dest.put("line", jsonLineProperties);
                }
            }
            if(!dest.has("fill")){
	            JSONObject jsonFillProperties = null;
	            if(shapePr.getSolidFill()!=null) {
	                jsonFillProperties = DMLHelper.createJsonFromSolidColorFillProperties(shapePr.getSolidFill());
	            }
	            else if(shapePr.getNoFill()!=null) {
	                jsonFillProperties = DMLHelper.createJsonFromNoFillProperties(shapePr.getNoFill());
	            }
	            if(jsonFillProperties!=null&&!jsonFillProperties.isEmpty()) {
	                dest.put("fill", jsonFillProperties);
	            }
            }
        }
        
        if(!dest.has("fill")){
        	final JSONObject jsonFillProperties = new JSONObject();
			jsonFillProperties.put("type", "solid");
			dest.put("fill", jsonFillProperties);
        }
        
        if(!dest.has("line")){
        	final JSONObject jsonFillProperties = new JSONObject();
			jsonFillProperties.put("type", "solid");
			dest.put("line", jsonFillProperties);
        }
    }
    
	public static void createChartShapePropertiesFromJson(CTShapeProperties docx, JSONObject data) throws JSONException {
		final JSONObject line = data.optJSONObject("line");
		if (line != null) {
			CTLineProperties ln = docx.getLn();
			if(ln == null){
				ln = new CTLineProperties();
				docx.setLn(ln);
			}
			createLinePropertiesFromJson(ln, line);
		}

		docx.setNoFill(null);
		docx.setSolidFill(null);
		docx.setGradFill(null);
		docx.setPattFill(null);
		docx.setBlipFill(null);
		docx.setGrpFill(null);

		final Object fillProp = createFillPropsFromJson(data.optJSONObject("fill"));
		if (fillProp instanceof CTNoFillProperties) {
			docx.setNoFill((CTNoFillProperties) fillProp);
		} else if (fillProp instanceof CTSolidColorFillProperties) {
			docx.setSolidFill((CTSolidColorFillProperties) fillProp);
		}
	}

    public static JSONObject createJsonFromLineProperties(CTLineProperties lineProperties)
        throws JSONException {

    	JSONObject jsonLineProperties = null;
        if(lineProperties!=null) {
            if(lineProperties.getNoFill()!=null) {
                jsonLineProperties = createJsonFromNoFillProperties(lineProperties.getNoFill());
            }
            else if(lineProperties.getSolidFill()!=null) {
                jsonLineProperties = createJsonFromSolidColorFillProperties(lineProperties.getSolidFill());
            }
        }
        if(jsonLineProperties==null) {
            jsonLineProperties = new JSONObject();
            jsonLineProperties.put("type", "solid");
        }
        if (lineProperties.getW() != null) {
            jsonLineProperties.put("width", lineWidthToPt(lineProperties.getW()));
        }
        return jsonLineProperties;
    }
    
	public static void createLinePropertiesFromJson(CTLineProperties docx, JSONObject json) throws JSONException {
		docx.setNoFill(null);
		docx.setSolidFill(null);
		docx.setGradFill(null);
		docx.setPattFill(null);

		final Object fillProp = createFillPropsFromJson(json);
		if (fillProp instanceof CTNoFillProperties) {
			docx.setNoFill((CTNoFillProperties) fillProp);
		} else if (fillProp instanceof CTSolidColorFillProperties) {
			docx.setSolidFill((CTSolidColorFillProperties) fillProp);
		}
		if (json.has("width")) {
			docx.setW(ptToLineWidth(json.getLong("width")));
		}
	}

	public static JSONObject createJsonFromSolidColorFillProperties(CTSolidColorFillProperties fillProperties)
        throws JSONException {

        if(fillProperties!=null) {
            final JSONObject jsonObject = createJsonColorFromSolidColorFillProperties(fillProperties);
            if(jsonObject!=null&&!jsonObject.isEmpty()) {
                jsonObject.put("type", "solid");
                return jsonObject;
            }
        }
        return null;
    }
    
    public static JSONObject createJsonColorFromSolidColorFillProperties(CTSolidColorFillProperties fillProperties)
        throws JSONException {

        if(fillProperties!=null) {
            JSONObject jsonColorProperties = null;
            if (fillProperties.getScrgbClr()!= null) {
                jsonColorProperties = createJsonFromScrgbClr(fillProperties.getScrgbClr());
            }
            else if(fillProperties.getSrgbClr()!= null) {
                jsonColorProperties = createJsonFromSrgbClr(fillProperties.getSrgbClr());
            }
            else if (fillProperties.getHslClr()!= null) {
                jsonColorProperties = createJsonFromHslClr(fillProperties.getHslClr());
            }
            else if (fillProperties.getSchemeClr()!= null) {
                jsonColorProperties = createJsonFromSchemeClr(fillProperties.getSchemeClr());
            }
            else if (fillProperties.getSysClr()!= null) {
                jsonColorProperties = createJsonFromSysClr(fillProperties.getSysClr());
            }
            else if (fillProperties.getPrstClr()!= null) {
                jsonColorProperties = createJsonFromPrstClr(fillProperties.getPrstClr());
            }
            if(jsonColorProperties!=null&&!jsonColorProperties.isEmpty()) {
                return jsonColorProperties;
            }
        }
        return null;
    }

	public static boolean createSolidColorFillPropertiesFromJson(CTSolidColorFillProperties docx, JSONObject json) throws JSONException {
		docx.setSysClr(null);
		docx.setScrgbClr(null);
		docx.setSrgbClr(null);
		docx.setHslClr(null);
		docx.setPrstClr(null);
		docx.setSchemeClr(null);
		

		final String type = json.getString("type");

		if (type.equals("auto")) {
			return false;
		}else if (type.equals("scheme")) {
			final CTSchemeColor schemeClr = new CTSchemeColor();
			createSchemeColorFromJson(schemeClr, json);
			docx.setSchemeClr(schemeClr);
		} else if (type.equals("rgb")) {
			final CTSRgbColor srgbClr = new CTSRgbColor();
			createSRgbColorFromJson(srgbClr, json);
			docx.setSrgbClr(srgbClr);
		} else {
			Logger.getAnonymousLogger().warning("colortype is not implemented " + type);
			// TODO: other colors?
		}
		return true;
	}

	public static Object createFillPropsFromJson(JSONObject prop) throws JSONException {
		if (prop == null) {
			return null;
		}
		final String type = prop.optString("type", null);
		if (type != null && type.equals("none")) {
			return new CTNoFillProperties();
		} else if (type == null || type.equals("solid")) {
			final CTSolidColorFillProperties res = new CTSolidColorFillProperties();
			final JSONObject color = prop.optJSONObject("color");
			
			if(color!=null){
				if(!createSolidColorFillPropertiesFromJson(res, color)){
					return null;
				}
			}
			return res;
		}
		return null;
	}
    
    
    public static JSONObject createJsonFromNoFillProperties(CTNoFillProperties fillProperties)
        throws JSONException {

        if(fillProperties!=null) {
            final JSONObject jsonObject = new JSONObject();
            jsonObject.put("type", "none");
            return jsonObject;
        }
        return null;
    }

    public static JSONObject createJsonFromScrgbClr(CTScRgbColor color)
        throws JSONException {

        final byte[] value = new byte[3];
        value[0] = JSONHelper.gammaChannel(color.getR());
        value[1] = JSONHelper.gammaChannel(color.getG());
        value[2] = JSONHelper.gammaChannel(color.getB());
        return Commons.surroundJSONObject("color", JSONHelper.makeColor("rgb", Commons.bytesToHexString(value, 0, 3), color.getEGColorTransform()));
    }

    public static JSONObject createJsonFromSrgbClr(CTSRgbColor color)
        throws JSONException {

        final byte[] value = color.getVal();
        return Commons.surroundJSONObject("color", JSONHelper.makeColor("rgb", Commons.bytesToHexString(value, 0, 3), color.getEGColorTransform()));
    }

    public static JSONObject createJsonFromHslClr(CTHslColor color)
        throws JSONException {

        return Commons.surroundJSONObject("color", JSONHelper.makeColor("hsl", JSONHelper.makeJSON("h", color.getHue(), "s", color.getSat(), "l", color.getLum()), color.getEGColorTransform()));
    }

    public static JSONObject createJsonFromSchemeClr(CTSchemeColor color)
        throws JSONException {

        String value = null;
        final STSchemeColorVal sourveVal = color.getVal();
        if (sourveVal != null) {
            switch (sourveVal) {
			case ACCENT_1:
			case ACCENT_2:
			case ACCENT_3:
			case ACCENT_4:
			case ACCENT_5:
			case ACCENT_6:
				value = sourveVal.value();
				break;
			case BG_1:
				value = "background1";
				break;
			case BG_2:
				value = "background2";
				break;
			case DK_1:
				value = "dark1";
				break;
			case DK_2:
				value = "dark2";
				break;
			case TX_1:
				value = "text1";
				break;
			case TX_2:
				value = "text2";
				break;
			case LT_1:
				value = "light1";
				break;
			case LT_2:
				value = "light2";
				break;
			case HLINK:
				value = "hyperlink";
				break;
			case FOL_HLINK:
				value = "followedHyperlink";
				break;
			case PH_CLR:
				value = "style";
				break;
			}
        }
        return Commons.surroundJSONObject("color", JSONHelper.makeColor("scheme", value, color.getEGColorTransform()));
    }
    
	public static void createSchemeColorFromJson(CTSchemeColor docx, JSONObject json) throws JSONException {
		final String value = json.getString("value");
		final STSchemeColorVal colorVal;
		if (value.equals("background1")) {
			colorVal = STSchemeColorVal.BG_1;
		} else if (value.equals("background2")) {
			colorVal = STSchemeColorVal.BG_2;
		} else if (value.equals("dark1")) {
			colorVal = STSchemeColorVal.DK_1;
		} else if (value.equals("dark2")) {
			colorVal = STSchemeColorVal.DK_2;
		} else if (value.equals("text1")) {
			colorVal = STSchemeColorVal.TX_1;
		} else if (value.equals("text2")) {
			colorVal = STSchemeColorVal.TX_2;
		} else if (value.equals("light1")) {
			colorVal = STSchemeColorVal.LT_1;
		} else if (value.equals("light2")) {
			colorVal = STSchemeColorVal.LT_2;
		} else if (value.equals("hyperlink")) {
			colorVal = STSchemeColorVal.HLINK;
		} else if (value.equals("followedHyperlink")) {
			colorVal = STSchemeColorVal.FOL_HLINK;
		} else if (value.equals("style")) {
			colorVal = STSchemeColorVal.PH_CLR;
		} else {
			STSchemeColorVal tmpVal;
			try {
				tmpVal = STSchemeColorVal.fromValue(value);
			} catch (final IllegalArgumentException e) {
				Logger.getAnonymousLogger().warning("could not find STSchemeColorVal for " + value);
				tmpVal = STSchemeColorVal.TX_1;
			}
			colorVal = tmpVal;
		}
		docx.setVal(colorVal);
		JSONHelper.saveTransformations(docx.getEGColorTransform(), json);
	}

	public static void createSRgbColorFromJson(CTSRgbColor docx, JSONObject json) throws JSONException {
		docx.setVal(Commons.hexStringToBytes(json.getString("value")));
		JSONHelper.saveTransformations(docx.getEGColorTransform(), json);
	}

    public static JSONObject createJsonFromSysClr(CTSystemColor color)
        throws JSONException {

        return JSONHelper.makeColor("system", color.getVal(), color.getEGColorTransform());
    }

    public static JSONObject createJsonFromPrstClr(CTPresetColor color) {
        throw new RuntimeException("CTPresetColor has not implemented toJSON function");
    }

    public static JSONObject createJsonFromChartColor(CTChartColor color)
        throws JSONException {

        JSONObject jsonChartColor = null;
        if(color!=null) {
            jsonChartColor = new JSONObject();
            jsonChartColor.put("id", color.getId());
            jsonChartColor.put("meth", color.getMeth());
            final List<CTSchemeColor> schemeClrList = color.getSchemeClr();
            if(schemeClrList!=null) {
                final JSONArray jsonArray = new JSONArray();
                for(final CTSchemeColor schemeClr:schemeClrList) {
                    jsonArray.put(createJsonFromSchemeClr(schemeClr));
                }
                jsonChartColor.put("schemeClr", jsonArray);
            }
            final List<CTSchemeColor> variationList = color.getVariation();
            if(variationList!=null ) {
                final JSONArray jsonArray = new JSONArray();
                for(final CTSchemeColor schemeClr:variationList) {
                	final JSONObject resColor = createJsonFromSchemeClr(schemeClr);
					if (resColor != null){
						jsonArray.put(resColor);
					}
                }
				if (!jsonArray.isEmpty()){
					jsonChartColor.put("variation", jsonArray);
				}
            }
        }
        return jsonChartColor!=null&&!jsonChartColor.isEmpty()?jsonChartColor:null;
    }

    public static void createJsonFromNonVisualDrawingProperties(JSONObject drawingProperties, CTNonVisualDrawingProps nonVisualDrawingProps)
        throws JSONException {

        if(nonVisualDrawingProps==null) {
            return;
        }
        final String name = nonVisualDrawingProps.getName();
        if(name!=null&&!name.isEmpty()) {
            drawingProperties.put("name", name);
        }
        final String description = nonVisualDrawingProps.getDescr();
        if(description!=null&&!description.isEmpty()) {
            drawingProperties.put("description", description);
        }
    }

    public static void applyNonVisualDrawingProperties(JSONObject drawingProperties, CTNonVisualDrawingProps nonVisualDrawingProps) {

        if(nonVisualDrawingProps==null) {
            return;
        }

        if(drawingProperties.has("name")) {
            final Object optName = drawingProperties.opt("name");
            if(optName instanceof String) {
                nonVisualDrawingProps.setName((String)optName);
            }
            else {
                // name is a required xml attribute... so we are setting a dummy
                nonVisualDrawingProps.setName("...");
            }
        }
        if(drawingProperties.has("description")) {
            final Object optDescription = drawingProperties.opt("description");
            if(optDescription instanceof String) {
                nonVisualDrawingProps.setDescr((String)optDescription);
            }
            else {
                nonVisualDrawingProps.setDescr(null);
            }
        }
    }

    public static CTTransform2D createTransform2D(int x100thmm, int y100thmm, int width100thmm, int height100thmm) {
        final org.docx4j.dml.ObjectFactory dmlObjectFactory = Context.getDmlObjectFactory();
        final CTTransform2D transform2D = dmlObjectFactory.createCTTransform2D();
        final CTPoint2D point2D = dmlObjectFactory.createCTPoint2D();
        point2D.setX(Commons.coordinateFrom100TH_MM(x100thmm));
        point2D.setY(Commons.coordinateFrom100TH_MM(y100thmm));
        transform2D.setOff(point2D);
        final CTPositiveSize2D positiveSize2D = dmlObjectFactory.createCTPositiveSize2D();
        positiveSize2D.setCx(Commons.coordinateFrom100TH_MM(width100thmm));
        positiveSize2D.setCy(Commons.coordinateFrom100TH_MM(height100thmm));
        return transform2D;
    }

    public static CTPresetGeometry2D createPresetGeometry2D(STShapeType presetType) {
        final CTPresetGeometry2D presetGeometry2D = Context.getDmlObjectFactory().createCTPresetGeometry2D();
        presetGeometry2D.setPrst(presetType);
        return presetGeometry2D;
    }

    public static CTBlipFillProperties createBlipFillProperties(OperationDocument operationDocument, Part part, String imageUrl)
        throws Exception {

        final org.docx4j.dml.ObjectFactory dmlObjectFactory = Context.getDmlObjectFactory();
        final CTBlipFillProperties blipFillProperties = dmlObjectFactory.createCTBlipFillProperties();
        final CTBlip blip = dmlObjectFactory.createCTBlip();
        blipFillProperties.setBlip(blip);
        blipFillProperties.setStretch(dmlObjectFactory.createCTStretchInfoProperties());
        final Relationship relationship = Commons.createAndApplyGraphic(operationDocument, part, imageUrl);
        if(relationship!=null) {
            final boolean linked = relationship.getTargetMode() != null && relationship.getTargetMode().equals("External");
            if(linked) {
                blip.setLink(relationship.getId());
            }
            else {
                blip.setEmbed(relationship.getId());
            }
        }
        return blipFillProperties;
    }

    public static CTNonVisualDrawingProps createNonVisualDrawingProps(String name) {
        final CTNonVisualDrawingProps nonVisualDrawingProps = Context.getDmlObjectFactory().createCTNonVisualDrawingProps();
        nonVisualDrawingProps.setName(name);
        return nonVisualDrawingProps;
    }

    public static void addTo(CTRegularTextRun textRun, StringBuffer stringBuffer) {
    	stringBuffer.append(textRun.getT());
    }

    public static void addTo(CTTextField textRun, StringBuffer stringBuffer) {
    	stringBuffer.append(textRun.getT());
    }

    public static void addTo(CTTextLineBreak textRun, StringBuffer stringBuffer) {
    	stringBuffer.append("\n");
    }

	public static String getText(CTTextBody rich) {
		final StringBuffer buf = new StringBuffer();

		final List<CTTextParagraph> paras = rich.getP();
		for (int i = 0; i < paras.size(); i++) {
			final CTTextParagraph p = paras.get(i);
			final List<Object> textes = p.getEGTextRun();
			for (final Object text : textes) {
				if(text instanceof CTRegularTextRun) {
					addTo((CTRegularTextRun)text, buf);
				}
				else if (text instanceof CTTextField) {
					addTo((CTTextField)text, buf);
				}
				else if (text instanceof CTTextLineBreak) {
					addTo((CTTextLineBreak)text, buf);
				}
			}
			if (i < paras.size() - 1) {
				buf.append("\n");
			}
		}
		return buf.toString();
	}

	public static JSONObject createJsonFromProperties(CTTitle title) throws JSONException {
		if (title == null) {
			return null;
		}
		final CTTx tx = title.getTx();
		if (tx == null) {
			return null;
		}
		final CTTextBody rich = tx.getRich();
		if (rich == null) {
			return null;
		}
		final String first = getText(rich);

		final JSONObject attrs = new JSONObject();
		final JSONArray link = new JSONArray();
		link.put(first);
		final JSONObject text = new JSONObject();
		text.put("link", link);
		attrs.put("text", text);
		
		final JSONObject character = DMLChartSpace.createJsonFromProperties(rich);
		if (character != null) {
			attrs.put("character", character);
		}
		return attrs;
	}

	public static void setTitleFromAttrs(CTTitle title, JSONObject attrs) throws JSONException {
		handleStandardTypes(title);
		final CTTx tx = title.getTx();
		final CTTextBody rich = tx.getRich();
		final CTTextBodyProperties body = rich.getBodyPr();
		final CTTextBody txPr = title.getTxPr();
		final CTTextBodyProperties body2 = txPr.getBodyPr();
		body.setRot(0);
		body2.setRot(0);
		
		final JSONObject text = attrs.optJSONObject("text");
		if(text==null){
			return;
		}
		

		final Object olink = text.opt("link");

		if (olink == null || !(olink instanceof JSONArray))
		{

			if (olink == null || olink.equals(JSONObject.NULL))
			{
				return;
			}
			else
			{
				Logger.getAnonymousLogger().warning("title text has no correct content " + text.opt("link"));
				return;
			}
		}

		final JSONArray links = (JSONArray) olink;
		if (links.length() > 0)
		{

		
			final List<CTTextParagraph> paras = rich.getP();
			paras.clear();
			for (int i = 0; i < links.length(); i++)
			{
				final CTTextParagraph p = new CTTextParagraph();
				setTextFromAttrs(p, attrs);
				paras.add(p);
				final List<Object> textHolder = p.getEGTextRun();
				textHolder.clear();

				final String link = links.getString(i);
				final String[] textes = link.split("\n");
				for (int j = 0; j < textes.length; j++)
				{
					final CTRegularTextRun reg = new CTRegularTextRun();
					reg.setRPr(GERMAN);
					reg.setT(textes[j]);
					textHolder.add(reg);
					if (j < textes.length - 1)
					{
						final CTTextLineBreak lb = new CTTextLineBreak();
						lb.setRPr(GERMAN);
						textHolder.add(lb);
					}
				}
			}
		}
	}

	public static void refreshTitleRotations(
		CTPlotArea area)
	{
		final List<IAxisDescription> axes = area.getValAxOrCatAxOrDateAx();
		final IListSer chart = area.getAreaChartOrArea3DChartOrLineChart().get(0);
		final String dir = chart.getBarDirection();
		for (int i = 0; i < axes.size(); i++)
		{
			boolean rotate = false;
			if (i == 1)
			{
				rotate = true;
			}
			if (dir == null || dir.equals("col"))
			{
				// normal
			}
			else
			{
				rotate = !rotate;
			}
			int rot = 0;
			if (rotate)
			{
				rot = -5400000;
			}
			final IAxisDescription axis = axes.get(i);
			final CTTitle title = axis.getTitle();

			if (title != null && title.getTx() != null)
			{
				final CTTx tx = title.getTx();
				final CTTextBody rich = tx.getRich();
				final CTTextBodyProperties body = rich.getBodyPr();
				final CTTextBody txPr = title.getTxPr();
				final CTTextBodyProperties body2 = txPr.getBodyPr();

				body.setRot(rot);
				body2.setRot(rot);
			}

			final CTTextBody lbl = axis.getTxPr();
			if (lbl != null && lbl.getBodyPr() != null)
			{
				final CTTextBodyProperties body = lbl.getBodyPr();
				body.setRot(0);
			}
		}
	}
	
	public static void handleStandardTypes(CTTextBody txPr) {
		CTTextBodyProperties body = txPr.getBodyPr();
		if(body==null){
			body = new CTTextBodyProperties();
			txPr.setBodyPr(body);
		}
		body.setSpcFirstLastPara(true);
		body.setVertOverflow(STTextVertOverflowType.ELLIPSIS);
		body.setVert(STTextVerticalType.HORZ);
		body.setWrap(STTextWrappingType.SQUARE);
		body.setAnchor(STTextAnchoringType.CTR);
		body.setAnchorCtr(true);
		
		CTTextListStyle lst2 = txPr.getLstStyle();
		if (lst2 == null) {
			lst2 = new CTTextListStyle();
			txPr.setLstStyle(lst2);
		}
	}

	private static void handleStandardTypes(CTTitle title) {
		CTTx tx = title.getTx();
		if (tx == null) {
			tx = new CTTx();
			title.setTx(tx);
		}
		CTTextBody rich = tx.getRich();
		if (rich == null) {
			rich = new CTTextBody();
			tx.setRich(rich);
		}
		handleStandardTypes(rich);

		CTLayout layout = title.getLayout();
		if (layout == null) {
			layout = new CTLayout();
			title.setLayout(layout);
		}
		CTBoolean overlay = title.getOverlay();
		if (overlay == null) {
			overlay = new CTBoolean();
			title.setOverlay(overlay);
		}
		overlay.setVal(false);
		CTShapeProperties spr = title.getSpPr();
		if (spr == null) {
			spr = getHiddenShapeProperties();
		}
		CTTextBody txpr = title.getTxPr();
		if (txpr == null) {
			txpr = new CTTextBody();
			title.setTxPr(txpr);
		}
		handleStandardTypes(txpr);
		
		final List<CTTextParagraph> ps = rich.getP();
		ps.clear();

		final List<CTTextParagraph> par2 = txpr.getP();
		par2.clear();

		final CTTextParagraph p = new CTTextParagraph();
		p.setPPr(getTextProp());
		par2.add(p);
		ps.add(p);

	}

	public static CTTextParagraphProperties getTextProp() {
		final CTTextParagraphProperties props = new CTTextParagraphProperties();
		final CTTextCharacterProperties def = new CTTextCharacterProperties();
		props.setDefRPr(def);

		def.setSz(1100);
		def.setB(false);
		def.setI(false);
		def.setU(STTextUnderlineType.NONE);
		def.setStrike(STTextStrikeType.NO_STRIKE);
		def.setKern(1200);
		def.setBaseline(0);

		final TextFont latin = new TextFont();
		latin.setTypeface("+mn-lt");
		def.setLatin(latin);
		final TextFont ea = new TextFont();
		ea.setTypeface("+mn-ea");
		def.setEa(ea);
		final TextFont cs = new TextFont();
		cs.setTypeface("+mn-cs");
		def.setCs(cs);
		
		def.setSolidFill(null);
		def.setNoFill(null);

		return props;
	}

	public static void addToOperations(CTPlotArea plotArea, JSONArray operations, JSONArray position) throws JSONException {
		final List<IAxisDescription> axes = plotArea.getValAxOrCatAxOrDateAx();
		if (axes != null && axes.size() > 0) {
			for (int i = 0; i < axes.size(); i++) {
				final IAxisDescription axe = (IAxisDescription) axes.get(i);

				final String axisParam;
				if (i == 0) {
					axisParam = "x";
				} else if (i == 1) {
					axisParam = "y";
				} else if (i == 2) {
					axisParam = "z";
				} else {
					return;
				}

				getAxeOperation(axe, operations, axisParam, position);
				getGridOperation(axe, operations, axisParam, position);
			}
		}
	}

	private static final void getAxeOperation(IAxisDescription axe, JSONArray operations, String axisParam, JSONArray position) throws JSONException {

		final JSONObject axis = new JSONObject();
        final CTScaling scaling = axe.getScaling();
        if(scaling!=null) {
            if (scaling.getMin()!=null) {
                axis.put("min", scaling.getMin().getVal());
            }
            if(scaling.getMax()!=null) {
                axis.put("max", scaling.getMax().getVal());
            }
        }
        JSONObject character = null;
		final JSONObject axisOp = new JSONObject();
		final CTBoolean del = axe.getDelete();
		if (del != null && del.isVal()) {
			final JSONObject line = new JSONObject();
			line.put("type", "none");
			axisOp.put("line", line);
		} else {
			final CTShapeProperties shape = axe.getSpPr();
			if (shape != null) {
                DMLHelper.createJsonFromChartShapeProperties(axisOp, shape);
			}
			if (!axisOp.has("line")) {
				axisOp.put("line", makeStandardShapeType());
			}

			final CTTickLblPos label = axe.getTickLblPos();
			if (label != null && label.getVal() != STTickLblPos.NONE) {
				axis.put("label", true);
				character = DMLChartSpace.createJsonFromProperties(axe.getTxPr());
			}

			getAxeTitleOperation(axe, operations, axisParam, position);
		}

		if (axis.length() > 0) {
			axisOp.put("axis", axis);
		}
		if (character != null) {
			axisOp.put("character", character);
		}


		if (axisOp.length() > 0) {
			final JSONObject op = new JSONObject();
			op.put("name", "setChartAxisAttributes");
			op.put("start", position);
			op.put("axis", axisParam);
			op.put("attrs", axisOp);
			operations.put(op);
		}
	}
	
	public static void setAxisAttributes(CTPlotArea plotArea, String axis, JSONObject attrs) throws JSONException {
		final IAxisDescription ax = getAxis(plotArea, axis);
		if (ax == null) {
			return;
		}

		CTShapeProperties shape = ax.getSpPr();
		if (shape == null) {
			shape = new CTShapeProperties();
			ax.setSpPr(shape);
		}
		
		createChartShapePropertiesFromJson(shape, attrs);
		CTTickMark tickmark = ax.getMajorTickMark();
		if (tickmark == null)
		{
			tickmark = new CTTickMark();
			ax.setMajorTickMark(tickmark);
		}
		if (shape.getLn().getNoFill() != null)
		{
			tickmark.setVal(STTickMark.NONE);
		}
		else
		{
			tickmark.setVal(STTickMark.OUT);
		}
		
		final JSONObject a = attrs.optJSONObject("axis");
		if (a != null) {
			CTTickLblPos lblPos = ax.getTickLblPos();
			if (lblPos == null) {
				lblPos = new CTTickLblPos();
				ax.setTickLblPos(lblPos);
			}
			final boolean label = a.optBoolean("label", false);

			if (label) {
				lblPos.setVal(STTickLblPos.LOW);

				
				CTTextBody txPr = ax.getTxPr();

				if (txPr == null) {
					txPr = new CTTextBody();
					ax.setTxPr(txPr);
					DMLHelper.handleStandardTypes(txPr);
				}
				setCharacterFromAttrs(txPr, attrs);
				
			} else {
				lblPos.setVal(STTickLblPos.NONE);
			}
		}
	}
	
	public static void setCharacterFromAttrs(CTTextBody txPr, JSONObject attrs) throws JSONException {
		final List<CTTextParagraph> ps = txPr.getP();
		ps.clear();
		final CTTextParagraph p = new CTTextParagraph();
		ps.add(p);
		setTextFromAttrs(p, attrs);
	}
	
	public static void setTextFromAttrs(CTTextParagraph p, JSONObject attrs) throws JSONException {
		final JSONObject character = attrs.optJSONObject("character");
		
		
		CTTextParagraphProperties prop = p.getPPr();
		if(prop == null){
			prop = getTextProp();
			p.setPPr(prop);
		}
		
		if (character != null)
		{

			final CTTextCharacterProperties def = prop.getDefRPr();

			def.setSz(ptTofontSize(character.optDouble("fontSize", 10)));

			CTSolidColorFillProperties solid = def.getSolidFill();
			if (solid == null)
			{
				solid = new CTSolidColorFillProperties();
				def.setSolidFill(solid);
			}

			final JSONObject color = character.optJSONObject("color");
			if (color != null)
			{
				if (!createSolidColorFillPropertiesFromJson(def.getSolidFill(), color))
				{
					def.setSolidFill(null);
				}
			}
		}
	}

	private static IAxisDescription getAxis(CTPlotArea plotArea, String axis) {
		final int index;
		if (axis.equals("x")) {
			index = 0;
		} else if (axis.equals("y")) {
			index = 1;
		} else if (axis.equals("z")) {
			index = 2;
		} else {
			return null;
		}
		final List<IAxisDescription> axes = plotArea.getValAxOrCatAxOrDateAx();
		
		if (index >= axes.size()) {
			return null;
		}
		
		IAxisDescription res = (IAxisDescription) axes.get(index);
		
		if (res == null) {
			if (axis.equals("x")) {
				final CTCatAx cat = new CTCatAx();
				cat.setMinorGridlines(getHiddenChartLines());
				cat.setMinorTickMark(getHiddenTickMark());
				cat.setNumFmt(createNumFmt());
				res = cat;
			} else {
				final CTValAx val = new CTValAx();
				val.setMinorGridlines(getHiddenChartLines());
				val.setMinorTickMark(getHiddenTickMark());
				val.setNumFmt(createNumFmt());
				res = val;
			}
			
			res.getAxId().setVal(index);
			axes.set(index, res);
		}
		if (res.getDelete() == null || res.getDelete().isVal()) {
			hideAxis(res);
		}
	
		return res;
	}

	private static void hideAxis(
		IAxisDescription axis)
	{
		final CTBoolean delete = new CTBoolean();
		delete.setVal(false);
		axis.setDelete(delete);

		axis.setSpPr(getHiddenShapeProperties());

		axis.setMajorGridlines(getHiddenChartLines());
		axis.setTickLblPos(getHiddenLblPos());

		axis.setTitle(null);
		axis.setTxPr(null);

	}
	
	public static CTNumFmt createNumFmt()
	{
		final CTNumFmt res = new CTNumFmt();
		res.setFormatCode("0");
		res.setSourceLinked(true);
		return res;
	}

	
	private static CTChartLines getHiddenChartLines()
	{
		final CTChartLines line = new CTChartLines();
		line.setSpPr(getHiddenShapeProperties());
		return line;
	}

	private static CTTickLblPos getHiddenLblPos()
	{
		final CTTickLblPos lbl = new CTTickLblPos();
		lbl.setVal(STTickLblPos.NONE);
		return lbl;
	}

	private static CTTickMark getHiddenTickMark()
	{
		final CTTickMark mark = new CTTickMark();
		mark.setVal(STTickMark.NONE);
		return mark;
	}
	
	public static CTShapeProperties getHiddenShapeProperties()
	{
		final CTShapeProperties shape = new CTShapeProperties();
		shape.setNoFill(new CTNoFillProperties());
		final CTLineProperties line = new CTLineProperties();
		line.setNoFill(new CTNoFillProperties());
		shape.setLn(line);
		return shape;
	}

	private static final void getAxeTitleOperation(IAxisDescription axe, JSONArray operations, String axisParam, JSONArray position) throws JSONException {
		final CTTitle title = axe.getTitle();

		final JSONObject attrs = DMLHelper.createJsonFromProperties(title);
		if (attrs != null) {

			final JSONObject op = new JSONObject();
			op.put("name", "setChartTitleAttributes");
			op.put("start", position);
			op.put("axis", axisParam);
			op.put("attrs", attrs);
			operations.put(op);
		}

	}

	public static void setChartTitleAttributes(CTPlotArea plotArea, String axis, JSONObject attrs) throws JSONException {
		final IAxisDescription ax = getAxis(plotArea, axis);
		if (ax == null) {
			return;
		}
		CTTitle title = ax.getTitle();
		if (title == null) {
			title = new CTTitle();
			ax.setTitle(title);
		}
		setTitleFromAttrs(title, attrs);
		refreshTitleRotations(plotArea);
	}

	private static final void getGridOperation(IAxisDescription axe, JSONArray operations, String axisParam, JSONArray position) throws JSONException {

		final CTChartLines grid = axe.getMajorGridlines();
		if (grid != null) {
			final JSONObject gridOp = new JSONObject();
			final CTShapeProperties shape = grid.getSpPr();
			if (shape != null) {
				createJsonFromChartShapeProperties(gridOp, shape);
			}
			if (!gridOp.has("line")) {
				gridOp.put("line", makeStandardShapeType());
			}
			final JSONObject op = new JSONObject();
			op.put("name", "setChartGridlineAttributes");
			op.put("start", position);
			op.put("axis", axisParam);
			op.put("attrs", gridOp);
			operations.put(op);
		}
	}

	public static void setChartGridlineAttributes(CTPlotArea plotArea, String axis, JSONObject attrs) throws JSONException {
		final IAxisDescription ax = getAxis(plotArea, axis);
		if (ax == null) {
			return;
		}
		
		CTChartLines grids = ax.getMajorGridlines();
		if (grids == null) {
			grids = new CTChartLines();
			ax.setMajorGridlines(grids);
		}
		CTShapeProperties shape = grids.getSpPr();
		if (shape == null) {
			shape = new CTShapeProperties();
			grids.setSpPr(shape);
		}
		createChartShapePropertiesFromJson(shape, attrs);
	}
	
    /**
     * @param part (partName Uri should be /word/document.xml)
     * @return the imageUrl in following form: word/media/xxxx.png
     */
    public static String getBlipUrl(Part part, CTBlip blip) {
        String imageUrl = "";
        if(blip!=null) {
            imageUrl = Commons.getUrl(part, blip.getEmbed().length() > 0 ? blip.getEmbed() : blip.getLink() );
        }
        return imageUrl;
    }
    
    public static float fontSizeToPt(Integer sz) {
		if (sz == null) {
			return 10;
		}
		return ((float) sz) / 100f;
	}
    
    public static Integer ptTofontSize(double pt) {
		return (int) (pt * 100);
	}

	public static float lineWidthToPt(Integer w) {
		if (w == null) {
			return 1;
		}
		return ((float) w) / 12700f;
	}

    private static Integer ptToLineWidth(long pt) {
		return (int) (pt * 12700);
	}
	
	//FIXME: nessecary?
	private static JSONObject makeStandardShapeType() throws JSONException {
		final JSONObject shape = new JSONObject();
		shape.put("type", "solid");
		final JSONObject color = new JSONObject();
		color.put("type", "auto");
		shape.put("color", color);
		return shape;
	}
}
