/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.docx.tools;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.Iterator;
import java.util.List;

import javax.xml.bind.JAXBElement;

import org.docx4j.jaxb.Context;
import org.docx4j.IndexedNode;
import org.docx4j.openpackaging.packages.WordprocessingMLPackage;
import org.docx4j.openpackaging.parts.WordprocessingML.MainDocumentPart;
import org.docx4j.wml.CTBorder;
import org.docx4j.wml.CTHeight;
import org.docx4j.wml.CTShd;
import org.docx4j.wml.CTTblCellMar;
import org.docx4j.wml.CTTblLayoutType;
import org.docx4j.wml.CTTblLook;
import org.docx4j.wml.CTTblPrBase;
import org.docx4j.wml.CTTblPrBase.TblStyle;
import org.docx4j.wml.CTTblPrBase.TblStyleColBandSize;
import org.docx4j.wml.CTTblPrBase.TblStyleRowBandSize;
import org.docx4j.wml.CTTrPrBase;
import org.docx4j.wml.ObjectFactory;
import org.docx4j.wml.STBorder;
import org.docx4j.wml.STTblLayoutType;
import org.docx4j.wml.SectPr;
import org.docx4j.wml.SectPr.PgMar;
import org.docx4j.wml.SectPr.PgSz;
import org.docx4j.wml.Tbl;
import org.docx4j.wml.TblBorders;
import org.docx4j.wml.TblGrid;
import org.docx4j.wml.TblGridBase;
import org.docx4j.wml.TblGridCol;
import org.docx4j.wml.TblPr;
import org.docx4j.wml.TblWidth;
import org.docx4j.wml.Tc;
import org.docx4j.wml.TcMar;
import org.docx4j.wml.TcPr;
import org.docx4j.wml.TcPrInner;
import org.docx4j.wml.TcPrInner.GridSpan;
import org.docx4j.wml.TcPrInner.TcBorders;
import org.docx4j.wml.TrPr;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import com.openexchange.office.ooxml.docx.OperationDocument;
import com.openexchange.office.ooxml.docx.tools.Component.TableContext;
import com.openexchange.office.ooxml.tools.Commons;

public class Table {

    // RowBandSize and ColBandSize needs to be set to 1, otherwise the
    // conditional table styles "band1Vert, band2Vert..." will be ignored
    // by word.
	public static void initLookAndRowBandSize(CTTblPrBase tblPr) {

		final ObjectFactory objectFactory = Context.getWmlObjectFactory();

		final CTTblLook tblLook = objectFactory.createCTTblLook();
		tblLook.setParent(tblPr);
        tblPr.setTblLook(tblLook);
        tblLook.setVal("4A0");

        TblStyleRowBandSize rowBandSize = tblPr.getTblStyleRowBandSize();
        if(rowBandSize==null) {
        	rowBandSize = objectFactory.createCTTblPrBaseTblStyleRowBandSize();
            rowBandSize.setParent(tblPr);
            tblPr.setTblStyleRowBandSize(rowBandSize);
        }
        rowBandSize.setVal(BigInteger.valueOf(1));
        TblStyleColBandSize colBandSize = tblPr.getTblStyleColBandSize();
        if(colBandSize==null) {
        	colBandSize = objectFactory.createCTTblPrBaseTblStyleColBandSize();
            colBandSize.setParent(tblPr);
            tblPr.setTblStyleColBandSize(colBandSize);
        }
        colBandSize.setVal(BigInteger.valueOf(1));
	}

    /**
     * @param objectFactory
     * @param tbl
     * @return TblPr are returned (they are created if not existing)
     */
    public static TblPr getTableProperties(Tbl tbl) {

    	TblPr tblPr = tbl.getTblPr();
        if(tblPr==null) {
            tblPr = Context.getWmlObjectFactory().createTblPr();
            tblPr.setParent(tbl);
            tbl.setTblPr(tblPr);
        }
        return tblPr;
    }

    public static JSONArray getTableGrid(TblGridBase tblGridBase) {

        JSONArray tableGrid = null;
        if (tblGridBase!=null) {
            List<TblGridCol> gridCols = tblGridBase.getGridCol();
            if(gridCols!=null) {
                tableGrid = new JSONArray();
                for(TblGridCol gridCol:gridCols) {
                    BigInteger w = gridCol.getW();
                    tableGrid.put(w!=null?w.longValue()*254/144:1);
                }
            }
        }
        return tableGrid;
    }

    public static void setTableGrid(OperationDocument operationDocument, Tbl tbl, TblGridBase tblGridBase, JSONArray tableGrid)
        throws JSONException {

        if (tblGridBase == null)
            return;

        if(tableGrid!=null) {
	        final List<TblGridCol> gridCols = tblGridBase.getGridCol();
	        gridCols.clear();
	        if(tableGrid!=null) {
	            for(int i=0; i<tableGrid.length(); i++) {
	                TblGridCol tblGridCol = Context.getWmlObjectFactory().createTblGridCol();
	                tblGridCol.setW(BigInteger.valueOf(tableGrid.getInt(i)*144/254));
	                gridCols.add(tblGridCol);
	            }
	        }
	        final TblPr tblPr = tbl.getTblPr();
	        if(tblPr==null) {
	        	return;
	        }
	        final TblWidth tblW = tblPr.getTblW();
	        if(tblW==null) {
	        	return;
	        }
	        if(tblW.getW()!=null) {
		        Double tableWidth = null;
		        if(tblW.getType()==null||tblW.getType().equals("dxa")) {
		        	tableWidth = tblW.getW().doubleValue();
		        }
		        else if(tbl.getParent() instanceof MainDocumentPart) {
		        	if(tblW.getType().equals("auto")||tblW.getType().equals("pct")) {
		        		tableWidth = Double.valueOf(Utils.getPageWidth(operationDocument.getPackage().getMainDocumentPart(), 19000));
		        		if(tblW.getType().equals("pct")) {
		                    tableWidth *= tblW.getW().doubleValue() / 5000;
		        		}
		        	}
		        }
		        // we could determine the width of the table, so it is possible to adjust also the gridValues to 1/20 pts
		        if(tableWidth!=null) {
			        // updating tableGrid to 1/20 points
			        double totalGridSize = 0;
			        for(int i=0; i<gridCols.size(); i++) {
			        	totalGridSize += gridCols.get(i).getW().doubleValue();
			        }
			        if(totalGridSize!=0.0) {
				        final double gridFactor = (tableWidth*1440.0d/2540.0d) / totalGridSize;
			        	for(int i=0; i<gridCols.size(); i++) {
			        		gridCols.get(i).setW(BigDecimal.valueOf(gridCols.get(i).getW().doubleValue()*gridFactor).toBigInteger());
			        	}
			        }
		        }
	        }
        }
    }

    public static void applyTableStyle(String styleId, CTTblPrBase tblPr) {
        if (styleId!=null) {
            TblStyle tblStyle = tblPr.getTblStyle();
            if (tblStyle==null) {
                tblStyle = Context.getWmlObjectFactory().createCTTblPrBaseTblStyle();
                tblStyle.setParent(tblPr);
            }
            tblStyle.setVal(styleId);
            tblPr.setTblStyle(tblStyle);
            initLookAndRowBandSize(tblPr);
        }
        else {
            tblPr.setTblStyle(null);
        }
    }

    public static void applyTableProperties(OperationDocument operationDocument, JSONObject tableProperties, Tbl tbl, TblGridBase tblGridBase, CTTblPrBase tblPr)
        throws JSONException {

        if(tableProperties==null)
            return;

        final ObjectFactory objectFactory = Context.getWmlObjectFactory();
        final Iterator<String> keys = tableProperties.keys();

        Object newTableWidth = null;
        JSONArray newTableGrid = null;

        while(keys.hasNext()) {
            String attr = keys.next();
            Object value = tableProperties.get(attr);

            if(attr.equals("tableGrid")) {
                if (value instanceof JSONArray) {
                	newTableGrid = (JSONArray)value;
                }
            } else if(attr.equals("width")) {
            	newTableWidth = value;
            }
            else if(attr.equals("borderLeft"))
                getTblBorders(objectFactory, tblPr).setLeft(createCTBorderFromJSON(operationDocument, objectFactory, tblPr, value));
            else if(attr.equals("borderTop"))
                getTblBorders(objectFactory, tblPr).setTop(createCTBorderFromJSON(operationDocument, objectFactory, tblPr, value));
            else if(attr.equals("borderRight"))
                getTblBorders(objectFactory, tblPr).setRight(createCTBorderFromJSON(operationDocument, objectFactory, tblPr, value));
            else if(attr.equals("borderBottom"))
                getTblBorders(objectFactory, tblPr).setBottom(createCTBorderFromJSON(operationDocument, objectFactory, tblPr, value));
            else if(attr.equals("borderInsideHor"))
                getTblBorders(objectFactory, tblPr).setInsideH(createCTBorderFromJSON(operationDocument, objectFactory, tblPr, value));
            else if(attr.equals("borderInsideVert"))
                getTblBorders(objectFactory, tblPr).setInsideV(createCTBorderFromJSON(operationDocument, objectFactory, tblPr, value));
            else if(attr.equals("paddingLeft"))
                getTblCellMar(objectFactory, tblPr).setLeft(createTblWidthFromJSON(objectFactory, tblPr, value));
            else if(attr.equals("paddingTop"))
                getTblCellMar(objectFactory, tblPr).setTop(createTblWidthFromJSON(objectFactory, tblPr, value));
            else if(attr.equals("paddingRight"))
                getTblCellMar(objectFactory, tblPr).setRight(createTblWidthFromJSON(objectFactory, tblPr, value));
            else if(attr.equals("paddingBottom"))
                getTblCellMar(objectFactory, tblPr).setBottom(createTblWidthFromJSON(objectFactory, tblPr, value));
            else if(attr.equals("fillColor")) {
                if(value instanceof JSONObject) {
                    CTShd shd = tblPr.getShd();
                    if(shd==null) {
                        shd = objectFactory.createCTShd();
                        shd.setParent(tblPr);
                        tblPr.setShd(shd);
                    }
                    Utils.initShdFromJSONColor(operationDocument, (JSONObject)value, shd);

                    if(tbl!=null) {
	                    Component tableRowComponent = TableContext.getNextComponent(0, new IndexedNode<Object>(tbl), -1);                    
	                    while(tableRowComponent!=null) {
	                    	Component tableCellComponent = tableRowComponent.getChildComponent();
	                        while(tableCellComponent!=null) {
	                            Tc tc = (Tc)tableCellComponent.getNode().getData();;
	                            TcPr tcPr = tc.getTcPr();
	                            if(tcPr==null) {
	                                tcPr = objectFactory.createTcPr();
	                                tcPr.setParent(tc);
	                                tc.setTcPr(tcPr);
	                            }
	                            shd = tcPr.getShd();
	                            if(shd==null) {
	                                shd = objectFactory.createCTShd();
	                                shd.setParent(tcPr);
	                                tcPr.setShd(shd);
	                            }
	                            Utils.initShdFromJSONColor(operationDocument, (JSONObject)value, shd);
	                            tableCellComponent = tableCellComponent.getNextComponent();
	                        }
	                        tableRowComponent = tableRowComponent.getNextComponent();
	                    }
                    }
                }
                else
                    tblPr.setShd(null);
            }
            else if(attr.equals("exclude")) {
                final CTTblLook tblLook = objectFactory.createCTTblLook();
                tblLook.setParent(tblPr);
                if(value instanceof JSONArray) {
                    short val = 0x1e0;
                    Iterator<Object> jsonArray = ((JSONArray)value).iterator();
                    while(jsonArray.hasNext()) {
                        Object o = jsonArray.next();
                        if(o instanceof String) {
                            if(o.equals("firstRow")) {
                                val&=~0x20;
                            } else if(o.equals("lastRow")) {
                                val&=~0x40;
                            } else if(o.equals("firstCol")) {
                                val&=~0x80;
                            } else if(o.equals("lastCol")) {
                                val&=~0x100;
                            } else if(o.equals("bandsHor")) {
                                val|=0x200;
                            } else if(o.equals("bandsVert")) {
                                val|=0x400;
                            }
                        }
                    }
                    tblLook.setVal(Integer.toHexString(val));
                }
                else
                    tblLook.setVal("01E0");
                tblPr.setTblLook(tblLook);
            }
        }
        if(newTableWidth!=null) {
	        tblPr.setTblW(createTblWidthFromJSON(objectFactory, tblPr, newTableWidth));
	
	        CTTblLayoutType tblLayout = tblPr.getTblLayout();
	        if(tblLayout==null) {
	            tblLayout = objectFactory.createCTTblLayoutType();
	            tblLayout.setParent(tblPr);
	            tblPr.setTblLayout(tblLayout);
	        }
	        tblLayout.setType(STTblLayoutType.FIXED);
        }
        if(newTableWidth!=null||newTableGrid!=null) {
            setTableGrid(operationDocument, tbl, tblGridBase, newTableGrid);
    	}
    }

    public static void createTableProperties(WordprocessingMLPackage document, CTTblPrBase tableProperties, JSONArray tableGrid, boolean isRootTable, int firstRowWidth, JSONObject attrs)
        throws JSONException {

        final JSONObject jsonTableProperties = new JSONObject();

        if(tableGrid!=null)
            jsonTableProperties.put("tableGrid", tableGrid);
        if(tableProperties!=null) {
            CTTblLayoutType ctLayoutType = tableProperties.getTblLayout();
            @SuppressWarnings("unused")
            STTblLayoutType tableLayoutType = ctLayoutType!=null ? ctLayoutType.getType() : STTblLayoutType.AUTOFIT;    // default = AUTOFIT
                                                                                                                        //           FIXED

            final long referenceWidth = isRootTable ? Utils.getPageWidth(document.getMainDocumentPart(), 19000) : 0;
            Commons.jsonPut(jsonTableProperties, "width", createJSONFromTblWidth(tableProperties.getTblW(), referenceWidth, firstRowWidth));

            // creating table borders
            TblBorders tblBorders = tableProperties.getTblBorders();
            if(tblBorders!=null) {
                Commons.jsonPut(jsonTableProperties, "borderLeft", createJSONfromCTBorder(tblBorders.getLeft()));
                Commons.jsonPut(jsonTableProperties, "borderTop", createJSONfromCTBorder(tblBorders.getTop()));
                Commons.jsonPut(jsonTableProperties, "borderRight", createJSONfromCTBorder(tblBorders.getRight()));
                Commons.jsonPut(jsonTableProperties, "borderBottom", createJSONfromCTBorder(tblBorders.getBottom()));
                Commons.jsonPut(jsonTableProperties, "borderInsideHor", createJSONfromCTBorder(tblBorders.getInsideH()));
                Commons.jsonPut(jsonTableProperties, "borderInsideVert", createJSONfromCTBorder(tblBorders.getInsideV()));
            }

            CTTblCellMar tblCellMar= tableProperties.getTblCellMar();
            if(tblCellMar!=null) {
                Commons.jsonPut(jsonTableProperties, "paddingLeft", createJSONFromTblWidth(tblCellMar.getLeft(), 0, 0));
                Commons.jsonPut(jsonTableProperties, "paddingTop", createJSONFromTblWidth(tblCellMar.getTop(), 0, 0));
                Commons.jsonPut(jsonTableProperties, "paddingRight", createJSONFromTblWidth(tblCellMar.getRight(), 0, 0));
                Commons.jsonPut(jsonTableProperties, "paddingBottom", createJSONFromTblWidth(tblCellMar.getBottom(), 0, 0));
            }

            CTShd ctShd = tableProperties.getShd();
            Commons.jsonPut(jsonTableProperties, "fillColor", Utils.createFillColor(ctShd));

            // taking care of used conditional styles..:
            final CTTblLook tblLook = tableProperties.getTblLook();
            if(tblLook!=null) {
                final String val = tblLook.getVal();
                final JSONArray jsonLook = new JSONArray();
                if(val!=null) {
                    int look = Integer.parseInt(val, 16);
                    if((look&0x20)==0)
                        jsonLook.put("firstRow");
                    if((look&0x40)==0)
                        jsonLook.put("lastRow");
                    if((look&0x80)==0)
                        jsonLook.put("firstCol");
                    if((look&0x100)==0)
                        jsonLook.put("lastCol");
                    if((look&0x200)>0) {
                        jsonLook.put("bandsHor");
                    }
                    if((look&0x400)>0) {
                        jsonLook.put("bandsVert");
                    }
                }
                else {
                    if(!tblLook.getFirstRow().getState()) {
                        jsonLook.put("firstRow");
                    }
                    if(!tblLook.getLastRow().getState()) {
                        jsonLook.put("lastRow");
                    }
                    if(!tblLook.getFirstColumn().getState()) {
                        jsonLook.put("firstCol");
                    }
                    if(!tblLook.getLastColumn().getState()) {
                        jsonLook.put("lastCol");
                    }
                    if(tblLook.getNoHBand().getState()) {
                        jsonLook.put("bandsHor");
                    }
                    if(tblLook.getNoVBand().getState()) {
                        jsonLook.put("bandsVert");
                    }
                }
                if(jsonLook.length()>0)
                    jsonTableProperties.put("exclude", jsonLook);
            }
        }
        if(jsonTableProperties!=null&&!jsonTableProperties.isEmpty()) {
        	attrs.put("table", jsonTableProperties);
        }
    }

    public static void applyRowProperties(JSONObject rowProperties, CTTrPrBase trPr)
        throws JSONException {

        if(rowProperties==null)
            return;

        final ObjectFactory objectFactory = Context.getWmlObjectFactory();
        final Iterator<String> keys = rowProperties.keys();
        while(keys.hasNext()) {
            String attr = keys.next();
            Object value = rowProperties.get(attr);
            if(attr.equals("height")) {
                if (value instanceof Integer) {
                    List<JAXBElement<?>> jaxbElements = trPr.getCnfStyleOrDivIdOrGridBefore();
                    JAXBElement<?> jaxbElement = Commons.findElement(jaxbElements, "trHeight");
                    CTHeight ctHeight;
                    if(jaxbElement!=null)
                        ctHeight = (CTHeight)jaxbElement.getValue();
                    else {
                        ctHeight = objectFactory.createCTHeight();
                        ctHeight.setParent(trPr);
                    }
                    ctHeight.setVal(BigInteger.valueOf((Integer)value*144/254));
                    if(jaxbElement!=null)
                        jaxbElement = objectFactory.createCTTrPrBaseTrHeight(ctHeight);
                    else
                        jaxbElements.add(objectFactory.createCTTrPrBaseTrHeight(ctHeight));
                }
            }
        }
    }

    public static void createRowProperties(CTTrPrBase rowProperties, JSONObject attrs)
        throws JSONException {

        JSONObject jsonRowProperties = new JSONObject();
        if(rowProperties!=null) {
            List<JAXBElement<?>> jaxbElements = rowProperties.getCnfStyleOrDivIdOrGridBefore();
            for(JAXBElement<?> jaxbElement:jaxbElements) {
                if(jaxbElement.getDeclaredType().getName().equals("org.docx4j.wml.CTHeight")) {
                    Object o = jaxbElement.getValue();
                    if(o instanceof CTHeight) {
                        BigInteger height = ((CTHeight)o).getVal();
                        if(height!=null)
                            jsonRowProperties.put("height", height.longValue()*254/144);
                    }
                    break;
                }
            }
        }
        if(jsonRowProperties!=null&&!jsonRowProperties.isEmpty()) {
        	attrs.put("row", jsonRowProperties);
        }
    }

    public static void applyCellProperties(OperationDocument operationDocument, JSONObject cellProperties, TcPrInner tcPr)
        throws JSONException {

        if(cellProperties==null)
            return;

        final ObjectFactory objectFactory = Context.getWmlObjectFactory();
        final Iterator<String> keys = cellProperties.keys();
        while(keys.hasNext()) {
            String attr = keys.next();
            Object value = cellProperties.get(attr);
            if(attr.equals("gridSpan")) {
                if (value instanceof Integer) {
                    TcPrInner.GridSpan span = tcPr.getGridSpan();
                    if (span==null) {
                        span = objectFactory.createTcPrInnerGridSpan();
                        span.setParent(tcPr);
                    }
                    span.setVal(BigInteger.valueOf((Integer)value));
                    tcPr.setGridSpan(span);
                }
                else
                    tcPr.setGridSpan(null);
            }
            else if(attr.equals("borderLeft"))
                getTblBorders(objectFactory, tcPr).setLeft(createCTBorderFromJSON(operationDocument, objectFactory, tcPr, value));
            else if(attr.equals("borderTop"))
                getTblBorders(objectFactory, tcPr).setTop(createCTBorderFromJSON(operationDocument, objectFactory, tcPr, value));
            else if(attr.equals("borderRight"))
                getTblBorders(objectFactory, tcPr).setRight(createCTBorderFromJSON(operationDocument, objectFactory, tcPr, value));
            else if(attr.equals("borderBottom"))
                getTblBorders(objectFactory, tcPr).setBottom(createCTBorderFromJSON(operationDocument, objectFactory, tcPr, value));
            else if(attr.equals("borderInsideHor"))
                getTblBorders(objectFactory, tcPr).setInsideH(createCTBorderFromJSON(operationDocument, objectFactory, tcPr, value));
            else if(attr.equals("borderInsideVert"))
                getTblBorders(objectFactory, tcPr).setInsideV(createCTBorderFromJSON(operationDocument, objectFactory, tcPr, value));
            else if(attr.equals("paddingLeft"))
                getTblCellMar(objectFactory, tcPr).setLeft(createTblWidthFromJSON(objectFactory, tcPr, value));
            else if(attr.equals("paddingTop"))
                getTblCellMar(objectFactory, tcPr).setTop(createTblWidthFromJSON(objectFactory, tcPr, value));
            else if(attr.equals("paddingRight"))
                getTblCellMar(objectFactory, tcPr).setRight(createTblWidthFromJSON(objectFactory, tcPr, value));
            else if(attr.equals("paddingBottom"))
                getTblCellMar(objectFactory, tcPr).setBottom(createTblWidthFromJSON(objectFactory, tcPr, value));
            else if(attr.equals("fillColor")) {
                if(value instanceof JSONObject) {
                    CTShd shd = tcPr.getShd();
                    if(shd==null) {
                        shd = objectFactory.createCTShd();
                        shd.setParent(tcPr);
                        tcPr.setShd(shd);
                    }
                    Utils.initShdFromJSONColor(operationDocument, (JSONObject)value, shd);
                }
                else
                    tcPr.setShd(null);
            }
        }
    }

    public static void createCellProperties(TcPrInner cellProperties, Tbl table, JSONObject attrs)
        throws JSONException {

        JSONObject jsonCellProperties = new JSONObject();
        if(cellProperties!=null) {

            GridSpan gridSpan = cellProperties.getGridSpan();
            if(gridSpan!=null) {
                jsonCellProperties.put("gridSpan", gridSpan.getVal().intValue());
            }

            // creating cell borders
            TcBorders tcBorder = cellProperties.getTcBorders();
            if(tcBorder!=null) {
                Commons.jsonPut(jsonCellProperties, "borderLeft", createJSONfromCTBorder(tcBorder.getLeft()));
                Commons.jsonPut(jsonCellProperties, "borderTop", createJSONfromCTBorder(tcBorder.getTop()));
                Commons.jsonPut(jsonCellProperties, "borderRight", createJSONfromCTBorder(tcBorder.getRight()));
                Commons.jsonPut(jsonCellProperties, "borderBottom", createJSONfromCTBorder(tcBorder.getBottom()));
                Commons.jsonPut(jsonCellProperties, "borderInsideHor", createJSONfromCTBorder(tcBorder.getInsideH()));
                Commons.jsonPut(jsonCellProperties, "borderInsideVert", createJSONfromCTBorder(tcBorder.getInsideV()));
            }

            // only create cell fill color if it differ from the table fill color
            final CTShd shd = cellProperties.getShd();
            boolean createCellFill = true;
            if(table!=null&&shd!=null) {
                final TblPr tblPr = table.getTblPr();
                if(tblPr!=null) {
                    final CTShd tblShd = tblPr.getShd();
                    if(tblShd!=null) {
                        if((tblShd.getVal() == shd.getVal())
                            && Utils.equals(tblShd.getFill(), shd.getFill())
                            && (tblShd.getThemeColor() == shd.getThemeColor())
                            && (tblShd.getThemeFill() == shd.getThemeFill())
                            && Utils.equals(tblShd.getThemeFillShade(), shd.getThemeFillShade())
                            && Utils.equals(tblShd.getThemeFillTint(), shd.getThemeFillTint())
                            && Utils.equals(tblShd.getThemeShade(), shd.getThemeShade())
                            && Utils.equals(tblShd.getThemeTint(), shd.getThemeTint()))

                            createCellFill = false;
                    }
                }
            }
            if(createCellFill)
                Commons.jsonPut(jsonCellProperties, "fillColor", Utils.createFillColor(shd));

            TcMar tcMar = cellProperties.getTcMar();
            if(tcMar!=null) {
                Commons.jsonPut(jsonCellProperties, "paddingLeft", createJSONFromTblWidth(tcMar.getLeft(), 0, 0));
                Commons.jsonPut(jsonCellProperties, "paddingTop", createJSONFromTblWidth(tcMar.getTop(), 0, 0));
                Commons.jsonPut(jsonCellProperties, "paddingRight", createJSONFromTblWidth(tcMar.getRight(), 0, 0));
                Commons.jsonPut(jsonCellProperties, "paddingBottom", createJSONFromTblWidth(tcMar.getBottom(), 0, 0));
            }
        }
        if(jsonCellProperties!=null&&!jsonCellProperties.isEmpty()) {
        	attrs.put("cell", jsonCellProperties);
        }
    }

    private static TcBorders getTblBorders(ObjectFactory objectFactory, TcPrInner tcPr) {
        TcBorders tcBorders = tcPr.getTcBorders();
        if(tcBorders==null) {
            tcBorders = objectFactory.createTcPrInnerTcBorders();
            tcBorders.setParent(tcPr);
            tcPr.setTcBorders(tcBorders);
        }
        return tcBorders;
    }

    private static TblBorders getTblBorders(ObjectFactory objectFactory, CTTblPrBase tblPr) {
        TblBorders tblBorders = tblPr.getTblBorders();
        if(tblBorders==null) {
            tblBorders = objectFactory.createTblBorders();
            tblBorders.setParent(tblPr);
            tblPr.setTblBorders(tblBorders);
        }
        return tblBorders;
    }

    private static TcMar getTblCellMar(ObjectFactory objectFactory, TcPrInner tcPr) {
        TcMar tcMar = tcPr.getTcMar();
        if(tcMar==null) {
            tcMar = objectFactory.createTcMar();
            tcMar.setParent(tcPr);
            tcPr.setTcMar(tcMar);
        }
        return tcMar;
    }

    private static CTTblCellMar getTblCellMar(ObjectFactory objectFactory, CTTblPrBase tblPr) {
        CTTblCellMar tblCellM = tblPr.getTblCellMar();
        if(tblCellM==null) {
            tblCellM = objectFactory.createCTTblCellMar();
            tblCellM.setParent(tblPr);
            tblPr.setTblCellMar(tblCellM);
        }
        return tblCellM;
    }

    private static CTBorder createCTBorderFromJSON(OperationDocument operationDocument, ObjectFactory objectFactory, Object parent, Object borderProperties )
        throws JSONException {

        CTBorder ctBorder = null;
        if(borderProperties instanceof JSONObject) {
             ctBorder = objectFactory.createCTBorder();
            ctBorder.setParent(parent);
            Iterator<String> keys = ((JSONObject)borderProperties).keys();
            while(keys.hasNext()) {
                String attr = keys.next();
                Object value = ((JSONObject)borderProperties).get(attr);
                if(attr.equals("style")) {
                    if(value instanceof String) {
                        STBorder borderStyle = STBorder.SINGLE;
                        if(((String)value).equals("none"))
                            borderStyle = STBorder.NONE;
                        else if (((String)value).equals("single"))
                            borderStyle = STBorder.SINGLE;
                        else if (((String)value).equals("none"))
                            borderStyle = STBorder.NONE;
                        else if (((String)value).equals("double"))
                            borderStyle = STBorder.DOUBLE;
                        else if (((String)value).equals("dotted"))
                            borderStyle = STBorder.DOTTED;
                        else if (((String)value).equals("dashed"))
                            borderStyle = STBorder.DASHED;
                        else if (((String)value).equals("outset"))
                            borderStyle = STBorder.OUTSET;
                        else if (((String)value).equals("inset"))
                            borderStyle = STBorder.INSET;
                        ctBorder.setVal(borderStyle);
                    }
                    else
                        ctBorder.setVal(STBorder.NONE);
                } else if(attr.equals("width")) {
                    if(value instanceof Integer) {
                        int val = ((Integer)value + 2 )* 576 / 2540;
                        ctBorder.setSz(BigInteger.valueOf(val));
                    }
                    else
                        ctBorder.setSz(null);
                } else if(attr.equals("color")) {
                    if(value instanceof JSONObject) {
                        Utils.initBorderColorFromJSONColor(operationDocument, (JSONObject)value, ctBorder);
                    }
                    else
                        ctBorder.setColor(null);
                }
            }
        }
        return ctBorder;
    }

    private static JSONObject createJSONfromCTBorder(CTBorder tblBorder)
        throws JSONException {

        if(tblBorder==null)
            return null;
        JSONObject jsonBorder = new JSONObject();
        STBorder borderStyle = tblBorder.getVal();
        if(borderStyle!=null) {
            String style = "single";
            if(borderStyle==STBorder.NONE)
                style = "none";
            else if(borderStyle==STBorder.NIL)
                style = "none";
            else if(borderStyle==STBorder.SINGLE)
                style = "single";
            else if(borderStyle==STBorder.DOUBLE)
                style = "double";
            else if(borderStyle==STBorder.DOTTED)
                style = "dotted";
            else if(borderStyle==STBorder.DASHED)
                style = "dashed";
            else if(borderStyle==STBorder.OUTSET)
                style = "outset";
            else if(borderStyle==STBorder.INSET)
                style = "inset";

            // TODO... each ooxml border style needs to be mapped to a corresponding html style... now they are all mapped to single border

            jsonBorder.put("style", style);
        }

        // creating border color
        Commons.jsonPut(jsonBorder, "color", Utils.createColor(tblBorder.getThemeColor(), tblBorder.getColor()));

        // creating border with
        BigInteger sz = tblBorder.getSz();
        if(sz!=null){
            int dv = (int)((sz.doubleValue() * 2540 + 288.)/ 576);
            jsonBorder.put("width", dv );    // 1/8 Pt to 1/100mm
        }
        return jsonBorder;
    }

    private static TblWidth createTblWidthFromJSON(ObjectFactory objectFactory, Object parent, Object jsonWidth) {

        TblWidth tblWidth = null;
        if (jsonWidth instanceof Integer||jsonWidth instanceof String) {

            tblWidth = objectFactory.createTblWidth();
            tblWidth.setParent(parent);

            if(jsonWidth instanceof Integer) {
                tblWidth.setType("dxa");
                tblWidth.setW(BigInteger.valueOf((Integer)jsonWidth * 72 * 20 / 2540));
            }
            else if (((String)jsonWidth).equals("auto")) {
                tblWidth.setType("pct");
                tblWidth.setW(BigInteger.valueOf(5000));
            }
        }
        return tblWidth;
    }

    private static Object createJSONFromTblWidth(TblWidth tblWidth, long referenceWidth, int firstRowWidth) {

        if(tblWidth==null)
            return null;
        Object jsonObject = null;

        String tableWidthType=null;     // default = "dxa"; (1/20 point)
                                        //           "auto"
                                        //           "pct"  (if percent sign is present... then val W represents whole percent points,
                                        //                   otherwise ????)
                                        //           "nil"  (zero width)
        int tableWidth = 0;             // default = 0;

        tableWidthType = tblWidth.getType();
        BigInteger bigInt = tblWidth.getW();
        if(bigInt!=null)
            tableWidth = bigInt.intValue();

        if ((tableWidthType != null) && (tableWidthType.equals("auto")) && (tableWidth == 0) && (firstRowWidth > 0)) {
            if(referenceWidth < firstRowWidth)
                jsonObject = new String("auto");
            else
                jsonObject = new Integer(firstRowWidth);  // using the width, that was calculated from all cells in the first row
        }
        else if ((tableWidthType!=null&&tableWidthType.equals("auto"))) {
            jsonObject = new String("auto");
        }
        else if (tableWidthType!=null&&tableWidthType.equals("pct")) {   // relative to page width excluding margins ... 5000==100%
            if(referenceWidth!=0) {
                tableWidth = (int)(referenceWidth * tableWidth / 5000);
                jsonObject = new Integer(tableWidth);
            }
            else {
                // we do not have a referenceWidth, percentage to absolute width can't be calculated, so we set width = auto here...
                jsonObject = new String("auto");
            }
        }
        else { // default: if(tableWidthType==null||tableWidthType.equals("dxa"))
            tableWidth = tableWidth * 2540 / 72 / 20;
            jsonObject = new Integer(tableWidth);
        }
        return jsonObject;
    }
}
