/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.docx.operations;

import java.math.BigInteger;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;

import javax.xml.bind.JAXBException;

import org.apache.commons.logging.Log;
import org.docx4j.IndexedNode;
import org.docx4j.IndexedNodeList;
import org.docx4j.XmlUtils;
import org.docx4j.jaxb.Context;
import org.docx4j.openpackaging.exceptions.InvalidFormatException;
import org.docx4j.openpackaging.exceptions.PartUnrecognisedException;
import org.docx4j.openpackaging.packages.WordprocessingMLPackage;
import org.docx4j.openpackaging.parts.WordprocessingML.MainDocumentPart;
import org.docx4j.openpackaging.parts.WordprocessingML.NumberingDefinitionsPart;
import org.docx4j.wml.BooleanDefaultTrue;
import org.docx4j.wml.Br;
import org.docx4j.wml.CTLanguage;
import org.docx4j.wml.CTMarkup;
import org.docx4j.wml.CTShd;
import org.docx4j.wml.CTSimpleField;
import org.docx4j.wml.CTTabStop;
import org.docx4j.wml.CTTblPrBase;
import org.docx4j.wml.CTTblStylePr;
import org.docx4j.wml.ContentAccessor;
import org.docx4j.wml.DocDefaults;
import org.docx4j.wml.DocDefaults.RPrDefault;
import org.docx4j.wml.IText;
import org.docx4j.wml.JcEnumeration;
import org.docx4j.wml.Lvl;
import org.docx4j.wml.NumFmt;
import org.docx4j.wml.NumberFormat;
import org.docx4j.wml.Numbering;
import org.docx4j.wml.Numbering.AbstractNum;
import org.docx4j.wml.Numbering.Num;
import org.docx4j.wml.Numbering.Num.AbstractNumId;
import org.docx4j.wml.ObjectFactory;
import org.docx4j.wml.P;
import org.docx4j.wml.PPr;
import org.docx4j.wml.PPrBase.Ind;
import org.docx4j.wml.CTSettings;
import org.docx4j.wml.R;
import org.docx4j.wml.RPr;
import org.docx4j.wml.STBrType;
import org.docx4j.wml.STTblStyleOverrideType;
import org.docx4j.wml.Style;
import org.docx4j.wml.Styles;
import org.docx4j.wml.Tabs;
import org.docx4j.wml.Tbl;
import org.docx4j.wml.TblGrid;
import org.docx4j.wml.TblPr;
import org.docx4j.wml.Tc;
import org.docx4j.wml.TcPr;
import org.docx4j.wml.Text;
import org.docx4j.wml.Tr;
import org.docx4j.wml.TrPr;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.jvnet.jaxb2_commons.ppp.Child;

import com.openexchange.log.LogFactory;
import com.openexchange.office.ooxml.docx.OperationDocument;
import com.openexchange.office.ooxml.docx.tools.Component;
import com.openexchange.office.ooxml.docx.tools.Component.FldSimpleComponent;
import com.openexchange.office.ooxml.docx.tools.Component.MainContext;
import com.openexchange.office.ooxml.docx.tools.Component.ParagraphComponent;
import com.openexchange.office.ooxml.docx.tools.Component.TableComponent;
import com.openexchange.office.ooxml.docx.tools.Component.TcComponent;
import com.openexchange.office.ooxml.docx.tools.Component.TextComponent;
import com.openexchange.office.ooxml.docx.tools.Component.TextRun_Base;
import com.openexchange.office.ooxml.docx.tools.Component.TrComponent;
import com.openexchange.office.ooxml.docx.tools.Paragraph;
import com.openexchange.office.ooxml.docx.tools.Table;
import com.openexchange.office.ooxml.docx.tools.TextUtils;
import com.openexchange.office.ooxml.docx.tools.Utils;

public class ApplyOperationHelper extends com.openexchange.office.ooxml.operations.ApplyOperationHelper {

    protected final static Log log = LogFactory.getLog(ApplyOperationHelper.class);

    private final OperationDocument operationDocument;
    private final WordprocessingMLPackage wordMLPackage;
    private final MainDocumentPart documentPart;
    private final ObjectFactory objectFactory;
    private final Map<String, String> conditionalTableAttrNames;

    public ApplyOperationHelper(OperationDocument _operationDocument) {
        super();

        operationDocument = _operationDocument;
        wordMLPackage = _operationDocument.getPackage();
        documentPart = wordMLPackage.getMainDocumentPart();
        objectFactory = Context.getWmlObjectFactory();

        conditionalTableAttrNames = new HashMap<String, String>();
        conditionalTableAttrNames.put("wholeTable", "wholeTable");
        conditionalTableAttrNames.put("firstRow", "firstRow");
        conditionalTableAttrNames.put("lastRow", "lastRow");
        conditionalTableAttrNames.put("firstCol", "firstCol");
        conditionalTableAttrNames.put("lastCol", "lastCol");
        conditionalTableAttrNames.put("band1Vert", "band1Vert");
        conditionalTableAttrNames.put("band2Vert", "band2Vert");
        conditionalTableAttrNames.put("band1Hor", "band1Horz");
        conditionalTableAttrNames.put("band2Hor", "band2Horz");
        conditionalTableAttrNames.put("northEastCell", "neCell");
        conditionalTableAttrNames.put("northWestCell", "nwCell");
        conditionalTableAttrNames.put("southEastCell", "seCell");
        conditionalTableAttrNames.put("southWestCell", "swCell");
    }

    @Override
    public OperationDocument getOperationDocument() {
        return operationDocument;
    }

    /**
     *
     * After applying operations to the document, this method is called
     * once to fix the output so that it fulfills the OOXML specification.
     * (E.g: A Tc element must have at least one P. In our operation model
     * we create Tc and P elements independently, so there might exist
     * a Tc without P... this is fixed here)
     *
     * additionally we are also creating unique ids for each changeTrack
     *  ... wtf, why is this id required
     *
     * @author Sven Jacobi <sven.jacobi@open-xchange.com>
     */
    public void postFixNonConformance() {

    	// TODO: applyNewIds should only be called if changes have been modified
    	applyNewIds(documentPart, new AtomicInteger(1000000));

    	Component component = MainContext.getNextComponent(0, new IndexedNode<Object>(documentPart), -1);
        while(component!=null) {
            if(component instanceof TableComponent)
                postFixTableComponent((TableComponent)component);
            component = component.getNextComponent();
        }
    }
    
    private void applyNewIds(ContentAccessor a, AtomicInteger nextId) {
    	for(Object o:a.getContent()) {
    		if (o instanceof CTMarkup) {
    			((CTMarkup)o).setId(BigInteger.valueOf(nextId.incrementAndGet()));
    		}
    		if(o instanceof ContentAccessor) {
    			applyNewIds((ContentAccessor)o, nextId);
    		}
    	}
    }

    private void postFixTableComponent(TableComponent tableComponent) {
    	Component component = tableComponent.getChildComponent();
        while(component!=null) {
            if(component instanceof TrComponent)
                postFixTrComponent((TrComponent)component);
            component = component.getNextComponent();
        }
    }

    private void postFixTrComponent(TrComponent trComponent) {
        Component component = trComponent.getChildComponent();
        while(component!=null) {
            if(component instanceof TcComponent)
                postFixTcComponent((TcComponent)component);
            component = component.getNextComponent();
        }
    }

    private void postFixTcComponent(TcComponent tcComponent) {
    	Component component = tcComponent.getChildComponent();
        boolean hasP = false;
        while(component!=null) {
            if(component instanceof TableComponent) {
                postFixTableComponent((TableComponent)component);
                hasP = false;                                           // :-( there must be a P within Tc... and there must be a P behind Tbl
            }
            else if(component instanceof ParagraphComponent)
                hasP = true;
            component = component.getNextComponent();
        }
        if(!hasP) {
            P p = objectFactory.createP();
            p.setRsidP("47110815");
            p.setParent(tcComponent.getObject());
            ((Tc)tcComponent.getObject()).getContent().add(p);
        }
    }

    public void delete(JSONArray start, JSONArray end)
        throws JSONException, InvalidFormatException, JAXBException, PartUnrecognisedException {

        if(start==null||start.length()==0)
            return;

        int startComponent = start.getInt(start.length()-1);
        int endComponent = startComponent;
        if(end!=null) {
            if(end.length()!=start.length())
                return;
            endComponent = end.getInt(end.length()-1);
        }
        Component component = TextUtils.getComponent(documentPart, start, start.length());
		component.splitStart(startComponent);
        final int startIndex = component.getContextChildNode().getIndex();
        while(component.getNextComponentNumber()<=endComponent) {
        	component = component.getNextComponent();
        }
		component.splitEnd(endComponent);
        final IndexedNodeList<Object> content = (IndexedNodeList<Object>)((ContentAccessor)component.getContextObject()).getContent();
        content.removeNodes(startIndex, (component.getContextChildNode().getIndex()-startIndex)+1);
    }

    /**
     * moving components, only DrawingComponents are supported
     *
     * @param startPosition
     * @param endPosition
     *
     */

    public void move(JSONArray startPosition, JSONArray endPosition, JSONArray toPosition)
        throws JSONException {

        if(startPosition.length()!=endPosition.length())
            throw new JSONException("ooxml export: move operation, size of startPosition != size of endPosition");

        int startIndex = startPosition.getInt(startPosition.length()-1);
        int endIndex = endPosition.getInt(startPosition.length()-1);
        int toIndex = toPosition.getInt(toPosition.length()-1);

        if((endIndex - startIndex) < 0)
            throw new JSONException("ooxml export: move operation, component count < 1");

        Component sourceParent = null;
        Component toParent = null;

        // check if split is necessary (only needed if parent is a paragraph, if length == 1 then the parent is always root)
        sourceParent = TextUtils.getComponent(documentPart, startPosition, startPosition.length()-1);
        if(sourceParent instanceof ParagraphComponent) {
        	Component c = TextUtils.getComponentNumber(sourceParent.getChildComponent(), startIndex);
        	if(c!=null) {
        		c.splitStart(startIndex);
        	}
        	c = TextUtils.getComponentNumber(c, endIndex+1);
        	if(c!=null) {
        		c.splitEnd(endIndex);
        	}
        }
        toParent = TextUtils.getComponent(documentPart, toPosition, toPosition.length()-1);
        if(toParent instanceof ParagraphComponent) {
        	Component c = TextUtils.getComponentNumber(toParent.getChildComponent(), toIndex);
        	if(c!=null) {
        		c.splitStart(toIndex);
        	}
        }

        Component startComponent = TextUtils.getComponentNumber(sourceParent.getChildComponent(), startIndex);
        if(startComponent==null)
        	throw new JSONException("ooxml export: move operation, could not get component for start position");

        final int parentStartIndex = startComponent.getNode().getIndex();
        int parentEndIndex = parentStartIndex;
        
        while(startComponent!=null&&startComponent.getNextComponentNumber()<=endIndex) {
            parentEndIndex = startComponent.getNode().getIndex();
            startComponent = startComponent.getNextComponent();
        }

        final IndexedNodeList<Object> sourceContent = (IndexedNodeList<Object>)((ContentAccessor)sourceParent.getObject()).getContent();
        final IndexedNodeList<Object> destContent = (IndexedNodeList<Object>)((ContentAccessor)toParent.getObject()).getContent();

        final Component toComponent = TextUtils.getComponentNumber(toParent.getChildComponent(), toIndex);
        final int destIndex = toComponent!=null?toComponent.getContextChildNode().getIndex():destContent.size();
    	IndexedNodeList.moveNodes(sourceContent, destContent, parentStartIndex, destIndex, (parentEndIndex - parentStartIndex)+1, toParent.getObject());
    }

    public void insertParagraph(JSONArray start, JSONObject attrs)
        throws JAXBException, InvalidFormatException, PartUnrecognisedException, JSONException {

    	final ParagraphComponent paragraphComponent = new ParagraphComponent(objectFactory.createP());
    	if(TextUtils.insertComponent(documentPart, start, paragraphComponent)) {
            if(attrs!=null) {
            	paragraphComponent.setAttributes(operationDocument, attrs);
            }
    	}
    }

    public void splitParagraph(JSONArray start)
        throws JSONException {

        final Component o = TextUtils.getComponent(documentPart, start, start.length()-1);
        if (o instanceof ParagraphComponent) {
            TextUtils.splitParagraph((ParagraphComponent)o, start.getInt(start.length()-1));
        }
    }

    public void mergeParagraph(JSONArray start) {

    	Component o = TextUtils.getComponent(documentPart, start, start.length());
        if(o instanceof ParagraphComponent) {
            TextUtils.mergeParagraph((ParagraphComponent)o);
        }
    }

    public void insertText(JSONArray start, String text, JSONObject attrs)
        throws JAXBException, JSONException, InvalidFormatException, PartUnrecognisedException {

    	if(text.length()>0) {
    		final Component paragraphComponent = TextUtils.getComponent(documentPart, start, start.length()-1);
        	final P paragraph = (P)paragraphComponent.getObject();
        	final IndexedNodeList<Object> paragraphContent = paragraph.getContent();
            final int textPosition = start.getInt(start.length()-1);

            IText t = null;
            Component childComponent = paragraphComponent.getChildComponent();
            TextRun_Base textRun_Base = null;

            if(childComponent!=null) {
            	if(textPosition>0) {
            		childComponent = TextUtils.getComponentNumber(childComponent, textPosition-1);
            	}
                // check if the character could be inserted into an existing text:
                if(childComponent instanceof TextComponent) {
                    t = (IText)childComponent.getObject();
                    final StringBuffer s = new StringBuffer(t.getValue());
                    s.insert(textPosition-((TextComponent)childComponent).getComponentNumber(), text);
                    t.setValue(s.toString());
                    textRun_Base = (TextRun_Base)childComponent;
                }
                else {
                	t = Context.getWmlObjectFactory().createText();
                    t.setValue(text);
                	if(childComponent instanceof TextRun_Base) {
                    	// we can insert the text into a existing run
                    	final TextRun_Base runBase = (TextRun_Base)childComponent;
                    	final R oldRun = runBase.getTextRun();
                        t.setParent(oldRun);
                        final int i = runBase.getNode().getIndex();
                        oldRun.getContent().add(textPosition>0?i+1:i, t);
                	}
                	else if(childComponent instanceof FldSimpleComponent) {
                    	// new run needs to be created ... (simpleField...)
                        final R newRun = Context.getWmlObjectFactory().createR();
                        newRun.setParent(paragraphComponent.getObject());
                        final int i = childComponent.getContextChildNode().getIndex();
                        paragraphContent.add(textPosition>0?i+1:i, newRun);
                    	final R run = ((FldSimpleComponent)childComponent).getTextRun();
                    	final RPr rPr = run.getRPr();
                    	if(rPr!=null) {
                			final RPr newRPr = XmlUtils.deepCopy(rPr);
                			if(newRPr!=null) {
                				newRun.setRPr(newRPr);
                				newRPr.setParent(newRun);
                			}
                    	}
                        t.setParent(newRun);
                        newRun.getContent().add(t);
                	}
                    if(textPosition>0) {
                        textRun_Base = (TextRun_Base)childComponent.getNextComponent();
                    }
                    else {
                        textRun_Base = (TextRun_Base)paragraphComponent.getChildComponent();
                    }
                }
            }
            else {
                // the paragraph is empty, we have to create R and its text
                final R newRun = Context.getWmlObjectFactory().createR();
                newRun.setParent(paragraphComponent.getObject());
                paragraphContent.add(newRun);
                if(paragraph.getPPr()!=null) {
                    final RPr rPr = TextUtils.cloneParaRPrToRPr(paragraph.getPPr().getRPr());
                    if(rPr!=null) {
                        rPr.setParent(newRun);
                        newRun.setRPr(rPr);
                    }
                }
                t = Context.getWmlObjectFactory().createText();
                t.setParent(newRun);
                t.setValue(text);
                newRun.getContent().add(t);
                textRun_Base = (TextRun_Base)paragraphComponent.getChildComponent();
            }
			if(t!=null) {
				TextComponent.preserveSpace(t);
			}
            if(attrs!=null||textRun_Base.getCT()!=Component.CT_NONE) {
            	textRun_Base.splitStart(textPosition);
            	textRun_Base.splitEnd(textPosition+text.length()-1);
            	TextUtils.setTextRunAttributes(operationDocument, attrs, textRun_Base, true);
            }
        }
    }

    public void insertTextRunChild(JSONArray start, JSONObject attrs, Child newTextRunChild)
    	throws JSONException, InvalidFormatException, PartUnrecognisedException, JAXBException {

    	final Component paragraphComponent = TextUtils.getComponent(documentPart, start, start.length()-1);
    	if(!(paragraphComponent instanceof ParagraphComponent))
    		return;

    	final P paragraph = (P)paragraphComponent.getObject();
    	final IndexedNodeList<Object> paragraphContent = paragraph.getContent();
    	final int componentNumber = start.getInt(start.length()-1);

    	Component childComponent = paragraphComponent.getChildComponent();
    	if(childComponent!=null) {
	    	if(componentNumber>0) {
	    		childComponent = TextUtils.getComponentNumber(childComponent, componentNumber-1);
	    		childComponent.splitEnd(componentNumber-1);
	    	}
	    	if(childComponent instanceof TextRun_Base) {
	    		final R run = ((TextRun_Base)childComponent).getTextRun();
	    		final int index = ((TextRun_Base)childComponent).getNode().getIndex();
	    		newTextRunChild.setParent(run);
	    		run.getContent().add(componentNumber>0?index+1:index, newTextRunChild);
	    	}
	    	else if(childComponent instanceof FldSimpleComponent) {
	            final R newRun = Context.getWmlObjectFactory().createR();
	            newRun.setParent(paragraph);
	            paragraphContent.add(newRun);
	            final int index = childComponent.getContextChildNode().getIndex();
	            paragraphContent.add(componentNumber>0?index+1:index, newRun);
	            final RPr rPr = ((FldSimpleComponent)childComponent).getTextRun().getRPr();
	            if(rPr!=null) {
	            	final RPr newRPr = XmlUtils.deepCopy(rPr);
	            	newRPr.setParent(newRun);
	            	newRun.setRPr(newRPr);
	            }
	    	}
    	}
    	else {
            final R newRun = Context.getWmlObjectFactory().createR();
            newRun.setParent(paragraph);
            paragraphContent.add(newRun);
            newTextRunChild.setParent(newRun);
            newRun.getContent().add(newTextRunChild);
            if(paragraph.getPPr()!=null) {
                final RPr rPr = TextUtils.cloneParaRPrToRPr(paragraph.getPPr().getRPr());
                if(rPr!=null) {
                    rPr.setParent(newRun);
                    newRun.setRPr(rPr);
                }
            }
    	}
        if(componentNumber>0) {
        	childComponent = childComponent.getNextComponent();
        }
        else {
        	childComponent = paragraphComponent.getChildComponent();
        }
    	if(attrs!=null||((TextRun_Base)childComponent).getCT()!=Component.CT_NONE) {
    		childComponent.splitStart(componentNumber);
    		childComponent.splitEnd(componentNumber);
        	TextUtils.setTextRunAttributes(operationDocument, attrs, (TextRun_Base)childComponent, true);
    	}
    }

    public void insertTab(JSONArray start, JSONObject attrs)
        throws JAXBException, InvalidFormatException, PartUnrecognisedException, JSONException {

        insertTextRunChild(start, attrs, objectFactory.createRTab());
    }

    public void insertHardBreak(JSONArray start, String type, JSONObject attrs)
        throws JAXBException, InvalidFormatException, PartUnrecognisedException, JSONException {

    	final Br br = objectFactory.createBr();
    	if(type!=null) {
    		if(type.equals("page")) {
    			br.setType(STBrType.PAGE);
    		}
    		else if(type.equals("column")) {
    			br.setType(STBrType.COLUMN);
    		}
    	}
        insertTextRunChild(start, attrs, br);
    }

    public void insertDrawing(JSONArray start, String type, JSONObject attrs)
        throws JAXBException, InvalidFormatException, PartUnrecognisedException, JSONException {

        insertTextRunChild(start, attrs, objectFactory.createDrawing());
    }

    public void insertTable(JSONArray start, JSONObject attrs)
        throws JSONException {

    	final TableComponent tableComponent = new TableComponent(objectFactory.createTbl());
    	if(TextUtils.insertComponent(documentPart, start, tableComponent)) {
            final TblPr tblPr = Table.getTableProperties((Tbl)tableComponent.getObject());

            // turning on each conditional Style, each of our conditional styles has to be used by default,
            // (they can be deactivated by using the exclude property)

            // TODO: tblLook changed in docx4j3.0.1...
            Table.initLookAndRowBandSize(tblPr);
            if(attrs!=null) {
            	tableComponent.setAttributes(operationDocument, attrs);
            }
    	}
    }

    public void splitTable(JSONArray start)
    	throws JSONException {

    	final int row = start.getInt(start.length()-1);
    	final TableComponent tableComponent = (TableComponent)TextUtils.getComponent(documentPart, start, start.length()-1);
    	final Tbl sourceTbl = (Tbl)tableComponent.getObject();

    	// to clone the table we temporarily remove the content
    	final IndexedNodeList<Object> sourceTblContent = sourceTbl.getContent();
    	sourceTbl.setContent(null);
    	final Tbl destTbl = XmlUtils.deepCopy(sourceTbl);
    	sourceTbl.setContent(sourceTblContent);
    	final ContentAccessor sourceTblParent = (ContentAccessor)tableComponent.getContextObject();
    	destTbl.setParent(sourceTblParent);
    	sourceTblParent.getContent().add(tableComponent.getNode().getIndex()+1, destTbl);
    	IndexedNodeList.moveNodes(sourceTblContent, destTbl.getContent(), row, 0, sourceTblContent.size()-row, destTbl);
    }

    public void mergeTable(JSONArray start) {
    	final TableComponent destTable = (TableComponent)TextUtils.getComponent(documentPart, start, start.length());
    	final TableComponent sourceTable = (TableComponent)destTable.getNextComponent();
    	final IndexedNodeList<Object> sourceContent = ((Tbl)sourceTable.getObject()).getContent();
    	final IndexedNodeList<Object> destContent = ((Tbl)destTable.getObject()).getContent();
    	IndexedNodeList.moveNodes(sourceContent, destContent, 0, destContent.size(), sourceContent.size(), destTable.getObject());
    	((ContentAccessor)((Child)sourceTable.getObject()).getParent()).getContent().remove(sourceTable.getNode().getIndex());
    }

    public void insertRows(JSONArray start, int count, boolean insertDefaultCells, int referenceRow, JSONObject attrs)
        throws JSONException {

    	TableComponent tableComponent = (TableComponent)TextUtils.getComponent(documentPart, start, start.length()-1);
    	if(tableComponent==null) {
    		return;
    	}
    	final int destinationRowComponent = start.getInt(start.length()-1);
    	final TrComponent trReferenceRow = referenceRow!=-1 ? (TrComponent)TextUtils.getComponentNumber(tableComponent.getChildComponent(), referenceRow) : null;
    	final Tbl tbl = (Tbl)tableComponent.getObject();
        final IndexedNodeList<Object> tblContent = tbl.getContent();
        final Component oldTr = TextUtils.getComponentNumber(tableComponent.getChildComponent(), destinationRowComponent);
        int destinationIndex = oldTr!=null ? oldTr.getNode().getIndex() : tblContent.size();

        for(int i = 0; i<count; i++) {
            final Tr tr = objectFactory.createTr();
            tr.setParent(tbl);
            tblContent.add(destinationIndex++, tr);
            if(insertDefaultCells) {
                JSONArray tableGrid = Table.getTableGrid(tbl.getTblGrid());
                if(tableGrid!=null) {
                    for(int j = 0; j<tableGrid.length(); j++) {
                        Tc tc = objectFactory.createTc();
                        tc.setParent(tr);
                        tr.getContent().add(tc);
                    }
                }
            }
            else if(trReferenceRow!=null) {       // we try to create the new row from the referenceRow
                final Tr trReference = (Tr)trReferenceRow.getObject();
                final TrPr sourceTrPr = trReference.getTrPr();
                if(sourceTrPr!=null) {
                    final TrPr newTrPr = XmlUtils.deepCopy(sourceTrPr);
                    newTrPr.setParent(tr);
                    tr.setTrPr(newTrPr);
                }
                TcComponent tcReferenceComponent = (TcComponent)trReferenceRow.getChildComponent();
                while(tcReferenceComponent!=null) {
                    final Tc tcReference = (Tc)tcReferenceComponent.getObject();
                    final Tc newTc = objectFactory.createTc();
                    newTc.setParent(tr);
                    tr.getContent().add(newTc);
                    TcPr tcPrReference = tcReference.getTcPr();
                    if(tcPrReference!=null) {
                        TcPr newTcPr = XmlUtils.deepCopy(tcPrReference);
                        newTcPr.setParent(newTc);
                        newTc.setTcPr(newTcPr);
                    }
                    tcReferenceComponent = (TcComponent)tcReferenceComponent.getNextComponent();
                }
            }
        }
        if(attrs!=null) {
        	Component c = TextUtils.getComponentNumber(tableComponent.getChildComponent(), destinationRowComponent);
	        for(int i=0; i<count; i++) {
	        	c.setAttributes(operationDocument, attrs);
	        	c = c.getNextComponent();
	        }
        }
    }

    public void insertCells(JSONArray start, int count, JSONObject attrs)
        throws JSONException {

    	final TrComponent trComponent = (TrComponent)TextUtils.getComponent(documentPart, start, start.length()-1);
        if (trComponent!=null) {
        	final int destinationCellComponent = start.getInt(start.length()-1);
        	Component c = TextUtils.getComponentNumber(trComponent.getChildComponent(), destinationCellComponent);
        	final Tr tr = (Tr)trComponent.getObject();
            final IndexedNodeList<Object> trContent = tr.getContent();
            final int destinationIndex = c!=null ? c.getNode().getIndex() : trContent.size();
            for (int i=0; i<count; i++) {
                final Tc tc = objectFactory.createTc();
                tc.setParent(tr);
                trContent.add(destinationIndex, tc);
            }
            if(attrs!=null) {
                c = TextUtils.getComponentNumber(trComponent.getChildComponent(), destinationCellComponent);
                for(int i=0; i<count; i++) {
                	c.setAttributes(operationDocument, attrs);
                	c = c.getNextComponent();
                }
            }
        }
    }

    public void insertColumn(JSONArray start, JSONArray tableGrid, int gridPosition, String insertMode)
        throws JSONException {

        Component o = TextUtils.getComponent(documentPart, start, start.length());
        if (o instanceof TableComponent) {

            boolean before = insertMode.equals("before");
            TrComponent trComponent = (TrComponent)o.getChildComponent();
            while(trComponent!=null) {
                TcComponent tcReference = (TcComponent)trComponent.getChildComponent();
                TcComponent destination = null;
                while(tcReference!=null) {
                    if(gridPosition>=tcReference.getGridPosition()&&gridPosition<tcReference.getNextGridPosition()) {
                        destination = tcReference;
                        break;
                    }
                    tcReference = (TcComponent)tcReference.getNextComponent();
                }
                List<Object> rowContent = ((Tr)trComponent.getObject()).getContent();
                int destinationIndex = destination==null?rowContent.size():before?destination.getNode().getIndex():destination.getNode().getIndex()+1;
                Tc tc = objectFactory.createTc();
                tc.setParent(trComponent.getObject());
                if(tcReference!=null) {
                    TcPr referenceTcPr = ((Tc)tcReference.getObject()).getTcPr();
                    if(referenceTcPr!=null) {
                        TcPr newTcPr = XmlUtils.deepCopy(referenceTcPr);
                        newTcPr.setParent(tc);
                        tc.setTcPr(newTcPr);
                    }
                }
                rowContent.add(destinationIndex, tc);
                trComponent = (TrComponent)trComponent.getNextComponent();
            }
            final Tbl tbl = (Tbl)o.getObject();
            TblGrid tblGrid = tbl.getTblGrid();
            if(tblGrid==null) {
            	tblGrid = Context.getWmlObjectFactory().createTblGrid();
            	tblGrid.setParent(tbl);
            	tbl.setTblGrid(tblGrid);
            }
            Table.setTableGrid(getOperationDocument(), tbl, tblGrid, tableGrid);
        }
    }

    public void deleteColumns(JSONArray position, int gridStart, int gridEnd)
        throws JSONException {

    	Component o = TextUtils.getComponent(documentPart, position, position.length());
        if (o instanceof TableComponent) {
        	Component trComponent = o.getChildComponent();
            while(trComponent!=null) {
                int startIndex = -1;
                int endIndex = -1;
                TcComponent tcComponent = (TcComponent)trComponent.getChildComponent();
                while(tcComponent!=null) {
                    if((tcComponent.getGridPosition()>=gridStart&&tcComponent.getGridPosition()<=gridEnd)||
                        (tcComponent.getNextGridPosition()-1>=gridStart&&tcComponent.getNextGridPosition()-1<=gridEnd)){
                        if (startIndex==-1)
                            startIndex = tcComponent.getNode().getIndex();
                        endIndex = tcComponent.getNode().getIndex();
                    }
                    if(tcComponent.getNextGridPosition()>gridEnd)
                        break;
                    tcComponent = (TcComponent)tcComponent.getNextComponent();
                }
                if(startIndex!=-1) {
                    List<Object> rowContent = ((Tr)trComponent.getObject()).getContent();
                    for(int i=endIndex;i>=startIndex;i--) {
                        rowContent.remove(i);
                    }
                }
                trComponent = trComponent.getNextComponent();
            }
            JSONArray tableGrid = Table.getTableGrid(((Tbl)o.getObject()).getTblGrid());
            JSONArray newTableGrid = new JSONArray();
            for(int i=0;i<tableGrid.length();i++) {
                if(i<gridStart||i>gridEnd) {
                    newTableGrid.put(tableGrid.get(i));
                }
            }
            final Tbl tbl = (Tbl)o.getObject();
            TblGrid tblGrid = tbl.getTblGrid();
            if(tblGrid==null) {
            	tblGrid = Context.getWmlObjectFactory().createTblGrid();
            	tblGrid.setParent(tbl);
            	tbl.setTblGrid(tblGrid);
            }
            Table.setTableGrid(getOperationDocument(), tbl, tblGrid, newTableGrid);
        }
        else
            System.out.println("OOX Error: ApplyOperationHelper:deleteColumns: Table requested...");
    }

    public void setAttributes(JSONObject attrs, JSONArray start, JSONArray end)
        throws JAXBException, InvalidFormatException, PartUnrecognisedException, JSONException {

    	if(attrs==null) {
    		return;
    	}
        int startIndex = start.getInt(start.length()-1);
        int endIndex = startIndex;

        if(end!=null) {
            if(end.length()!=start.length())
                return;
            endIndex = end.getInt(end.length()-1);
        }
    	Component component = TextUtils.getComponent(documentPart, start, start.length());
		component.splitStart(startIndex);
        while(component!=null&&component.getComponentNumber()<=endIndex) {
        	if(component.getNextComponentNumber()>=endIndex+1) {
        		component.splitEnd(endIndex);
        	}
        	component.setAttributes(operationDocument, attrs);
            component = component.getNextComponent();
        }
    }

    public void insertField(JSONArray start, String type, String representation, JSONObject attrs)
            throws JAXBException, InvalidFormatException, PartUnrecognisedException, JSONException {

        final Component o = TextUtils.getComponent(documentPart, start, start.length()-1);
        if (o instanceof ParagraphComponent) {

        	final ParagraphComponent paragraphComponent = (ParagraphComponent)o;
        	final P paragraph = (P)paragraphComponent.getObject();
        	final IndexedNodeList<Object> paragraphContent = paragraph.getContent();
            final int fieldPosition = start.getInt(start.length()-1);

            final CTSimpleField ctSimpleField = objectFactory.createCTSimpleField();
            ctSimpleField.setParent(paragraph);
            ctSimpleField.setInstr(type);
            final R simpleFieldRun = objectFactory.createR();
            simpleFieldRun.setParent(ctSimpleField);
            ctSimpleField.getContent().add(simpleFieldRun);
            final Text simpleFieldText = objectFactory.createText();
            simpleFieldText.setValue(representation);
            simpleFieldText.setParent(simpleFieldRun);
            simpleFieldRun.getContent().add(simpleFieldText);

            RPr destRPr = null;
            int simpleFieldIndex = 0;

            Component childComponent = paragraphComponent.getChildComponent();
            if(childComponent!=null) {
            	if(fieldPosition>0) {
            		childComponent = TextUtils.getComponentNumber(childComponent, fieldPosition-1);
            		childComponent.splitEnd(fieldPosition);
            		simpleFieldIndex = childComponent.getContextNode().getIndex()+1;
            	}
            	RPr sourceRPr = null;
            	if(childComponent instanceof TextRun_Base) {
            		sourceRPr = ((TextRun_Base)childComponent).getTextRun().getRPr();
            	}
            	else if(childComponent instanceof FldSimpleComponent) {
            		sourceRPr = ((FldSimpleComponent)childComponent).getTextRun().getRPr();
            	}
            	if(sourceRPr!=null) {
            		destRPr = XmlUtils.deepCopy(sourceRPr);
            	}
            	paragraphContent.add(simpleFieldIndex, ctSimpleField);
            }
            else {
            	if(paragraph.getPPr()!=null) {
            		destRPr = TextUtils.cloneParaRPrToRPr(paragraph.getPPr().getRPr());
            	}
                paragraphContent.add(simpleFieldIndex, ctSimpleField);
            }
    		if(destRPr!=null) {
    			destRPr.setParent(simpleFieldRun);
    			simpleFieldRun.setRPr(destRPr);
    		}
            if(attrs!=null) {
            	if(fieldPosition>0) {
            		childComponent = childComponent.getNextComponent();
            	}
            	else {
            		childComponent = paragraphComponent.getChildComponent();
            	}
            	childComponent.setAttributes(operationDocument, attrs);
            }
        }
    }

    public void insertStyleSheet(String type, String styleId, String styleName, JSONObject attrs, String parentId, Boolean hidden, Integer uipriority, Boolean isDefault)
        throws Exception {

        final Styles styles = getOperationDocument().getStyles(true);
        final List<Style> stylesList = styles.getStyle();

        // removing style if it already exists
        Iterator<Style> styleIter = stylesList.iterator();
        while(styleIter.hasNext()){
            Style aStyle = styleIter.next();
            if ((aStyle.getType().equals(type)) && (aStyle.getStyleId().equals(styleId))) {
                styleIter.remove();
                break;
            }
        };

        // always creating a new style...
        Style style = objectFactory.createStyle();
        stylesList.add(style);

        // style properties
        style.setParent(styles);
        style.setType(type);
        style.setStyleId(styleId);
        if (styleName != null) {
            Style.Name name = objectFactory.createStyleName();
            name.setParent(style);
            name.setVal(styleName);
            style.setName(name);
        }
        if (hidden != null) {
            BooleanDefaultTrue hiddenValue = objectFactory.createBooleanDefaultTrue();
            hiddenValue.setParent(style);
            hiddenValue.setVal(hidden.booleanValue());
            style.setHidden(hiddenValue);
        }
        if (isDefault != null) {
            style.setDefault(isDefault.booleanValue());
        }
        if (uipriority != null) {
            Style.UiPriority uiPrio = objectFactory.createStyleUiPriority();
            uiPrio.setParent(style);
            uiPrio.setVal(BigInteger.valueOf(uipriority.intValue()));
            style.setUiPriority(uiPrio);
        }
        if (parentId != null) {
            Style.BasedOn basedOn = objectFactory.createStyleBasedOn();
            basedOn.setParent(style);
            basedOn.setVal(parentId);
            style.setBasedOn(basedOn);
        }

        // paragraph properties
        JSONObject paragraphProps = attrs.optJSONObject("paragraph");
        if (paragraphProps != null) {
            String nextStyle = paragraphProps.optString("nextStyleId");
            if (nextStyle != null) {
               Style.Next next = objectFactory.createStyleNext();
               next.setParent(style);
               next.setVal(nextStyle);
               style.setNext(next);
            }
            PPr ppr = style.getPPr();
            if (ppr == null) {
                ppr = objectFactory.createPPr();
                ppr.setParent(style);
                style.setPPr(ppr);
            }
            Paragraph.applyParagraphProperties(operationDocument, paragraphProps, ppr);
        }

        // character properties
        JSONObject characterProps = attrs.optJSONObject("character");
        if (characterProps != null) {
            RPr rpr = style.getRPr();
            if (rpr == null) {
                rpr = objectFactory.createRPr();
                rpr.setParent(style);
                style.setRPr(rpr);
            }
            com.openexchange.office.ooxml.docx.tools.Character.applyCharacterProperties(operationDocument, characterProps, rpr);
        }

        if(type.equals("table")) {
            Iterator<String> keys = attrs.keys();
            while(keys.hasNext()) {
                String key = keys.next();
                if(conditionalTableAttrNames.containsKey(key)) {
                    JSONObject conditionalTableProps = attrs.optJSONObject(key);
                    if (conditionalTableProps != null) {
                        String conditionalTableName = conditionalTableAttrNames.get(key);
                        List<CTTblStylePr> conditionalStyles = style.getTblStylePr();
                        CTTblStylePr conditionalStyle = objectFactory.createCTTblStylePr();
                        conditionalStyle.setType(STTblStyleOverrideType.fromValue(conditionalTableName));
                        Iterator<String> tablePropKeys = conditionalTableProps.keys();
                        while(tablePropKeys.hasNext()) {
                            String tablePropKey = tablePropKeys.next();
                            JSONObject tablePropAttrs = conditionalTableProps.optJSONObject(tablePropKey);
                            if(tablePropAttrs!=null) {
                                if(tablePropKey.equals("table")) {
                                    CTTblPrBase tblPrBase = conditionalStyle.getTblPr();
                                    if (tblPrBase==null) {
                                        tblPrBase = objectFactory.createCTTblPrBase();
                                        tblPrBase.setParent(conditionalStyle);
                                        conditionalStyle.setTblPr(tblPrBase);
                                    }
                                    Table.applyTableProperties(operationDocument, tablePropAttrs, null, null, tblPrBase);
                                } else if(tablePropKey.equals("row")) {
                                     TrPr trPr = conditionalStyle.getTrPr();
                                    if (trPr==null) {
                                        trPr = objectFactory.createTrPr();
                                        trPr.setParent(conditionalStyle);
                                        conditionalStyle.setTrPr(trPr);
                                    }
                                    Table.applyRowProperties(tablePropAttrs, trPr);
                                } else if(tablePropKey.equals("cell")) {
                                    TcPr tcPr = conditionalStyle.getTcPr();
                                   if (tcPr==null) {
                                       tcPr = objectFactory.createTcPr();
                                       tcPr.setParent(conditionalStyle);
                                       conditionalStyle.setTcPr(tcPr);
                                   }
                                   Table.applyCellProperties(operationDocument, tablePropAttrs, tcPr);
                                } else if(tablePropKey.equals("paragraph")) {
                                    PPr pPr = conditionalStyle.getPPr();
                                    if (pPr==null) {
                                        pPr = objectFactory.createPPr();
                                        pPr.setParent(conditionalStyle);
                                        conditionalStyle.setPPr(pPr);
                                    }
                                    Paragraph.applyParagraphProperties(operationDocument, tablePropAttrs, pPr);
                                } else if(tablePropKey.equals("character")) {
                                    RPr rPr = conditionalStyle.getRPr();
                                    if (rPr==null) {
                                        rPr = objectFactory.createRPr();
                                        rPr.setParent(conditionalStyle);
                                        conditionalStyle.setRPr(rPr);
                                    }
                                    com.openexchange.office.ooxml.docx.tools.Character.applyCharacterProperties(operationDocument, tablePropAttrs, rPr);
                                }
                            }
                        }

                        // we will not create a whole_table style, because it does not seem to work,
                        // (even the TblLook does not mention the conditional WholeTable style) instead
                        // we will set the properties direct as tblPr
                        if(conditionalStyle.getType()==STTblStyleOverrideType.WHOLE_TABLE) {
                            style.setTblPr(conditionalStyle.getTblPr());
                            style.setTrPr(conditionalStyle.getTrPr());
                            style.setTcPr(conditionalStyle.getTcPr());
                            style.setPPr(conditionalStyle.getPPr());
                            style.setRPr(conditionalStyle.getRPr());

                            // if no shading is used within the TblPr, then we have to copy the shading from
                            // the cell into TblPr (LO does not take care about cell shading, so we have to do this workaround)
                            CTTblPrBase tblPr = style.getTblPr();
                            if(tblPr==null||tblPr.getShd()==null&&style.getTcPr()!=null) {
                            	final CTShd tableShade = style.getTcPr().getShd();
                            	if(tableShade!=null) {
	                            	if(tblPr==null) {
	                                	tblPr = objectFactory.createCTTblPrBase();
	                                	style.setTblPr(tblPr);
	                                	tblPr.setParent(style);
	                                }
	                            	tblPr.setShd(XmlUtils.deepCopy(tableShade));
	                            	tblPr.getShd().setParent(tblPr);
                            	}
                            }
                        }
                        else {
                            conditionalStyles.add(conditionalStyle);
                        }
                    }
                }
            }
        }
    }

    @SuppressWarnings("unused")
    public void deleteStylesheet(String type, String styleName) {
        //
    }
    public void insertListStyle(String listStyleId, JSONObject listDefinition)
        throws InvalidFormatException, JAXBException, JSONException {

        NumberingDefinitionsPart numDefPart = wordMLPackage.getMainDocumentPart().getNumberingDefinitionsPart();
        if( numDefPart == null ) {
            numDefPart = new NumberingDefinitionsPart();
            numDefPart.unmarshalDefaultNumbering();
            Numbering numbering = numDefPart.getJaxbElement();
            // remove defaults
            numbering.getAbstractNum().clear();
            numbering.getNum().clear();
            wordMLPackage.getMainDocumentPart().addTargetPart(numDefPart);
        }
        Numbering numbering = numDefPart.getJaxbElement();
        List<AbstractNum> abstractNumList = numbering.getAbstractNum();

        //find a 'free' abstractNumId
        Iterator<AbstractNum> abstractIter = abstractNumList.iterator();
        HashSet<Long> existingAbstractIds = new HashSet<Long>();
        while( abstractIter.hasNext() ){
            AbstractNum abstractNum = abstractIter.next();
            long abstractNumId = abstractNum.getAbstractNumId().longValue();
            existingAbstractIds.add(new Long(abstractNumId));
        }
        long newAbstractId = 0;
        while( existingAbstractIds.contains( new Long( newAbstractId ) ) )
            ++newAbstractId;

        AbstractNum newAbstractNum = objectFactory.createNumberingAbstractNum();
        newAbstractNum.setParent(numDefPart);
        BigInteger abstractId = BigInteger.valueOf(newAbstractId);
        newAbstractNum.setAbstractNumId(abstractId);
        List<Lvl> lvlList = newAbstractNum.getLvl();
        for( long lvlIndex = 0; lvlIndex < 9; ++lvlIndex){
            JSONObject levelObject = listDefinition.getJSONObject("listLevel" + lvlIndex);
            Lvl lvl = objectFactory.createLvl();
            lvl.setParent(newAbstractNum);
            lvl.setIlvl(BigInteger.valueOf(lvlIndex));
            if( levelObject.has("numberFormat")) {
                NumFmt numFmt = new NumFmt();
                numFmt.setVal( NumberFormat.fromValue( levelObject.getString("numberFormat") ));
                lvl.setNumFmt(numFmt);
            }
            if( levelObject.has("listStartValue")) {
                Lvl.Start lvlStart = objectFactory.createLvlStart();
                lvlStart.setVal(BigInteger.valueOf(levelObject.getLong("listStartValue")));
                lvl.setStart(lvlStart);
            }

            if( levelObject.has("levelText")) {
                if( lvl.getLvlText() == null )
                    lvl.setLvlText(objectFactory.createLvlLvlText());
                lvl.getLvlText().setVal(levelObject.getString("levelText"));
            }
            if( levelObject.has("textAlign") ){
                lvl.setLvlJc( objectFactory.createJc() );
                lvl.getLvlJc().setVal(JcEnumeration.fromValue(levelObject.getString("textAlign")));
            }
            RPr runProperties = objectFactory.createRPr();
            lvl.setRPr(runProperties);
            if( levelObject.has("fontName") ){
                runProperties.setRFonts(objectFactory.createRFonts());
                String font = levelObject.getString("fontName");
                runProperties.getRFonts().setAscii(font);
                runProperties.getRFonts().setHAnsi(font);
            }
            if( levelObject.has("paraStyle") ){
                Lvl.PStyle pstyle = objectFactory.createLvlPStyle();
                pstyle.setVal(levelObject.getString("paraStyle"));
            }

            PPr paraProperties = objectFactory.createPPr();
            lvl.setPPr(paraProperties);
            Ind ind = objectFactory.createPPrBaseInd();
            if( levelObject.has("indentLeft")) {
                ind.setLeft(Utils.map100THMMToTwip( levelObject.getInt("indentLeft")) );
            }
            if( levelObject.has("indentFirstLine")) {
                int firstline = levelObject.getInt("indentFirstLine");
                if( firstline > 0){
                    ind.setFirstLine( Utils.map100THMMToTwip( firstline ));
                    ind.setHanging( null );
                } else if( firstline < 0) {
                    ind.setFirstLine( null );
                    ind.setHanging( Utils.map100THMMToTwip( -firstline));
                }
            }
            paraProperties.setInd( ind );
            if( levelObject.has("tabStopPosition")){
                Tabs tabs = objectFactory.createTabs();
                CTTabStop tabStop = objectFactory.createCTTabStop();
                tabStop.setPos( Utils.map100THMMToTwip( levelObject.getInt("tabStopPosition") ) );
                tabs.getTab().add(tabStop);
                paraProperties.setTabs(tabs);
            }
            lvlList.add(lvl);
        }
        abstractNumList.add(newAbstractNum);
//        newAbstractNum.setName(value);
//        newAbstractNum.setMultiLevelType(value);

        List<Num> numList = numbering.getNum();
        Num num = objectFactory.createNumberingNum();

        num.setParent(numDefPart);
        AbstractNumId abstractNumId = new AbstractNumId();
        abstractNumId.setVal(abstractId);
        num.setAbstractNumId( abstractNumId );
        BigInteger listId;
        try {
            if(!Character.isDigit(listStyleId.charAt(0))){
                listId = BigInteger.valueOf(Integer.parseInt(listStyleId.substring(1)));
            } else {
                listId = BigInteger.valueOf(Integer.parseInt(listStyleId));
            }
            num.setNumId(listId);

            // removing list style if it already exists
            Iterator<Num> numIter = numList.iterator();
            while(numIter.hasNext()){
                Num aNum = numIter.next();
                if (aNum.getNumId().equals(listId)) {
                    numIter.remove();
                    break;
                }
            };
            numList.add(num);
        }
        catch (NumberFormatException e) {
            log.info("docx export insertListStyle: invalid listStyleId found!");
        }
    }

    public void setDocumentLanguageIfNotSet(String langStr)
        throws Exception {

        final Styles styles = getOperationDocument().getStyles(true);
        DocDefaults docDefaults = styles.getDocDefaults();
        if (docDefaults == null)
            docDefaults = objectFactory.createDocDefaults();
        if(docDefaults!=null) {
            RPrDefault rDefault = docDefaults.getRPrDefault();
            if (rDefault == null)
                rDefault = objectFactory.createDocDefaultsRPrDefault();
            if(rDefault!=null) {
                RPr rPrDefault = rDefault.getRPr();
                if (rPrDefault == null)
                    rPrDefault = objectFactory.createRPr();
                if(rPrDefault!=null) {
                    CTLanguage ctLanguage = rPrDefault.getLang();
                    if (ctLanguage == null)
                        ctLanguage = objectFactory.createCTLanguage();

                    if (ctLanguage != null) {
                        String lang = ctLanguage.getVal();
                        if (lang == null || lang.length() == 0) {
                            if (langStr != null && langStr.length() > 0)
                                ctLanguage.setVal(langStr);
                            else
                                ctLanguage.setVal("en-US"); // default according to MS
                        }
                        lang = ctLanguage.getEastAsia();
                        if (lang == null || lang.length() == 0) {
                            ctLanguage.setEastAsia("en-US"); // default according to MS
                        }
                        lang = ctLanguage.getBidi();
                        if (lang == null || lang.length() == 0) {
                            ctLanguage.setBidi("ar-SA"); // default according to MS
                        }
                    }
                }
            }
        }
    }
    
    public void setDocumentAttributes(JSONObject attrs)
    	throws Exception {
    	
    	final JSONObject documentAttrs = attrs.optJSONObject("document");
    	if(documentAttrs!=null) {
			final Object changeTracking = documentAttrs.opt("changeTracking");
			if(changeTracking instanceof Boolean) {
    			final CTSettings settings = getOperationDocument().getSettings(true);
    			if((Boolean)changeTracking) {
	    			final BooleanDefaultTrue trackRevision = Context.getWmlObjectFactory().createBooleanDefaultTrue();
	    			trackRevision.setVal(true);
	    			settings.setTrackRevisions(trackRevision);
    			}
    			else {
    				settings.setTrackRevisions(null);
    			}
    		}
    	}
    }
}

