/*
 * Copyright 2012 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.odftoolkit.odfdom.component;

import java.util.AbstractList;
import java.util.List;
import java.util.logging.Logger;
import org.json.JSONArray;
import org.json.JSONObject;
import org.odftoolkit.odfdom.doc.table.OdfTable;
import org.odftoolkit.odfdom.doc.table.OdfTableRow;
import org.odftoolkit.odfdom.dom.element.table.TableTableElement;
import org.odftoolkit.odfdom.pkg.OdfElement;
import org.w3c.dom.Node;

/**
 * A MultiCoomponent uses a single XML element to represent multiple components.
 * This container can be used for spreadsheet row and cell components using
 * repeated elements via an attribute.
 *
 * @author svante.schubertATgmail.com
 */
public class Table<T> extends Component {

    public Table(OdfElement componentElement, Component parent) {
        super(componentElement, parent);
    }
    private static final Logger LOG = Logger.getLogger(Table.class.getName());
    /** Used to indicate that the end position is not existing */
    public static final int ETERNITY = -1;
    /** Used to indicate that the start position is before the table
     * If a column is inserted in the first place, there is no prior column to inherit styles from.  */
    static final int BEFORE_START = -1;

    private TableTraversal mTableTraversal;

    /**
     * The maximal number of rows being generated. Unspecified by ODF but commonly used. Counting starts with 0.
     */
    public static final Integer MAX_ROW_NUMBER = 1048575;

    /**
     * The maximal number of columns being generated. Unspecified by ODF but commonly used. Counting starts with 0.
     */
    public static final Integer MAX_COLUMN_NUMBER = 16383;

    /**
     * A multiple components can be represented by a single XML element
     *
     * @return the number of components the elements represents
     */
    @Override
    public int repetition() {
        return mRootElement.getRepetition();
    }

    @Override
    // TABLE ONLY -- Svante REMOVE THIS LATER AS THIS IT IS ONLY USED BY TABLES
    public List getChildren() {
        return list(this);
    }

    // Svante ToDo: After all the refactoring this looks like something to change after the release as well.
    private List<Component> list(final Table tableComponent) {
        return new AbstractList<Component>() {
            public int size() {
                return tableComponent.size();
            }

            public Component get(int index) {
                Component c = null;
                TableTableElement tableRoot = (TableTableElement) tableComponent.getRootElement();
                OdfTable table = OdfTable.getInstance(tableRoot);
                OdfTableRow row = table.getRowByIndex(index);
                if (row != null) {
                    c = row.getOdfElement().getComponent();
                }
                return c;
            }
        };
    }

    /**
     * Adds the given component to the root element
     *
     * @param c the component of the row to be added
     */
    @Override
    public void addChild(int index, Component c) {
        mRootElement.insert(c.getRootElement(), index);
// 2DO: Svante: ARE THE ABOVE AND THE BELOW EQUIVALENT?
//		OdfElement rootElement = c.getRootElement();
//		if (index >= 0) {
//			mRootElement.insertBefore(rootElement, ((OdfElement) mRootElement).receiveNode(index));
//		} else {
//			mRootElement.appendChild(rootElement);
//		}
    }

    /**
     * @param index the position of the row being returned
     * @return either a text node of size 1 or an element being the root element
     * of a component
     */
    @Override
    public Node getChildNode(int index) {
        return ((OdfElement) mRootElement).receiveNode(index);

    }

    /**
     * Removes a component from the text element container. Removes either an
     * element representing a component or text node of size 1
     *
     * @param index row position to be removed
     * @return the row being removed
     */
    @Override
    public Node remove(int index) {
        Node node = (Node) this.getChildNode(index);
        return mRootElement.removeChild(node);
    }

    /**
     * All children of the root element will be traversed. If it is a text node
     * the size is added, if it is an element and a component a size of one is
     * added, if it is a marker, for known text marker elements (text:span,
     * text:bookmark) the children are recursive checked
     *
     * @return the number of child components
     */
    @Override
    public int size() {
        OdfTable table = OdfTable.getInstance((TableTableElement) mRootElement);
        return table.getRowCount();
    }

    /**
     * Formats the complete column of a table starting from columnStartNo to
     * columnEndNo. Counting starts with 0.
     *
     * @param columnStartNo the first column to be formated.
     * @param columnEndNo the last column to be formated
     * @param attrs the formatting attributes
     */
    public void formatColumn(int columnStartNo, int columnEndNo, JSONObject attrs) {
        int rowEndNo = -1;
        // in ODF cell and character attributes are on a default style, that is only applied in an application to the cells existing in XML
        if (attrs.has("cell") || attrs.has("cell")) {
            JSONObject cellProps = attrs.optJSONObject("cell");
            if (cellProps != null && cellProps.length() > 0) {
                rowEndNo = MAX_ROW_NUMBER;
            } else {
                JSONObject characterProps = attrs.optJSONObject("character");
                if (characterProps != null && characterProps.length() > 0) {
                    rowEndNo = MAX_ROW_NUMBER;
                }
            }
        }
        this.modifyRange(columnStartNo, columnEndNo, 0, rowEndNo, TableChanges.newFormatation(getTableTraversal(), attrs, columnStartNo, columnEndNo, 0, rowEndNo));
    }

    /**
     * Formats the complete row of a table starting from rowStartNo to rowEndNo.
     * Counting starts with 0.
     *
     * @param rowStartNo the first row to be formated.
     * @param rowEndNo the last row to be formated.
     * @param attrs the formatting attributes
     */
    public void formatRow(int rowStartNo, int rowEndNo, JSONObject attrs) {
        int columnEndNo = -1;
        // in ODF cell and character attributes are on a default style, that is only applied in an application to the cells existing in XML
        if (attrs.has("cell") || attrs.has("cell")) {
            JSONObject cellProps = attrs.optJSONObject("cell");
            if (cellProps != null && cellProps.length() > 0) {
                columnEndNo = MAX_COLUMN_NUMBER;
            } else {
                JSONObject characterProps = attrs.optJSONObject("character");
                if (characterProps != null && characterProps.length() > 0) {
                    columnEndNo = MAX_COLUMN_NUMBER;
                }
            }
        }
        modifyRange(0, columnEndNo, rowStartNo, rowEndNo, TableChanges.newFormatation(getTableTraversal(), attrs, 0, columnEndNo, rowStartNo, rowEndNo));
    }

    /**
     * Selects a cell range of the given table and applies the given changes.
     *
     * @param tableElement the table the cell range will be selected from
     * @param rowStartNo the upper delimiter of the cell range. All positions
     * start counting with zero.
     * @param rowEndNo the lower delimiter of the cell range. Any negative value
     * represents eternity, ie. to the end of table.
     * @param columnStartNo the left delimiter of the cell range.
     * @param columnEndNo the right delimiter of the cell range. Any negative
     * value represents eternity, ie. to the end of table.
     * @param changes a list of changes to be applied to each row and cells
     * during traversing the tree. TableChange may define a subset of selection
     * and the modification (e.g. insert, format, etc.)
     */
    // ToDo: Currently only a single Change is supported!!
    public static void modifyRange(TableTableElement tableElement, int columnStartNo, int columnEndNo, int rowStartNo, int rowEndNo, List<TableChange> changes) {
        if (tableElement != null) {
            ((Table) tableElement.getComponent()).modifyRange(columnStartNo, columnEndNo, rowStartNo, rowEndNo, changes.get(0));
        }
    }

    /**
     * Adds one or more empty columns to the table
     */
    public static void addColumns(TableTableElement tableElement, int columnStart, int columnEnd) {
        Table.addEmptyRange(tableElement, columnStart, columnEnd, 0, -1);
    }

    /**
     * Adds one or more empty columns to the table
     */
    public static void addRows(TableTableElement tableElement, int rowStart, int rowEnd) {
        Table.addEmptyRange(tableElement, 0, -1, rowStart, rowEnd);
    }

    /**
     * Deletes one or more columns from the table
     */
    public static void deleteColumns(TableTableElement tableElement, int columnStartNo, int columnEndNo) {
        if (tableElement != null) {
            Table table = ((Table) tableElement.getComponent());
            table.modifyRange(columnStartNo, columnEndNo, 0, MAX_ROW_NUMBER, TableChanges.newColumnDeletion(table.getTableTraversal(), columnStartNo, columnEndNo, 0, MAX_ROW_NUMBER));
        }
    }

    /**
     * Selects a cell range of the given table and applies the given changes.
     *
     * @param tableElement the table the cell range will be selected from
     * @param rowStartNo the upper delimiter of the cell range. All positions
     * start counting with zero.
     * @param rowEndNo the lower delimiter of the cell range. Any negative value
     * represents eternity, ie. to the end of table.
     * @param columnStartNo the left delimiter of the cell range.
     * @param columnEndNo the right delimiter of the cell range. Any negative
     * value represents eternity, ie. to the end of table.
     */
    public static void addEmptyRange(TableTableElement tableElement, int columnStartNo, int columnEndNo, int rowStartNo, int rowEndNo) {
        if (tableElement != null) {
            Table table = (Table) tableElement.getComponent();
            if (table != null) {
                TableChange tableChange = TableChanges.newCellChange(tableElement, columnStartNo, columnEndNo, rowStartNo, rowEndNo, null);
				// The preceding row/column is influencing the format of the new elements
                // Unless it is the first row/column always the preceding row/column have to be taken into account..

                // Adjust parameter for insertion
                // unless the end is eternity, adapt the new inserted column positions to a range of before/after the insertion point
                if (columnEndNo != ETERNITY) {
                    columnEndNo = columnStartNo;
                    // aside of the first all inherit from the one ealier
                    columnStartNo--;
                }
                // unless the end is eternity, adapt the new inserted row positions to a range of before/after the insertion point
                if (rowEndNo != ETERNITY) {
                    rowEndNo = rowStartNo;
                    // aside of the first all inherit from the one ealier
                    rowStartNo--;
                }
                table.modifyRange(columnStartNo, columnEndNo, rowStartNo, rowEndNo, tableChange);
            }
        }
    }

    /**
     * Modifies the format and/or value of multiple cells in multiple rows. The
     * rows might be of different length.
     *
     * @param columnStartNo horizontal start point.
     * @param rowStartNo vertical start point.
     * @param dataRange The values and attribute sets to be written into the
     * cell range. MUST be a 2-dimensional array. The outer array contains rows
     * of cell contents, and the inner row arrays contain the cell contents for
     * each single row. The lengths of the inner arrays may be different. Cells
     * not covered by a row array will not be modified. See next table for
     * details about the format of the single cell content objects.
     */
    public void modifyCells(int columnStartNo, int rowStartNo, JSONArray dataRange) {

        if (dataRange != null && columnStartNo >= 0 && rowStartNo >= 0) {
            // starting with -1 position for row & column as the 0 would already be the first position found
            TableChange tableChange = TableChanges.newCellsChange(getTableTraversal(), dataRange, columnStartNo, rowStartNo);
            tableChange.start();
            // ToDo: perhaps check if the change had been sucessfull... Is a separate start() necessary??
        }
    }

    /**
     * Modifies the format and/or value of multiple cells in multiple rows. The
     * rows might be of different length.
     *
     * @param columnStartNo horizontal start point.
     * @param rowStartNo vertical start point.
     * @param dataRange The values and attribute sets to be written into the
     * cell range. MUST be a 2-dimensional array. The outer array contains rows
     * of cell contents, and the inner row arrays contain the cell contents for
     * each single row. The lengths of the inner arrays may be different. Cells
     * not covered by a row array will not be modified. See next table for
     * details about the format of the single cell content objects.
     */
    public void modifyCells(int columnStartNo, int columnEndNo, int rowStartNo, int rowEndNo, Object cellValue, JSONObject cellFormat) {

        // starting with -1 position for row & column as the 0 would already be the first position found
        TableChange tableChange = TableChanges.newCellsChange(getTableTraversal(), cellValue, cellFormat, columnStartNo, columnEndNo, rowStartNo, rowEndNo);
        tableChange.start();
        // ToDo: perhaps check if the change had been sucessfull... Is a separate start() necessary??

    }

    /**
     * The given change will be applied to the table descendants (column, row
     * and cell elements) of the range that is selected.
     *
     * @param columnStartNo horizontal start point.
     * @param columnEndNo horizontal end point.
     * @param rowStartNo vertical start point.
     * @param rowEndNo vertical end point.
     * @param change an object of type TableChange, which execute() method will
     * be called with the table descendant as parameter.
     */
    public void modifyRange(int columnStartNo, int columnEndNo, int rowStartNo, int rowEndNo, TableChange change) {
        change.setColumnStart(columnStartNo);
        change.setColumnEnd(columnEndNo);
        change.setRowStart(rowStartNo);
        change.setRowEnd(rowEndNo);
        change.start();
    }

    TableTraversal getTableTraversal() {
        // starting with -1 position for row & column as the 0 would already be the first position found
        if (mTableTraversal == null) {
            // ToDo: The traversal moves only in document order, any change after the current point will not be found
            // ONLY multiple changes, which are ordered are allowed to be provided or a new traversal have to be taken
            // perhaps provide X/Y coordinate and return a VALID
            mTableTraversal = new TableTraversal((TableTableElement) this.getRootElement());
        }
        return mTableTraversal;
    }
}
