/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.json.actions;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.ajax.requesthandler.AJAXRequestData;
import com.openexchange.ajax.requesthandler.AJAXRequestResult;
import com.openexchange.documentconverter.Feature;
import com.openexchange.documentconverter.IManager;
import com.openexchange.documentconverter.Properties;
import com.openexchange.exception.OXException;
import com.openexchange.file.storage.DefaultFile;
import com.openexchange.file.storage.File;
import com.openexchange.file.storage.File.Field;
import com.openexchange.file.storage.FileStorageFileAccess;
import com.openexchange.file.storage.composition.IDBasedFileAccess;
import com.openexchange.file.storage.composition.IDBasedFileAccessFactory;
import com.openexchange.file.storage.composition.IDBasedFolderAccess;
import com.openexchange.file.storage.composition.IDBasedFolderAccessFactory;
import com.openexchange.folderstorage.FolderService;
import com.openexchange.groupware.ldap.User;
import com.openexchange.office.DocumentProperties;
import com.openexchange.office.FilterException;
import com.openexchange.office.IImporter;
import com.openexchange.office.IResourceProvider;
import com.openexchange.office.json.tools.EmptyDocumentCache;
import com.openexchange.office.json.tools.FilterExceptionToErrorCode;
import com.openexchange.office.json.tools.OXDocumentHelper;
import com.openexchange.office.json.tools.OXDocumentHelper.OfficeDocumentTypeProperties;
import com.openexchange.office.tools.DocumentFormatHelper;
import com.openexchange.office.tools.DocumentFormatHelper.OfficeDocumentType;
import com.openexchange.office.tools.DocumentType;
import com.openexchange.office.tools.ErrorCode;
import com.openexchange.office.tools.ExceptionToErrorCode;
import com.openexchange.office.tools.FileDescriptor;
import com.openexchange.office.tools.FileHelper;
import com.openexchange.office.tools.FileHelper.StreamInfo;
import com.openexchange.office.tools.ResourceManager;
import com.openexchange.office.tools.UserConfigurationHelper;
import com.openexchange.office.tools.UserConfigurationHelper.Mode;
import com.openexchange.office.tools.htmldoc.ConfHelper;
import com.openexchange.office.tools.htmldoc.GenericHtmlDocumentBuilder;
import com.openexchange.office.tools.htmldoc.OperationReducer;
import com.openexchange.office.tools.htmldoc.TextTableLimits;
import com.openexchange.office.tools.message.MessagePropertyKey;
import com.openexchange.office.tools.monitoring.CreateEvent;
import com.openexchange.office.tools.monitoring.DocumentEvent;
import com.openexchange.office.tools.monitoring.ErrorType;
import com.openexchange.office.tools.monitoring.Statistics;
import com.openexchange.server.ServiceLookup;
import com.openexchange.tools.session.ServerSession;
import com.openexchange.tx.TransactionAwares;

/**
 * {@link CreateDefaultDocumentAction}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @author <a href="mailto:carsten.driesner@open-xchange.com">Carsten Driesner</a>
 */

/*
 * MH/KA comment this Action to make it compile
 * @Action(method = RequestMethod.GET, name = "createdefaultdocument", description =
 * "Create a default (empty) OOXML document based on given document type.", parameters = {
 * @Parameter(name = "folder_id", description = "Folder ID of the requested infoitem."),
 * @Parameter(name = "session", description = "A session ID previously obtained from the login module."),
 * @Parameter(name = "uid", description = "The unique id of the client application."),
 * @Parameter(name = "document_type", optional=true, description =
 * "Specifies the type of default document to be created. Possible values are: 'text' (default), 'spreadsheet' and 'presentation' ."),
 * responseDescription =
 * @Parameter(name = "as_template", optional = true, description = "Specifies that a template document should stay a template format, even if conversion is needed.") , },
 * "Response with timestamp: A JSON object containing the attributes of the created filestore item: 'id', 'folder_id', 'version' and 'filename'."
 * )
 */

public class CreateDefaultDocumentAction extends DocumentFilterAction {

    private static final Log LOG = com.openexchange.log.Log.loggerFor(CreateDefaultDocumentAction.class);

    private final EmptyDocumentCache defaultDocCache;

    private class TemplateResult {
        public final JSONObject jsonResult;
        public final DocumentType eventDocumentType;

        public TemplateResult(JSONObject jsonResult, DocumentType eventDocumentType) {
            this.jsonResult = jsonResult;
            this.eventDocumentType = eventDocumentType;
        }
    }

    /**
     * Initializes a new {@link CreateDefaultDocumentAction}.
     *
     * @param services
     * @param oqm
     */
    public CreateDefaultDocumentAction(ServiceLookup services, EmptyDocumentCache defaultDocCache) {
        super(services);
        this.defaultDocCache = defaultDocCache;
    }

    @Override
    public AJAXRequestResult perform(AJAXRequestData request, ServerSession session) {
        AJAXRequestResult requestResult = null;
        DocumentEvent documentEvent = null;
        DocumentType eventDocumentType = DocumentType.NONE;

        if ((null != request) && (null != session)) {
            JSONObject jsonResult = null;
            ErrorCode errorCode = ErrorCode.NO_ERROR;

            try {
                final String fileId = request.getParameter("template_id");

                if (null == fileId) {
                    // create a new empty default document - this will be handled
                    // differently to speed-up the process
                    jsonResult = impl_createDefaultDocument(session, request);
                    eventDocumentType = getDocumentType(request);
                } else {
                    // create a new document based on a template document
                    TemplateResult templateResult = impl_createDocumentOnTemplate(session, request);
                    eventDocumentType = templateResult.eventDocumentType;
                    jsonResult = templateResult.jsonResult;
                }
            } catch (Exception e) {
                errorCode = ErrorCode.GENERAL_UNKNOWN_ERROR;
                if (e instanceof OXException) {
                    errorCode = ExceptionToErrorCode.map((OXException)e, errorCode, false);
                } else if (e instanceof FilterException) {
                    errorCode = FilterExceptionToErrorCode.map((FilterException)e, errorCode);
                }
                LOG.error(errorCode.getDescription(), e);
                jsonResult = errorCode.getAsJSONResultObject();
            } finally {
                // create the final result and document event
                documentEvent = errorCode.isError() ? new DocumentEvent(eventDocumentType, ErrorType.CREATE) :
                    new CreateEvent(eventDocumentType);
                requestResult = new AJAXRequestResult(jsonResult);
            }
        }

        // update statistics with a possibly created event
        Statistics.handleDocumentEvent(documentEvent);

        return requestResult;
    }

    /**
     * Determines the document type dependent on the request data from an AJAX
     * request.
     *
     * @param request
     *  The AJAX request data to be checked for the document type.
     *
     * @return
     *  The document type, can be DocmentType.NONE if the type couldn't be
     *  extracted.
     */
    DocumentType getDocumentType(AJAXRequestData request) {
        DocumentType result = DocumentType.NONE;

        if (null != request) {
            final String docTypeName = request.getParameter("document_type").toLowerCase();

            if (docTypeName.equals("spreadsheet")) {
                result = DocumentType.SPREADSHEET;
            } else if (docTypeName.equals("presentation")) {
                result = DocumentType.PRESENTATION;
            } else {
                result = DocumentType.TEXT;
            }
        }
        return result;
    }

    /**
     * Handles the case that a new, empty document should be created, where we
     * can use several assumptions to speed-up the document generation.
     *
     * @param session
     *  The session of the client which requested to create a new empty document.
     *
     * @param request
     *  The AJAX request fill with the data what file should be used a template
     *  and where the file should be created.
     *
     * @return
     *  The JSON result for the request specific for creating a new empty document.
     *
     * @throws OXException
     * @throws FilterException
     */
    private JSONObject impl_createDefaultDocument(ServerSession session, AJAXRequestData request) throws Exception {
        JSONObject jsonResult = null;

        InputStream inputStream = null;
        InputStream resourceInputStream = null;
        final IResourceProvider resourceProvider = m_services.getService(com.openexchange.office.IResourceProvider.class);

        final File file = new DefaultFile();
        final String docTypeName = request.getParameter("document_type").toLowerCase();
        String folderId = request.getParameter("folder_id");
        final String initialSheetname = request.getParameter("initial_sheetname");
        final String initialFileName = request.getParameter("initial_filename");
        // will be needed for speed-up access if creating a new empty document
        String htmlDoc = null;
        JSONObject documentOperations = null;

        // determine requested document type and corresponding extensionType
        OfficeDocumentTypeProperties docTypeProps = OXDocumentHelper.determineDefaultOfficeProperties(docTypeName);
        OfficeDocumentType officeDocumentType = docTypeProps.docType;
        String module = docTypeProps.module;
        String extensionType = docTypeProps.extension;
        String mimeType = docTypeProps.mimeType;

        try {
            // look into the soft cache for stored empty default document data
            final IImporter importer = OXDocumentHelper.getImporter(m_services, officeDocumentType);
            final EmptyDocumentCache.EmptyDocData cacheData = defaultDocCache.getDocData(module);
            if (null != cacheData) {
                // cache hit - provide the fastLoad empty document data
                jsonResult = new JSONObject();
                jsonResult.put(MessagePropertyKey.KEY_ERROR_DATA, ErrorCode.NO_ERROR.getAsJSON());
                jsonResult.put(MessagePropertyKey.KEY_HTMLDOCUMENT, cacheData.htmlDoc);
                jsonResult.put(MessagePropertyKey.KEY_OPERATIONS, cacheData.operations);
                // create a input stream buffered by the byte array stored in the cache
                inputStream = new ByteArrayInputStream(cacheData.documentData);
                LOG.debug("RT connection: cache hit for document type: " + module + " data send from cache.");
            } else {
                LOG.debug("RT connection: cache miss for document type: " + module + " data must be retrieved from file.");
                // no cache hit - load the default document
                // try to get default document from resource first
                if (null != resourceProvider) {
                    // !!! resource leak warning has been checked:
                    // if resourceInputStream is returned unchanged as the result
                    // InputStream, it will be closed by the caller; otherwise it will
                    // be closed instantly after retrieving the real result InputStream from the
                    // resourceInputStream after calling importer.getDefaultDocument(...) !!!
                    resourceInputStream = resourceProvider.getResource(resourceProvider.getEntry(module, "template", "Default"));
                }

                if (null != importer) {
                    final DocumentProperties docProps = new DocumentProperties();
                    final User user = session.getUser();

                    if (null != user) {
                        docProps.put(DocumentProperties.PROP_CREATOR, user.getDisplayName());
                        if (initialSheetname != null) {
                            docProps.put(DocumentProperties.PROP_INITIAL_SHEETNAME, initialSheetname);
                        }
                    }
                    inputStream = importer.getDefaultDocument(resourceInputStream, docProps);
                }

                if (resourceInputStream != inputStream) {
                    // Close resource input stream if default document stream
                    // was created by the import getDefaultDocument function. Don't
                    // do that if inputStm refers to resourceInputStream
                    IOUtils.closeQuietly(resourceInputStream);
                }
            }

            // set the required file properties
            final String createFileName = ((null == initialFileName) || (initialFileName.length() < 1)) ? "unnamed" : initialFileName;
            file.setId(FileStorageFileAccess.NEW);
            file.setFolderId(folderId);
            file.setFileName(createFileName + "." + extensionType);
            file.setFileMIMEType(mimeType);

            // initialize json result object
            jsonResult = new JSONObject();

            ErrorCode errorCode = ErrorCode.NO_ERROR;
            if (module.equalsIgnoreCase("text") && (null == cacheData) && (null != inputStream)) {
                // just continue with the slower way - no cache hit
                OXDocumentHelper oxDocHelper = new OXDocumentHelper(m_services, session, inputStream, new ResourceManager(m_services));
                documentOperations = oxDocHelper.getOperations(session, importer);
                // retrieve buffered input stream from document helper - original
                // stream will be closed by the OXDocumentHelper ctor!!
                inputStream = oxDocHelper.getInputStream();

                // now we have to create the new document file using the input stream
                errorCode = impl_createFileAndWriteStream(session, null, file, inputStream);
                if (!errorCode.isError()) {
                    UserConfigurationHelper userConfHelper = new UserConfigurationHelper(m_services, session, "io.ox/office", Mode.WRITE_BACK);
                    final Boolean fastLoad = userConfHelper.getBoolean("text/documentFastLoad", true);
                    if (fastLoad != null && fastLoad) {
                        final TextTableLimits textTableLimits = ConfHelper.getTextTableLimits(userConfHelper);
                        OperationReducer.reduceOperationCount(documentOperations);
                        htmlDoc = GenericHtmlDocumentBuilder.buildHtmlDocument(documentOperations, file.getId(), file.getFolderId(), file.getVersion(), textTableLimits);
                        defaultDocCache.setDocData(module,
                            defaultDocCache.new EmptyDocData(htmlDoc,
                                                             documentOperations.getJSONArray("operations"),
                                                             oxDocHelper.getDocumentBuffer()));
                        jsonResult.put(MessagePropertyKey.KEY_HTMLDOCUMENT, htmlDoc);
                    }
                    jsonResult.put(MessagePropertyKey.KEY_OPERATIONS, documentOperations.getJSONArray("operations"));
                }
            } else {
                // now we have to create the new document file using the input stream
                errorCode = impl_createFileAndWriteStream(session, null, file, inputStream);
                if (!errorCode.isError() && (null != cacheData)) {
                    // set result data in case of a cache hit and the new file could be created
                    if (null != cacheData.htmlDoc) {
                        jsonResult.put(MessagePropertyKey.KEY_HTMLDOCUMENT, cacheData.htmlDoc);
                    }
                    jsonResult.put(MessagePropertyKey.KEY_OPERATIONS, cacheData.operations);
                }
            }

            jsonResult.put("error", errorCode.getAsJSON());
            if (!errorCode.isError()) {
                // add the file descriptor to the json object
                jsonResult.put(MessagePropertyKey.KEY_FILEDESCRIPTOR, FileDescriptor.createJSONObject(file, null));
            }
        } catch (JSONException e) {
            // if we encountered a JSON exception we have to provide a general
            // error to the client by just return a null result
            jsonResult = null;
        } finally {
            IOUtils.closeQuietly(inputStream);
        }

        return jsonResult;
    }

    /**
     * Handles the case that a new, empty document should be created, where we
     * can use several assumptions to speed-up the document generation.
     *
     * @param session
     *  The session of the client which requested to create a new empty document
     *  based on a arbitrary template file.
     *
     * @param request
     *  The AJAX request fill with the data what file should be used a template
     *  and where the file should be created.
     *
     * @return
     *  A template result containing the JSONObject to be sent to the client and
     *  specific information for event logging.
     *
     * @throws OXException
     * @throws FilterException
     */
    private TemplateResult impl_createDocumentOnTemplate(ServerSession session, AJAXRequestData request) throws Exception {
        TemplateResult result = null;

        final File file = new DefaultFile();
        final String fileId = request.getParameter("template_id");
        final String versionStr = request.getParameter("version");
        final boolean preserveFileName = Boolean.parseBoolean(request.getParameter("preserve_filename"));
        final boolean asTemplate = Boolean.parseBoolean(request.getParameter("as_template"));
        String folderId = request.getParameter("folder_id");
        final String templateAccessId = FileHelper.getFileAccessId(fileId, folderId);
        String version = (null != versionStr) ? versionStr : "";
        InputStream inputStm = null;
        DocumentType eventDocumentType = DocumentType.NONE;

        // determine requested document type and corresponding extensionType
        HashMap<String, String> conversionFormat = null;
        String extensionType;
        String mimeType;

        // create document on template must have a template id
        if ((null == fileId) || fileId.length() < 1)
            return new TemplateResult(new JSONObject().put("error", ErrorCode.GENERAL_ARGUMENTS_ERROR), eventDocumentType);

        // Create document based on template file. It could be that we have to
        // convert the template document to the related ODF document format.
        if (version.length() == 0) {
            version = FileStorageFileAccess.CURRENT_VERSION;
        }

        final IDBasedFileAccessFactory fileFactory = m_services.getService(IDBasedFileAccessFactory.class);
        final IDBasedFolderAccessFactory folderFactory = m_services.getService(IDBasedFolderAccessFactory.class);
        final IDBasedFileAccess fileAccess = fileFactory.createAccess(session);
        final IDBasedFolderAccess folderAccess = folderFactory.createAccess(session);
        final File templateFile = fileAccess.getFileMetadata(templateAccessId, version);
        final String templateFileName = templateFile.getFileName();

        extensionType = FileHelper.getExtension(templateFileName);
        mimeType = templateFile.getFileMIMEType();
        conversionFormat = DocumentFormatHelper.getConversionFormatInfo(mimeType, extensionType);

        if (conversionFormat != null) {
            final IManager dcManager = m_services.getService(IManager.class);

            if ((null != dcManager) && dcManager.hasFeature(Feature.DOCUMENTCONVERTER)) {
                AJAXRequestData loadRequest = new AJAXRequestData();
                loadRequest.putParameter("id", fileId);
                loadRequest.putParameter("folder_id", folderId);
                loadRequest.putParameter("version", version);
                loadRequest.putParameter("fileName", templateFileName);

                final StreamInfo streamInfo = FileHelper.getDocumentStream(m_services, session, loadRequest);
                final InputStream documentInputStm = streamInfo.documentStream;

                if (null != documentInputStm) {
                    HashMap<String, Object> jobProperties = new HashMap<String, Object>(4);
                    HashMap<String, Object> resultProperties = new HashMap<String, Object>(8);
                    final String filterShortName = conversionFormat.get(Properties.PROP_FILTER_SHORT_NAME);
                    final String convertedExtensionType = conversionFormat.get(Properties.PROP_INPUT_TYPE);

                    jobProperties.put(Properties.PROP_INPUT_STREAM, documentInputStm);
                    jobProperties.put(Properties.PROP_FILTER_SHORT_NAME, filterShortName);

                    if (null != templateFileName) {
                        jobProperties.put(Properties.PROP_INFO_FILENAME, templateFileName);
                    }

                    inputStm = dcManager.convert(filterShortName, jobProperties, resultProperties);
                    IOUtils.closeQuietly(documentInputStm);

                    // set new mime type and extensionType
                    mimeType = conversionFormat.get(Properties.PROP_MIME_TYPE);
                    extensionType = conversionFormat.get(Properties.PROP_INPUT_TYPE);

                    if (asTemplate) {
                        // target format should be a template => retrieve mime & extension type
                        HashMap<String, String> templateFormatInfo = DocumentFormatHelper.getTemplateFormatInfoForDocument(mimeType, convertedExtensionType);
                        if (null != templateFormatInfo) {
                            mimeType = templateFormatInfo.get(Properties.PROP_MIME_TYPE);
                            extensionType = templateFormatInfo.get(Properties.PROP_INPUT_TYPE);
                        }
                    }
                } else {
                    // provide error information to the client side
                    return new TemplateResult(new JSONObject().put(MessagePropertyKey.KEY_ERROR_DATA, streamInfo.errorCode), eventDocumentType);
                }
            }

            eventDocumentType = DocumentType.valueOf(conversionFormat.get("DocumentType"));
        } else {
            inputStm = fileAccess.getDocument(templateAccessId, version);
        }

        boolean canWriteToFolder = FileHelper.folderHasWriteAccess(folderAccess, folderId);
        boolean canCreateFilesInFolder = FileHelper.folderHasCreateAccess(folderAccess, folderId);
        if ((!canWriteToFolder || !canCreateFilesInFolder) && !preserveFileName) {
            // We cannot create files/write to the folder which contains our source template document (Edit as New).
            // Create the document in the user's default folder instead.
            folderId = FileHelper.getUserFolderId(m_services.getService(FolderService.class), session);
        }

        // create a new file store item with default content
        file.setId(FileStorageFileAccess.NEW);
        file.setFolderId(folderId);

        if (!preserveFileName) {
            final String initialFileName = request.getParameter("initial_filename");
            final String createFileName = ((null == initialFileName) || (initialFileName.length() < 1)) ? "unnamed" : initialFileName;
            file.setFileName(createFileName + "." + extensionType);
        } else {
            file.setFileName(FileHelper.getBaseName(templateFile.getFileName()) + "." + extensionType);
        }

        file.setFileMIMEType(mimeType);

        ErrorCode errorCode = ErrorCode.NO_ERROR;
        JSONObject jsonResult = new JSONObject();

        try {
            if (null != inputStm) {
                errorCode = impl_createFileAndWriteStream(session, fileAccess, file, inputStm);

                if (!errorCode.isError()) {
                    // set the file descriptor
                    jsonResult.put(MessagePropertyKey.KEY_FILEDESCRIPTOR, FileDescriptor.createJSONObject(file, null));
                }
            } else {
                errorCode = ErrorCode.CREATEDOCUMENT_CANNOT_READ_DEFAULTTEMPLATEFILE_ERROR;
                // error case: we don't have a input stream
                if (null != conversionFormat) {
                    errorCode = ErrorCode.CREATEDOCUMENT_CONVERSION_FAILED_ERROR;
                } else if (null != fileId) {
                    errorCode = ErrorCode.CREATEDOCUMENT_CANNOT_READ_TEMPLATEFILE_ERROR;
                }
                LOG.warn(errorCode.getDescription());
            }
            jsonResult.put(MessagePropertyKey.KEY_ERROR_DATA, errorCode.getAsJSON());
            result = new TemplateResult(jsonResult, eventDocumentType);
        } catch (JSONException e) {
            // if we encountered a JSON exception we have to provide a general
            // error to the client by just return a null result
            jsonResult = null;
        } finally {
            IOUtils.closeQuietly(inputStm);
        }

        return result;
    }

    /**
     * Creates a OX Drive file and writes the provided input stream to it.
     *
     * @param session
     *  The session of the user who requested to create a new file.
     *
     * @param fileAccess
     *  The file access instance to be used by this method. Can be null
     *  where the method creates it's own file access.
     *
     * @param file
     *  The new file instance, which is correctly initialized.
     *
     * @param inputStream
     *  The input stream of the document to be written. Attention: The stream
     *  is not closed by the method. So the caller is responsible to close
     *  the stream.
     *
     * @return
     *  The error code of the operation which is ErrorCode.NO_ERROR on success
     *  or any other code in case of an error.
     */
    private ErrorCode impl_createFileAndWriteStream(final ServerSession session, final IDBasedFileAccess fileAccess, final File file, final InputStream inputStream) {
        IDBasedFileAccess myFileAccess = fileAccess;
        if (null == myFileAccess) {
            final IDBasedFileAccessFactory fileFactory = m_services.getService(IDBasedFileAccessFactory.class);
            myFileAccess = fileFactory.createAccess(session);
        }
        ErrorCode errorCode = ErrorCode.NO_ERROR;
        boolean rollback = true;

        try {
            myFileAccess.startTransaction();
            rollback = true;
            myFileAccess.saveDocument(file, inputStream, FileStorageFileAccess.DISTANT_FUTURE, new ArrayList<Field>());
            myFileAccess.commit();
            rollback = false;
        } catch (final Exception e) {
            errorCode = ErrorCode.GENERAL_PERMISSION_CREATE_MISSING_ERROR;
            if (e instanceof OXException) {
                errorCode = ExceptionToErrorCode.map((OXException)e, errorCode, false);
            }
            LOG.error(errorCode.getDescription(), e);
        } finally {
            // Roll-back (if needed) and finish
            if (rollback) {
                TransactionAwares.rollbackSafe(myFileAccess);
            }

            TransactionAwares.finishSafe(myFileAccess);
        }

        return errorCode;
    }
}
