/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2014 Open-Xchange Inc., Tarrytown, NY, USA. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define(['io.ox/office/editframework/utils/mixedborder'], function (MixedBorder) {

    'use strict';

    var AUTO = { type: 'auto' },
        SOLID = { style: 'solid', width: 1, color: AUTO },
        DOTTED = { style: 'dotted', width: 1, color: AUTO },
        THICK = { style: 'solid', width: 10, color: AUTO },
        NONE = { style: 'none' };

    describe('OX Documents Edit Framework', function () {
        describe('MixedBorder module', function () {

            it('should exist', function () {
                expect(MixedBorder).to.be.an('object');
            });

            // methods ========================================================

            describe('method "isVisibleBorder"', function () {
                it('should exist', function () {
                    expect(MixedBorder).itself.to.respondTo('isVisibleBorder');
                });
                it('should return true for visible borders', function () {
                    expect(MixedBorder.isVisibleBorder(SOLID)).to.be['true'];
                    expect(MixedBorder.isVisibleBorder(DOTTED)).to.be['true'];
                    expect(MixedBorder.isVisibleBorder(THICK)).to.be['true'];
                    expect(MixedBorder.isVisibleBorder(_.extend({ mixed: false }, SOLID))).to.be['true'];
                    expect(MixedBorder.isVisibleBorder(_.extend({ mixed: false }, DOTTED))).to.be['true'];
                    expect(MixedBorder.isVisibleBorder(_.extend({ mixed: false }, THICK))).to.be['true'];
                });
                it('should return false for invisible borders', function () {
                    expect(MixedBorder.isVisibleBorder(NONE)).to.be['false'];
                    expect(MixedBorder.isVisibleBorder(_.extend({ mixed: false }, NONE))).to.be['false'];
                    expect(MixedBorder.isVisibleBorder({ style: 'solid', width: 0, color: AUTO })).to.be['false'];
                });
                it('should return true for visible ambiguous borders', function () {
                    expect(MixedBorder.isVisibleBorder({ mixed: false, style: null, width: 1, color: AUTO })).to.be['true'];
                    expect(MixedBorder.isVisibleBorder({ mixed: false, style: 'solid', width: null, color: AUTO })).to.be['true'];
                    expect(MixedBorder.isVisibleBorder({ mixed: false, style: 'solid', width: 1, color: null })).to.be['true'];
                    expect(MixedBorder.isVisibleBorder({ mixed: false, style: null, width: null, color: null })).to.be['true'];
                });
                it('should return true for mixed borders', function () {
                    expect(MixedBorder.isVisibleBorder(_.extend({ mixed: true }, SOLID))).to.be['true'];
                    expect(MixedBorder.isVisibleBorder(_.extend({ mixed: true }, DOTTED))).to.be['true'];
                    expect(MixedBorder.isVisibleBorder(_.extend({ mixed: true }, THICK))).to.be['true'];
                });
                it('should return true for mixed ambiguous borders', function () {
                    expect(MixedBorder.isVisibleBorder({ mixed: true, style: null, width: 1, color: AUTO })).to.be['true'];
                    expect(MixedBorder.isVisibleBorder({ mixed: true, style: 'solid', width: null, color: AUTO })).to.be['true'];
                    expect(MixedBorder.isVisibleBorder({ mixed: true, style: 'solid', width: 1, color: null })).to.be['true'];
                    expect(MixedBorder.isVisibleBorder({ mixed: true, style: null, width: null, color: null })).to.be['true'];
                });
            });

            // methods ========================================================

            describe('method "mixBorders"', function () {
                it('should exist', function () {
                    expect(MixedBorder).itself.to.respondTo('mixBorders');
                });
                it('should return equal borders unmodified', function () {
                    expect(MixedBorder.mixBorders(SOLID)).to.deep.equal(_.extend({ mixed: false }, SOLID));
                    expect(MixedBorder.mixBorders(SOLID, SOLID)).to.deep.equal(_.extend({ mixed: false }, SOLID));
                    expect(MixedBorder.mixBorders(SOLID, SOLID, SOLID)).to.deep.equal(_.extend({ mixed: false }, SOLID));
                    expect(MixedBorder.mixBorders(DOTTED, DOTTED)).to.deep.equal(_.extend({ mixed: false }, DOTTED));
                    expect(MixedBorder.mixBorders(THICK, THICK)).to.deep.equal(_.extend({ mixed: false }, THICK));
                    expect(MixedBorder.mixBorders(NONE, NONE)).to.deep.equal(_.extend({ mixed: false }, NONE));
                });
                it('should mix visible borders', function () {
                    expect(MixedBorder.mixBorders(SOLID, DOTTED)).to.deep.equal({ mixed: false, style: null, width: 1, color: AUTO });
                    expect(MixedBorder.mixBorders(SOLID, THICK)).to.deep.equal({ mixed: false, style: 'solid', width: null, color: AUTO });
                    expect(MixedBorder.mixBorders(DOTTED, THICK)).to.deep.equal({ mixed: false, style: null, width: null, color: AUTO });
                    expect(MixedBorder.mixBorders(SOLID, DOTTED, THICK)).to.deep.equal({ mixed: false, style: null, width: null, color: AUTO });
                });
                it('should mix visible and invisible borders', function () {
                    expect(MixedBorder.mixBorders(SOLID, NONE)).to.deep.equal(_.extend({ mixed: true }, SOLID));
                    expect(MixedBorder.mixBorders(NONE, SOLID)).to.deep.equal(_.extend({ mixed: true }, SOLID));
                    expect(MixedBorder.mixBorders(DOTTED, NONE)).to.deep.equal(_.extend({ mixed: true }, DOTTED));
                    expect(MixedBorder.mixBorders(NONE, DOTTED)).to.deep.equal(_.extend({ mixed: true }, DOTTED));
                    expect(MixedBorder.mixBorders(THICK, NONE)).to.deep.equal(_.extend({ mixed: true }, THICK));
                    expect(MixedBorder.mixBorders(NONE, THICK)).to.deep.equal(_.extend({ mixed: true }, THICK));
                    expect(MixedBorder.mixBorders(SOLID, DOTTED, NONE)).to.deep.equal({ mixed: true, style: null, width: 1, color: AUTO });
                    expect(MixedBorder.mixBorders(SOLID, NONE, DOTTED)).to.deep.equal({ mixed: true, style: null, width: 1, color: AUTO });
                    expect(MixedBorder.mixBorders(NONE, SOLID, DOTTED)).to.deep.equal({ mixed: true, style: null, width: 1, color: AUTO });
                    expect(MixedBorder.mixBorders(SOLID, THICK, NONE)).to.deep.equal({ mixed: true, style: 'solid', width: null, color: AUTO });
                    expect(MixedBorder.mixBorders(NONE, SOLID, THICK)).to.deep.equal({ mixed: true, style: 'solid', width: null, color: AUTO });
                    expect(MixedBorder.mixBorders(DOTTED, THICK, NONE)).to.deep.equal({ mixed: true, style: null, width: null, color: AUTO });
                    expect(MixedBorder.mixBorders(NONE, DOTTED, THICK)).to.deep.equal({ mixed: true, style: null, width: null, color: AUTO });
                    expect(MixedBorder.mixBorders(SOLID, DOTTED, THICK, NONE)).to.deep.equal({ mixed: true, style: null, width: null, color: AUTO });
                    expect(MixedBorder.mixBorders(NONE, SOLID, DOTTED, THICK)).to.deep.equal({ mixed: true, style: null, width: null, color: AUTO });
                });
                it('should accept an array as parameter', function () {
                    expect(MixedBorder.mixBorders([SOLID])).to.deep.equal(_.extend({ mixed: false }, SOLID));
                    expect(MixedBorder.mixBorders([SOLID, NONE])).to.deep.equal(_.extend({ mixed: true }, SOLID));
                    expect(MixedBorder.mixBorders([SOLID, DOTTED], THICK, NONE)).to.deep.equal({ mixed: true, style: null, width: null, color: AUTO });
                    expect(MixedBorder.mixBorders(SOLID, [DOTTED, THICK], NONE)).to.deep.equal({ mixed: true, style: null, width: null, color: AUTO });
                    expect(MixedBorder.mixBorders(SOLID, DOTTED, [THICK, NONE])).to.deep.equal({ mixed: true, style: null, width: null, color: AUTO });
                    expect(MixedBorder.mixBorders([SOLID, DOTTED], [THICK, NONE])).to.deep.equal({ mixed: true, style: null, width: null, color: AUTO });
                });
            });

            // ================================================================

        });
    });
});
