/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/object/baseobject', ['io.ox/office/tk/utils'], function (Utils) {

    'use strict';

    // class BaseObject =======================================================

    /**
     * An abstract base class for all classes that want to register destructor
     * code that will be executed when the public method BaseObject.destoy()
     * has been invoked.
     *
     * @constructor
     */
    function BaseObject() {

        var // internal unique object identifier
            uid = _.uniqueId('obj'),

            // all event source objects and listeners
            listeners = null,

            // the last registered clone constructor
            cloneConstructor = null,

            // all clone getter callbacks
            cloneGetters = null,

            // all clone setter callbacks
            cloneSetters = null,

            // all destructor callbacks
            destructors = null;

        // protected methods --------------------------------------------------

        /**
         * Registers a callback function used when cloning this instance with
         * the method BaseObject.clone(). The last registered clone constructor
         * function (of the top-most sub class) will be used to receive a new
         * cloned instance of this instance. If no clone constructor has been
         * registered, the clone will be constructed by invoking the class
         * constructor directly.
         *
         * @attention
         *  Intended to be used by the internal implementations of sub classes.
         *  DO NOT CALL from external code!
         *
         * @param {Function} constructorFunc
         *  A callback function that must return a cloned instance of this
         *  instance. Will be called in the context of this object. If several
         *  sub classes register a clone constructor, only the last registered
         *  one will be used when cloning this instance. See method
         *  BaseObject.registerCloneHandler() for an advanced way of
         *  implementing cloning of complex sub classes.
         *
         * @returns {BaseObject}
         *  A reference to this instance.
         */
        this.registerCloneConstructor = function (constructorFunc) {
            cloneConstructor = constructorFunc;
            return this;
        };

        /**
         * Registers callback functions used when cloning this instance with
         * the method BaseObject.clone(). Used by subclasses to implement
         * cloning their private data members without having to expose them to
         * the public interface of the class.
         *
         * @attention
         *  Intended to be used by the internal implementations of sub classes.
         *  DO NOT CALL from external code!
         *
         * @param {Function} getterFunc
         *  A callback function that returns all internal data to be cloned.
         *  When cloning an object, the getter function of the source object
         *  will be invoked (in the context of the source object).
         *
         * @param {Function} setterFunc
         *  A callback function that sets the internal data of the new clone.
         *  When cloning an object, the setter function of the new clone will
         *  be invoked (in the context of that clone). The function receives
         *  the following parameters:
         *  (1) {Any} sourceData
         *      The return value of the getter function that has been invoked
         *      at the source object.
         *  (2) {BaseObject} sourceObj
         *      The source object itself.
         *
         * @returns {BaseObject}
         *  A reference to this instance.
         */
        this.registerCloneHandler = function (getterFunc, setterFunc) {
            (cloneGetters || (cloneGetters = [])).push(getterFunc);
            (cloneSetters || (cloneSetters = [])).push(setterFunc);
            return this;
        };

        /**
         * Registers a destructor callback function that will be invoked when
         * the method BaseObject.destroy() has been called.
         *
         * @attention
         *  Intended to be used by the internal implementations of sub classes.
         *  DO NOT CALL from external code!
         *
         * @param {Function} destructor
         *  A destructor callback function. The BaseObject.destroy() method
         *  will invoke all registered destructor callbacks in reverse order of
         *  registration. The callback will be invoked in the context of this
         *  instance.
         *
         * @returns {BaseObject}
         *  A reference to this instance.
         */
        this.registerDestructor = function (destructor) {
            (destructors || (destructors = [])).unshift(destructor);
            return this;
        };

        // methods ------------------------------------------------------------

        /**
         * Returns the globally unique identifier for this instance.
         *
         * @returns {String}
         *  The globally unique identifier for this instance.
         */
        this.getUid = function () {
            return uid;
        };

        /**
         * Registers an event listener at the specified event source object, or
         * a callback function for a specific event at a promise object. On
         * destruction of this instance, all event listeners registered with
         * this method will be removed from the source object, and the promise
         * callbacks will not be invoked anymore (it is not possible to remove
         * callbacks from promises explicitly).
         *
         * @param {jQuery|Events|jQuery.Promise} source
         *  The event source object. Can be any object that provides a method
         *  'on()' to register an event listener (for example, jQuery objects,
         *  or any object extended with the core Events mix-in class), or the
         *  promise of a Deferred object.
         *
         * @param {String} type
         *  The type of the event to listen to. In case the passed source
         *  object is an event source, this value can be a space-separated list
         *  of event type names. In case the passed source object is a promise,
         *  this type MUST be the name of one of the callback registration
         *  methods provided by the promise (e.g. 'done', 'fail', 'always',
         *  'progress', etc.).
         *
         * @param {Function} listener
         *  The event listener that will be invoked for the events triggered by
         *  the event source object, or the callback function that will be
         *  invoked when the promise will be resolved, rejected, or notified.
         *
         * @returns {BaseObject}
         *  A reference to this instance.
         */
        this.listenTo = function (source, type, listener) {

            if (_.isFunction(source.on)) {
                // register an event listener (prefer 'on' over 'promise' as jQuery collections provide a 'promise' method too)
                (listeners || (listeners = [])).push({ source: source, type: type, listener: listener });
                source.on(type, listener);
            } else if (_.isFunction(source.promise)) {
                // register promise callbacks
                source.promise()[type](_.bind(function () {
                    if (!this.destroyed) { return listener.apply(this, arguments); }
                }, this));
            }

            return this;
        };

        /**
         * Removes all event handlers that have been registered for the passed
         * event source object using the method BaseObject.listenTo().
         *
         * @param {jQuery|Events} source
         *  The event source object. Note that it is not possible to detach
         *  callback handlers from promises.
         *
         * @returns {BaseObject}
         *  A reference to this instance.
         */
        this.stopListeningTo = function (source) {

            // detach all event listeners for the passed object
            _.each(listeners, function (data) {
                if (data.source === source) {
                    source.off(data.type, data.listener);
                }
            });

            return this;
        };

        /**
         * Returns a clone of this instance. If a clone constructor has been
         * registered (see method BaseObject.registerCloneConstructor() above),
         * it will be invoked. Otherwise, the class constructor of this
         * instance will be invoked directly. Afterwards, all registered clone
         * handlers (see method BaseObject.registerCloneHandler() above) will
         * be called.
         *
         * @param {Any} ...
         *  All parameters required by the clone constructor or class
         *  constructor of this instance to create a new instance of the same
         *  class. After creating the new instance, the registered clone
         *  handlers will be used to copy the contents of this instance to the
         *  new instance.
         *
         * @returns {BaseObject}
         *  AThe new clone of this instance.
         */
        this.clone = function () {

            // helper function to construct a new instance using the apply() method
            function Creator(ctor, args) { return ctor.apply(this, args); }
            Creator.prototype = this.constructor.prototype;

            var // the arguments passed to this method, as array
                args = _.toArray(arguments),
                // the default-constructed clone
                cloned = _.isFunction(cloneConstructor) ? cloneConstructor.apply(this, args) : new Creator(this.constructor, args),
                // the clone setter callbacks
                setters = cloned.getCloneSetters();

            // invoke all registered clone handlers
            if (_.isArray(cloneGetters) && _.isArray(setters) && (cloneGetters.length === setters.length)) {
                _.each(cloneGetters, function (getterFunc, index) {
                    var setterFunc = setters[index];
                    setterFunc.call(cloned, getterFunc.call(this), this);
                }, this);
            } else if (_.isArray(cloneGetters) || _.isArray(setters)) {
                Utils.error('BaseObject.clone(): length mismatch of data getters and setters');
            }

            return cloned;
        };

        /**
         * Destroys this object. Invokes all destructor callback functions that
         * have been registered for this instance in reverse order. Afterwards,
         * all public properties and methods of this instance will be deleted,
         * and a single property 'destroyed' will be inserted and set to the
         * value true.
         */
        this.destroy = function () {

            // unregister all event listeners
            _.each(listeners, function (data) {
                // source object may have been destroyed already
                if (data.source.off) { data.source.off(data.type, data.listener); }
            });
            listeners = null;

            // reset the clone handlers
            cloneConstructor = cloneGetters = cloneSetters = null;

            // invoke all destructor callbacks
            _.each(destructors, function (destructor) { destructor.call(this); }, this);
            destructors = null;

            // delete all public members, to detect any misuse after destruction
            _.each(this, function (member, name) { delete this[name]; }, this);
            this.destroyed = true;
            this.uid = uid;
        };

        // initialization -----------------------------------------------------

        // INTERNAL: used by the method BaseObject.clone() to get the setters of the new clone
        this.getCloneSetters = function () { return cloneSetters; };

    } // class BaseObject

    // static methods ---------------------------------------------------------

    /**
     * Clones the objects contained in the passed array, by invoking their
     * clone() method, passing the specified parameters.
     *
     * @param {Array} array
     *  An array of objects (instances of subclasses of BaseObject), providing
     *  a clone() method.
     *
     * @param {Any} [...]
     *  Additional parameters that will be passed to the clone() methods of all
     *  objects contained in the array.
     *
     * @returns {Array}
     *  A new array containing clones of all passed objects.
     */
    BaseObject.cloneArray = function (array) {
        var args = _.toArray(arguments).splice(1);
        return _.map(array, function (entry) { return entry.clone.apply(entry, args); });
    };

    /**
     * Clones the objects contained in the passed map, by invoking their
     * clone() method, passing the specified parameters.
     *
     * @param {Object} map
     *  A map of objects (instances of subclasses of BaseObject), providing a
     *  clone() method.
     *
     * @param {Any} [...]
     *  Additional parameters that will be passed to the clone() methods of all
     *  objects contained in the map.
     *
     * @returns {Object}
     *  A new map containing clones of all passed objects, keyed by the same
     *  keys as in the source map.
     */
    BaseObject.cloneMap = function (map) {
        var args = _.toArray(arguments).splice(1), result = {};
        _.each(map, function (entry, key) { result[key] = entry.clone.apply(entry, args); });
        return result;
    };

    // exports ================================================================

    return _.makeExtendable(BaseObject);

});
