/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/control/button',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/forms',
     'io.ox/office/tk/control/group',
     'io.ox/office/tk/control/captionmixin',
     'io.ox/office/tk/control/widthmixin'
    ], function (Utils, Forms, Group, CaptionMixin, WidthMixin) {

    'use strict';

    // class Button ===========================================================

    /**
     * Creates a container element used to hold a push button or a toggle
     * button.
     *
     * @constructor
     *
     * @extends Group
     * @extends CaptionMixin
     * @extends WidthMixin
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options of the Group base class, all
     *  formatting options for button control elements supported by the method
     *  Forms.createButtonMarkup(), and all options of the mix-in class
     *  WidthMixin. Additionally, the following options are supported:
     *  @param {Any} [initOptions.value]
     *      A value, object, or function that will be used as fixed value for
     *      the button when it has been clicked. Must not be null.
     *      Ignored, if this button is a toggle button (see option 'toggle'
     *      below).
     *  @param {String} [initOptions.dataValue]
     *      A string that will be inserted into the 'data-value' attribute of
     *      the button node. If omitted, the JSON string representation of the
     *      'value' option will be used instead (all double-quote characters
     *      will be removed from the string though), unless the passed value is
     *      a function.
     *  @param {Boolean} [initOption.toggle=false]
     *      If set to true, the button represents a Boolean value and toggles
     *      its state when clicked.
     *  @param {Function} [initOptions.highlight]
     *      A predicate function that will be called every time after the value
     *      of the button has been set. The button will be highlighted if this
     *      handler function returns true. Receives the value of the button as
     *      first parameter. Will be called in the context of this button
     *      instance. Toggle buttons will highlight themselves automatically,
     *      if this option is omitted.
     *  @param {Object} [initOptions.dropDownVersion]
     *      If specified, the button will known, how it should act, if it will
     *      shrinked in a dropdown-menu.
     *  @param {Object} [initOptions.smallerVersion]
     *      If specified, the button will known, how it should act, if there
     *      will be not enough free place to show normal view
     */
    function Button(initOptions) {

        var // self reference
            self = this,

            // create the DOM button element
            buttonNode = $(Forms.createButtonMarkup(initOptions)),

            // toggle button or push button
            toggle = Utils.getBooleanOption(initOptions, 'toggle', false),

            // custom predicate callback for button highlighting
            highlightHandler = Utils.getFunctionOption(initOptions, 'highlight'),

            // special options for a drop-down-version of this button
            dropDownVersion = Utils.getObjectOption(initOptions, 'dropDownVersion'),

            // saves the (old) styles in case of switching to small version
            savedStyles = {},

            // smaller version of group (to fit in smaller resolutions)
            smallerVersion = Utils.getOption(initOptions, 'smallerVersion', false);

        // base constructor ---------------------------------------------------

        Group.call(this, initOptions);
        CaptionMixin.call(this, buttonNode);
        WidthMixin.call(this, buttonNode, initOptions);

        // private methods ----------------------------------------------------

        /**
         * The update handler for this button.
         */
        function updateHandler(value) {

            if (_.isFunction(highlightHandler)) {
                Forms.selectButtonNodes(buttonNode, highlightHandler.call(self, value));
            } else if (toggle) {
                Forms.selectButtonNodes(buttonNode, value);
            }

            // set value attributes for Selenium testing
            if (toggle) { Forms.setButtonValue(self.getNode(), value); }

            // do not change the fixed value of push buttons (several buttons
            // can be used to change the value of an item independently)
        }

        /**
         * Modifies the button for the dropdown-view
         *
         * @param {Object} dropDownVersion
         *  The following options are supported:
         *  @param {String} [dropDownVersion.label]
         *      If specified, the button gets a label
         */
        function createDropDownVersion(dropDownVersion){
            if (_.has(dropDownVersion, 'label')) {
                buttonNode.find('.caption').append(
                    Forms.createSpanMarkup(dropDownVersion.label, {
                        classes: 'drop-down'
                    })
                );
            }
        }

        /**
         * Activates/Deactivates a smaller version of the Button
         *
         * @param {Boolean} value
         *  decides wether the Button should be small or wide
         */
        function smallButton(value){
            // show the small version
            if (value === true) {
                // set new css if exists
                if (_.isObject(smallerVersion.css)) {
                    if (_.isEmpty(savedStyles.css) && !_.isEmpty(buttonNode.attr('style'))) {
                        savedStyles.css = buttonNode.attr('style');
                    }
                    buttonNode.css(smallerVersion.css);
                }

                // hide text of label (if set)
                if (smallerVersion.hideLabel) { buttonNode.addClass('hideLabel'); }

            // show the default
            } else {
                // re-set the old (saved) styles
                if (_.has(smallerVersion, 'css') && !_.isNull(savedStyles.css)) {
                    buttonNode.removeAttr('style');
                    buttonNode.attr('style',savedStyles.css);
                    savedStyles.css = null;
                // if there were noch old styles, remove the style-attribute
                } else if (_.has(smallerVersion, 'css')){
                    buttonNode.removeAttr('style');
                }

                // show text of label (if set)
                if (smallerVersion.hideLabel) { buttonNode.removeClass('hideLabel'); }
            }
        }

        // public methods -----------------------------------------------------

        /**
         * Returns the DOM button element, as jQuery object.
         *
         * @returns {jQuery}
         *  The DOM button element, as jQuery object.
         */
        this.getButtonNode = function () {
            return buttonNode;
        };

        /**
         * Overwrites the base-methods (group.js) to
         * activate/deactivate the small version of the button
         */
        this.activateSmallVersion = function(){
            if (_.isObject(smallerVersion)) {
                smallButton(true);
            }
        };
        this.deactivateSmallVersion = function(){
            if (_.isObject(smallerVersion)) {
                smallButton(false);
            }
        };

        // initialization -----------------------------------------------------

        // set the fixed value of push buttons
        if (!toggle) {
            Forms.setButtonValue(buttonNode, Utils.getOption(initOptions, 'value'), initOptions);
        }

        // creates the dropdown-version of this button, if specified
        if (dropDownVersion) {
            createDropDownVersion(dropDownVersion);
        }

        // insert the button into this group
        this.addChildNodes(buttonNode);

        // update handler for selected state
        this.registerUpdateHandler(updateHandler);

        // button click handler (convert ENTER and SPACE keys to click events)
        Forms.setButtonKeyHandler(buttonNode);
        buttonNode.on('click', function (event) {
            var value = toggle ? !self.getValue() : Forms.getButtonValue(buttonNode);
            self.triggerChange(value, { sourceEvent: event });
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            buttonNode.remove();
            initOptions = self = buttonNode = highlightHandler = null;
        });

    } // class Button

    // exports ================================================================

    // derive this class from class Group
    return Group.extend({ constructor: Button });

});
