/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/text/view/controls',
    ['io.ox/contacts/api',
     'io.ox/office/tk/utils',
     'io.ox/office/tk/forms',
     'io.ox/office/tk/errorcode',
     'io.ox/office/editframework/utils/lineheight',
     'io.ox/office/editframework/view/editcontrols',
     'io.ox/office/text/dom',
     'io.ox/office/text/format/listcollection',
     'gettext!io.ox/office/text'
    ], function (ContactsAPI, Utils, Forms, ErrorCode, LineHeight, EditControls, DOM, ListCollection, gt) {

    'use strict';

    var // class name shortcuts
        RadioGroup = EditControls.RadioGroup,
        RadioList = EditControls.RadioList,
        Picture = EditControls.Picture;

    // static class TextControls ==============================================

    /**
     * Additional classes defining specialized GUI controls for the OX Text
     * application.
     *
     * @extends EditControls
     */
    var TextControls = _.clone(EditControls);

    // class ParagraphAlignmentPicker =========================================

    /**
     * A picker control for horizontal text alignment in paragraphs.
     *
     * @constructor
     *
     * @extends RadioGroup
     */
    TextControls.ParagraphAlignmentPicker = RadioGroup.extend({ constructor: function (options) {

        // base constructor ---------------------------------------------------

        RadioGroup.call(this, Utils.extendOptions({
            role: 'radiogroup'
        }, options));

        // initialization -----------------------------------------------------

        var // accessible name for the widget
            ariaName =  /*#. horizontal text alignment in a paragraph */ gt('Paragraph alignment');

        // set accessible name - WCAG 2.0 Success Criterion 4.1.2
        this.getNode().attr({ title: ariaName, 'aria-label': ariaName, 'aria-labelledby': 'dummy' });

        this.createOptionButton('left',    { tooltip: gt('Left'),     attributes: { role: 'radio' }, icon: 'docs-para-align-left' })
            .createOptionButton('center',  { tooltip: gt('Center'),   attributes: { role: 'radio' }, icon: 'docs-para-align-center' })
            .createOptionButton('right',   { tooltip: gt('Right'),    attributes: { role: 'radio' }, icon: 'docs-para-align-right' })
            .createOptionButton('justify', { tooltip: gt('Justify'),  attributes: { role: 'radio' }, icon: 'docs-para-align-justify' });

    }}); // class ParagraphAlignmentPicker

    // class ParagraphAlignmentPickerDropDown =========================================

    /**
     * A picker control for horizontal text alignment in paragraphs.
     *
     * @constructor
     *
     * @extends RadioGroup
     */
    TextControls.ParagraphAlignmentPickerDropDown = RadioList.extend({ constructor: function (initOptions) {

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({
            icon: 'docs-para-align-left',
            tooltip: /*#. horizontal text alignment in a paragraph */ gt('Paragraph alignment'),
            updateCaptionMode: 'icon'
        }, initOptions));

        // initialization -----------------------------------------------------

        this.createOptionButton('left',    { icon: 'docs-para-align-left',      label: gt('Left')     })
            .createOptionButton('center',  { icon: 'docs-para-align-center',    label: gt('Center')   })
            .createOptionButton('right',   { icon: 'docs-para-align-right',     label: gt('Right')    })
            .createOptionButton('justify', { icon: 'docs-para-align-justify',   label: gt('Justify')  });

    }}); // class ParagraphAlignmentPickerDropDown

    // class EscapementGroup ==================================================

    /**
     * A button group control for text escapement (subscript/superscript).
     *
     * @constructor
     *
     * @extends RadioGroup
     */
    TextControls.EscapementGroup = RadioGroup.extend({ constructor: function (initOptions) {

        // base constructor ---------------------------------------------------

        RadioGroup.call(this, Utils.extendOptions({ toggleValue: 'baseline', role: 'radiogroup' }, {dropDownVersion: Utils.getOption(initOptions, 'dropDownVersion')}));

        // initialization -----------------------------------------------------

        var // accessible name for the widget
            ariaName =  /*#. text escapement (subscript/superscript) */ gt('Text escapement');

        // set accessible name - WCAG 2.0 Success Criterion 4.1.2
        this.getNode().attr({ title: ariaName, 'aria-label': ariaName, 'aria-labelledby': 'dummy' });

        this.createOptionButton('sub',   { icon: 'docs-font-subscript',   tooltip: gt('Subscript'),     attributes: { role: 'radio' }, dropDownVersion: { label: gt('Subscript')} })
            .createOptionButton('super', { icon: 'docs-font-superscript', tooltip: gt('Superscript'),   attributes: { role: 'radio' }, dropDownVersion: { label: gt('Superscript')} });

    }});

    // class LineHeightPicker =================================================

    /**
     * A picker control for the line height in paragraphs.
     *
     * @constructor
     *
     * @extends RadioList
     */
    TextControls.LineHeightPicker = RadioList.extend({ constructor: function () {

        // base constructor ---------------------------------------------------

        RadioList.call(this, {
            icon: 'docs-para-line-spacing-100',
            tooltip: /*#. distance between lines of text in a paragraph */ gt('Line spacing'),
            updateCaptionMode: 'icon',
            dropDownVersion: {
                label: gt('Line spacing')
            }
        });

        // initialization -----------------------------------------------------

        this.createOptionButton(LineHeight.SINGLE,   { icon: 'docs-para-line-spacing-100', label: /*#. text line spacing in paragraphs */ gt('100%') })
            .createOptionButton(LineHeight._115,     { icon: 'docs-para-line-spacing-115', label: /*#. text line spacing in paragraphs */ gt('115%') })
            .createOptionButton(LineHeight.ONE_HALF, { icon: 'docs-para-line-spacing-150', label: /*#. text line spacing in paragraphs */ gt('150%') })
            .createOptionButton(LineHeight.DOUBLE,   { icon: 'docs-para-line-spacing-200', label: /*#. text line spacing in paragraphs */ gt('200%') });

    }}); // class LineHeightPicker

    // class ParagraphStylePicker =============================================

    /**
     * A drop-down menu control for paragraph style sheets.
     *
     * @constructor
     *
     * @extends EditControls.StyleSheetPicker
     */
    TextControls.ParagraphStylePicker = EditControls.StyleSheetPicker.extend({ constructor: function (app) {

        var // self reference
            self = this,

            // paragraph style sheets
            paragraphStyles = app.getModel().getStyleCollection('paragraph');

        // base constructor ---------------------------------------------------

        EditControls.StyleSheetPicker.call(this, app, 'paragraph', {
            width: 175,
            icon: 'docs-pilcrow',
            tooltip: gt('Paragraph style'),
            gridColumns: 4,
            i18nModulePath: 'io.ox/office/text/resource/paragraphstylenames',
            smallerVersion: {
                css: {
                    width: 50
                },
                hideLabel: true
            }
        });

        // private methods ----------------------------------------------------

        /**
         * Formats the passed DOM element according to the specified paragraph
         * style sheet.
         */
        function setParagraphFormatting(buttonNode, styleId) {
            var captionNode = Forms.getCaptionNode(buttonNode);
            paragraphStyles.setElementAttributes(captionNode, { styleId: styleId }, { preview: true });
        }

        /**
         * Initializes a new list item according to the paragraph style sheet.
         */
        function createItemHandler(event, buttonNode, styleId) {
            setParagraphFormatting(buttonNode, styleId);
        }

        /**
         * Updates the drop-down menu button.
         */
        function updateHandler(styleId) {
            setParagraphFormatting(self.getMenuButton(), styleId);
        }

        // initialization -----------------------------------------------------

        this.getMenu().on('create:item', createItemHandler);
        this.registerUpdateHandler(updateHandler);

    }}); // class ParagraphStylePicker

    // class ListStylePicker ==================================================

    TextControls.ListStylePicker = RadioList.extend({ constructor: function (app, predefinedListStyles, initOptions) {

        var // the collection of fonts of the edited document
            fontCollection = app.getModel().getFontCollection(),
            // font attributes for the drop-down list entries
            labelStyle = 'font-family:' + fontCollection.getCssFontFamily('Cambria') + ';font-size:18px;min-width:18px;';

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({
            highlight: function (listStyleId) { return listStyleId !== ''; },
            splitValue: ListCollection.DEFAULT_VALUE,
            updateCaptionMode: 'none',
            itemDesign: 'grid',
            sortItems: true
        }, initOptions));

        // initialization -----------------------------------------------------

        this.getMenuNode().addClass('list-style-picker');

        this.createMenuSection('default').createMenuSection('special', { gridColumns: 1 });
        _(predefinedListStyles).each(function (listStyle, listStyleId) {
            if (Utils.trimAndCleanString(listStyle.listLabel).length > 0) {
                this.createOptionButton(listStyleId, {
                    section: 'default',
                    label: _.noI18n(listStyle.listLabel),
                    labelStyle: labelStyle,
                    tooltip: _.noI18n(listStyle.tooltip),
                    dataValue: listStyle.listKey,
                    sortIndex: listStyleId
                });
            } else {
                this.createOptionButton(listStyleId, {
                    section: 'special',
                    label: _.noI18n(listStyle.tooltip),
                    dataValue: listStyle.listKey
                });
            }
        }, this);

    }}); // class ListStylePicker

    // class BulletListStylePicker ============================================

    TextControls.BulletListStylePicker = TextControls.ListStylePicker.extend({ constructor: function (app, initOptions) {

        // base constructor ---------------------------------------------------

        TextControls.ListStylePicker.call(this, app, ListCollection.getPredefinedBulletListStyles(), Utils.extendOptions({
            icon: 'docs-list-bullet',
            tooltip: gt('Bullet list'),
            gridColumns: 4,
            dropDownVersion: {
                label: gt('Bullet list')
            }
        }, initOptions));

    }}); // class BulletListStylePicker

    // class NumberedListStylePicker ==========================================

    TextControls.NumberedListStylePicker = TextControls.ListStylePicker.extend({ constructor: function (app, initOptions) {

        // base constructor ---------------------------------------------------

        TextControls.ListStylePicker.call(this, app, ListCollection.getPredefinedNumberedListStyles(), Utils.extendOptions({
            icon: 'docs-list-numbered',
            tooltip: gt('Numbered list'),
            gridColumns: 4,
            dropDownVersion: {
                label: gt('Numbered list')
            }
        }, initOptions));

    }}); // class NumberedListStylePicker


    // class ZoomTypePicker ===================================================

    /**
     * The selector for additional zoom types.
     *
     * @constructor
     *
     * @extends RadioList
     */
    TextControls.ZoomTypePicker = RadioList.extend({ constructor: function (initOptions) {

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({
            icon: 'fa-ellipsis-v',
            tooltip: gt('More zoom settings'),
            caret: false,
            updateCaptionMode: 'none'
        }, initOptions));

        // initialization -----------------------------------------------------

        this.createMenuSection('fix')
            .createOptionButton(50,  { section: 'fix', label: gt('50%') })
            .createOptionButton(75,  { section: 'fix', label: gt('75%') })
            .createOptionButton(100, { section: 'fix', label: gt('100%') })
            .createOptionButton(150, { section: 'fix', label: gt('150%') })
            .createOptionButton(200, { section: 'fix', label: gt('200%') })
            .createMenuSection('fit')
            .createOptionButton('width', { section: 'fit', label: gt('Fit to screen width') });

    }}); // class ZoomTypePicker

    // class TableStylePicker =================================================

    /**
     * @constructor
     *
     * @extends EditControls.StyleSheetPicker
     */
    TextControls.TableStylePicker = EditControls.StyleSheetPicker.extend({ constructor: function (app) {

        var // self reference
            self = this,

            // table style sheets
            tableStyles = app.getModel().getStyleCollection('table'),

            /**
             * Registers a new table that will be formatted delayed with a
             * specific table style sheet.
             *
             * @param {jQuery} table
             *  The new table to be formatted.
             *
             * @param {String} styleId
             *  The identifier of the table style sheet.
             */
            updateTableFormatting = (function () {
                // this function MUST be defined before calling the base c'tor, as it will
                // be called from the 'create:item' event handler triggered by the base class

                var // which tables in list items need formatting
                    pendingTables = [],
                    // background loop processing all pending tables
                    formatTimer = null;

                // direct callback: called every time when updateTableFormatting() has been called
                function registerTable(tableNode, styleId) {
                    pendingTables.push({ tableNode: tableNode, styleId: styleId });
                }

                // deferred callback: called once after the specified timeout
                function formatTables() {

                    // check if the background loop is already running
                    if (formatTimer) { return; }

                    // create a new background loop that processes all pending tables
                    formatTimer = app.repeatDelayed(function () {

                        var // table info with table and style identifier of the next table to be formatted
                            tableInfo = null;

                        // find a pending table that is still in the drop-down menu (menu may have been cleared
                        // in the meantime), abort the background loop, if no more tables are available
                        while ((tableInfo = pendingTables.shift()) && !Utils.containsNode(self.getMenuNode(), tableInfo.tableNode)) {}
                        if (!tableInfo) { return Utils.BREAK; }

                        // update the formatting of the table
                        tableStyles.setElementAttributes(tableInfo.tableNode, {
                            styleId: tableInfo.styleId,
                            table: { exclude: ['lastRow', 'lastCol', 'bandsVert'] }
                        }, { preview: true });
                        tableInfo.tableNode.find(DOM.PARAGRAPH_NODE_SELECTOR).css({ margin: 0 });  // no margins in micro tables allowed
                    });

                    // forget reference to the timer, when all tables have been formatted
                    formatTimer.always(function () { formatTimer = null; });
                }

                // create and return the debounced updateTableFormatting() method (with a delay of 10ms per table)
                return app.createDebouncedMethod(registerTable, formatTables, { delay: 10 });

            }()); // updateTableFormatting()

        // base constructors --------------------------------------------------

        EditControls.StyleSheetPicker.call(this, app, 'table', {
            width: 200,
            icon: 'docs-table-style',
            tooltip: gt('Table style'),
            gridColumns: 7,
            i18nModulePath: 'io.ox/office/text/resource/tablestylenames',
            smallerVersion: {
                css: {
                    width: 50
                },
                hideLabel: true
            }
        });

        // private methods ----------------------------------------------------

        /**
         * Initializes a new list item according to the table style sheet.
         */
        function createItemHandler(event, buttonNode, styleId) {

            var paragraphNode = DOM.createParagraphNode().css({ minWidth: 8, minHeight: 4 }),
                cellNode = DOM.createTableCellNode(paragraphNode, { plain: true }),
                cellMarkup = cellNode[0].outerHTML,
                rowMarkup = '<tr>' + Utils.repeatString(cellMarkup, 5) + '</tr>',
                tableNode = $('<table><tbody>' + Utils.repeatString(rowMarkup, 5) + '</tbody></table>');

            buttonNode.addClass('mini-caption').prepend(tableNode);
            updateTableFormatting(tableNode, styleId);
        }

        // initialization -----------------------------------------------------

        // register a handler that inserts a table element into each list item
        this.getMenu().on('create:item', createItemHandler);

    }}); // class TableStylePicker

    // class DrawingPositionPicker ============================================

    /**
     *
     * Drop-down menu control for position and text floating of drawing
     * objects in text documents.
     *
     * @constructor
     *
     * @extends RadioList
     */
    TextControls.DrawingPositionPicker = RadioList.extend({ constructor: function () {

        // base constructor ---------------------------------------------------

        RadioList.call(this, {
            icon: 'docs-drawing-inline',
            label: /*#. alignment and text floating of drawing objects in text documents */ gt.pgettext('drawing-pos', 'Position'),
            tooltip: /*#. alignment and text floating of drawing objects in text documents */ gt.pgettext('drawing-pos', 'Drawing position'),
            updateCaptionMode: 'icon'
        });

        // initialization -----------------------------------------------------

        this.createMenuSection('inline')
            .createOptionButton('inline',       { section: 'inline', icon: 'docs-drawing-inline',       label: /*#. drawing object position in paragraph */ gt.pgettext('drawing-pos', 'Inline with text') })
            .createMenuSection('left')
            .createOptionButton('left:none',    { section: 'left',   icon: 'docs-drawing-left-none',    label: /*#. drawing object position in paragraph */ gt.pgettext('drawing-pos', 'Left aligned, no text wrapping') })
            .createOptionButton('left:right',   { section: 'left',   icon: 'docs-drawing-left-right',   label: /*#. drawing object position in paragraph */ gt.pgettext('drawing-pos', 'Left aligned, text wraps at right side') })
            .createMenuSection('right')
            .createOptionButton('right:none',   { section: 'right',  icon: 'docs-drawing-right-none',   label: /*#. drawing object position in paragraph */ gt.pgettext('drawing-pos', 'Right aligned, no text wrapping') })
            .createOptionButton('right:left',   { section: 'right',  icon: 'docs-drawing-right-left',   label: /*#. drawing object position in paragraph */ gt.pgettext('drawing-pos', 'Right aligned, text wraps at left side') })
            .createMenuSection('center')
            .createOptionButton('center:none',  { section: 'center', icon: 'docs-drawing-center-none',  label: /*#. drawing object position in paragraph */ gt.pgettext('drawing-pos', 'Centered, no text wrapping') })
            .createOptionButton('center:left',  { section: 'center', icon: 'docs-drawing-center-left',  label: /*#. drawing object position in paragraph */ gt.pgettext('drawing-pos', 'Centered, text wraps at left side') })
            .createOptionButton('center:right', { section: 'center', icon: 'docs-drawing-center-right', label: /*#. drawing object position in paragraph */ gt.pgettext('drawing-pos', 'Centered, text wraps at right side') });

    }}); // class DrawingPositionPicker


    // class ChangeTrackBadge ============================================

    /**
     *
     * Drop-down menu control for position and text floating of drawing
     * objects in text documents.
     *
     * @constructor
     *
     * @extends Image
     */
    TextControls.ChangeTrackBadge = Picture.extend({ constructor: function (app, initOptions) {

        var authorLinkNode = $('<a>').addClass('change-track-author').text(gt('Unknown')),
            dateNode = $('<div>').addClass('change-track-date').text(gt('Unknown')),
            actionNode = $('<div>').addClass('change-track-action').text(gt('Unknown')),
            descriptionNode = $('<div>').addClass('change-track-description'),
            placeHolderImageUrl = ox.base + '/apps/themes/default/dummypicture.png',
            imageOptions = { tooltip: gt('Author picture'), alt: gt('Author picture') },
            pictureHeight = _.device('retina') ? 96 : 48,
            pictureWidth = _.device('retina') ? 96 : 48;

        // base constructor ---------------------------------------------------

        Picture.apply(this, [placeHolderImageUrl, Utils.extendOptions(initOptions, imageOptions)]);

        // public methods -----------------------------------------------------

        // updates properties of the badge
        this.update = function (data) {
            if (!data) { return; }
            var self = this;
            // set placeholder picture as default and reset author halo link
            self.setSource(placeHolderImageUrl);
            authorLinkNode.removeClass('halo-link').removeData();
            // fetch user photo and enable halo view only if author OX User ID is available
            if (data.authorUserId) {
                // try to get the authors user data on the local AppSuite
                require(['io.ox/office/tk/io'], function (IO) {
                    IO.sendRequest({
                        module: 'oxodocumentfilter',
                        params: {
                            action: 'getuserinfo',
                            userId: data.authorUserId
                        }
                    }).done(function (userInfo) {
                        // quit if user data can not be found
                        var errorCode = new ErrorCode(userInfo);
                        if (errorCode.isError()) {
                            Utils.warn('TextControls.ChangeTrackBadge.update(): ' + errorCode.getErrorText());
                            return;
                        }
                        // if we found the author in the local user DB, try getting his picture
                        if (data.author && (userInfo.userData.displayName === data.author)) {
                            // enable author halo view
                            authorLinkNode.addClass('halo-link').data({ internal_userid : data.authorUserId });
                            // build user photo url
                            var pictureParams = { internal_userid: data.authorUserId, width: pictureWidth, height: pictureHeight, scaleType: 'cover' },
                                userPhotoSource = ContactsAPI.pictureHalo(null, pictureParams);
                            self.setSource(userPhotoSource);
                            // set also placeHolder image, when this particular user doesn't have an profile image yet.
                            var testImage = new Image();
                            testImage.src = userPhotoSource;
                            testImage.onload = function () { if (testImage.width === 1) { self.setSource(placeHolderImageUrl); } };
                            testImage.onerror = function () { Utils.warn('TextControls.ChangeTrackBadge.update(): Failed to load user profile picture'); };
                        }
                    });
                });
            }
            authorLinkNode.text(data.author);
            dateNode.text(data.date);
            actionNode.text(data.action);
        };

        // initialization -----------------------------------------------------
        // show placeholder image initially
        this.setSource(placeHolderImageUrl);

        // insert nodes to group node
        descriptionNode.append(authorLinkNode, actionNode);
        this.addChildNodes(descriptionNode, dateNode);

        // ARIA
        this.getNode().attr({ role: 'dialog'});

    }}); // class ChangeTrackBadge



    // exports ================================================================

    return TextControls;

});
