/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/text/format/characterstyles',
    ['io.ox/office/tk/utils',
     'io.ox/office/editframework/utils/color',
     'io.ox/office/editframework/utils/attributeutils',
     'io.ox/office/editframework/model/stylecollection',
     'io.ox/office/text/dom'
    ], function (Utils, Color, AttributeUtils, StyleCollection, DOM) {

    'use strict';

    var // definitions for character attributes
        DEFINITIONS = {

            fontName: {
                def: 'Arial',
                format: function (element, fontName) {
                    element.css('font-family', this.getCssFontFamily(fontName));
                },
                preview: true // 'true' calls the own 'format' method
            },

            fontSize: {
                def: 11,
                // format: done together with escapement in the format handler
                preview: function (element, fontSize) {
                    fontSize = Utils.round(10 + (fontSize - 10) / 1.5, 0.1);
                    element.css('font-size', Utils.minMax(fontSize, 6, 22) + 'pt');
                }
            },

            bold: {
                def: false,
                format: function (element, state) {
                    element.css('font-weight', state ? 'bold' : 'normal');
                },
                preview: true // 'true' calls the own 'format' method
            },

            italic: {
                def: false,
                format: function (element, state) {
                    element.css('font-style', state ? 'italic' : 'normal');
                },
                preview: true // 'true' calls the own 'format' method
            },

            underline: {
                def: false
            },

            strike: {
                def: 'none'
            },

            vertAlign: { def: 'baseline' },

            color: {
                def: Color.AUTO,
                // format: color will be set in format handler, depending on fill colors
                preview: function (element, color) {
                    element.css('color', this.getCssColor(color, 'text'));
                }
            },

            fillColor: {
                def: Color.AUTO,
                format: function (element, color) {
                    element.css('background-color', this.getCssColor(color, 'fill'));
                }
            },

            language: {
                def: '',
                format: function (element, value) {
                    element.attr('lang', value);
                }
            },

            url: {
                def: '',
                scope: 'element',
                format: function (element, url) {
                    if (_.isString(url) && (url.length > 0)) {
                        element.attr('title', url);
                    } else {
                        element.removeAttr('title');
                    }
                }
            },

            // special attributes

            spellerror: {
                def: false,
                scope: 'element',
                format: function (element, state) {
                    element.toggleClass('spellerror', state);
                },
                special: true
            },

            highlight: {
                def: false,
                scope: 'element',
                format: function (element, state) {
                    element.toggleClass('highlight', state);
                },
                special: true
            }

        },

        PARENT_RESOLVERS = {
            paragraph: function (span) { return span.closest(DOM.PARAGRAPH_NODE_SELECTOR); }
        };

    // class CharacterStyles ==================================================

    /**
     * Contains the style sheets for character formatting attributes. The CSS
     * formatting will be written to text span elements contained somewhere in
     * the paragraph elements.
     *
     * @constructor
     *
     * @extends StyleCollection
     *
     * @param {TextApplication} app
     *  The root application instance.
     *
     * @param {TextDocumentStyles} documentStyles
     *  Collection with the style containers of all style families.
     */
    function CharacterStyles(app, documentStyles) {

        var // self reference
            self = this;

        // base constructor ---------------------------------------------------

        StyleCollection.call(this, app, documentStyles, 'character', {
            parentResolvers: PARENT_RESOLVERS,
            formatHandler: updateCharacterFormatting,
            additionalFamilies: 'changes'
        });

        // private methods ----------------------------------------------------

        /**
         * Will be called for every text span whose character attributes have
         * been changed.
         *
         * @param {jQuery} textSpan
         *  The text span whose character attributes have been changed, as
         *  jQuery object.
         *
         * @param {Object} mergedAttributes
         *  A map of attribute maps (name/value pairs), keyed by attribute
         *  family, containing the effective attribute values merged from style
         *  sheets and explicit attributes.
         */
        function updateCharacterFormatting(textSpan, mergedAttributes) {

            var // the parent paragraph of the node (may be a grandparent)
                paragraph = $(textSpan).closest(DOM.PARAGRAPH_NODE_SELECTOR),
                // the closest cell node (if it exists)
                cell = $(textSpan).closest(DOM.TABLE_CELLNODE_SELECTOR),
                // the paragraph style container
                paragraphStyles = documentStyles.getStyleCollection('paragraph'),
                // the merged attributes of the paragraph
                paragraphAttributes = paragraphStyles.getElementAttributes(paragraph).paragraph,

                // the character attributes of the passed attribute map
                characterAttributes = mergedAttributes.character,
                // effective text color, font size, and vertical offset
                textColor = characterAttributes.color,
                fontSize = characterAttributes.fontSize,
                textDecoration = self.getCssTextDecoration(characterAttributes),
                verticalAlign = 'baseline',
                // the background color of the cell
                cellBackgroundColor = null,
                // an array containing all affected fill colors of text, paragraph and cell
                allFillColors = [];

            //restriction of max font size for small devices
            if (Utils.SMALL_DEVICE && fontSize > 16) {
                fontSize = 16;
            }

            // Checking also the background-color of an affected cell, if there is one (28988).
            // For performance reasons, not using attributes (might be assigned to row or table, too),
            // but checking css style 'background-color' of the table directly.
            if (cell.length > 0) {
                cellBackgroundColor = cell.css('background-color');
                if (cellBackgroundColor) { allFillColors.push(Color.convertCssColorToRgbColor(cellBackgroundColor)); }
            }

            // adding fill color of paragraph and character (order is important)
            allFillColors.push(paragraphAttributes.fillColor);
            allFillColors.push(characterAttributes.fillColor);

            // calculate effective text color, according to fill colors
            textColor = self.getCssTextColor(textColor, allFillColors);

            // calculate font height and vertical alignment (according to escapement)
            switch (characterAttributes.vertAlign) {
            case 'sub':
                fontSize = Utils.round(fontSize * 0.66, 0.1);
                verticalAlign = '-30%';
                break;
            case 'super':
                fontSize = Utils.round(fontSize * 0.66, 0.1);
                verticalAlign = '50%';
                break;
            }

            // update CSS of the text span
            textSpan.css({
                color: textColor,
                fontSize: fontSize + 'pt',
                textDecoration: textDecoration,
                verticalAlign: verticalAlign
            });

            // update line height due to changed font settings
            self.updateElementLineHeight(textSpan, paragraphAttributes.lineHeight, characterAttributes);

            // change track attribute handling
            app.getModel().getChangeTrack().updateChangeTrackAttributes(textSpan, mergedAttributes);

            // TODO: set bullet character formatting according to paragraph attributes
            // Bug 30794: We want have a consistent list-label formatting, therefore only the
            // code in updateList in editor should be responsible to set character formatting.
            // This should be extended as the user is currently only able to change the list-label
            // formatting via character attributes in paragraph styles.
            //$(paragraph).find('> ' + DOM.LIST_LABEL_NODE_SELECTOR + ' > span').css('font-size', characterAttributes.fontSize + 'pt');
        }

        // methods ------------------------------------------------------------

        /**
         * Sets the text line height of the specified DOM element.
         *
         * @param {HTMLElement|jQuery} element
         *  The element whose line height will be changed. If this object is a
         *  jQuery collection, uses the first DOM node it contains.
         *
         * @param {Object} lineHeight
         *  The new line height value. Must contain the properties 'type' and
         *  'value'.
         *
         * @param {Object} attributes
         *  Character formatting attributes influencing the normal line height.
         *  @param {String} attributes.fontName
         *      The name of the original font family (case-insensitive).
         *  @param {Number} attributes.fontSize
         *      The font size, in points.
         *  @param {Boolean} [attributes.bold=false]
         *      Whether the text will be rendered in bold characters.
         *  @param {Boolean} [attributes.italic=false]
         *      Whether the text will be rendered in italic characters.
         */
        this.updateElementLineHeight = function (element, lineHeight, charAttributes) {

            var // the font collection
                fontCollection = documentStyles.getFontCollection(),
                // effective line height in pixels (start with passed value, converted from 1/100 mm)
                height = Utils.convertHmmToLength(lineHeight.value, 'px');

            // calculate effective line height
            switch (lineHeight.type) {
            case 'fixed':
                // line height as passed
                break;
            case 'leading':
                height += fontCollection.getNormalLineHeight(charAttributes);
                break;
            case 'atLeast':
                height = Math.max(height, fontCollection.getNormalLineHeight(charAttributes));
                break;
            case 'percent':
                height = fontCollection.getNormalLineHeight(charAttributes) * lineHeight.value / 100;
                break;
            case 'normal':
                height = fontCollection.getNormalLineHeight(charAttributes);
                break;
            default:
                Utils.error('CharacterStyles.updateElementLineHeight(): invalid line height type');
            }

            // set the CSS formatting
            $(element).first().css('line-height', Math.ceil(height) + 'px');
        };

        // initialization -----------------------------------------------------

        // register the attribute definitions for the style family
        documentStyles.registerAttributeDefinitions('character', DEFINITIONS);

    } // class CharacterStyles

    // static methods ---------------------------------------------------------

    /**
     * Tries to merge the passed text span with its next or previous sibling
     * text span. To be able to merge two text spans, they must contain equal
     * formatting attributes. If merging was successful, the sibling span will
     * be removed from the DOM.
     *
     * @param {HTMLElement|jQuery} node
     *  The DOM node to be merged with its sibling text span. If this object is
     *  a jQuery object, uses the first DOM node it contains.
     *
     * @param {Boolean} next
     *  If set to true, will try to merge with the next span, otherwise with
     *  the previous text span.
     */
    CharacterStyles.mergeSiblingTextSpans = function (node, next) {

        var // the sibling text span, depending on the passed direction
            sibling = null,
            // text in the passed and in the sibling node
            text = null, siblingText = null;

        // passed node and sibling node, as DOM nodes
        node = Utils.getDomNode(node);
        sibling = node[next ? 'nextSibling' : 'previousSibling'];

        // both nodes must be text spans with the same attributes
        if (sibling && DOM.isTextSpan(node) && DOM.isTextSpan(sibling) && AttributeUtils.hasEqualElementAttributes(node, sibling)) {

            // add text of the sibling text node to the passed text node
            text = node.firstChild.nodeValue;
            siblingText = sibling.firstChild.nodeValue;
            node.firstChild.nodeValue = next ? (text + siblingText) : (siblingText + text);

            // remove the entire sibling span element
            $(sibling).remove();
        }
    };

    // exports ================================================================

    // derive this class from class StyleCollection
    return StyleCollection.extend({ constructor: CharacterStyles });

});
