/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/render/renderutils',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/logger',
     'io.ox/office/editframework/utils/border',
     'io.ox/office/spreadsheet/utils/sheetutils'
    ], function (Utils, Logger, Border, SheetUtils) {

    'use strict';

    var // border line styles, mapping rendering priority (less numbers are higher priority)
        BORDER_STYLE_PRIOS = { solid: 1, dashed: 2, dotted: 3, dashDot: 4, dashDotDot: 5 };

    // static class RenderUtils ===============================================

    /**
     * The static class 'RenderUtils' is a console logger bound to the URL hash
     * flag 'spreadsheet:log-renderer', that logs detailed information about
     * the cell caching and rendering process.
     */
    var RenderUtils = new Logger({ enable: 'spreadsheet:log-renderer', prefix: 'RENDER' });

    // public methods ---------------------------------------------------------

    /**
     * Converts the passed border attribute value used in document operations
     * to a border style descriptor with specific properties used in the cell
     * renderer and rendering cache.
     *
     * @param {Object} border
     *  A border attribute value, as used in document operations.
     *
     * @param {SpreadsheetDocumentStyles} documentStyles
     *  The document styles collection needed to resolve scheme colors.
     *
     * @param {Object} gridColor
     *  The color of the grid lines, needed to resolve automatic border color.
     *
     * @returns {Object|Null}
     *  A border style descriptor for a visible border, or null for an
     *  invisible border. The descriptor contains the following properties:
     *  - {String} style
     *      Effective dash style of the border line: one of 'solid', 'dotted',
     *      'dashed', 'dashDot', or 'dashDotDot'.
     *  - {Number} width
     *      Effective width of the border line, in pixels (positive integer).
     *      In case of double lines, this value will not be less than 3, and
     *      specifies the total width of the entire border. In case of hair
     *      lines, this value will be 1.
     *  - {Boolean} hair
     *      Whether the border consists of a hair line (original width is less
     *      than one pixel).
     *  - {Boolean} double
     *      Whether the border consists of two parallel lines.
     *  - {Object} color
     *      A descriptor for the line color. See method Color.getColorDetails()
     *      for details about this object.
     */
    RenderUtils.getBorderStyle = function (border, documentStyles, gridColor) {

        // style is 'none', or the 'width' property is not positive: no border
        if (!Border.isVisibleBorder(border)) { return null; }

        var // the line width, in pixels (rounded, zero is hair line), restricted to minimum  cell
            // size to prevent that left/right or top/bottom border of a cell overlay each other
            width = Math.min(Math.round(Utils.convertHmmToLength(border.width, 'px')), SheetUtils.MIN_CELL_SIZE),
            // double lines
            double = border.style === 'double',
            // hair lines
            hair = !double && (width === 0),
            // convert color object to CSS color (semi-transparency for hair lines)
            color = documentStyles.getColorDetails(border.color, gridColor, hair ? { alphaMod: 0.4 } : null);

        // set effective minimum width in pixels (3 pixels for double lines: two lines, one pixel gap)
        width = Math.max(width, double ? 3 : 1);

        // create the border style object
        return { style: border.style, width: width, hair: hair, double: double, color: color };
    };

    /**
     * Returns the border with stronger visual appearance.
     *
     * @param {Object|Null} border1
     *  The first border style descriptor, as returned by the method
     *  RenderUtils.getBorderStyle(), or null for invisible borders.
     *
     * @param {Object|Null} border2
     *  The second border style descriptor, as returned by the method
     *  RenderUtils.getBorderStyle(), or null for invisible borders.
     *
     * @returns {Object|Null}
     *  The one of the passed borders with stronger visual appearance. If both
     *  borders are missing (invisible), null will be returned. If one of the
     *  passed borders is missing (invisible), the other border will be
     *  returned. If both borders are visible, one will be selected by checking
     *  the following conditions (in the specified order):
     *  - Compare border width: Thicker border wins over thinner border.
     *  - Compare line style: Double over solid over dashed over dotted over
     *      dash-dot over dash-dot-dot.
     *  - Compare line color by luma: Darker color wins over lighter color.
     */
    RenderUtils.getStrongerBorder = function (border1, border2) {

        // handle missing borders (return the other border)
        if (!border1) { return border2; }
        if (!border2) { return border1; }

        // compare border width (hair lines first)
        if (border1.hair && !border2.hair) { return border2; }
        if (!border1.hair && border2.hair) { return border1; }

        // compare pixel border width
        if (border1.width > border2.width) { return border1; }
        if (border1.width < border2.width) { return border2; }

        // compare border style (double border wins first)
        if (border1.double && !border2.double) { return border1; }
        if (!border1.double && border2.double) { return border2; }

        // compare remaining border styles by priority
        if (border1.style !== border2.style) {
            var prio1 = BORDER_STYLE_PRIOS[border1.style] || 9999,
                prio2 = BORDER_STYLE_PRIOS[border2.style] || 9999;
            return (prio1 < prio2) ? border1 : border2;
        }

        // compare luma value (darker color wins!)
        return (border1.color.y < border2.color.y) ? border1 : border2;
    };

    // exports ================================================================

    return RenderUtils;

});
