/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/utils/sheetutils',
    ['io.ox/office/tk/utils',
     'gettext!io.ox/office/spreadsheet'
    ], function (Utils, gt) {

    'use strict';

    var // maximum number of columns that will ever be allowed in a sheet (A to ZZZZ)
        MAX_COL_COUNT = 475254,

        // maximum number of rows that will ever be allowed in a sheet (1 to 99,999,999)
        MAX_ROW_COUNT = 99999999,

        // regular expression pattern matching a valid column name (A to ZZZZ)
        COL_PATTERN = '[A-Z]{1,4}',

        // regular expression pattern matching a row name (1 to 99,999,999)
        ROW_PATTERN = '0*[1-9][0-9]{0,7}',

        // regular expression object matching a column name only
        RE_COL_NAME = new RegExp('^' + COL_PATTERN + '$', 'i'),

        // regular expression object matching a row name only
        RE_ROW_NAME = new RegExp('^' + ROW_PATTERN + '$'),

        // regular expression object matching a cell name
        RE_CELL_NAME_A1 = new RegExp('^(' + COL_PATTERN + ')(' + ROW_PATTERN + ')$', 'i'),

        // regular expression object matching a range name
        RE_RANGE_NAME_A1 = new RegExp('^(' + COL_PATTERN + ROW_PATTERN + '):(' + COL_PATTERN + ROW_PATTERN + ')$', 'i');

    // class ErrorCode ========================================================

    /**
     * Instances of this class represent literal error codes, as used in cells,
     * in formula expressions, and as the result of formulas. Implemented as
     * named class to be able to use the 'instanceof' operator to distinguish
     * error codes from other cell values.
     *
     * @constructor
     *
     * @param {String} code
     *  The native string representation of the error code.
     *
     * @property {String} code
     *  The native string representation of the error code, as passed to the
     *  constructor.
     */
    function ErrorCode(code) { this.code = code; }

    // prototype methods ------------------------------------------------------

    /**
     * Returns whether the passed value is an error code object (an instance of
     * the class ErrorCode) of the same type as represented by this error code
     * object.
     *
     * @param {Any} value
     *  Any literal value that can be used in formulas.
     *
     * @returns {Boolean}
     *  Whether the passed value is an error code object with the same type as
     *  this error code.
     */
    ErrorCode.prototype.equals = function (value) {
        return (value instanceof ErrorCode) && (value.code === this.code);
    };

    // string representation for debug logging
    ErrorCode.prototype.toString = function () { return this.code; };

    // static class SheetUtils ================================================

    var SheetUtils = {};

    // constants --------------------------------------------------------------

    // TODO: get maximum size of columns/rows from somewhere
    /**
     * Maximum width of columns, in 1/100 millimeters.
     *
     * @constant
     */
    SheetUtils.MAX_COLUMN_WIDTH = 20000;

    // TODO: get maximum size of columns/rows from somewhere
    /**
     * Maximum height of rows, in 1/100 millimeters.
     *
     * @constant
     */
    SheetUtils.MAX_ROW_HEIGHT = 20000;

    /**
     * The maximum number of cells to be filled with one range operation, or
     * auto-fill operation.
     *
     * @constant
     */
    SheetUtils.MAX_FILL_CELL_COUNT = 2000;

    /**
     * The maximum number of entire columns/rows to be filled with one
     * auto-fill operation.
     *
     * @constant
     */
    SheetUtils.MAX_AUTOFILL_COL_ROW_COUNT = 10;

    /**
     * Maximum number of merged ranges to be created with a single operation.
     *
     * @constant
     */
    SheetUtils.MAX_MERGED_RANGES_COUNT = 1000;

    /**
     * The maximum number of cells to be filled when unmerging a range.
     *
     * @constant
     */
    SheetUtils.MAX_UNMERGE_CELL_COUNT = 5000;

    /**
     * The maximum number of columns to be changed with one row operation.
     *
     * @constant
     */
    SheetUtils.MAX_CHANGE_COLS_COUNT = 2000;

    /**
     * The maximum number of rows to be changed with one row operation.
     *
     * @constant
     */
    SheetUtils.MAX_CHANGE_ROWS_COUNT = 2000;

    /**
     * Maximum number of cells contained in the selection for local update of
     * selection settings (prevent busy JS if selection is too large).
     *
     * @constant
     */
    SheetUtils.MAX_SELECTION_CELL_COUNT = 100000;

    /**
     * Maximum number of characters for the absolute part of a number formatted
     * with the 'General' number format (in cells).
     *
     * @constant
     */
    SheetUtils.MAX_LENGTH_STANDARD_CELL = 11;

    /**
     * Maximum number of characters for the absolute part of a number formatted
     * with the 'General' number format (in cell edit mode).
     *
     * @constant
     */
    SheetUtils.MAX_LENGTH_STANDARD_EDIT = 21;

    /**
     * Minimum effective size of a cell, in pixels. Also used as maximum width
     * of cell border lines, to prevent that the left/right borders or the top/
     * bottom borders of a cell overlay each other.
     *
     * @constant
     */
    SheetUtils.MIN_CELL_SIZE = 5;

    /**
     * Minimum allowed zoom factor in the spreadsheet view.
     *
     * @constant
     */
    SheetUtils.MIN_ZOOM = 0.5;

    /**
     * Maximum allowed zoom factor in the spreadsheet view.
     *
     * @constant
     */
    SheetUtils.MAX_ZOOM = 8;

    /**
     * Whether multi-selection (multiple ranges at the same time) is supported.
     *
     * @constant
     */
    SheetUtils.MULTI_SELECTION = !Modernizr.touch;


    /**
     * Predefined native error codes, as used in operations.
     *
     * @constant
     */
    SheetUtils.ErrorCodes = {
        DIV0:  new ErrorCode('#DIV/0!'),
        NA:    new ErrorCode('#N/A'),
        NAME:  new ErrorCode('#NAME?'),
        NULL:  new ErrorCode('#NULL!'),
        NUM:   new ErrorCode('#NUM!'),
        REF:   new ErrorCode('#REF!'),
        VALUE: new ErrorCode('#VALUE!')
    };

    // public methods ---------------------------------------------------------

    /**
     * Returns whether the passed value is an error code object (an instance of
     * the internal class ErrorCode).
     *
     * @param {Any} value
     *  Any literal value that can be used in formulas.
     *
     * @returns {Boolean}
     *  Whether the passed value is an error code object.
     */
    SheetUtils.isErrorCode = function (value) {
        return value instanceof ErrorCode;
    };

    /**
     * Creates an error code object for the passed string representation of an
     * error code.
     *
     * @param {String} code
     *  The native string representation of an error code.
     *
     * @returns {ErrorCode}
     *  The error code object for the passed error code.
     */
    SheetUtils.makeErrorCode = function (code) {
        return new ErrorCode(code);
    };

    // cell and range addresses -----------------------------------------------

    /**
     * Checks the column and row indexes contained in the passed range, and
     * swaps them (in-place) if they are not ordered.
     *
     * @param {Object} range
     *  (in/out) The address of a range, that will be adjusted in-place if
     *  required.
     *
     * @returns {Object}
     *  A reference to the passed range.
     */
    SheetUtils.adjustRange = function (range) {
        var tmp = null;
        // sort columns and rows independently
        if (range.start[0] > range.end[0]) { tmp = range.start[0]; range.start[0] = range.end[0]; range.end[0] = tmp; }
        if (range.start[1] > range.end[1]) { tmp = range.start[1]; range.start[1] = range.end[1]; range.end[1] = tmp; }
        return range;
    };

    /**
     * Returns a new range with column and row indexes in the correct order.
     *
     * @param {Object} range
     *  The (unordered) address of a range.
     *
     * @returns {Object}
     *  A new adjusted cell range address.
     */
    SheetUtils.getAdjustedRange = function (range) {
        return {
            start: [Math.min(range.start[0], range.end[0]), Math.min(range.start[1], range.end[1])],
            end: [Math.max(range.start[0], range.end[0]), Math.max(range.start[1], range.end[1])]
        };
    };

    /**
     * Returns the number of columns covered by the passed cell range.
     *
     * @param {Object} range
     *  The address of the range.
     *
     * @returns {Number}
     *  The number of columns in the passed range.
     */
    SheetUtils.getColCount = function (range) {
        return range.end[0] - range.start[0] + 1;
    };

    /**
     * Returns the number of rows covered by the passed cell range.
     *
     * @param {Object} range
     *  The address of the range.
     *
     * @returns {Number}
     *  The number of rows in the passed range.
     */
    SheetUtils.getRowCount = function (range) {
        return range.end[1] - range.start[1] + 1;
    };

    /**
     * Returns the number of columns or rows covered by the passed cell range.
     *
     * @param {Object} range
     *  The address of the range.
     *
     * @param {Boolean} columns
     *  Whether to return the number of columns (true), or the number of rows
     *  (false) in the range.
     *
     * @returns {Number}
     *  The number of columns or rows in the passed range.
     */
    SheetUtils.getIndexCount = function (range, columns) {
        return (columns ? SheetUtils.getColCount : SheetUtils.getRowCount)(range);
    };

    /**
     * Returns the number of cells covered by the passed cell range.
     *
     * @param {Object} range
     *  The address of the range.
     *
     * @returns {Number}
     *  The number of cells in the passed range.
     */
    SheetUtils.getCellCount = function (range) {
        return SheetUtils.getColCount(range) * SheetUtils.getRowCount(range);
    };

    /**
     * Returns whether the passed range contains the specified column index.
     *
     * @param {Object} range
     *  The address of the range.
     *
     * @param {Number} col
     *  The zero-based column index to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed range contains the column index.
     */
    SheetUtils.rangeContainsCol = function (range, col) {
        return (range.start[0] <= col) && (col <= range.end[0]);
    };

    /**
     * Returns whether the passed range contains the specified row index.
     *
     * @param {Object} range
     *  The address of the range.
     *
     * @param {Number} row
     *  The zero-based row index to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed range contains the row index.
     */
    SheetUtils.rangeContainsRow = function (range, row) {
        return (range.start[1] <= row) && (row <= range.end[1]);
    };

    /**
     * Returns whether the passed range contains the specified column index.
     *
     * @param {Object} range
     *  The address of the range.
     *
     * @param {Number} index
     *  The zero-based column or row index to be checked.
     *
     * @param {Boolean} columns
     *  Whether to check the column index (true), or the row index (false).
     *
     * @returns {Boolean}
     *  Whether the passed range contains the column or row index.
     */
    SheetUtils.rangeContainsIndex = function (range, index, columns) {
        return (columns ? SheetUtils.rangeContainsCol : SheetUtils.rangeContainsRow)(range, index);
    };

    /**
     * Returns whether the passed bounding cell range contains the specified
     * cell.
     *
     * @param {Object} boundRange
     *  The addresses of the bounding cell range.
     *
     * @param {Number[]} address
     *  The cell address to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed bounding cell range contains the cell address.
     */
    SheetUtils.rangeContainsCell = function (boundRange, address) {
        return SheetUtils.rangeContainsCol(boundRange, address[0]) && SheetUtils.rangeContainsRow(boundRange, address[1]);
    };

    /**
     * Returns whether the cell address is one of the outer cells in the passed
     * bounding cell range (a cell located inside the range, but at its left,
     * right, top, or bottom border).
     *
     * @param {Object} boundRange
     *  The addresses of the bounding cell range.
     *
     * @param {Number[]} address
     *  The cell address to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed cell is one of the outer cells in the range.
     */
    SheetUtils.rangeContainsOuterCell = function (boundRange, address) {
        return SheetUtils.rangeContainsCell(boundRange, address) &&
            ((address[0] === boundRange.start[0]) || (address[0] === boundRange.end[0]) || (address[1] === boundRange.start[1]) || (address[1] === boundRange.end[1]));
    };

    /**
     * Returns whether the passed bounding cell range contains another cell
     * range.
     *
     * @param {Object} boundRange
     *  The addresses of the bounding cell range.
     *
     * @param {Object} range
     *  The cell range address to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed bounding cell range contains the other cell range.
     */
    SheetUtils.rangeContainsRange = function (boundRange, range) {
        return SheetUtils.rangeContainsCell(boundRange, range.start) && SheetUtils.rangeContainsCell(boundRange, range.end);
    };

    /**
     * Returns whether any of the cell ranges in the passed array contains the
     * specified column index.
     *
     * @param {Array} ranges
     *  The addresses of the cell ranges to be checked.
     *
     * @param {Number} col
     *  The zero-based column index to be checked.
     *
     * @returns {Boolean}
     *  Whether any of the passed cell ranges contains the column index.
     */
    SheetUtils.rangesContainCol = function (ranges, col) {
        return _.any(ranges, function (range) { return SheetUtils.rangeContainsCol(range, col); });
    };

    /**
     * Returns whether any of the cell ranges in the passed array contains the
     * specified row index.
     *
     * @param {Array} ranges
     *  The addresses of the cell ranges to be checked.
     *
     * @param {Number} row
     *  The zero-based row index to be checked.
     *
     * @returns {Boolean}
     *  Whether any of the passed cell ranges contains the row index.
     */
    SheetUtils.rangesContainRow = function (ranges, row) {
        return _.any(ranges, function (range) { return SheetUtils.rangeContainsRow(range, row); });
    };

    /**
     * Returns whether any of the cell ranges in the passed array contains the
     * specified column or row index.
     *
     * @param {Array} ranges
     *  The addresses of the cell ranges to be checked.
     *
     * @param {Number} index
     *  The zero-based column or row index to be checked.
     *
     * @param {Boolean} columns
     *  Whether to check the column index (true), or the row index (false).
     *
     * @returns {Boolean}
     *  Whether any of the passed cell ranges contains the column or row index.
     */
    SheetUtils.rangesContainIndex = function (ranges, index, columns) {
        return (columns ? SheetUtils.rangesContainCol : SheetUtils.rangesContainRow)(ranges, index);
    };

    /**
     * Returns whether any of the cell ranges in the passed array contains the
     * specified cell.
     *
     * @param {Array} ranges
     *  The addresses of the cell ranges to be checked.
     *
     * @param {Number[]} address
     *  The cell address to be checked.
     *
     * @returns {Boolean}
     *  Whether any of the passed cell ranges contains the cell address.
     */
    SheetUtils.rangesContainCell = function (ranges, address) {
        return _.any(ranges, function (range) { return SheetUtils.rangeContainsCell(range, address); });
    };

    /**
     * Returns the first range that contains the specified cell.
     *
     * @param {Array} ranges
     *  The addresses of the cell ranges to be searched.
     *
     * @param {Number[]} address
     *  The cell address to be checked.
     *
     * @returns {Object|Null}
     *  The address of the first range in the passed array that contains the
     *  cell, or null if no range could be found.
     */
    SheetUtils.findFirstRange = function (ranges, address) {
        var result = _.find(ranges, function (range) { return SheetUtils.rangeContainsCell(range, address); });
        return result ? result : null;
    };

    /**
     * Compares the passed cell addresses.
     *
     * @returns {Number}
     *  - A negative number, if address1 is located before address2 (either in
     *      a preceding row; or in the same row and in a preceding column.
     *  - A positive number, if address1 is located after address2 (either in a
     *      following row; or in the same row and in a following column.
     *  - Zero, if the addresses are equal.
     */
    SheetUtils.compareCells = function (address1, address2) {
        var rowDiff = address1[1] - address2[1];
        return (rowDiff === 0) ? (address1[0] - address2[0]) : rowDiff;
    };

    // range intersections and bounding range ---------------------------------

    /**
     * Returns the unique ranges of the passed range addresses. Does NOT join
     * overlapping or adjacent ranges.
     *
     * @param {Object|Array} ranges
     *  The address of a single cell range, or an array with range addresses.
     *
     * @returns {Array}
     *  The address of the resulting unique cell ranges.
     */
    SheetUtils.getUniqueRanges = function (ranges) {
        return _.unique(_.getArray(ranges), SheetUtils.getRangeName);
    };

    /**
     * Returns whether the passed ranges overlap with at least one cell.
     *
     * @param {Object} range1
     *  The address of the first range.
     *
     * @param {Object} range2
     *  The address of the second range.
     *
     * @returns {Boolean}
     *  Whether the passed ranges are overlapping.
     */
    SheetUtils.rangeOverlapsRange = function (range1, range2) {
        return (range1.start[0] <= range2.end[0]) && (range2.start[0] <= range1.end[0]) && (range1.start[1] <= range2.end[1]) && (range2.start[1] <= range1.end[1]);
    };

    /**
     * Returns whether at least one range in the first range list overlaps with
     * at least one range in the second range list.
     *
     * @param {Object|Array} ranges1
     *  The address of a single cell range, or an array with range addresses.
     *
     * @param {Object|Array} ranges2
     *  The address of a single cell range, or an array with range addresses.
     *
     * @returns {Boolean}
     *  Whether any ranges in the first list overlaps with any range in the
     *  second list.
     */
    SheetUtils.rangesOverlapRanges = function (ranges1, ranges2) {
        ranges1 = SheetUtils.getUniqueRanges(ranges1);
        ranges2 = SheetUtils.getUniqueRanges(ranges2);
        return _.any(ranges1, function (range1) {
            return _.any(ranges2, function (range2) {
                return SheetUtils.rangeOverlapsRange(range1, range2);
            });
        });
     };

    /**
     * Returns whether any two ranges in the passed array of cell range
     * addresses overlap each other.
     *
     * @param {Array} ranges
     *  The array of range addresses.
     *
     * @returns {Boolean}
     *  Whether any two of the passed ranges are overlapping each other.
     */
    SheetUtils.anyRangesOverlap = function (ranges) {
        // TODO: this has O(n^2) complexity, better algorithm?
        return _.any(ranges, function (range, index) {
            return SheetUtils.rangesOverlapRanges(ranges.slice(index + 1), range);
        });
    };

    /**
     * Returns the intersecting range of the passed ranges.
     *
     * @param {Object} range1
     *  The address of the first range.
     *
     * @param {Object} range2
     *  The address of the second range.
     *
     * @returns {Object|Null}
     *  The address of the cell range covered by both passed ranges, if
     *  existing; otherwise null.
     */
    SheetUtils.getIntersectionRange = function (range1, range2) {
        return SheetUtils.rangeOverlapsRange(range1, range2) ? {
            start: [Math.max(range1.start[0], range2.start[0]), Math.max(range1.start[1], range2.start[1])],
            end: [Math.min(range1.end[0], range2.end[0]), Math.min(range1.end[1], range2.end[1])]
        } : null;
    };

    /**
     * Returns the range addresses resulting from intersecting all ranges in
     * the passed array and the bounding range.
     *
     * @param {Object|Array} ranges
     *  The address of a single cell range, or an array with range addresses.
     *
     * @param {Object} boundRange
     *  The address of the bounding range.
     *
     * @returns {Array}
     *  The address of the resulting cell ranges covered by the bounding range.
     *  May be an empty array, if the bounding range does not contain any of
     *  the cells in the passed ranges.
     */
    SheetUtils.getIntersectionRanges = function (ranges, boundRange) {
        var result = [];
        _.each(_.getArray(ranges), function (range) {
            var intersectRange = SheetUtils.getIntersectionRange(range, boundRange);
            if (intersectRange) { result.push(intersectRange); }
        });
        return result;
    };

    /**
     * Iterates the range addresses resulting from intersecting all ranges in
     * the passed array and the bounding range. Ranges not contained in the
     * bounding range at all will be skipped in the iteration process.
     *
     * @param {Object|Array} ranges
     *  The address of a single cell range, or an array with range addresses.
     *
     * @param {Object} boundRange
     *  The address of the bounding range.
     *
     * @param {Function} iterator
     *  The iterator function called for all intersection ranges. Receives the
     *  following parameters:
     *  (1) {Object} intersectionRange
     *      The intersection of the original range and the bounding range.
     *  (2) {Object} originalRange
     *      The original range.
     *  (3) {Number} index
     *      The index of the original range in the passed array of ranges.
     *  If the iterator returns the Utils.BREAK object, the iteration
     *  process will be stopped immediately.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Object} [options.context]
     *      If specified, the iterator will be called with this context (the
     *      symbol 'this' will be bound to the context inside the iterator
     *      function).
     *  @param {Boolean} [options.reverse=false]
     *      If set to true, the resulting intersection ranges will be visited
     *      in reversed order.
     *
     * @returns {Utils.BREAK|Undefined}
     *  A reference to the Utils.BREAK object, if the iterator has returned
     *  Utils.BREAK to stop the iteration process, otherwise undefined.
     */
    SheetUtils.iterateIntersectionRanges = function (ranges, boundRange, iterator, options) {

        var // the calling context for the iterator function
            context = Utils.getOption(options, 'context'),
            // whether to iterate in reversed order
            reverse = Utils.getBooleanOption(options, 'reverse', false);

        return Utils.iterateArray(_.getArray(ranges), function (range, index) {

            var // the intersection between current range and bound range
                intersectRange = SheetUtils.getIntersectionRange(range, boundRange);

            // invoke the iterator, if the intersection is not empty
            return intersectRange ? iterator.call(context, intersectRange, range, index) : undefined;
        }, { reverse: reverse });
    };

    /**
     * Returns the bounding range of the passed cell ranges (the smallest range
     * that contains all passed ranges).
     *
     * @param {Object|Array} [...]
     *  The address of a single cell range, or an array with range addresses.
     *  The number of parameters that can be passed to this method is not
     *  limited.
     *
     * @returns {Object|Null}
     *  The address of the bounding range containing all passed ranges; or
     *  null, if no range address has been passed.
     */
    SheetUtils.getBoundingRange = function () {
        var boundRange = null;
        _.each(arguments, function (ranges) {
            _.each(_.getArray(ranges), function (range) {
                if (!boundRange) {
                    boundRange = _.copy(range, true);
                } else {
                    boundRange.start[0] = Math.min(boundRange.start[0], range.start[0]);
                    boundRange.start[1] = Math.min(boundRange.start[1], range.start[1]);
                    boundRange.end[0] = Math.max(boundRange.end[0], range.end[0]);
                    boundRange.end[1] = Math.max(boundRange.end[1], range.end[1]);
                }
            });
        });
        return boundRange;
    };

    // column/row intervals ---------------------------------------------------

    /**
     * Returns the number of indexes covered by the passed index interval.
     *
     * @param {Object} interval
     *  A single index interval, with the zero-based index properties 'first'
     *  and 'last'.
     *
     * @returns {Number}
     *  The number of indexes covered by the passed interval.
     */
    SheetUtils.getIntervalSize = function (interval) {
        return interval.last - interval.first + 1;
    };

    /**
     * Returns whether the passed interval contains the specified index.
     *
     * @param {Object} interval
     *  The index interval, with the zero-based index properties 'first' and
     *  'last'.
     *
     * @param {Number} index
     *  The zero-based index to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed interval contains the index.
     */
    SheetUtils.intervalContainsIndex = function (interval, index) {
        return (interval.first <= index) && (index <= interval.last);
    };

    /**
     * Returns whether the passed bounding interval contains another interval.
     *
     * @param {Object} boundInterval
     *  The bounding index interval, with the zero-based index properties
     *  'first' and 'last'.
     *
     * @param {Object} range
     *  The index interval to be checked, with the zero-based index properties
     *  'first' and 'last'.
     *
     * @returns {Boolean}
     *  Whether the passed bounding interval contains the other interval.
     */
    SheetUtils.intervalContainsInterval = function (boundInterval, interval) {
        return (boundInterval.first <= interval.first) && (interval.last <= boundInterval.last);
    };

    /**
     * Returns whether the passed intervals overlap with at least one index.
     *
     * @param {Object} interval1
     *  The first index interval, with the zero-based index properties 'first'
     *  and 'last'.
     *
     * @param {Object} interval2
     *  The second index interval, with the zero-based index properties 'first'
     *  and 'last'.
     *
     * @returns {Boolean}
     *  Whether the passed intervals are overlapping.
     */
    SheetUtils.intervalOverlapsInterval = function (interval1, interval2) {
        return (interval1.first <= interval2.last) && (interval2.first <= interval1.last);
    };

    /**
     * Returns the unique intervals form the passed index intervals. Does NOT
     * join overlapping or adjacent intervals.
     *
     * @param {Object|Array} intervals
     *  A single index interval, or an array with index intervals.
     *
     * @returns {Array}
     *  The resulting unique index intervals.
     */
    SheetUtils.getUniqueIntervals = function (intervals) {
        return _.unique(_.getArray(intervals), function (interval) { return interval.first + ':' + interval.last; });
    };

    /**
     * Returns the intersection of the passed index intervals.
     *
     * @param {Object} interval1
     *  The first index interval, with the zero-based index properties 'first'
     *  and 'last'.
     *
     * @param {Object} interval2
     *  The second index interval, with the zero-based index properties 'first'
     *  and 'last'.
     *
     * @returns {Object|Null}
     *  The intersection interval covered by both passed intervals, if
     *  existing; otherwise null.
     */
    SheetUtils.getIntersectionInterval = function (interval1, interval2) {
        var first = Math.max(interval1.first, interval2.first),
            last = Math.min(interval1.last, interval2.last);
        return (first <= last) ? { first: first, last: last } : null;
    };

    /**
     * Returns the index intervals resulting from intersecting all intervals in
     * the passed array and the bounding interval.
     *
     * @param {Object|Array} intervals
     *  A single index interval, or an array with index intervals.
     *
     * @param {Object} boundInterval
     *  The bounding index interval.
     *
     * @returns {Array}
     *  The resulting index intervals covered by the bounding interval. May be
     *  an empty array, if the bounding interval does not contain any of the
     *  passed intervals.
     */
    SheetUtils.getIntersectionIntervals = function (intervals, boundInterval) {
        var result = [];
        _.each(_.getArray(intervals), function (interval) {
            var intersectInterval = SheetUtils.getIntersectionInterval(interval, boundInterval);
            if (intersectInterval) { result.push(intersectInterval); }
        });
        return result;
    };

    /**
     * Returns the bounding interval of the passed index intervals (the
     * smallest interval that contains all passed intervals).
     *
     * @param {Object|Array} [...]
     *  A single index interval, or an array with index intervals. The number
     *  of parameters that can be passed to this method is not limited.
     *
     * @returns {Object|Null}
     *  The bounding interval containing all passed index intervals; or null,
     *  if no interval has been passed.
     */
    SheetUtils.getBoundingInterval = function () {
        var boundInterval = null;
        _.each(arguments, function (intervals) {
            _.each(_.getArray(intervals), function (interval) {
                if (!boundInterval) {
                    boundInterval = _.clone(interval);
                } else {
                    boundInterval.first = Math.min(boundInterval.first, interval.first);
                    boundInterval.last = Math.max(boundInterval.last, interval.last);
                }
            });
        });
        return boundInterval;
    };

    /**
     * Merges the passed column or row intervals.
     *
     * @param {Object|Array} intervals
     *  A single index interval, or an array with index intervals.
     *
     * @returns {Array}
     *  An array of intervals exactly covering the passed intervals. The
     *  intervals are sorted, and overlapping or adjacent source intervals have
     *  been merged into a single interval object.
     */
    SheetUtils.getUnifiedIntervals = function (intervals) {

        // make deep copy of the passed parameter (for in-place manipulation)
        intervals = _.chain(intervals).copy(true).getArray().value();

        // make deep copy, and sort the intervals by start index
        intervals.sort(function (i1, i2) { return i1.first - i2.first; });

        // merge overlapping intervals
        _.each(intervals, function (interval, index) {
            var nextInterval = null;
            // extend current interval, as long as adjacent or overlapping with next interval
            while ((nextInterval = intervals[index + 1]) && (interval.last + 1 >= nextInterval.first)) {
                interval.last = Math.max(interval.last, nextInterval.last);
                intervals.splice(index + 1, 1);
            }
        });

        return intervals;
    };

    /**
     * Returns the column interval covered by the passed cell range.
     *
     * @param {Object} range
     *  The address of the cell range.
     *
     * @returns {Object}
     *  The column interval covering the passed cell range, containing the
     *  zero-based index properties 'first' and 'last'.
     */
    SheetUtils.getColInterval = function (range) {
        return { first: range.start[0], last: range.end[0] };
    };

    /**
     * Returns the sorted and merged column intervals covering the passed range
     * addresses.
     *
     * @param {Object|Array} ranges
     *  The address of a single cell range, or an array with range addresses.
     *
     * @returns {Array}
     *  The column intervals covering the passed cell ranges. Each interval
     *  object contains the zero-based index properties 'first' and 'last'. The
     *  intervals are sorted, and overlapping or adjacent intervals have been
     *  merged into a single interval object.
     */
    SheetUtils.getColIntervals = function (ranges) {
        var intervals = _.map(_.getArray(ranges), SheetUtils.getColInterval);
        return SheetUtils.getUnifiedIntervals(intervals);
    };

    /**
     * Returns the row interval covered by the passed cell range.
     *
     * @param {Object} range
     *  The address of the cell range.
     *
     * @returns {Object}
     *  The row interval covering the passed cell range, containing the
     *  zero-based index properties 'first' and 'last'.
     */
    SheetUtils.getRowInterval = function (range) {
        return { first: range.start[1], last: range.end[1] };
    };

    /**
     * Returns the sorted and merged row intervals covering the passed range
     * addresses.
     *
     * @param {Object|Array} ranges
     *  The address of a single cell range, or an array with range addresses.
     *
     * @returns {Array}
     *  The row intervals covering the passed cell ranges. Each interval object
     *  contains the zero-based index properties 'first' and 'last'. The
     *  intervals are sorted, and overlapping or adjacent intervals have been
     *  merged into a single interval object.
     */
    SheetUtils.getRowIntervals = function (ranges) {
        var intervals = _.map(_.getArray(ranges), SheetUtils.getRowInterval);
        return SheetUtils.getUnifiedIntervals(intervals);
    };

    /**
     * Returns the column or row interval covered by the passed cell range.
     *
     * @param {Object} range
     *  The address of the cell range.
     *
     * @param {Boolean} columns
     *  Whether to return the column interval (true), or the row interval
     *  (false) of the range.
     *
     * @returns {Object}
     *  The column or row interval covering the passed cell range, containing
     *  the zero-based index properties 'first' and 'last'.
     */
    SheetUtils.getInterval = function (range, columns) {
        return (columns ? SheetUtils.getColInterval : SheetUtils.getRowInterval)(range);
    };

    /**
     * Returns the sorted and merged column or row intervals covering the
     * passed range addresses.
     *
     * @param {Object|Array} ranges
     *  The address of a single cell range, or an array with range addresses.
     *
     * @param {Boolean} columns
     *  Whether to return the merged column intervals (true), or the merged row
     *  intervals (false) of the ranges.
     *
     * @returns {Array}
     *  The column or row intervals covering the passed cell ranges. Each
     *  interval object contains the zero-based index properties 'first' and
     *  'last'. The intervals are sorted, and overlapping or adjacent intervals
     *  have been merged into a single interval object.
     */
    SheetUtils.getIntervals = function (ranges, columns) {
        return (columns ? SheetUtils.getColIntervals : SheetUtils.getRowIntervals)(ranges);
    };

    /**
     * Builds a range address from the passed column and row interval.
     *
     * @param {Object|Number} colInterval
     *  The column interval, in the zero-based column index properties 'first'
     *  and 'last', or a single zero-based column index.
     *
     * @param {Object|Number} rowInterval
     *  The row interval, in the zero-based row index properties 'first' and
     *  'last', or a single zero-based row index.
     *
     * @returns {Object}
     *  The range address built from the passed intervals.
     */
    SheetUtils.makeRangeFromIntervals = function (colInterval, rowInterval) {
        return {
            start: [_.isNumber(colInterval) ? colInterval : colInterval.first, _.isNumber(rowInterval) ? rowInterval : rowInterval.first],
            end: [_.isNumber(colInterval) ? colInterval : colInterval.last, _.isNumber(rowInterval) ? rowInterval : rowInterval.last]
        };
    };

    /**
     * Builds an array of range addresses from the passed lists of column and
     * row intervals.
     *
     * @param {Object|Number|Array} colIntervals
     *  A column interval, in the zero-based column index properties 'first'
     *  and 'last', or a single zero-based column index, or an array of column
     *  intervals (with interval objects, and/or zero-based column indexes).
     *
     * @param {Object|Number|Array} rowIntervals
     *  A row interval, in the zero-based row index properties 'first' and
     *  'last', or a single zero-based row index, or an array of row intervals
     *  (with interval objects, and/or zero-based row indexes).
     *
     * @returns {Array}
     *  The range addresses built from the passed intervals.
     */
    SheetUtils.makeRangesFromIntervals = function (colIntervals, rowIntervals) {

        var ranges = [];

        colIntervals = _.getArray(colIntervals);
        rowIntervals = _.getArray(rowIntervals);

        if ((colIntervals.length > 0) && (rowIntervals.length > 0)) {
            _.each(colIntervals, function (colInterval) {
                _.each(rowIntervals, function (rowInterval) {
                    ranges.push(SheetUtils.makeRangeFromIntervals(colInterval, rowInterval));
                });
            });
        }
        return ranges;
    };

    // unified ranges ---------------------------------------------------------

    /**
     * Divides the passed cell ranges into an array of ordered row intervals
     * which contain the matching parts of the original ranges which do not
     * overlap each other.
     *
     * @param {Array} ranges
     *  An array of range addresses.
     *
     * @returns {Array}
     *  An array of row band descriptors. Each row band descriptor contains the
     *  following properties:
     *  - {Number} first
     *      The zero-based index of the first row in the row band interval.
     *  - {Number} last
     *      The zero-based index of the last row in the row band interval.
     *  - {Array} intervals
     *      An array with column intervals covering the passed original cell
     *      ranges in this row band.
     */
    function getUnifiedRowBands(ranges) {

        var // the row bands containing the matching column intervals
            rowBands = [],
            // the start index in rowBands used to search for matching bands
            startIndex = 0;

        // sort the ranges by start row index
        ranges = _.clone(ranges);
        ranges.sort(function (r1, r2) { return r1.start[1] - r2.start[1]; });

        // build the row bands array
        _.each(ranges, function (range) {

            var // the start and end row index of the range
                firstRow = range.start[1], lastRow = range.end[1],
                // the current row band
                index = 0, rowBand = null;

            // update the row bands containing the range
            for (index = startIndex; index < rowBands.length; index += 1) {
                rowBand = rowBands[index];

                if (rowBand.last < firstRow) {
                    // row band is above the range, ignore it in the next iterations
                    startIndex = index + 1;
                } else if (rowBand.first < firstRow) {
                    // row band needs to be split (range starts inside the row band)
                    rowBands.splice(index, 0, { first: rowBand.first, last: firstRow - 1, ranges: _.clone(rowBand.ranges) });
                    rowBand.first = firstRow;
                    startIndex = index + 1;
                    // next iteration of the for loop will process the current row band again
                } else {

                    if (lastRow < rowBand.last) {
                        // row band needs to be split (range ends inside the row band)
                        rowBands.splice(index + 1, 0, { first: lastRow + 1, last: rowBand.last, ranges: _.clone(rowBand.ranges) });
                        rowBand.last = lastRow;
                    }
                    // row band contains the range completely
                    rowBand.ranges.push(range);
                    // continue with next range, if end of range found
                    if (lastRow === rowBand.last) { return; }
                    // adjust start index in case a new row band needs to be appended
                    firstRow = rowBand.last + 1;
                }
            }

            // no row band found, create and append a new row band
            rowBands.push({ first: firstRow, last: lastRow, ranges: [range] });
        });

        // create the column intervals in all row bands
        _.each(rowBands, function (rowBand) {
            rowBand.intervals = SheetUtils.getColIntervals(rowBand.ranges);
            delete rowBand.ranges;
        });

        return rowBands;
    }

    /**
     * Converts the passed row bands to an array of cell range addresses.
     */
    function convertRowBandsToRanges(rowBands) {

        var // resulting cell ranges
            resultRanges = [],
            // current ranges whose last row ands in the current row band
            bandRanges = [];

        // returns whether the passed range precedes the specified column interval,
        // or whether it starts at the interval, but ends at another column
        function isPrecedingUnextendableRange(range, interval) {
            return (range.start[0] < interval.first) || ((range.start[0] === interval.first) && (range.end[0] !== interval.last)) || ((interval.first < range.start[0]) && (range.start[0] <= interval.last));
        }

        // build ranges from the row bands and their column intervals
        _.each(rowBands, function (rowBand, bandIndex) {

            var // the preceding row band
                prevRowBand = (bandIndex > 0) ? rowBands[bandIndex - 1] : null,
                // array index into the bandRanges array
                bandRangeIndex = 0;

            // try to extend ranges from the previous row band with ranges from this row band
            if (prevRowBand && (prevRowBand.last + 1 === rowBand.first)) {

                // process all intervals of the current row band, extend ranges or move them to the result array
                _.each(rowBand.intervals, function (interval) {

                    // move all ranges from bandRanges to resultRanges that cannot be merged with the current interval anymore
                    // (taking advantage of the fact that all ranges in bandRanges are distinct and ordered by column)
                    while ((bandRangeIndex < bandRanges.length) && isPrecedingUnextendableRange(bandRanges[bandRangeIndex], interval)) {
                        resultRanges = resultRanges.concat(bandRanges.splice(bandRangeIndex, 1));
                    }

                    // extend the current band range to this row band, if column interval matches
                    var bandRange = bandRanges[bandRangeIndex];
                    if (bandRange && (bandRange.start[0] === interval.first) && (bandRange.end[0] === interval.last)) {
                        bandRange.end[1] = rowBand.last;
                    } else {
                        bandRanges.splice(bandRangeIndex, 0, SheetUtils.makeRangeFromIntervals(interval, rowBand));
                    }
                    bandRangeIndex += 1;
                });

                // move all remaining unextended ranges from bandRanges to resultRanges
                resultRanges = resultRanges.concat(bandRanges.splice(bandRangeIndex));

            } else {

                // store all old band ranges in the result array
                resultRanges = resultRanges.concat(bandRanges);

                // create new ranges for this row band
                bandRanges = SheetUtils.makeRangesFromIntervals(rowBand.intervals, rowBand);
            }
        });

        return resultRanges.concat(bandRanges);
    }

    /**
     * Converts the passed cell ranges to an array of cell ranges that do not
     * overlap each other.
     *
     * @param {Object|Array} ranges
     *  The address of a single range, or an array of range addresses.
     *
     * @returns {Array}
     *  An array of cell ranges covering every cell contained in the cell
     *  ranges passed to this function, but that do not overlap each other.
     */
    SheetUtils.getUnifiedRanges = function (ranges) {
        ranges = _.getArray(ranges);
        // at most one range: return directly instead of running the complex row band code
        return (ranges.length <= 1) ? ranges : convertRowBandsToRanges(getUnifiedRowBands(ranges));
    };

    /**
     * Converts the passed array of cell addresses to a unified array of range
     * addresses, trying to join as much cells as possible.
     *
     * @param {Array} addresses
     *  An array of cell addresses.
     *
     * @returns {Array}
     *  An array of cell ranges covering every cell contained in the array
     *  passed to this function.
     */
    SheetUtils.joinCellsToRanges = function (addresses) {

        var // the row bands containing column intervals for all existing rows
            rowBands = [];

        // single cell: return directly instead of running the complex row band code
        if (addresses.length === 1) {
            return [{ start: _.clone(addresses[0]), end: _.clone(addresses[0]) }];
        }

        // sort and unify the cell addresses
        addresses = _.clone(addresses);
        addresses.sort(SheetUtils.compareCells);
        addresses = _.unique(addresses, true, SheetUtils.getCellName);

        // create row bands with all rows that contains any cells
        _.each(addresses, function (address) {

            var // the last row band in the array of all row bands
                lastRowBand = _.last(rowBands),
                // the last interval in the current row band
                lastInterval = null;

            // create a new row band on demand
            if (!lastRowBand || (lastRowBand.first !== address[1])) {
                rowBands.push(lastRowBand = { first: address[1], last: address[1], intervals: [] });
            }

            // get the last interval, extend it or create a new interval
            lastInterval =  _.last(lastRowBand.intervals);
            if (lastInterval && (lastInterval.last + 1 === address[0])) {
                lastInterval.last += 1;
            } else {
                lastRowBand.intervals.push({ first: address[0], last: address[0] });
            }
        });

        // convert the row bands to an array of cell ranges
        return convertRowBandsToRanges(rowBands);
    };

    // range difference -------------------------------------------------------

    /**
     * Returns the difference of the passed cell ranges (all cells contained in
     * the first list of ranges, that are not contained in the second list of
     * ranges).
     *
     * @param {Object|Array} ranges1
     *  The address of a single range, or an array of range addresses.
     *
     * @param {Object|Array} ranges2
     *  The address of a single range, or an array of range addresses.
     *
     * @returns {Array}
     *  The addresses of all ranges that are contained in the parameter
     *  'ranges1' but not in 'ranges2'. May be an empty array, if the ranges in
     *  'ranges2' completely cover all ranges in 'ranges1'.
     */
    SheetUtils.getRemainingRanges = function (ranges1, ranges2) {

        var // the source ranges to be reduced while iterating
            sourceRanges = null,
            // the resulting ranges
            resultRanges = _.getArray(ranges1);

        // reduce the ranges from 'ranges1' with each range from 'ranges2'
        Utils.iterateArray(_.getArray(ranges2), function (range2) {

            // initialize the arrays for this iteration
            sourceRanges = resultRanges;
            resultRanges = [];

            // process each range in the source range list
            _.each(sourceRanges, function (range1) {

                // check if the ranges cover each other at all
                if (!SheetUtils.rangeOverlapsRange(range1, range2)) {
                    resultRanges.push(range1);
                    return;
                }

                // do nothing if range2 covers range1 completely (delete range1 from the result)
                if (SheetUtils.rangeContainsRange(range2, range1)) {
                    return;
                }

                // if range1 starts above range2, extract the upper part of range1
                if (range1.start[1] < range2.start[1]) {
                    resultRanges.push({
                        start: range1.start,
                        end: [range1.end[0], range2.start[1] - 1]
                    });
                }

                // if range1 starts left of range2, extract the left part of range1
                if (range1.start[0] < range2.start[0]) {
                    resultRanges.push({
                        start: [range1.start[0], Math.max(range1.start[1], range2.start[1])],
                        end: [range2.start[0] - 1, Math.min(range1.end[1], range2.end[1])]
                    });
                }

                // if range1 ends right of range2, extract the right part of range1
                if (range1.end[0] > range2.end[0]) {
                    resultRanges.push({
                        start: [range2.end[0] + 1, Math.max(range1.start[1], range2.start[1])],
                        end: [range1.end[0], Math.min(range1.end[1], range2.end[1])]
                    });
                }

                // if range1 ends below range2, extract the lower part of range1
                if (range1.end[1] > range2.end[1]) {
                    resultRanges.push({
                        start: [range1.start[0], range2.end[1] + 1],
                        end: range1.end
                    });
                }
            });

            // early exit the loop, if all source ranges have been deleted already
            return (resultRanges.length === 0) ? Utils.BREAK : undefined;
        });

        return SheetUtils.getUnifiedRanges(resultRanges);
    };

    /**
     * Shortens the passed cell ranges so that the resulting ranges do not
     * contain more cells than specified. If a range in the list does not fit
     * completely, it will be shortened to the appropriate number of upper rows
     * of the range. If the range cannot be shortened exactly to entire rows,
     * it will be split into two ranges (the latter range being the leading
     * part of an inner row). Overlapping ranges will NOT be merged. Example:
     * Shortening the ranges A1:C3,D1:F3 to 14 cells results in the ranges
     * A1:C3,D1:F1,D2:E2.
     *
     * @param {Object|Array} ranges
     *  The address of a single range, or an array of range addresses.
     *
     * @param {Number} maxCount
     *  The maximum number of cells allowed in the resulting ranges.
     *
     * @returns {Array}
     *  The shortened range list.
     */
    SheetUtils.shortenRangesByCellCount = function (ranges, maxCount) {

        // create deep copy of the passed ranges
        ranges = _.chain(ranges).getArray().copy(true).value();

        Utils.iterateArray(ranges, function (range, index) {

            // no more cells left (e.g., previous range has occupied the last cell)
            if (maxCount <= 0) {
                ranges.splice(index);
                return Utils.BREAK;
            }

            var // number of cells in the current range
                count = SheetUtils.getCellCount(range);

            // skip entirely fitting ranges
            if (count <= maxCount) {
                maxCount -= count;
                return;
            }

            var // number of columns in the range
                cols = SheetUtils.getColCount(range),
                // number of entire rows fitting in the result
                rows = Math.floor(maxCount / cols);

            // shorten range to entire rows
            range.end[1] = range.start[1] + rows - 1;
            if (rows > 0) {
                maxCount -= (cols * rows);
                ranges.splice(index + 1);
            } else {
                ranges.splice(index);
            }

            // add part of next row if needed
            if (maxCount > 0) {
                var nextRange = _.copy(range, true);
                nextRange.start[1] = nextRange.end[1] = range.end[1] + 1;
                nextRange.end[0] = nextRange.start[0] + maxCount - 1;
                ranges.push(nextRange);
            }

            // stop iteration
            return Utils.BREAK;
        });

        return ranges;
    };

    // unique keys for maps of cells/ranges -----------------------------------

    /**
     * Returns a unique string key for the passed cell address which can by
     * used internally e.g. as keys for associative maps. Uses a faster
     * implementation than the method SheetUtils.getCellName().
     *
     * @param {Number[]} address
     *  The cell address.
     *
     * @returns {String}
     *  A unique string key for the cell position (the zero-based integral
     *  column and row indexes, separated by a forward slash character).
     */
    SheetUtils.getCellKey = function (address) {
        return address[0] + ',' + address[1];
    };

    /**
     * Returns the cell address for the passed unique cell key.
     *
     * @param {String} key
     *  The unique cell key, as returned by the method SheetUtils.getCellKey().
     *
     * @returns {Number[]}
     *  The cell address parsed from the passed unique cell key.
     */
    SheetUtils.parseCellKey = function (key) {
        return _.map(key.split(','), function (index) { return parseInt(index, 10); });
    };

    /**
     * Returns a unique string key for the passed cell range address which can
     * by used internally e.g. as keys for associative maps. Uses a faster
     * implementation than the method SheetUtils.getRangeName().
     *
     * @param {Object} range
     *  The address of a cell range.
     *
     * @returns {String}
     *  A unique string key for the cell range (the cell keys for the start and
     *  end cells, see method SheetUtils.getCellKey(), separated by a colon).
     */
    SheetUtils.getRangeKey = function (range) {
        return SheetUtils.getCellKey(range.start) + ':' + SheetUtils.getCellKey(range.end);
    };

    /**
     * Returns the cell range address for the passed unique range key.
     *
     * @param {String} key
     *  The unique cell range key, as returned by the method
     *  SheetUtils.getRangeKey().
     *
     * @returns {Object}
     *  The cell range address parsed from the passed unique range key.
     */
    SheetUtils.parseRangeKey = function (key) {
        var cellKeys = key.split(':');
        return { start: SheetUtils.parseCellKey(cellKeys[0]), end: SheetUtils.parseCellKey(cellKeys[1]) };
    };

    // generate UI strings ----------------------------------------------------

    /**
     * Returns the string representation of the passed column index.
     *
     * @param {Number} col
     *  The zero-based column index.
     *
     * @returns {String}
     *  The upper-case string representation of the column index.
     */
    SheetUtils.getColName = function (col) {
        var name = '';
        while (col >= 0) {
            name = String.fromCharCode(65 + col % 26) + name;
            col = Math.floor(col / 26) - 1;
        }
        return name;
    };

    /**
     * Returns the string representation of the passed column interval.
     *
     * @param {Object|Number} interval
     *  The column interval, in the zero-based column index properties 'first'
     *  and 'last', or a single zero-based column index.
     *
     * @returns {String}
     *  The upper-case string representation of the column interval.
     */
    SheetUtils.getColIntervalName = function (interval) {
        return SheetUtils.getColName(_.isNumber(interval) ? interval : interval.first) + ':' + SheetUtils.getColName(_.isNumber(interval) ? interval : interval.last);
    };

    /**
     * Returns the string representation of the passed column intervals.
     *
     * @param {Number|Object|Array} intervals
     *  A single column index, a single column interval, or an array of column
     *  indexes and/or intervals.
     *
     * @param {String} [separator=',']
     *  The separator text inserted between the column interval names.
     *
     * @returns {String}
     *  The upper-case string representation of the column intervals.
     */
    SheetUtils.getColIntervalsName = function (intervals, separator) {
        return _.map(_.getArray(intervals), SheetUtils.getColIntervalName).join(separator);
    };

    /**
     * Returns the string representation of the passed row index.
     *
     * @param {Number} row
     *  The zero-based row index.
     *
     * @returns {String}
     *  The one-based string representation of the row index.
     */
    SheetUtils.getRowName = function (row) {
        return String(row + 1);
    };

    /**
     * Returns the string representation of the passed row interval.
     *
     * @param {Object|Number} interval
     *  The row interval, in the zero-based row index properties 'first' and
     *  'last', or a single zero-based row index.
     *
     * @returns {String}
     *  The one-based string representation of the row interval.
     */
    SheetUtils.getRowIntervalName = function (interval) {
        return SheetUtils.getRowName(_.isNumber(interval) ? interval : interval.first) + ':' + SheetUtils.getRowName(_.isNumber(interval) ? interval : interval.last);
    };

    /**
     * Returns the string representation of the passed row intervals.
     *
     * @param {Number|Object|Array} intervals
     *  A single row index, a single row interval, or an array of row indexes
     *  and/or intervals.
     *
     * @param {String} [separator=',']
     *  The separator text inserted between the row interval names.
     *
     * @returns {String}
     *  The one-based string representation of the row intervals.
     */
    SheetUtils.getRowIntervalsName = function (intervals, separator) {
        return _.map(_.getArray(intervals), SheetUtils.getRowIntervalName).join(separator);
    };

    /**
     * Returns the string representation of the passed column or row index.
     *
     * @param {Number} index
     *  The zero-based column or row index.
     *
     * @param {Boolean} columns
     *  Whether to return a column name (true), or a row name (false).
     *
     * @returns {String}
     *  The string representation of the column or row index.
     */
    SheetUtils.getIndexName = function (index, columns) {
        return (columns ? SheetUtils.getColName : SheetUtils.getRowName)(index);
    };

    /**
     * Returns the string representation of the passed column or row interval.
     *
     * @param {Object|Number} interval
     *  The column or row interval, in the zero-based index properties 'first'
     *  and 'last', or a single zero-based column/row index.
     *
     * @param {Boolean} columns
     *  Whether to return the name of a column interval (true), or a row
     *  interval (false).
     *
     * @returns {String}
     *  The string representation of the interval.
     */
    SheetUtils.getIntervalName = function (interval, columns) {
        return (columns ? SheetUtils.getColIntervalName : SheetUtils.getRowIntervalName)(interval);
    };

    /**
     * Returns the string representation of the passed column or row intervals.
     *
     * @param {Number|Object|Array} intervals
     *  A single column/row index, a single column/row interval, or an array of
     *  column/row indexes and/or intervals.
     *
     * @param {Boolean} columns
     *  Whether to return the name of column intervals (true), or row intervals
     *  (false).
     *
     * @param {String} [separator=',']
     *  The separator text inserted between the interval names.
     *
     * @returns {String}
     *  The string representation of the column/row intervals.
     */
    SheetUtils.getIntervalsName = function (intervals, columns, separator) {
        return (columns ? SheetUtils.getColIntervalsName : SheetUtils.getRowIntervalsName)(intervals, separator);
    };

    /**
     * Returns the string representation of the passed cell address.
     *
     * @param {Number[]} address
     *  The cell address.
     *
     * @returns {String}
     *  The string representation of the cell position, in A1 notation.
     */
    SheetUtils.getCellName = function (address) {
        return SheetUtils.getColName(address[0]) + SheetUtils.getRowName(address[1]);
    };

    /**
     * Returns the string representation of the passed range address.
     *
     * @param {Object} range
     *  The cell range address.
     *
     * @returns {String}
     *  The string representation of the cell range, in A1:A1 notation.
     */
    SheetUtils.getRangeName = function (range) {
        return SheetUtils.getCellName(range.start) + ':' + SheetUtils.getCellName(range.end);
    };

    /**
     * Returns the string representation of the passed range addresses.
     *
     * @param {Object|Array} ranges
     *  The address of a single cell range, or an array with range addresses.
     *
     * @param {String} [separator=',']
     *  The separator text inserted between the range names.
     *
     * @returns {String}
     *  The string representation of the cell ranges, in A1:A1 notation.
     */
    SheetUtils.getRangesName = function (ranges, separator) {
        return _.map(_.getArray(ranges), SheetUtils.getRangeName).join(separator);
    };

    /**
     * Generates a localized sheet name with the specified index.
     *
     * @param {Number} sheet
     *  The zero-based sheet index to be inserted into the sheet name. The
     *  number inserted into the sheet name will be increased by one.
     *
     * @returns {String}
     *  The generated sheet name (e.g. 'Sheet1', 'Sheet2', etc.).
     */
    SheetUtils.generateSheetName = function (sheet) {
        return (
            //#. Default sheet names in spreadsheet documents. Should be equal to the sheet
            //#. names used in existing spreadsheet applications (Excel, OpenOffice Calc, ...).
            //#. %1$d is the numeric index in the sheet name (e.g. "Sheet1", "Sheet2", etc.)
            //#. No white-space between the "Sheet" label and sheet index!
            //#, c-format
            gt('Sheet%1$d', _.noI18n(sheet + 1)));
    };

    // parse UI strings -------------------------------------------------------

    /**
     * Returns the column index of the passed column string representation.
     *
     * @param {String} name
     *  The string representation of a column (case-insensitive).
     *
     * @returns {Number}
     *  The zero-based column index, if the passed column name is valid,
     *  otherwise -1.
     */
    SheetUtils.parseColName = function (name) {
        var col = -1;
        if (RE_COL_NAME.test(name)) {
            name = name.toUpperCase();
            for (var ichar = 0; ichar < name.length; ichar += 1) {
                col = (col + 1) * 26 + name.charCodeAt(ichar) - 65;
            }
        }
        return (col < MAX_COL_COUNT) ? col : -1;
    };

    /**
     * Returns the row index of the passed row string representation.
     *
     * @param {String} name
     *  The string representation of a row (one-based).
     *
     * @returns {Number}
     *  The zero-based row index, if the passed row name is valid, otherwise
     *  the value -1.
     */
    SheetUtils.parseRowName = function (name) {
        var row = RE_ROW_NAME.test(name) ? (parseInt(name, 10) - 1) : -1;
        return (row < MAX_ROW_COUNT) ? row : -1;
    };

    /**
     * Returns the cell address for the passed string representation.
     *
     * @param {String} name
     *  The string representation of a cell address, in A1 notation.
     *
     * @returns {Number[]|Null}
     *  The cell address, if the passed cell name is valid, otherwise null.
     */
    SheetUtils.parseCellName = function (name) {
        var matches = RE_CELL_NAME_A1.exec(name), col = -1, row = -1;
        if (matches) {
            col = SheetUtils.parseColName(matches[1]);
            row = SheetUtils.parseRowName(matches[2]);
        }
        return ((col >= 0) && (row >= 0)) ? [col, row] : null;
    };

    /**
     * Returns the range address for the passed string representation.
     *
     * @param {String} name
     *  The string representation of a cell range, in A1 notation, with a colon
     *  as separator between start and end address.
     *
     * @returns {Object|Null}
     *  The range address (with adjusted column and row indexes), if the passed
     *  range name is valid, otherwise null.
     */
    SheetUtils.parseRangeName = function (name) {
        var matches = RE_RANGE_NAME_A1.exec(name), start = null, end = null;
        if (matches) {
            start = SheetUtils.parseCellName(matches[1]);
            end = SheetUtils.parseCellName(matches[2]);
        }
        return (start && end) ? SheetUtils.adjustRange({ start: start, end: end }) : null;
    };

    // exports ================================================================

    return SheetUtils;

});
