/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/tokenutils',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/logger',
     'io.ox/office/spreadsheet/utils/sheetutils'
    ], function (Utils, Logger, SheetUtils) {

    'use strict';

    // static private functions ===============================================

    /**
     * Creates a special error code for internal usage in the formula engine.
     */
    function makeInternalErrorCode(code) {
        var errorCode = SheetUtils.makeErrorCode('[#' + code + ']');
        errorCode.internal = code;
        return errorCode;
    }

    // static class TokenUtils ================================================

    /**
     * The static class 'TokenUtils' is a console logger bound to the URL hash
     * flag 'spreadsheet:log-formulas', that logs detailed information about
     * the formula parser, compiler, and interpreter.
     */
    var TokenUtils = new Logger({ enable: 'spreadsheet:log-formulas', prefix: 'FMLA' });

    // constants --------------------------------------------------------------

    /**
     * Distance between the number 1 and the smallest number greater than 1
     * (the value of 2^-52). Provided as non-standard property Number.EPSILON
     * by Chrome and Firefox, but not supported by IE10 and IE11.
     *
     * @constant
     */
    TokenUtils.EPSILON = Math.pow(2, -52);

    /**
     * The smallest positive normalized floating-point number (browsers may
     * support smaller numbers though, which are stored with a denormalized
     * mantissa).
     *
     * @constant
     */
    TokenUtils.MIN_NUMBER = Math.pow(2, -1022);

    /**
     * Maximum number of function parameters (CalcEngine limitation).
     *
     * @constant
     */
    TokenUtils.MAX_PARAM_COUNT = 254;

    /**
     * Maximum number of rows supported in matrix literals.
     *
     * @constant
     */
    TokenUtils.MAX_MATRIX_ROW_COUNT = 16;

    /**
     * Maximum number of columns supported in matrix literals.
     *
     * @constant
     */
    TokenUtils.MAX_MATRIX_COL_COUNT = 16;

    /**
     * Maximum number of references in a reference list literal.
     *
     * @constant
     */
    TokenUtils.MAX_REF_LIST_SIZE = 1024;

    /**
     * Special error code literal to transport the 'unsupported' state.
     */
    TokenUtils.UNSUPPORTED_ERROR = makeInternalErrorCode('unsupported');

    /**
     * Special error code literal to transport the 'circular reference' result.
     */
    TokenUtils.CIRCULAR_ERROR = makeInternalErrorCode('circular');

    // public methods ---------------------------------------------------------

    /**
     * Returns the string representation of the passed literal value for debug
     * logging.
     *
     * @param {Number|String|Boolean|ErrorCode|Null}
     *  The value to be converted to the string representation.
     *
     * @returns {String}
     *  The string representation of the value.
     */
    TokenUtils.valueToString = function (value) {
        return _.isString(value) ? ('"' + value.replace('"', '\\"') + '"') :
            (value instanceof Date) ? value.toISOString() :
            _.isNull(value) ? '[empty]' :
            String(value).toUpperCase();
    };

    /**
     * Writes all passed tokens to the browser console.
     *
     * @param {String} message
     *  A message string written before the formula tokens.
     *
     * @param {Array} tokens
     *  An array of formula tokens to be written to the browser console.
     */
    TokenUtils.logTokens = TokenUtils.isLoggingActive() ? function (message, tokens) {
        TokenUtils.log(message + ': ' + tokens.join(' '));
    } : $.noop;

    // exports ================================================================

    return TokenUtils;

});
