/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/tokenizer',
    ['io.ox/office/tk/utils',
     'io.ox/office/baseframework/model/modelobject',
     'io.ox/office/spreadsheet/utils/sheetutils',
     'io.ox/office/spreadsheet/model/formula/tokenutils'
    ], function (Utils, ModelObject, SheetUtils, TokenUtils) {

    'use strict';

    var // shortcut for the map of error code literals
        ErrorCodes = SheetUtils.ErrorCodes;

    // private global functions ===============================================

    /**
     * Little helper function that calls the toUpperCase() method at its string
     * argument and returns the result.
     */
    function toUpperCase(text) {
        return text.toUpperCase();
    }

    function createBaseConfig(app, grammar) {

        var // the configuration object to be returned
            config = {
                // zero-based index of last available column
                MAXCOL: app.getModel().getMaxCol(),
                // zero-based index of last available row
                MAXROW: app.getModel().getMaxRow()
            };

        // grammar dependent configuration items
        switch (grammar) {

        case 'op':
            _.extend(config, {
                // the decimal separator character
                DEC: '.',
                // the list separator character
                SEP: ',',
                // the column separator character in constant matrixes
                COL: ',',
                // the row separator character in constant matrixes
                ROW: ';',
                // the Boolean TRUE literal
                TRUE: 'TRUE',
                // the Boolean FALSE literal
                FALSE: 'FALSE',
                // all supported error code literals, as strings
                ERRORS: app.getNativeErrorCodes(),
                // converts error code object to text representation
                getErrorCodeText: _.property('code'),
                // converts text representation to error code object
                getErrorCodeValue: SheetUtils.makeErrorCode,
                // converts native function name to text representation
                getFunctionText: toUpperCase,
                // converts text representation to native function name
                getFunctionValue: toUpperCase
            });
            break;

        case 'ui':
            _.extend(config, {
                // the decimal separator character
                DEC: app.getDecimalSeparator(),
                // the list separator character
                SEP: app.getListSeparator(),
                // the column separator character in constant matrixes
                COL: ';',
                // the row separator character in constant matrixes
                ROW: '|',
                // the Boolean TRUE literal
                TRUE: app.getBooleanLiteral(true),
                // the Boolean FALSE literal
                FALSE: app.getBooleanLiteral(false),
                // all supported error code literals, as strings
                ERRORS: app.getLocalizedErrorCodes(),
                // converts error code object to text representation
                getErrorCodeText: _.bind(app.convertErrorCodeToString, app),
                // converts text representation to error code object
                getErrorCodeValue: _.bind(app.convertStringToErrorCode, app),
                // converts native function name to text representation
                getFunctionText: _.bind(app.getLocalizedFunctionName, app),
                // converts text representation to native function name
                getFunctionValue: _.bind(app.getNativeFunctionName, app)
            });
            break;

        default:
            Utils.error('Tokenizer.createBaseConfig(): invalid formula grammar "' + grammar + '"');
        }

        // shortcut: the error code text for #REF! errors
        config.REF_ERROR = config.getErrorCodeText(ErrorCodes.REF);

        return config;
    }

    /**
     * Creates the configuration structure for the specified formula grammar.
     *
     * @param {SpreadsheetApplication} app
     *  The spreadsheet application instance.
     *
     * @param {String} grammar
     *  The formula grammar identifier (See the class constructor of the class
     *  Tokenizer for details.).
     *
     * @returns {Object}
     *  The configuration structure for the specified formula grammar.
     */
    function createConfig(app, grammar) {

        var // the configuration object to be returned
            config = createBaseConfig(app, grammar),

            // RE pattern snippet for the leading character of a simple sheet name (without apostrophes).
            SHEET_LEADING_CHAR_TOKEN = '[a-z_\\xa1-\\u2027\\u202a-\\uffff]',

            // RE pattern snippet for other characters of a simple sheet name (without apostrophes).
            SHEET_SIMPLE_CHAR_TOKEN = '[\\w.\\xa1-\\u2027\\u202a-\\uffff]',

            // RE pattern snippet for any character of a complex sheet name (enclosed in apostrophes).
            SHEET_COMPLEX_CHAR_TOKEN = '[^\\x00-\\x1f\\x80-\\x9f\\[\\]\'*?:/\\\\]',

            // RE pattern snippet for the leading character of a name or identifier.
            NAME_LEADING_CHAR_TOKEN = '[a-z_\\xa1-\\u2027\\u202a-\\uffff]',

            // RE pattern snippet for other characters of a name or identifier.
            NAME_INNER_CHAR_TOKEN = '[\\w.\\xa1-\\u2027\\u202a-\\uffff]',

            // RE pattern for a simple sheet name.
            SHEET_NAME_SIMPLE_PATTERN = SHEET_LEADING_CHAR_TOKEN + SHEET_SIMPLE_CHAR_TOKEN + '*',

            // RE pattern for a complex sheet name (with embedded double-apostrophes).
            SHEET_NAME_COMPLEX_PATTERN = SHEET_COMPLEX_CHAR_TOKEN + '+(?:(?:\'\')+' + SHEET_COMPLEX_CHAR_TOKEN + '+)*',

            // RE pattern for a number literal: With or without decimal separator, digits preceding
            // and/or following the separator, optional scientific prefix. No leading sign character.
            NUMBER_PATTERN = '(\\d*' + _.escapeRegExp(config.DEC) + '\\d+|\\d+' + _.escapeRegExp(config.DEC) + '?)([eE][-+]?\\d+)?',

            // RE pattern for a #REF! error code
            REF_ERROR_PATTERN = _.escapeRegExp(config.REF_ERROR),

            // RE look-ahead pattern to exclude an opening parenthesis and an exclamation mark
            // after another pattern. All valid inner characters of names must be excluded too,
            // otherwise the previous groups would match less characters than they should.
            TERMINATE_REF_PATTERN = '(?!\\(|!|' + NAME_INNER_CHAR_TOKEN + ')',

            // RE pattern for a sheet reference used in formulas, e.g. Sheet1! in the formula =Sheet1!A1,
            // or a broken sheet reference after deleting the referenced sheet, e.g. #REF!! in the formula =#REF!!A1.
            // - Group 1: The name of the sheet (simple sheet name).
            // - Group 2: The name of the sheet (complex sheet name).
            // - Group 3: The error code of a broken sheet reference.
            // Either group 1 or group 2 will match the sheet name, or group 3 will contain the error code (the other groups will be empty).
            SHEET_REF_PATTERN = '(?:(' + SHEET_NAME_SIMPLE_PATTERN + ')|\'(' + SHEET_NAME_COMPLEX_PATTERN + ')\'|(' + REF_ERROR_PATTERN + '))!',

            // RE pattern for an optional sheet reference. See SHEET_REF_PATTERN for details.
            SHEET_REF_PATTERN_OPT = '(?:' + SHEET_REF_PATTERN + ')?',

            // RE pattern for a sheet range reference used in formulas, e.g. Sheet1:Sheet2! in the formula =SUM(Sheet1:Sheet2!A1).
            // - Group 1: The name of the first sheet (simple sheet names).
            // - Group 2: The name of the second sheet (simple sheet names).
            // - Group 3: The name of the first sheet (complex sheet names).
            // - Group 4: The name of the second sheet (complex sheet names).
            // Either group 1 or group 3 will match the first sheet name (the other group will be empty).
            // Either group 2 or group 4 will match the second sheet name (the other group will be empty).
            SHEET_RANGE_REF_PATTERN = '(?:(' + SHEET_NAME_SIMPLE_PATTERN + '):(' + SHEET_NAME_SIMPLE_PATTERN + ')|\'(' + SHEET_NAME_COMPLEX_PATTERN + '):(' + SHEET_NAME_COMPLEX_PATTERN + ')\')!',

            // RE pattern for a column name (absolute or relative).
            // - Group 1: The absolute marker.
            // - Group 2: The column name.
            COL_REF_PATTERN = '(\\$)?([a-z]+)',

            // RE pattern for a row name (absolute or relative).
            // - Group 1: The absolute marker.
            // - Group 2: The row name.
            ROW_REF_PATTERN = '(\\$)?([0-9]+)',

            // RE pattern for a cell address.
            // - Group 1: The absolute marker for the column.
            // - Group 2: The name of the column.
            // - Group 3: The absolute marker for the row.
            // - Group 4: The name of the row.
            CELL_REF_PATTERN = COL_REF_PATTERN + ROW_REF_PATTERN,

            // RE pattern for a cell range address.
            // - Group 1: The absolute marker for the first column.
            // - Group 2: The name of the first column.
            // - Group 3: The absolute marker for the first row.
            // - Group 4: The name of the first row.
            // - Group 5: The absolute marker for the second column.
            // - Group 6: The name of the second column.
            // - Group 7: The absolute marker for the second row.
            // - Group 8: The name of the second row.
            CELL_RANGE_REF_PATTERN = CELL_REF_PATTERN + ':' + CELL_REF_PATTERN,

            // RE pattern for a column range reference, e.g. $C:$C.
            // - Group 1: The absolute marker for the first column.
            // - Group 2: The name of the first column.
            // - Group 3: The absolute marker for the second column.
            // - Group 4: The name of the second column.
            COL_RANGE_REF_PATTERN = COL_REF_PATTERN + ':' + COL_REF_PATTERN,

            // RE pattern for a row range reference, e.g. $2:$2.
            // - Group 1: The absolute marker for the first row.
            // - Group 2: The name of the first row.
            // - Group 3: The absolute marker for the second row.
            // - Group 4: The name of the second row.
            ROW_RANGE_REF_PATTERN = ROW_REF_PATTERN + ':' + ROW_REF_PATTERN,

            // RE pattern for a complete name or identifier used in formulas.
            // - Group 1: The entire name.
            NAME_REF_PATTERN = '(' + NAME_LEADING_CHAR_TOKEN + NAME_INNER_CHAR_TOKEN + '*)',

            // RE pattern for remaining invalid characters
            // - Group 1: The matched invalid character.
            INVALID_CHAR_PATTERN = '([!§$?°~#;,.|])';

        // create the map of all regular expressions
        config.RE = {

            // Number literals: With or without decimal separator, digits preceding
            // and/or following the separator, optional scientific prefix. No
            // leading sign character, will be parsed as operator, otherwise a formula
            // such as =1-1 becomes an invalid sequence of two numbers.
            NUMBER_LIT: new RegExp('^' + NUMBER_PATTERN),

            // String literals: Enclosed in double quotes, embedded double quotes
            // are represented by a sequence of two double quote characters.
            STRING_LIT: /^("[^"]*")+/,

            // Boolean literals: Must not be followed by any character valid inside an
            // identifier. This prevents to match 'trueA' or 'false_1' or similar name tokens.
            BOOLEAN_LIT: new RegExp('^(' + _.escapeRegExp(config.TRUE) + '|' + _.escapeRegExp(config.FALSE) + ')(?!' + NAME_INNER_CHAR_TOKEN + ')', 'i'),

            // Error code literals (only supported error codes, not any string
            // starting with a hash character).
            ERROR_LIT: new RegExp('^(' + _.map(config.ERRORS, _.escapeRegExp).join('|') + ')(?!' + NAME_INNER_CHAR_TOKEN + ')', 'i'),

            // Opening parenthesis of a matrix literal.
            MATRIX_OPEN: /^{/,

            // Numbers in matrix literals: always with leading sign character (plus and minus).
            MATRIX_NUMBER_LIT: new RegExp('^[-+]?' + NUMBER_PATTERN),

            // The column separator in a matrix literal.
            MATRIX_COL_SEPARATOR: new RegExp('^' + _.escapeRegExp(config.COL)),

            // The row separator in a matrix literal.
            MATRIX_ROW_SEPARATOR: new RegExp('^' + _.escapeRegExp(config.ROW)),

            // Closing parenthesis of a matrix literal.
            MATRIX_CLOSE: /^}/,

            // All unary and binary operators, mapped by operator type.
            OPERATORS: {
                // comparison
                '=': /^(<>|<=?|>=?|=)/,
                // string concatenation
                '&': /^&/,
                // addition and subtraction (also as unary operators)
                '+': /^[-+]/,
                // multiplication and division
                '*': /^[*\/]/,
                // power operator
                '^': /^\^/,
                // unary percent operator
                '%': /^%/,
                // range operator
                ':': /^:/
            },

            // The list/parameter separator character.
            SEPARATOR: new RegExp('^' + _.escapeRegExp(config.SEP)),

            // The opening parenthesis.
            OPEN: /^\(/,

            // The closing parenthesis.
            CLOSE: /^\)/,

            // A cell reference, e.g. A1 or Sheet1!$A$1, but without trailing parenthesis, e.g. F2().
            // - Group 1: The simple sheet name.
            // - Group 2: The complex sheet name.
            // - Group 3: The error code of a broken sheet reference.
            // - Group 4: The absolute marker for the column.
            // - Group 5: The column name.
            // - Group 6: The absolute marker for the row.
            // - Group 7: The row name.
            CELL_REF: new RegExp('^' + SHEET_REF_PATTERN_OPT + CELL_REF_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A cell reference in multiple sheets, e.g. Sheet1:Sheet2!$A$1.
            // - Group 1: The simple name of the first sheet.
            // - Group 2: The simple name of the second sheet.
            // - Group 3: The complex name of the first sheet.
            // - Group 4: The complex name of the second sheet.
            // - Group 5: The absolute marker for the column.
            // - Group 6: The column name.
            // - Group 7: The absolute marker for the row.
            // - Group 8: The row name.
            CELL_CUBE_REF: new RegExp('^' + SHEET_RANGE_REF_PATTERN + CELL_REF_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A range reference, e.g. A1:B2 or Sheet1!$A$1:$B$2, but without trailing parenthesis, e.g. A1:F2().
            // - Group 1: The simple sheet name.
            // - Group 2: The complex sheet name.
            // - Group 3: The error code of a broken sheet reference.
            // - Group 4: The absolute marker for the first column.
            // - Group 5: The name of the first column.
            // - Group 6: The absolute marker for the first row.
            // - Group 7: The name of the first row.
            // - Group 8: The absolute marker for the second column.
            // - Group 9: The name of the second column.
            // - Group 10: The absolute marker for the second row.
            // - Group 11: The name of the second row.
            CELL_RANGE_REF: new RegExp('^' + SHEET_REF_PATTERN_OPT + CELL_RANGE_REF_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A range reference in multiple sheets, e.g. Sheet1:Sheet2!$A$1:$A$1.
            // - Group 1: The simple name of the first sheet.
            // - Group 2: The simple name of the second sheet.
            // - Group 3: The complex name of the first sheet.
            // - Group 4: The complex name of the second sheet.
            // - Group 5: The absolute marker for the first column.
            // - Group 6: The name of the first column.
            // - Group 7: The absolute marker for the first row.
            // - Group 8: The name of the first row.
            // - Group 9: The absolute marker for the second column.
            // - Group 10: The name of the second column.
            // - Group 11: The absolute marker for the second row.
            // - Group 12: The name of the second row.
            CELL_RANGE_CUBE_REF: new RegExp('^' + SHEET_RANGE_REF_PATTERN + CELL_RANGE_REF_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A column range, e.g. A:B or Sheet1!$C:$C (always with colon).
            // - Group 1: The simple sheet name.
            // - Group 2: The complex sheet name.
            // - Group 3: The error code of a broken sheet reference.
            // - Group 4: The absolute marker for the first column.
            // - Group 5: The name of the first column.
            // - Group 6: The absolute marker for the second column.
            // - Group 7: The name of the second column.
            COL_RANGE_REF: new RegExp('^' + SHEET_REF_PATTERN_OPT + COL_RANGE_REF_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A column range in multiple sheets, Sheet1:Sheet2!$C:$C (always with colon).
            // - Group 1: The simple name of the first sheet.
            // - Group 2: The simple name of the second sheet.
            // - Group 3: The complex name of the first sheet.
            // - Group 4: The complex name of the second sheet.
            // - Group 5: The absolute marker for the first column.
            // - Group 6: The name of the first column.
            // - Group 7: The absolute marker for the second column.
            // - Group 8: The name of the second column.
            COL_RANGE_CUBE_REF: new RegExp('^' + SHEET_RANGE_REF_PATTERN + COL_RANGE_REF_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A row range, e.g. 1:2 or $3:$3 (always with colon).
            // - Group 1: The simple sheet name.
            // - Group 2: The complex sheet name.
            // - Group 3: The error code of a broken sheet reference.
            // - Group 4: The absolute marker for the first row.
            // - Group 5: The name of the first row.
            // - Group 6: The absolute marker for the second row.
            // - Group 7: The name of the second row.
            ROW_RANGE_REF: new RegExp('^' + SHEET_REF_PATTERN_OPT + ROW_RANGE_REF_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A row range in multiple sheets, e.g. Sheet1:Sheet2!$3:$3 (always with colon).
            // - Group 1: The simple name of the first sheet.
            // - Group 2: The simple name of the second sheet.
            // - Group 3: The complex name of the first sheet.
            // - Group 4: The complex name of the second sheet.
            // - Group 5: The absolute marker for the first row.
            // - Group 6: The name of the first row.
            // - Group 7: The absolute marker for the second row.
            // - Group 8: The name of the second row.
            ROW_RANGE_CUBE_REF: new RegExp('^' + SHEET_RANGE_REF_PATTERN + ROW_RANGE_REF_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A reference error with sheet reference, e.g. Sheet1!#REF!.
            // - Group 1: The simple sheet name.
            // - Group 2: The complex sheet name.
            // - Group 3: The error code of a broken sheet reference.
            ERROR_REF: new RegExp('^' + SHEET_REF_PATTERN + REF_ERROR_PATTERN, 'i'),

            // A reference error in multiple sheets, e.g. Sheet1:Sheet2!#REF!.
            // - Group 1: The simple name of the first sheet.
            // - Group 2: The simple name of the second sheet.
            // - Group 3: The complex name of the first sheet.
            // - Group 4: The complex name of the second sheet.
            ERROR_CUBE_REF: new RegExp('^' + SHEET_RANGE_REF_PATTERN + REF_ERROR_PATTERN, 'i'),

            // Any function name (names followed by an opening parenthesis), e.g. SUM() or Module1!macro().
            // - Group 1: The simple sheet name.
            // - Group 2: The complex sheet name.
            // - Group 3: The error code of a broken sheet reference.
            // - Group 4: The name of the function.
            FUNCTION_REF: new RegExp('^' + SHEET_REF_PATTERN_OPT + NAME_REF_PATTERN + '(?=\\()', 'i'),

            // Defined names with sheet reference, e.g. Sheet1!my_name.
            // - Group 1: The simple sheet name.
            // - Group 2: The complex sheet name.
            // - Group 3: The error code of a broken sheet reference.
            // - Group 4: The name.
            SHEET_NAME_REF: new RegExp('^' + SHEET_REF_PATTERN + NAME_REF_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // Defined names without sheet reference, e.g. my_name.
            // - Group 1: The name.
            GLOBAL_NAME_REF: new RegExp('^' + NAME_REF_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // Any whitespace characters.
            WHITESPACE: /^[\s\x00-\x1f\x80-\x9f]+/,

            // A single unsupported character.
            INVALID_CHAR: new RegExp('^' + INVALID_CHAR_PATTERN),

            // Matches a simple sheet name that does not need to be enclosed in apostrophes.
            SIMPLE_SHEET_NAME: new RegExp('^' + SHEET_NAME_SIMPLE_PATTERN + '$', 'i')
        };

        TokenUtils.info('Tokenizer.createConfig(): grammar=' + grammar + ' config=', config);
        return config;
    }

    // class Tokenizer ========================================================

    /**
     * Parses formula expressions to arrays of formula tokens.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {SpreadsheetApplication} app
     *  The spreadsheet application instance.
     *
     * @param {String} grammar
     *  The formula grammar identifier. Supported values are:
     *  - 'op': Fixed token representations as used in operations (matches the
     *      formula grammar used in OOXML spreadsheet files).
     *  - 'ui': The localized token representations according to the current
     *      GUI language of the application.
     *
     * @param {SheetModel} [sheetModel]
     *  The model of the sheet that contains this tokenizer. If omitted, this
     *  tokenizer is owned by the document.
     */
    function Tokenizer(app, grammar, sheetModel) {

        var // self reference
            self = this,

            // the document model
            model = app.getModel(),

            // the formula grammar configuration
            config = app.getOrCreateSingleton('tokenizer:config' + grammar, function () {
                return createConfig(app, grammar);
            });

        // base constructor ---------------------------------------------------

        ModelObject.call(this, app);

        // private methods ----------------------------------------------------

        /**
         * Generates the display text for the passed literal value.
         *
         * @param {Number|String|Boolean|ErrorCode|Null} value
         *  The value to be converted to a number.
         *
         * @returns {String|Null}
         *  The string representation of the value, if it is valid; otherwise
         *  null.
         */
        function generateLiteralText(value) {

            // numbers
            if (_.isNumber(value)) {
                return !_.isFinite(value) ? config.getErrorCodeText(ErrorCodes.NUM) :
                    (Math.abs(value) < TokenUtils.MIN_VALUE) ? '0' :
                    app.convertNumberToString(value, SheetUtils.MAX_LENGTH_STANDARD_EDIT, config.SEP);
            }

            // strings
            if (_.isString(value)) {
                // duplicate inner quotes, enclose in quotes
                return '"' + value.replace(/"/g, '""') + '"';
            }

            // Boolean values
            if (_.isBoolean(value)) {
                return value ? config.TRUE : config.FALSE;
            }

            // error codes
            if (SheetUtils.isErrorCode(value)) {
                return config.getErrorCodeText(value);
            }

            // empty function parameters
            if (!_.isNull(value)) {
                Utils.warn('generateLiteralText(): unsupported value type');
            }
            return '';
        }

        /**
         * Returns a sheet reference structure for the passed simple or complex
         * sheet name. At most one of the passed parameters can be a string,
         * the other parameters must be null or undefined.
         *
         * @param {String|Null} simpleSheet
         *  A simple sheet name without any special characters that would cause
         *  to enclose the sheet name in apostrophes.
         *
         * @param {String|Null} complexSheet
         *  A complex sheet name with special characters that requires the
         *  sheet name to be enclosed in apostrophes.
         *
         * @param {String|Null} refError
         *  The error code of a reference error, used a sheet has been deleted,
         *  e.g. in the formula =#REF!!A1.
         *
         * @returns {Object|Null}
         *  A sheet reference structure, if one of the passed parameters is a
         *  string, otherwise null. The sheet reference structure contains the
         *  following properties:
         *  - {Number|String} sheet
         *      The zero-based sheet index, if a sheet with the passed name
         *      exists in the document; or -1 for a broken sheet reference;
         *      otherwise the sheet name as string.
         *  - {Boolean} abs
         *      Whether the sheet reference is marked to be absolute.
         */
        function makeSheetRef(simpleSheet, complexSheet, refError) {

            var sheetRef = null,
                sheetIndex = null;

            if (_.isString(simpleSheet)) {
                sheetIndex = model.getSheetIndex(simpleSheet);
                sheetRef = { sheet: (sheetIndex >= 0) ? sheetIndex : simpleSheet, abs: true };
            } else if (_.isString(complexSheet)) {
                complexSheet = complexSheet.replace(/''/g, '\'');
                sheetIndex = model.getSheetIndex(complexSheet);
                sheetRef = { sheet: (sheetIndex >= 0) ? sheetIndex : complexSheet, abs: true };
            } else if (_.isString(refError)) {
                sheetRef = { sheet: -1, abs: true };
            }

            return sheetRef;
        }

        /**
         * Returns whether the passed sheet reference structure points to an
         * existing sheet.
         *
         * @param {Object|Null} sheetRef
         *  A sheet reference structure. Can be null, in that case the method
         *  returns false.
         *
         * @returns {Boolean}
         *  Whether the sheet reference structure points to an existing sheet.
         */
        function isValidSheetRef(sheetRef) {
            return _.isObject(sheetRef) && _.isNumber(sheetRef.sheet) && (sheetRef.sheet >= 0);
        }

        /**
         * Returns whether the passed sheet reference structure points to a
         * non-existing sheet.
         *
         * @param {Object|Null} sheetRef
         *  A sheet reference structure. Can be null, in that case the method
         *  returns false.
         *
         * @returns {Boolean}
         *  Whether the sheet reference structure points to a non-existing
         *  sheet.
         */
        function isInvalidSheetRef(sheetRef) {
            return _.isObject(sheetRef) && !(_.isNumber(sheetRef.sheet) && (sheetRef.sheet >= 0));
        }

        /**
         * Transforms the passed sheet reference, after a sheet has been
         * inserted, deleted, or moved in the document.
         *
         * @param {Object|Null} sheetRef
         *  A sheet reference structure. Can be null, in that case the method
         *  does nothing and returns false.
         *
         * @param {Number} to
         *  The new index of the inserted or moved sheet.
         *
         * @param {Number} from
         *  The old index of the deleted or moved sheet.
         *
         * @returns {Boolean}
         *  Whether the sheet reference structure has been modified.
         */
        function transformSheetRef(sheetRef, to, from) {

            // no sheet reference passed, or invalid sheet name in reference
            if (!isValidSheetRef(sheetRef) || (to === from)) {
                return false;
            }

            // sheet inserted
            if (!_.isNumber(from)) {
                // referenced sheet follows inserted sheet
                if (to <= sheetRef.sheet) { sheetRef.sheet += 1; return true; }
            }

            // sheet deleted
            else if (!_.isNumber(to)) {
                // referenced sheet follows deleted sheet
                if (from < sheetRef.sheet) { sheetRef.sheet -= 1; return true; }
                // referenced sheet has been deleted
                if (from === sheetRef.sheet) { sheetRef.sheet = -1; return true; }
            }

            // sheet moved
            else {
                // referenced sheet moved to another position
                if (from === sheetRef.sheet) { sheetRef.sheet = to; return true; }
                // sheet moved backwards, referenced sheet follows
                if ((to < from) && (to <= sheetRef.sheet) && (sheetRef.sheet < from)) { sheetRef.sheet += 1; return true; }
                // sheet moved forwards, referenced sheet follows
                if ((from < to) && (from < sheetRef.sheet) && (sheetRef.sheet <= to)) { sheetRef.sheet -= 1; return true; }
            }

            return false;
        }

        /**
         * Changes the passed sheet reference to the new sheet index, if it
         * refers to the old sheet index.
         *
         * @param {Object|Null} sheetRef
         *  A sheet reference structure. Can be null, in that case the method
         *  does nothing and returns false.
         *
         * @param {Number} oldSheet
         *  The old index of the sheet to be relocated.
         *
         * @param {Number} newSheet
         *  The new index of the sheet to be relocated.
         *
         * @returns {Boolean}
         *  Whether the sheet reference structure has been modified.
         */
        function relocateSheetRef(sheetRef, oldSheet, newSheet) {

            // no sheet reference passed, or invalid sheet name in reference
            if (!isValidSheetRef(sheetRef) || (oldSheet === newSheet)) {
                return false;
            }

            // TODO: always for Excel formulas, but only relative sheet references for ODF
            if (sheetRef.sheet === oldSheet) {
                sheetRef.sheet = newSheet;
                return true;
            }

            return false;
        }

        /**
         * Returns the sheet reference prefix used by cell references and
         * defined names, including the trailing exclamation mark.
         */
        function generateSheetName(sheet1Ref, sheet2Ref) {

            var sheet1Name = null,
                sheet2Name = null,
                simpleNames = false;

            // convert first sheet name
            if (!sheet1Ref) { return ''; }
            sheet1Name = _.isNumber(sheet1Ref.sheet) ? model.getSheetName(sheet1Ref.sheet) : sheet1Ref.sheet;
            if (!_.isString(sheet1Name)) { return config.REF_ERROR + '!'; }
            simpleNames = config.RE.SIMPLE_SHEET_NAME.test(sheet1Name);

            // convert second sheet name
            if (sheet2Ref && !_.isEqual(sheet1Ref, sheet2Ref)) {
                sheet2Name = _.isNumber(sheet2Ref.sheet) ? model.getSheetName(sheet2Ref.sheet) : sheet2Ref.sheet;
                if (!_.isString(sheet2Name)) { return config.REF_ERROR + '!'; }
                simpleNames = simpleNames && config.RE.SIMPLE_SHEET_NAME.test(sheet2Name);
                sheet1Name += ':' + sheet2Name;
            }

            // enclose complex names in apostrophes
            return simpleNames ? (sheet1Name + '!') : ('\'' + sheet1Name.replace(/'/g, '\'\'') + '\'!');
        }

        /**
         * Returns a cell reference structure for the passed cell address
         * components.
         *
         * @param {String|Boolean} absCol
         *  The absolute marker for the column. The column will be marked to be
         *  absolute, if this parameter is the string '$' or the Boolean value
         *  true.
         *
         * @param {String|Number} col
         *  The column name, or the zero-based column index.
         *
         * @param {String|Boolean} absRow
         *  The absolute marker for the row. The row will be marked to be
         *  absolute, if this parameter is the string '$' or the Boolean value
         *  true.
         *
         * @param {String|Number} row
         *  The row name, or the zero-based row index.
         *
         * @returns {Object|Null}
         *  Null, if the passed column or row is not valid. Otherwise, a cell
         *  reference structure with the following properties:
         *  - {Array} address
         *      The cell address.
         *  - {Boolean} absCol
         *      Whether the column reference is absolute (as in $C2).
         *  - {Boolean} absRow
         *      Whether the row reference is absolute (as in C$2).
         */
        function makeCellRef(absCol, col, absRow, row) {

            var address = [_.isNumber(col) ? col : SheetUtils.parseColName(col), _.isNumber(row) ? row : SheetUtils.parseRowName(row)];

            return model.isValidAddress(address) ? {
                address: address,
                absCol: _.isBoolean(absCol) ? absCol : (absCol === '$'),
                absRow: _.isBoolean(absRow) ? absRow : (absRow === '$')
            } : null;
        }

        /**
         * Relocates the relative column/row components in the passed cell
         * reference in-place.
         */
        function relocateCellRef(cellRef, refAddress, targetAddress, wrapReferences) {

            // relocates a single column/row index
            function relocate(addrIndex, maxIndex) {
                var index = cellRef.address[addrIndex] + (targetAddress[addrIndex] - refAddress[addrIndex]);
                if (wrapReferences) {
                    if (index < 0) {
                        index += (maxIndex + 1);
                    } else if (index > maxIndex) {
                        index -= (maxIndex + 1);
                    }
                }
                cellRef.address[addrIndex] = index;
            }

            // relocate relative column and row
            if (!cellRef.absCol) { relocate(0, config.MAXCOL); }
            if (!cellRef.absRow) { relocate(1, config.MAXROW); }
        }

        /**
         * Generates the display text for the passed references.
         */
        function generateReference(cell1Ref, cell2Ref, sheet1Ref, sheet2Ref) {

            var range = cell1Ref ? SheetUtils.getAdjustedRange({ start: cell1Ref.address, end: (cell2Ref || cell1Ref).address }) : null,
                text = generateSheetName(sheet1Ref, sheet2Ref);

            // returns the complete name of the column described by the passed cell reference
            function generateColName(cellRef) {
                return (cellRef.absCol ? '$' : '') + SheetUtils.getColName(cellRef.address[0]);
            }

            // returns the complete name of the row described by the passed cell reference
            function generateRowName(cellRef) {
                return (cellRef.absRow ? '$' : '') + SheetUtils.getRowName(cellRef.address[1]);
            }

            // returns the complete name of the cell described by the passed cell reference
            function generateCellName(cellRef) {
                return generateColName(cellRef) + generateRowName(cellRef);
            }

            if (!model.isValidRange(range)) {
                text += config.REF_ERROR;
            }

            // generate row interval (preferred over column interval for entire sheet range)
            else if (cell1Ref.absCol && cell2Ref && cell2Ref.absCol && model.isRowRange(range)) {
                text += generateRowName(cell1Ref) + ':' + generateRowName(cell2Ref);
            }

            // generate column interval
            else if (cell1Ref.absRow && cell2Ref && cell2Ref.absRow && model.isColRange(range)) {
                text += generateColName(cell1Ref) + ':' + generateColName(cell2Ref);
            }

            // generate range or cell address
            else {
                text += generateCellName(cell1Ref);
                if (cell2Ref) { text += ':' + generateCellName(cell2Ref); }
            }

            return text;
        }

        // class Token --------------------------------------------------------

        /**
         * Base class for all formula tokens used in the tokenizer.
         *
         * @constructor
         *
         * @param {String} type
         *  The token type identifier.
         *
         * @param {String} [text]
         *  The current text representation of the token. This text may differ
         *  from the default text representation the token would generate for
         *  its current value. Used for example while the formula is edited in
         *  the GUI where the same token value can be represented by different
         *  text strings, e.g. '1' vs. '1.000'. Can be omitted, if a text
         *  generator callback function has been passed (see next parameter).
         *  In this case, the initial text of this token will be set to the
         *  return value of the callback function.
         *
         * @param {Function} [generateText]
         *  A callback function that returns the current text representation of
         *  the token. If omitted, the token will always keep its initial text
         *  representation.
         */
        var Token = _.makeExtendable(function (type, text, generateText) {

            /**
             * Returns the type identifier of this token.
             *
             * @returns {String}
             *  The type identifier of this token.
             */
            this.getType = function () {
                return type;
            };

            /**
             * Returns whether the type identifier of this token matches the
             * passed type specifier.
             *
             * @param {String|RegExp} typeSpec
             *  Either a string with a single type identifier that must match
             *  the type of this token exactly, or a regular expression that
             *  will be tested against the type of this token.
             *
             * @returns {Boolean}
             *  Whether the type of this token matches the passed type.
             */
            this.isType = function (typeSpec) {
                return _.isString(typeSpec) ? (type === typeSpec) :
                    _.isRegExp(typeSpec) ? typeSpec.test(type) :
                    false;
            };

            /**
             * Returns the current text representation of this token.
             *
             * @returns {String}
             *  The current text representation of this token.
             */
            this.getText = function () {
                return text;
            };

            /**
             * Changes the current text representation of this token.
             *
             * @param {String} newText
             *  The new text representation of this token.
             *
             * @returns {Boolean}
             *  Whether the text representation has been changed.
             */
            this.setText = function (newText) {
                if (text !== newText) {
                    text = newText;
                    return true;
                }
                return false;
            };

            /**
             * Regenerates and stores the text representation for the current
             * value of this token.
             *
             * @returns {Boolean}
             *  Whether the text representation has been changed.
             */
            this.updateText = _.isFunction(generateText) ? function () {
                return this.setText(generateText());
            } : _.constant(false);

            // set initial text, if missing
            if (!_.isString(text)) {
                text = _.isFunction(generateText) ? generateText() : '';
            }

        }); // class Token

        /**
         * Returns a text description of the token for debugging purposes.
         */
        Token.prototype.toString = function () {
            var text = this.getText();
            return this.getType() + (text ? ('[' + text + ']') : '');
        };

        // class OperatorToken ------------------------------------------------

        /**
         * A formula token that represents a unary or binary operator in a
         * formula. See method Tokenizer.createOperatorToken() for details
         * about the parameters.
         *
         * @constructor
         *
         * @extends Token
         */
        var OperatorToken = Token.extend({ constructor: function (opType, opText) {

            // base constructor
            Token.call(this, 'op', opText);

            /**
             * Returns the operator type of this token.
             */
            this.getOpType = function () {
                return opType;
            };

        }}); // class OperatorToken

        // class LiteralToken -------------------------------------------------

        /**
         * A formula token that represents a literal value (a number, a string,
         * a Boolean value, or an error code) in a formula. See method
         * Tokenizer.createLiteralToken() for details about the parameters.
         *
         * @constructor
         *
         * @extends Token
         */
        var LiteralToken = Token.extend({ constructor: function (value, text) {

            // base constructor
            Token.call(this, 'lit', text, generateText);

            /**
             * Generates the display text for the current value.
             */
            function generateText() {
                return generateLiteralText(value);
            }

            /**
             * Returns the current value of this token.
             *
             * @returns {Number|String|Boolean|ErrorCode}
             *  The current value of this token.
             */
            this.getValue = function () {
                return value;
            };

            /**
             * Changes the current value of this token.
             *
             * @param {Number|String|Boolean|ErrorCode} newValue
             *  The new value for this token.
             *
             * @returns {Boolean}
             *  Whether the text representation of the token has been changed.
             */
            this.setValue = function (newValue) {
                value = newValue;
                return this.updateText();
            };

        }}); // class LiteralToken

        // class MatrixToken --------------------------------------------------

        /**
         * A formula token that represents a constant matrix literal in a
         * formula. See method Tokenizer.createMatrixToken() for details about
         * the parameters.
         *
         * @constructor
         *
         * @extends Token
         */
        var MatrixToken = Token.extend({ constructor: function (matrix, text) {

            // base constructor
            Token.call(this, 'mat', text, generateText);

            /**
             * Generates the display text for the matrix.
             */
            function generateText() {
                return '{' + _.map(matrix.getArray(), function (elems) {
                    return _.map(elems, generateLiteralText).join(config.COL);
                }).join(config.ROW) + '}';
            }

            /**
             * Returns the matrix literal contained in this token.
             *
             * @returns {Matrix}
             *  The matrix literal of this token.
             */
            this.getMatrix = function () {
                return matrix;
            };

        }}); // class MatrixToken

        // class ReferenceToken -----------------------------------------------

        /**
         * A formula token that represents a cell range reference in a formula.
         * See method Tokenizer.createReferenceToken() for details about the
         * parameters.
         *
         * @constructor
         *
         * @extends Token
         */
        var ReferenceToken = Token.extend({ constructor: function (cell1Ref, cell2Ref, sheet1Ref, sheet2Ref, text) {

            // base constructor
            Token.call(this, 'ref', text, generateText);

            /**
             * Generates the display text for the current references.
             */
            function generateText() {
                return generateReference(cell1Ref, cell2Ref, sheet1Ref, sheet2Ref);
            }

            /**
             * Returns copies of the own cell reference objects that have been
             * relocated according to the passed settings.
             */
            function makeRelocatedCellRefs(refAddress, targetAddress, wrapReferences) {

                var // the result object returned by this method
                    result = { cell1Ref: _.copy(cell1Ref, true), cell2Ref: _.copy(cell2Ref, true) };

                if (result.cell1Ref) {
                    relocateCellRef(result.cell1Ref, refAddress, targetAddress, wrapReferences);
                    if (!model.isValidAddress(result.cell1Ref.address)) { result.cell1Ref = null; }
                }
                if (result.cell1Ref && result.cell2Ref) {
                    relocateCellRef(result.cell2Ref, refAddress, targetAddress, wrapReferences);
                    if (!model.isValidAddress(result.cell2Ref.address)) { result.cell1Ref = result.cell2Ref = null; }
                }

                return result;
            }

            /**
             * Returns the sheet range referred by this token.
             *
             * @returns {Object|Null}
             *  The sheet range referred by this token, in the properties
             *  'sheet1' and 'sheet2'. If the token does not contain a valid
             *  sheet range, returns null.
             */
            this.getSheetRange = function () {
                var sheet1 = sheet1Ref ? sheet1Ref.sheet : sheetModel ? sheetModel.getIndex() : -1,
                    sheet2 = sheet2Ref ? sheet2Ref.sheet : sheet1;
                return (_.isNumber(sheet1) && (sheet1 >= 0) && _.isNumber(sheet2) && (sheet2 >= 0)) ? { sheet1: Math.min(sheet1, sheet2), sheet2: Math.max(sheet1, sheet2) } : null;
            };

            /**
             * Returns whether this reference token refers exactly to the
             * specified sheet.
             *
             * @returns {Boolean}
             *  Whether this reference token refers to the specified sheet.
             */
            this.isSheet = function (sheet) {
                var sheetRange = this.getSheetRange();
                return _.isObject(sheetRange) && (sheetRange.sheet1 === sheet) && (sheetRange.sheet2 === sheet);
            };

            /**
             * Returns whether this reference token refers to the specified
             * sheet, or refers to a sheet range that contains the specified
             * sheet.
             *
             * @returns {Boolean}
             *  Whether this reference token contains the specified sheet.
             */
            this.containsSheet = function (sheet) {
                var sheetRange = this.getSheetRange();
                return _.isObject(sheetRange) && (sheetRange.sheet1 <= sheet) && (sheet <= sheetRange.sheet2);
            };

            /**
             * Returns whether the sheet reference of this token is invalid
             * (it exists but points to a non-existing sheet).
             *
             * @returns {Boolean}
             *  Whether the sheet reference of this token is invalid.
             */
            this.hasSheetError = function () {
                return isInvalidSheetRef(sheet1Ref) || isInvalidSheetRef(sheet2Ref);
            };

            /**
             * Returns the reference expression of the relocated cell range for
             * this token.
             */
            this.generateReference = function (refAddress, targetAddress, wrapReferences) {
                var result = makeRelocatedCellRefs(refAddress, targetAddress, wrapReferences);
                return generateReference(result.cell1Ref, result.cell2Ref, sheet1Ref, sheet2Ref);
            };

            /**
             * Returns the address of the original cell range represented by
             * this token, or null if the range address is invalid.
             *
             * @returns {Object|Null}
             *  The address of the range referenced by this reference token; or
             *  null, if the token does not refer to a valid range.
             */
            this.getRange = function () {
                return cell1Ref ? SheetUtils.getAdjustedRange({ start: cell1Ref.address, end: (cell2Ref || cell1Ref).address }) : null;
            };

            /**
             * Returns the address of a relocated cell range.
             *
             * @returns {Object|Null}
             *  The address of the relocated cell range; or null, if the
             *  resulting range address is invalid.
             */
            this.getRelocatedRange = function (refAddress, targetAddress, wrapReferences) {
                var result = makeRelocatedCellRefs(refAddress, targetAddress, wrapReferences);
                return result.cell1Ref ? SheetUtils.getAdjustedRange({ start: result.cell1Ref.address, end: (result.cell2Ref || result.cell1Ref).address }) : null;
            };

            /**
             * Changes the current cell range of this token, keeps all absolute
             * flags and sheet references intact.
             *
             * @returns {Boolean}
             *  Whether the token has been changed.
             */
            this.setRange = function (range) {
                if (!cell1Ref) { cell1Ref = { absCol: true, absRow: true }; cell2Ref = null; }
                cell1Ref.address = _.clone(range.start);
                if (!cell2Ref && !_.isEqual(range.start, range.end)) { cell2Ref = _.copy(cell1Ref, true); }
                if (cell2Ref) { cell2Ref.address = _.clone(range.end); }
                return this.updateText();
            };

            /**
             * Transforms this reference token after columns or rows have been
             * inserted into or deleted from a sheet in the document.
             *
             * @returns {Boolean}
             *  Whether the token has been changed.
             */
            this.transformToken = function (sheet, interval, insert, columns) {

                // nothing to do, if this token represents a reference error,
                // or points to another sheet, or sheet reference is invalid
                if (!cell1Ref || !this.isSheet(sheet)) { return false; }

                var // the original range
                    oldRange = this.getRange(),
                    // the transformed range address
                    newRange = model.transformRange(oldRange, interval, insert, columns);

                // nothing to do, if the range does not change
                if (_.isEqual(oldRange, newRange)) { return false; }

                // check that the range has been transformed successfully
                if (newRange) {
                    this.setRange(newRange);
                } else {
                    cell1Ref = cell2Ref = null;
                    this.updateText();
                }
                return true;
            };

            /**
             * Relocates this reference token to a new cell position.
             *
             * @returns {Boolean}
             *  Whether the token has been changed.
             */
            this.relocateToken = function (refAddress, targetAddress, wrapReferences) {

                // do not touch tokens representing a reference error
                if (!cell1Ref) { return false; }

                // relocate both cell references
                relocateCellRef(cell1Ref, refAddress, targetAddress, wrapReferences);
                if (cell2Ref) { relocateCellRef(cell2Ref, refAddress, targetAddress, wrapReferences); }

                // check validity of the new references
                if (!model.isValidAddress(cell1Ref.address) || (cell2Ref && !model.isValidAddress(cell2Ref.address))) {
                    cell1Ref = cell2Ref = null;
                }

                // recalculate and set the new text representation
                this.updateText();
                return true;
            };

            /**
             * Refreshes the sheet references after a sheet has been inserted,
             * deleted, or moved in the document.
             *
             * @returns {Boolean}
             *  Whether the token has been changed.
             */
            this.transformSheet = function (to, from) {
                var changed1 = transformSheetRef(sheet1Ref, to, from),
                    changed2 = transformSheetRef(sheet2Ref, to, from);
                return changed1 || changed2;
            };

            /**
             * Changes all sheet references containing the specified old sheet
             * index to the new sheet index. Used for example when copying
             * existing sheets, and adjusting all references and defined names
             * that point to the original sheet.
             *
             * @param {Number} oldSheet
             *  The zero-based index of the original sheet.
             *
             * @param {Number} newSheet
             *  The zero-based index of the target sheet.
             *
             * @returns {Boolean}
             *  Whether the token has been changed.
             */
            this.relocateSheet = function (oldSheet, newSheet) {
                var changed1 = relocateSheetRef(sheet1Ref, oldSheet, newSheet),
                    changed2 = relocateSheetRef(sheet2Ref, oldSheet, newSheet);
                if (changed1 || changed2) {
                    this.updateText();
                    return true;
                }
                return false;
            };

            /**
             * Refreshes the text representation after a sheet has been renamed
             * in the document.
             *
             * @param {Number} sheet
             *  The zero-based index of the renamed sheet.
             *
             * @returns {Boolean}
             *  Whether the token has been changed.
             */
            this.renameSheet = function (sheet) {
                var sheetRange = this.getSheetRange();
                if (sheetRange && ((sheetRange.sheet1 === sheet) || (sheetRange.sheet2 === sheet))) {
                    this.updateText();
                    return true;
                }
                return false;
            };

        }}); // class ReferenceToken

        // class NameToken ----------------------------------------------------

        /**
         * A formula token that represents a defined name in a formula. See
         * methods Tokenizer.createNameToken() and
         * Tokenizer.createFunctionToken() for details about the parameters.
         *
         * @constructor
         *
         * @extends Token
         *
         * @param {String} value
         *  The name of the defined name or function.
         *
         * @param {Boolean} isFunc
         *  Whether the token represents a function call (true), or a defined
         *  name (false).
         */
        var NameToken = Token.extend({ constructor: function (value, sheetRef, isFunc, text) {

            // base constructor
            Token.call(this, isFunc ? 'func' : 'name', text, generateText);

            /**
             * Generates the display text for the current token contents.
             */
            function generateText() {
                return generateSheetName(sheetRef) + ((isFunc && !sheetRef) ? config.getFunctionText(value) : value);
            }

            /**
             * Returns the name of the defined name, or the native name of the
             * function according to the grammar of the tokenizer.
             *
             * @returns {String}
             *  The native name, according to the tokenizer grammar.
             */
            this.getValue = function () {
                return value;
            };

            /**
             * Returns whether the token contains a sheet reference structure,
             * regardless if it is valid.
             *
             * @returns {Boolean}
             *  Whether the token contains a sheet reference structure.
             */
            this.hasSheetRef = function () {
                return _.isObject(sheetRef);
            };

            /**
             * Returns the index of the sheet referred by this token.
             *
             * @returns {Number}
             *  The index of the sheet referred by this token. If no sheet
             *  index can be resolved (no sheet reference in the token, and no
             *  sheet model in the token array), returns -1.
             */
            this.getSheet = function () {
                var sheet = sheetRef ? sheetRef.sheet : sheetModel ? sheetModel.getIndex() : -1;
                return (_.isNumber(sheet) && (sheet >= 0)) ? sheet : -1;
            };

            /**
             * Returns whether this name token refers to the specified sheet.
             *
             * @returns {Boolean}
             *  Whether this name token refers to the specified sheet.
             */
            this.isSheet = function (sheet) {
                return (sheet >= 0) && (this.getSheet() === sheet);
            };

            /**
             * Returns whether the sheet reference of this token is invalid
             * (it exists but points to a non-existing sheet).
             *
             * @returns {Boolean}
             *  Whether the sheet reference of this token is invalid.
             */
            this.hasSheetError = function () {
                return isInvalidSheetRef(sheetRef);
            };

            /**
             * Returns the token array with the definition of this name.
             *
             * @returns {TokenArray|Null}
             *  The token array, if the token refers to an existing defined
             *  name, otherwise null.
             */
            this.getTokenArray = function () {

                // do nothing for functions
                if (isFunc) { return null; }

                // sheet reference exists (e.g.: Sheet2!name): get name from specified sheet
                if (sheetRef) {
                    var refSheetModel = model.getSheetModel(this.getSheet());
                    return refSheetModel ? refSheetModel.getNameCollection().getTokenArray(value) : null;
                }

                // no sheet reference: try sheet names (if token array in sheet context), then global names
                return (sheetModel && sheetModel.getNameCollection().getTokenArray(value)) || model.getNameCollection().getTokenArray(value);
            };

            /**
             * Refreshes the sheet reference after a sheet has been inserted,
             * deleted, or moved in the document.
             *
             * @returns {Boolean}
             *  Whether the token has been changed.
             */
            this.transformSheet = function (to, from) {
                return transformSheetRef(sheetRef, to, from);
            };

            /**
             * Changes the sheet reference to the new sheet index, if it
             * contains the specified old sheet index. Used for example when
             * copying existing sheets, and adjusting all references and
             * defined names that point to the original sheet.
             *
             * @param {Number} oldSheet
             *  The zero-based index of the original sheet.
             *
             * @param {Number} newSheet
             *  The zero-based index of the target sheet.
             *
             * @returns {Boolean}
             *  Whether the token has been changed.
             */
            this.relocateSheet = function (oldSheet, newSheet) {
                if (relocateSheetRef(sheetRef, oldSheet, newSheet)) {
                    this.updateText();
                    return true;
                }
                return false;
            };

            /**
             * Refreshes the text representation after a sheet has been renamed
             * in the document.
             *
             * @returns {Boolean}
             *  Whether the token has been changed.
             */
            this.renameSheet = function (sheet) {
                if (sheetRef && (sheetRef.sheet === sheet)) {
                    this.updateText();
                    return true;
                }
                return false;
            };

        }}); // class NameToken

        // public methods -----------------------------------------------------

        /**
         * Returns the identifier of the formula grammar used by this tokenizer
         * instance.
         *
         * @returns {String}
         *  The formula grammar. See class constructor for details.
         */
        this.getGrammar = function () {
            return grammar;
        };

        /**
         * Encloses the passed sheet name in apostrophes, if it contains any
         * characters that would be ambiguous in a formula expression (e.g.
         * whitespace or any operator characters). Apostrophes embedded in the
         * sheet name will be doubled.
         *
         * @param {String} sheetName
         *  The sheet name to be encoded.
         *
         * @returns {String}
         *  The encoded sheet name ready to be used in a formula expression.
         */
        this.encodeSheetName = function (sheetName) {

            var // whether encoding is not needed
                simpleName = config.RE.SIMPLE_SHEET_NAME.test(sheetName);

            // enclose complex names in apostrophes
            return simpleName ? sheetName : ('\'' + sheetName.replace(/'/g, '\'\'') + '\'');
        };

        /**
         * Creates a new token instance with the passed type and display text.
         *
         * @param {String} type
         *  The token type identifier.
         *
         * @param {String} text
         *  The current text representation of the token.
         *
         * @returns {Token}
         *  The new token instance.
         */
        this.createToken = function (type, text) {
            return new Token(type, text);
        };

        /**
         * Create a new token that represents a literal value (a number, a
         * string, a Boolean value, or an error code).
         *
         * @param {Number|String|Boolean|ErrorCode|Null} value
         *  The value for the token. The special value null is reserved for
         *  representing the empty parameter in a function call, e.g. in the
         *  formula =SUM(1,,2).
         *
         * @param {String} [text]
         *  The exact text representation of the token as appearing in the
         *  formula expression. This text may differ from the default text
         *  representation the token would generate for its current value. Used
         *  for example while the formula is edited in the GUI where the same
         *  token value can be represented by different text strings, e.g. '1'
         *  vs. '1.000'. If omitted, a default text representation will be
         *  generated automatically.
         *
         * @returns {Token}
         *  The new token instance.
         */
        this.createLiteralToken = function (value, text) {
            return new LiteralToken(value, text);
        };

        /**
         * Create a new token that represents a constant matrix literal.
         *
         * @param {Matrix} matrix
         *  The matrix to be inserted into the token.
         *
         * @param {String} [text]
         *  The exact text representation of the token as appearing in the
         *  formula expression. This text may differ from the default text
         *  representation the token would generate for its current value. Used
         *  for example while the formula is edited in the GUI where the same
         *  token value can be represented by different text strings, e.g.
         *  '{1}' vs. '{1.000}'. If omitted, a default text representation will
         *  be generated automatically.
         *
         * @returns {Token}
         *  The new token instance.
         */
        this.createMatrixToken = function (matrix, text) {
            return new MatrixToken(matrix, text);
        };

        /**
         * Creates a new operator token.
         *
         * @param {String} opType
         *  The type of the operator. Must be one of '=' (comparison), '&'
         *  (string concatenation), '+' (addition or subtraction, also used for
         *  unary operators), '*' (multiplication or division), '^' (power
         *  operator), ',' (reference list operator), '!' (reference
         *  intersection operator), or ':' (reference range operator).
         *
         * @param {String} opText
         *  The exact text representation of the operator.
         *
         * @returns {Token}
         *  The new token instance.
         */
        this.createOperatorToken = function (opType, opText) {
            return new OperatorToken(opType, opText);
        };

        /**
         * Creates a new separator token used as list operator, or as function
         * parameter separator.
         *
         * @returns {Token}
         *  The new token instance.
         */
        this.createSeparatorToken = function () {
            return new Token('sep', config.SEP);
        };

        /**
         * Creates a new formula token that represents an opening or closing
         * parenthesis.
         *
         * @param {Boolean} open
         *  Whether this token represents the opening parenthesis.
         *
         * @returns {Token}
         *  The new token instance.
         */
        this.createParenthesisToken = function (open) {
            return new Token(open ? 'open' : 'close', open ? '(' : ')');
        };

        /**
         * Creates a new formula token that represents an opening or closing
         * parenthesis of a matrix literal.
         *
         * @param {Boolean} open
         *  Whether this token represents the opening parenthesis.
         *
         * @returns {Token}
         *  The new token instance.
         */
        this.createMatrixParenthesisToken = function (open) {
            return new Token(open ? 'mat_open' : 'mat_close', open ? '{' : '}');
        };

        /**
         * Creates a new formula token that represents a column or a row
         * separator in a matrix literal.
         *
         * @param {Boolean} row
         *  Whether this token represents the row separator.
         *
         * @returns {Token}
         *  The new token instance.
         */
        this.createMatrixSeparatorToken = function (row) {
            return new Token(row ? 'mat_row' : 'mat_col', row ? config.ROW : config.COL);
        };

        /**
         * Creates a new cell range reference token.
         *
         * @param {Object|Null} cell1Ref
         *  The first cell reference structure (e.g. the cell address 'A1' in
         *  the formula A1:B2). If null or omitted, the token will represent a
         *  reference error (e.g. Sheet1!#REF!, after deleting the referenced
         *  column or row).
         *
         * @param {Object|Null} cell2Ref
         *  The second cell reference structure (e.g. the cell address 'B2' in
         *  the formula A1:B2). If null or omitted, the token will represent a
         *  single cell (e.g. Sheet1!A1).
         *
         * @param {Object|Null} sheet1Ref
         *  The first sheet reference structure (e.g. 'Sheet1' in the formula
         *  Sheet1:Sheet2!A1). If null or undefined, the token represents a
         *  local cell range reference into the passed sheet (e.g. A1:B2).
         *
         * @param {Object|Null} sheet2Ref
         *  The second sheet reference structure (e.g. 'Sheet2' in the formula
         *  Sheet1:Sheet2!A1). If null or undefined, the token represents a
         *  single sheet reference (e.g. Sheet1!A1).
         *
         * @param {String} [text]
         *  The current text representation of the token. This text may differ
         *  from the default text representation the reference token would
         *  generate for its current value. Used for example while the formula
         *  is edited in the GUI where the same token value can be represented
         *  by different text strings, e.g. 'A1' vs. 'a1' vs. 'A001'. If
         *  omitted, an appropriate text representation will be generated
         *  automatically.
         *
         * @returns {Token}
         *  The new token instance.
         */
        this.createReferenceToken = function (cell1Ref, cell2Ref, sheet1Ref, sheet2Ref, text) {
            return new ReferenceToken(cell1Ref, cell2Ref, sheet1Ref, sheet2Ref, text);
        };

        /**
         * Creates a new defined name token.
         *
         * @param {String} name
         *  The actual name (e.g. 'my_name' in the formula Sheet1!my_name).
         *
         * @param {Object|Null} sheetRef
         *  The sheet reference structure (e.g. 'Sheet1' in the formula
         *  Sheet1!my_name). If null or undefined, the token represents a local
         *  name in the passed sheet, or a global name.
         *
         * @param {String} [text]
         *  The current text representation of the token. This text may differ
         *  from the default text representation the name token would generate
         *  for its current value. Used for example while the formula is edited
         *  in the GUI where the same token value can be represented by
         *  different text strings, e.g. 'my_name' vs. 'MY_NAME'. If omitted,
         *  an appropriate text representation will be generated automatically.
         *
         * @returns {Token}
         *  The new token instance.
         */
        this.createNameToken = function (name, sheetRef, text) {
            return new NameToken(name, sheetRef, false, text);
        };

        /**
         * Creates a new function token.
         *
         * @param {String} funcName
         *  The function name (e.g. 'myFunc' in the formula Module1!myFunc()).
         *
         * @param {Object|Null} sheetRef
         *  The sheet reference structure (e.g. 'Module1' in the formula
         *  Module1!myFunc()). If null or undefined, the token represents a
         *  built-in function, or a globally available macro function.
         *
         * @param {String} [text]
         *  The current text representation of the token. This text may differ
         *  from the default text representation the function token would
         *  generate for its current value. Used for example while the formula
         *  is edited in the GUI where the same token value can be represented
         *  by different text strings, e.g. 'myFunc' vs. 'MYFUNC'. If omitted,
         *  an appropriate text representation will be generated automatically.
         *
         * @returns {Token}
         *  The new token instance.
         */
        this.createFunctionToken = function (funcName, sheetRef, text) {
            return new NameToken(funcName, sheetRef, true, text);
        };

        /**
         * Creates a new white-space token.
         *
         * @param {String} text
         *  The white-space text represented by the token (space characters and
         *  new-line characters).
         *
         * @returns {Token}
         *  The new token instance.
         */
        this.createWhiteSpaceToken = function (text) {
            return new Token('ws', text);
        };

        /**
         * Creates a new bad token containing any invalid text.
         *
         * @param {String} text
         *  The text represented by the token.
         *
         * @returns {Token}
         *  The new token instance.
         */
        this.createBadToken = function (text) {
            return new Token('bad', text);
        };

        /**
         * Converts the specified formula expression to an array of formula
         * tokens.
         *
         * @param {String} formula
         *  The formula string, without the leading equality sign.
         *
         * @returns {Array}
         *  An array of formula tokens.
         */
        this.parseFormula = function (formula) {

            var // the matches of the regular expressions
                matches = null,
                // whether a matrix literal is currently being parsed
                matrix = false,
                // start and end reference structures for sheet references
                sheet1Ref = null, sheet2Ref = null,
                // start and end reference structures for column/row references
                cell1Ref = null, cell2Ref = null,
                // parsed literal
                value = null,
                // the resulting formula tokens
                tokens = [];

            // callback for the _.any() function (JSHint prevents definition inside while loop)
            function matchOperator(regExp, opType) {
                if ((matches = regExp.exec(formula))) {
                    tokens.push(self.createOperatorToken(opType, matches[0]));
                    return true;
                }
            }

            // extends the last 'bad' token, or appends a new 'bad' token
            function appendBadText(text) {
                var token = _.last(tokens);
                if (token && token.isType('bad')) {
                    token.setText(token.getText() + text);
                } else {
                    tokens.push(self.createBadToken(text));
                }
            }

            // extract all supported tokens from start of formula string
            TokenUtils.info('Tokenizer.parseFormula(): formula="' + formula + '"');
            while (formula.length > 0) {

                // try/finally to be able to continue the while-loop at any place
                try {

                    // the following tokens are invalid in matrix literals
                    if (!matrix) {

                        // opening parenthesis of a matrix literal
                        if ((matches = config.RE.MATRIX_OPEN.exec(formula))) {
                            tokens.push(this.createMatrixParenthesisToken(true));
                            matrix = true;
                            continue;
                        }

                        // unary/binary operator
                        if (_.any(config.RE.OPERATORS, matchOperator)) {
                            continue;
                        }

                        // list/parameter separator
                        if ((matches = config.RE.SEPARATOR.exec(formula))) {
                            tokens.push(this.createSeparatorToken());
                            continue;
                        }

                        // opening parenthesis
                        if ((matches = config.RE.OPEN.exec(formula))) {
                            tokens.push(this.createParenthesisToken(true));
                            continue;
                        }

                        // closing parenthesis
                        if ((matches = config.RE.CLOSE.exec(formula))) {
                            tokens.push(this.createParenthesisToken(false));
                            continue;
                        }

                        // cell range reference in multiple sheets (before other references)
                        if ((matches = config.RE.CELL_RANGE_CUBE_REF.exec(formula))) {
                            sheet1Ref = makeSheetRef(matches[1], matches[3]);
                            sheet2Ref = makeSheetRef(matches[2], matches[4]);
                            cell1Ref = makeCellRef(matches[5], matches[6], matches[7], matches[8]);
                            cell2Ref = makeCellRef(matches[9], matches[10], matches[11], matches[12]);
                            if (sheet1Ref && sheet2Ref && cell1Ref && cell2Ref) {
                                tokens.push(this.createReferenceToken(cell1Ref, cell2Ref, sheet1Ref, sheet2Ref, matches[0]));
                                continue;
                            }
                            // otherwise, continue with other REs, it may be a local name such as 'Sheet1:Sheet2!A1:name42'
                        }

                        // cell reference in multiple sheets
                        if ((matches = config.RE.CELL_CUBE_REF.exec(formula))) {
                            sheet1Ref = makeSheetRef(matches[1], matches[3]);
                            sheet2Ref = makeSheetRef(matches[2], matches[4]);
                            cell1Ref = makeCellRef(matches[5], matches[6], matches[7], matches[8]);
                            if (cell1Ref) {
                                tokens.push(this.createReferenceToken(cell1Ref, null, sheet1Ref, sheet2Ref, matches[0]));
                                continue;
                            }
                            // otherwise, continue with other REs, it may be a local name such as 'Sheet1:Sheet2!name42'
                        }

                        // column range reference in multiple sheets
                        if ((matches = config.RE.COL_RANGE_CUBE_REF.exec(formula))) {
                            sheet1Ref = makeSheetRef(matches[1], matches[3]);
                            sheet2Ref = makeSheetRef(matches[2], matches[4]);
                            cell1Ref = makeCellRef(matches[5], matches[6], true, 0);
                            cell2Ref = makeCellRef(matches[7], matches[8], true, config.MAXROW);
                            if (cell1Ref && cell2Ref) {
                                tokens.push(this.createReferenceToken(cell1Ref, cell2Ref, sheet1Ref, sheet2Ref, matches[0]));
                                continue;
                            }
                            // otherwise, continue with other REs, it may be a range of names such as 'Sheet1:Sheet2!name:name'
                        }

                        // row range reference in multiple sheets
                        if ((matches = config.RE.ROW_RANGE_CUBE_REF.exec(formula))) {
                            sheet1Ref = makeSheetRef(matches[1], matches[3]);
                            sheet2Ref = makeSheetRef(matches[2], matches[4]);
                            cell1Ref = makeCellRef(true, 0, matches[5], matches[6]);
                            cell2Ref = makeCellRef(true, config.MAXCOL, matches[7], matches[8]);
                            if (cell1Ref && cell2Ref) {
                                tokens.push(this.createReferenceToken(cell1Ref, cell2Ref, sheet1Ref, sheet2Ref, matches[0]));
                                continue;
                            }
                            // otherwise, continue with other REs
                        }

                        // reference error in multiple sheets
                        if ((matches = config.RE.ERROR_CUBE_REF.exec(formula))) {
                            sheet1Ref = makeSheetRef(matches[1], matches[3]);
                            sheet2Ref = makeSheetRef(matches[2], matches[4]);
                            tokens.push(this.createReferenceToken(null, null, sheet1Ref, sheet2Ref, matches[0]));
                            continue;
                        }

                        // cell range reference (before cell references)
                        if ((matches = config.RE.CELL_RANGE_REF.exec(formula))) {
                            sheet1Ref = makeSheetRef(matches[1], matches[2], matches[3]);
                            cell1Ref = makeCellRef(matches[4], matches[5], matches[6], matches[7]);
                            cell2Ref = makeCellRef(matches[8], matches[9], matches[10], matches[11]);
                            if (cell1Ref && cell2Ref) {
                                tokens.push(this.createReferenceToken(cell1Ref, cell2Ref, sheet1Ref, null, matches[0]));
                                continue;
                            }
                            // otherwise, continue with other REs, it may be a local name such as 'A1:name42'
                        }

                        // cell reference
                        if ((matches = config.RE.CELL_REF.exec(formula))) {
                            sheet1Ref = makeSheetRef(matches[1], matches[2], matches[3]);
                            cell1Ref = makeCellRef(matches[4], matches[5], matches[6], matches[7]);
                            if (cell1Ref) {
                                tokens.push(this.createReferenceToken(cell1Ref, null, sheet1Ref, null, matches[0]));
                                continue;
                            }
                            // otherwise, continue with other REs, it may be a local name such as 'name42'
                        }

                        // column range reference
                        if ((matches = config.RE.COL_RANGE_REF.exec(formula))) {
                            sheet1Ref = makeSheetRef(matches[1], matches[2], matches[3]);
                            cell1Ref = makeCellRef(matches[4], matches[5], true, 0);
                            cell2Ref = makeCellRef(matches[6], matches[7], true, config.MAXROW);
                            if (cell1Ref && cell2Ref) {
                                tokens.push(this.createReferenceToken(cell1Ref, cell2Ref, sheet1Ref, null, matches[0]));
                                continue;
                            }
                            // otherwise, continue with other REs, it may be a range of names such as 'name:name'
                        }

                        // row range reference
                        if ((matches = config.RE.ROW_RANGE_REF.exec(formula))) {
                            sheet1Ref = makeSheetRef(matches[1], matches[2], matches[3]);
                            cell1Ref = makeCellRef(true, 0, matches[4], matches[5]);
                            cell2Ref = makeCellRef(true, config.MAXCOL, matches[6], matches[7]);
                            if (cell1Ref && cell2Ref) {
                                tokens.push(this.createReferenceToken(cell1Ref, cell2Ref, sheet1Ref, null, matches[0]));
                                continue;
                            }
                            // otherwise, continue with other REs
                        }

                        // reference error
                        if ((matches = config.RE.ERROR_REF.exec(formula))) {
                            sheet1Ref = makeSheetRef(matches[1], matches[2], matches[3]);
                            tokens.push(this.createReferenceToken(null, null, sheet1Ref, null, matches[0]));
                            continue;
                        }

                        // defined name with sheet reference
                        if ((matches = config.RE.SHEET_NAME_REF.exec(formula))) {
                            sheet1Ref = makeSheetRef(matches[1], matches[2], matches[3]);
                            tokens.push(this.createNameToken(matches[4], sheet1Ref, matches[0]));
                            continue;
                        }

                        // function call (with and without sheet reference)
                        if ((matches = config.RE.FUNCTION_REF.exec(formula))) {
                            sheet1Ref = makeSheetRef(matches[1], matches[2], matches[3]);
                            value = sheet1Ref ? matches[4] : config.getFunctionValue(matches[4]);
                            tokens.push(this.createFunctionToken(value, sheet1Ref, matches[0]));
                            continue;
                        }
                    }

                    // tokens valid exclusively in matrix literals
                    if (matrix) {

                        // column separator in a matrix literal
                        if ((matches = config.RE.MATRIX_COL_SEPARATOR.exec(formula))) {
                            tokens.push(this.createMatrixSeparatorToken(false));
                            continue;
                        }

                        // row separator in a matrix literal
                        if ((matches = config.RE.MATRIX_ROW_SEPARATOR.exec(formula))) {
                            tokens.push(this.createMatrixSeparatorToken(true));
                            continue;
                        }

                        // closing parenthesis of a matrix literal
                        if ((matches = config.RE.MATRIX_CLOSE.exec(formula))) {
                            tokens.push(this.createMatrixParenthesisToken(false));
                            matrix = false;
                            continue;
                        }
                    }

                    // literals are valid in regular formula structure, and in matrix literals

                    // number literal (with leading sign character in matrix literals)
                    if ((matches = (matrix ? config.RE.MATRIX_NUMBER_LIT : config.RE.NUMBER_LIT).exec(formula))) {
                        value = app.convertStringToNumber(matches[0], config.DEC);
                        tokens.push(this.createLiteralToken(value, matches[0]));
                        continue;
                    }

                    // string literal
                    if ((matches = config.RE.STRING_LIT.exec(formula))) {
                        value = matches[0].slice(1, matches[0].length - 1).replace(/""/g, '"');
                        tokens.push(this.createLiteralToken(value, matches[0]));
                        continue;
                    }

                    // Boolean literal (before defined names)
                    if ((matches = config.RE.BOOLEAN_LIT.exec(formula))) {
                        value = matches[0].toUpperCase() === config.TRUE;
                        tokens.push(this.createLiteralToken(value, matches[0]));
                        continue;
                    }

                    // error code literal
                    if ((matches = config.RE.ERROR_LIT.exec(formula))) {
                        value = config.getErrorCodeValue(matches[0]);
                        tokens.push(this.createLiteralToken(value, matches[0]));
                        continue;
                    }

                    // whitespace
                    if ((matches = config.RE.WHITESPACE.exec(formula))) {
                        tokens.push(this.createWhiteSpaceToken(matches[0]));
                        continue;
                    }

                    // defined name without sheet reference (weakest pattern, try last)
                    if (!matrix && (matches = config.RE.GLOBAL_NAME_REF.exec(formula))) {
                        tokens.push(this.createNameToken(matches[1], null, matches[0]));
                        continue;
                    }

                    // invalid characters
                    if ((matches = config.RE.INVALID_CHAR.exec(formula))) {
                        appendBadText(matches[0]);
                        continue;
                    }

                    // error, or other unexpected/unsupported tokens:
                    // push entire remaining formula text as 'bad' token
                    matches = [formula];
                    appendBadText(formula);
                }

                // remove the current match from the formula string
                finally {
                    formula = formula.slice(matches[0].length);
                }
            }
            TokenUtils.logTokens('  tokenize', tokens);

            // post-process the token array
            Utils.iterateArray(tokens, function (token, index) {

                var // the token preceding and following the current token
                    prevToken = tokens[index - 1],
                    nextToken = tokens[index + 1],
                    // value of the next literal token
                    nextValue = null,
                    // the next non-whitespace token preceding the current token
                    prevNonWsToken = (prevToken && prevToken.isType('ws')) ? tokens[index - 2] : prevToken,
                    // text and character index of a space character
                    spaceText = '', spaceIndex = 0;

                // replace combination of 'op[-] lit[number]' with a negative number, if the
                // minus is the leading token or preceded by an operator or opening parenthesis
                if (token.isType('op') && (token.getText() === '-') &&
                    nextToken && nextToken.isType('lit') && _.isNumber(nextValue = nextToken.getValue()) && (nextValue > 0) &&
                    (!prevNonWsToken || prevNonWsToken.isType(/^(op|open|sep)$/))
                ) {
                    token = tokens[index] = this.createLiteralToken(-nextValue, '-' + nextToken.getText());
                    tokens.splice(index + 1, 1);
                }

                // insert a range intersection operator for a white-space token
                // containing a SPACE character and surrounded by reference/name
                // tokens or other subexpressions in parentheses
                if (token.isType('ws') && ((spaceIndex = (spaceText = token.getText()).indexOf(' ')) >= 0) &&
                    prevToken && prevToken.isType(/^(ref|name|close)$/) &&
                    nextToken && nextToken.isType(/^(ref|name|open|func)$/)
                ) {
                    // insert the remaining white-space following the first SPACE character
                    if (spaceIndex + 1 < spaceText.length) {
                        tokens.splice(index + 1, 0, this.createWhiteSpaceToken(spaceText.slice(spaceIndex + 1)));
                    }
                    // insert an intersection operator for the SPACE character after the current token
                    tokens.splice(index + 1, 0, this.createOperatorToken('!', ' '));
                    // shorten the current white-space token to the text preceding the first SPACE character, or remove it
                    if (spaceIndex > 0) {
                        token.setText(spaceText.slice(0, spaceIndex));
                    } else {
                        tokens.splice(index, 1);
                        token = tokens[index];
                    }
                }

            }, { reverse: true, context: this });
            TokenUtils.logTokens('  postprocess', tokens);

            return tokens;
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            app = sheetModel = self = model = config = null;
        });

    } // class Tokenizer

    // exports ================================================================

    // derive this class from class ModelObject
    return ModelObject.extend({ constructor: Tokenizer });

});
