/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/spreadsheet/controller/drawingcontroller',
    ['io.ox/office/tk/utils',
     'io.ox/office/drawinglayer/utils/drawingutils',
     'io.ox/office/spreadsheet/model/operations',
     'io.ox/office/spreadsheet/model/formula/tokenarray',
     'io.ox/office/spreadsheet/view/controls',
     'io.ox/office/spreadsheet/view/chartcreator',
     'gettext!io.ox/office/spreadsheet'
    ], function (Utils, DrawingUtils, Operations, TokenArray, Controls, ChartCreator, gt) {

    'use strict';

    // class DrawingController ================================================

    /**
     * DrawingController (will) encapsulate(s) all Drawing-Features of the Controller
     * For now Chart are part of Drawings!
     */
    function DrawingController(app) {

        var self = this,
            model = null,
            view = null,
            chartTypes = Controls.CHART_TYPES,
            sourceSelector = false,
            chartSourceToken = null,
            // the token arrays containing the highlighted ranges of the selected drawing object
            tokenArrays = [];

        // private methods ----------------------------------------------------

        /**
         * Shows a warning alert box for the 'exchange direction' function for
         * chart series.
         */
        function yellChartDirection(type) {

            switch (type) {
            case 'sheets':
                view.yell({
                    type: 'warning',
                    message:
                        //#. Warning text: A chart object in a spreadsheet contains complex data source (cells from multiple sheets).
                        gt('Data references are located on different sheets.')
                });
                break;

            case 'directions':
                view.yell({
                    type: 'warning',
                    message:
                        //#. Warning text: A chart object in a spreadsheet contains complex data source (not a simple cell range).
                        gt('Data references are too complex for this operation.')
                });
                break;

            default:
                Utils.warn('DrawingController.yellChartDirection(): unknown warning type: "' + type + '"');
            }
        }

        /**
         * changeSelection
         * if only one Drawing is selected and this is a Chart, the function highlights its source-cells via TokenArray
         */
        var changeSelection = (function () {

            return function (evt, selection) {
                if (tokenArrays.length > 0) {
                    view.endRangeHighlighting();
                    tokenArrays = [];
                }

                if (selection.drawings.length === 1) {

                    var chartModel = view.getDrawingCollection().findModel(selection.drawings[0], { type: 'chart' });

                    if (chartModel) {
                        chartModel.getTokenArrays(tokenArrays);
                    }
                }

                if (tokenArrays.length > 0) {
                    view.startRangeHighlighting(tokenArrays, { draggable: false, resolveNames: true });
                }
            };
        }());

        /**
         * @return the ID of the current selected Drawing
         */
        function getDrawingIndices() {
            var selection = view.getSelection();
            if (selection.drawings.length === 1) {
                return selection.drawings[0];
            }
        }

        /**
         * @return the current selected Drawing
         */
        function getDrawingModel(type) {
            var drawingIndices = getDrawingIndices();
            if (drawingIndices) {
                return view.getDrawingCollection().findModel(drawingIndices, { type: type });
            }
        }

        function generateDrawingOperation(opName, params, disableAxes) {
            var generator = model.createOperationsGenerator();
            var position = getDrawingIndices();
            var sheet = view.getActiveSheet();

            if (disableAxes) {
                var noShape = {line: ChartCreator.getNoneShape(), label: false};
                generator.generateDrawingOperation(Operations.SET_CHART_AXIS_ATTRIBUTES,        sheet, position, { attrs: noShape, axis: 'x' });
                generator.generateDrawingOperation(Operations.SET_CHART_AXIS_ATTRIBUTES,        sheet, position, { attrs: noShape, axis: 'y' });
                generator.generateDrawingOperation(Operations.SET_CHART_GRIDLINE_ATTRIBUTES,    sheet, position, { attrs: noShape, axis: 'x' });
                generator.generateDrawingOperation(Operations.SET_CHART_GRIDLINE_ATTRIBUTES,    sheet, position, { attrs: noShape, axis: 'y' });
                generator.generateDrawingOperation(Operations.SET_CHART_TITLE_ATTRIBUTES,       sheet, position, { attrs: { text: { link: []}}, axis: 'x' });
                generator.generateDrawingOperation(Operations.SET_CHART_TITLE_ATTRIBUTES,       sheet, position, { attrs: { text: { link: []}}, axis: 'y' });
            }

            generator.generateDrawingOperation(opName, sheet, position, params);

            model.applyOperations(generator.getOperations());
        }

        /**
         * Returns the title text from the passed attribute set.
         *
         * @param {Object} [attributes]
         *  The merged attribute set for a chart title. If omitted, returns an
         *  empty string.
         *
         * @returns {String}
         *  The title text from the passed attribute set.
         */
        function getChartTitle(attributes) {
            return _.isObject(attributes) ? (attributes.text.link[0] || '') : '';
        }

        /**
         * Changes the attributes of the specified title in the selected chart
         * object.
         *
         * @param {String} titleId
         *  The identifier of the title object. Must be one of 'main', 'x',
         *  'y', or 'z'.
         *
         * @param {Object} attrs
         *  The attribute set to be set for the title.
         */
        function setChartTitleAttributes(titleId, attrs) {
            generateDrawingOperation(Operations.SET_CHART_TITLE_ATTRIBUTES, { axis: titleId, attrs: attrs });
        }

        /**
         * Changes the text of the specified title in the selected chart
         * object.
         *
         * @param {String} titleId
         *  The identifier of the title object. Must be one of 'main', 'x',
         *  'y', or 'z'.
         *
         * @param {String} title
         *  The new text contents of the title. An empty string will remove the
         *  title from the chart.
         */
        function setChartTitle(titleId, title) {
            setChartTitleAttributes(titleId, { text: { link: [title] }, character: ChartCreator.getHeadChar() });
        }

        /**
         * generates an insert-Chart-Op and Insert-Chart-DataSeries chosen by
         * current Selection
         *
         * @returns {jQuery.Deferred}
         *  a deferred object, which is already resolved
         */
        function insertChart(id) {
            return new ChartCreator(app, chartTypes[id]).deferred();
        }

        function setChartSource() {

            function applySelection(selection) {
                var range = selection.ranges[0];
                var sheet = view.getActiveSheet();

                if (!chartSourceToken) {
                    chartSourceToken = new TokenArray(app, view.getSheetModel());
                }

                chartSourceToken.clear().appendRange(range, { sheet: sheet, abs: true });

                var rangeHolder = chartSourceToken.resolveRangeList()[0];
                var chart = getDrawingModel('chart');

                var possSources = chart.getExchangeInfo();
                if (possSources.warn) {
                    ChartCreator.generateOperations(app, rangeHolder, rangeHolder.sheet, chart);
                } else {
                    ChartCreator.generateOperations(app, rangeHolder, rangeHolder.sheet, chart, (possSources.axis - 1) * - 1);
                }
            }

            view.enterCustomSelectionMode(applySelection, { statusLabel: /*#. change source data for a chart object in a spreadsheet */ gt('Select source data') });
        }

        /**
         * changeChartType maps the assigned id to chart-data
         * all ids are mapped in DrawingControls.CHART_TYPES
         *
         * There is a special behavior for bubble-chart, change to bubble or from bubble.
         * all series-data will be removed an initialized complete new by the ChartCreator
         */
        function changeChartType(id) {
            var chart = getDrawingModel('chart');
            var data = chartTypes[id];
            var oldData = chart.getMergedAttributes().chart;
            if (data.type !== oldData.type && (data.type.indexOf('bubble') === 0 || oldData.type.indexOf('bubble') === 0)) {
                //special behavior for bubble charts!!!
                var possSources = chart.getExchangeInfo();
                if (possSources.warn) {
                    yellChartDirection(possSources.warn);
                } else {
                    ChartCreator.generateOperations(app, possSources.range, possSources.sheet, chart, (possSources.axis - 1) * - 1, data);
                }
            } else {
                generateDrawingOperation(Operations.SET_DRAWING_ATTRIBUTES, { attrs: { chart: data} }, (data.type === 'pie2d' || data.type === 'donut2d'));
            }
        }

        function addAxisDef(axisId, defs) {

            var // the base path of axis items
                keyPath = 'drawing/chart/axis/' + axisId + '/';

            function getCorrectAxisId(chartModel) {
                var chartType = chartModel ? chartModel.getMergedAttributes().chart.type : '';
                if (chartType.indexOf('bar') === 0) {
                    if (axisId === 'x') { return 'y'; }
                    if (axisId === 'y') { return 'x'; }
                }
                return axisId;
            }

            function getStandardLineAttributes(visible) {
                return { line: visible ? ChartCreator.getStandardShape() : ChartCreator.getNoneShape() };
            }

            function setAxisAttributes(chartModel, attrs) {
                generateDrawingOperation(Operations.SET_CHART_AXIS_ATTRIBUTES, { axis: getCorrectAxisId(chartModel), attrs: attrs });
            }

            function setGridLineAttributes(chartModel, attrs) {
                generateDrawingOperation(Operations.SET_CHART_GRIDLINE_ATTRIBUTES, { axis: getCorrectAxisId(chartModel), attrs: attrs });
            }

            // *** axis items ***

            // parent item providing access to a chart axis model
            defs[keyPath + 'model'] = {
                parent: ['drawing/chart', 'drawing/chart/axes/enabled'],
                get: function (chartModel) { return chartModel ? chartModel.getAxisModel(getCorrectAxisId(chartModel)) : null; }
            };

            // parent item providing access to the attributes of a chart axis model
            defs[keyPath + 'attributes'] = {
                parent: [keyPath + 'model', 'drawing/chart'],
                get: function (axisModel) { return axisModel ? axisModel.getMergedAttributes() : null; },
                set: function (attributes) { setAxisAttributes(this.getParentValue(1), attributes); }
            };

            // return or modify the visibility of the axis caption labels
            defs[keyPath + 'labels/visible'] = {
                parent: [keyPath + 'attributes', 'drawing/chart'],
                get: function (attributes) { return _.isObject(attributes) && (attributes.axis.label === true) && this.areParentsEnabled(); },
                set: function (visible) { setAxisAttributes(this.getParentValue(1), { axis: { label: visible } }); }
            };

            // return or modify the visibility of the axis line
            defs[keyPath + 'line/visible'] = {
                parent: [keyPath + 'attributes', 'drawing/chart'],
                get: function (attributes) { return _.isObject(attributes) && (attributes.line.type !== 'none'); },
                set: function (visible) { setAxisAttributes(this.getParentValue(1), getStandardLineAttributes(visible)); }
            };

            // *** grid line items ***

            // parent item providing access to a chart axis grid model
            defs[keyPath + 'grid/model'] = {
                parent: keyPath + 'model',
                get: function (axisModel) { return axisModel ? axisModel.getGrid() : null; }
            };

            // parent item providing access to the attributes of a chart axis grid model
            defs[keyPath + 'grid/attributes'] = {
                parent: [keyPath + 'grid/model', 'drawing/chart'],
                get: function (gridModel) { return gridModel ? gridModel.getMergedAttributes() : null; },
                set: function (attributes) { setGridLineAttributes(this.getParentValue(1), attributes); }
            };

            // return or modify the visibility of the axis grid lines
            defs[keyPath + 'grid/visible'] = {
                parent: [keyPath + 'grid/attributes', 'drawing/chart'],
                get: function (attributes) { return _.isObject(attributes) && (attributes.line.type !== 'none'); },
                set: function (visible) { setGridLineAttributes(this.getParentValue(1), getStandardLineAttributes(visible)); }
            };

            // *** title items ***

            // parent item providing access to a chart axis title model
            defs[keyPath + 'title/model'] = {
                parent: keyPath + 'model',
                get: function (axisModel) { return axisModel ? axisModel.getTitle() : null; }
            };

            // parent item providing access to the attributes of a chart axis title model
            defs[keyPath + 'title/attributes'] = {
                parent: [keyPath + 'title/model', 'drawing/chart'],
                get: function (titleModel) { return titleModel ? titleModel.getMergedAttributes() : null; },
                set: function (attributes) { setChartTitleAttributes(getCorrectAxisId(this.getParentValue(1)), attributes); }
            };

            // return or modify the text contents of a chart axis title
            defs[keyPath + 'title/text'] = {
                parent: [keyPath + 'title/attributes', 'drawing/chart'],
                get: function (attributes) { return getChartTitle(attributes); },
                set: function (title) { setChartTitle(getCorrectAxisId(this.getParentValue(1)), title); }
            };

            // remove the main title from a chart (disabled if title does not exist)
            defs[keyPath + 'title/delete'] = {
                parent: [keyPath + 'title/text', 'drawing/chart'],
                enable: function () { return this.getValue().length > 0; },
                set: function () { setChartTitle(getCorrectAxisId(this.getParentValue(1)), ''); }
            };
        }


        /**
         * private registerDefinitions handles all controller definitions for Drawings and Charts.
         * Including the Insert-Drawing button which is physical part of the original Sidepane but is better placed here
         */
        function registerDefinitions() {
            var defs = {};

            defs['drawing/operations'] = {
                parent: ['sheet/unlocked', 'view/cell/editmode/off'] // allow insertion of drawings only if sheet is not locked
            };

            defs['image/insert/dialog'] = {
                parent: 'drawing/operations',
                set: function () { return view.showInsertImageDialog(); }
            };

            defs['chart/insert'] = {
                parent: 'drawing/operations',
                enable: function () {
                    var ranges = view.getSelectedRanges();
                    return (ranges.length === 1) && !view.isSingleCellInRange(ranges[0]);
                },
                set: insertChart
            };

            defs['document/editable/drawing'] = {
                parent: 'sheet/unlocked',
                enable: function () { return view.hasDrawingSelection(); }
            };

            defs['drawing/model'] = {
                parent: 'document/editable/drawing',
                get: getDrawingModel
            };

            defs['drawing/type/label'] = {
                parent: 'drawing/model',
                get: function (drawingModel) { return DrawingUtils.getDrawingTypeLabel(drawingModel ? drawingModel.getType() : ''); }
            };

            defs['drawing/delete'] = {
                parent: 'document/editable/drawing',
                set: function () { view.deleteDrawings(); }
            };

            defs['drawing/chart'] = {
                parent: 'document/editable/drawing',
                enable: function () { return _.isObject(this.getValue()); },
                get: function () { return getDrawingModel('chart'); }
            };

            defs['drawing/chartexchange'] = {
                parent: 'document/editable/drawing',
                set: function () {
                    var chart = getDrawingModel('chart');
                    var possSources = chart.getExchangeInfo();
                    if (possSources.warn) {
                        yellChartDirection(possSources.warn);
                    } else {
                        ChartCreator.generateOperations(app, possSources.range, possSources.sheet, chart, possSources.axis);
                    }
                },
                enable: function () {
                    var ch = getDrawingModel('chart');
                    return _.isObject(ch) && (ch.getSeriesCount() > 0);
                }
            };

            defs['drawing/charttype'] = {
                parent: 'drawing/chart',
                get: function (chartModel) { return chartModel ? chartModel.getChartTypeForGui() : null; },
                set: changeChartType
            };

            defs['drawing/chartvarycolor'] = {
                parent: 'drawing/chart',
                get: function (chartModel) { return (chartModel && chartModel.isVaryColorEnabled()) ? chartModel.getMergedAttributes().chart.varyColors : false; },
                set: function (state) { getDrawingModel('chart').changeVaryColors(state); },
                enable: function () {
                    var chartModel = getDrawingModel('chart');
                    return chartModel && chartModel.isVaryColorEnabled();
                }
            };

            defs['drawing/chartcolorset'] = {
                parent: 'drawing/chart',
                get: function (chartModel) { return chartModel ? chartModel.getColorSet() : null; },
                set: function (colorset) { getDrawingModel('chart').changeColorSet(colorset); }
            };

            defs['drawing/chartstyleset'] = {
                parent: 'drawing/chart',
                get: function (chartModel) { return chartModel ? chartModel.getStyleSet() : null; },
                set: function (colorset) { getDrawingModel('chart').changeStyleSet(colorset); }
            };

            defs['drawing/chartsource'] = {
                parent: 'drawing/chart',
                get: function () { return sourceSelector; },
                set: setChartSource
            };

            defs['drawing/chartlegend/pos'] = {
                parent: 'drawing/chart',
                get: function (chartModel) { return chartModel ? chartModel.getLegendModel().getMergedAttributes().legend.pos : null; },
                set: function (pos) { generateDrawingOperation(Operations.SET_CHART_LEGEND_ATTRIBUTES, {attrs: {legend: {pos: pos}, character: ChartCreator.getStandardChar()}}); }
            };

            defs['drawing/chartdatalabel'] = {
                parent: 'drawing/chart',
                enable: function () { var ch = getDrawingModel('chart'); return _.isObject(ch) && ch.isDataLabelEnabled(); },
                get: function (ch) { return (ch && ch.isDataLabelEnabled()) ? ch.getMergedAttributes().chart.dataLabel : false; },
                set: function (state) {  generateDrawingOperation(Operations.SET_DRAWING_ATTRIBUTES, { attrs: { chart: { dataLabel: state } } }); }
            };

            defs['drawing/chart/axes/enabled'] = {
                parent: 'drawing/chart',
                enable: function (chartModel) { return _.isObject(chartModel) && chartModel.isAxesEnabled(); }
            };

            addAxisDef('x', defs);
            addAxisDef('y', defs);

            // parent item providing access to the main title model of a chart
            defs['drawing/chart/title/model'] = {
                parent: 'drawing/chart',
                get: function (chartModel) { return chartModel ? chartModel.getTitleModel('main') : null; }
            };

            // parent item providing access to the attributes of the main title model of a chart
            defs['drawing/chart/title/attributes'] = {
                parent: 'drawing/chart/title/model',
                get: function (titleModel) { return titleModel ? titleModel.getMergedAttributes() : null; },
                set: function (attributes) { setChartTitleAttributes('main', attributes); }
            };

            // return or modify the text contents of the main title of a chart
            defs['drawing/chart/title/text'] = {
                parent: 'drawing/chart/title/attributes',
                get: function (attributes) { return getChartTitle(attributes); },
                set: function (title) { setChartTitle('main', title); }
            };

            // remove the main title from a chart (disabled if title does not exist)
            defs['drawing/chart/title/delete'] = {
                parent: 'drawing/chart/title/text',
                enable: function () { return this.getValue().length > 0; },
                set: function () { setChartTitle('main', ''); }
            };

            self.registerDefinitions(defs);
        }

        // initialization -----------------------------------------------------

        registerDefinitions();

        //View and Listener initialization
        app.onInit(function () {
            model = app.getModel();
            view = app.getView();

            view.on('change:selection', changeSelection);
        });

    } // class DrawingController

    // exports ================================================================

    return DrawingController;

});
