/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Edy Haryono <edy.haryono@open-xchange.com>
 */
define('io.ox/office/portal/portalutils',
    ['gettext!io.ox/office/portal',
     'io.ox/core/date',
     'io.ox/core/capabilities',
     'io.ox/office/baseframework/app/extensionregistry',
     'io.ox/office/tk/keycodes',
     'io.ox/office/tk/utils'
    ],
function (gt, CoreDate, Capabilities, ExtensionRegistry, KeyCodes, Utils) {

    'use strict';

    // TODO maybe there is a better place for this. Maybe backend office settings?
    var OFFICE_CAPABILITY_NAMES = ['text', 'spreadsheet', 'presentation'];

    // copy tk/utils methods
    var PortalUtils = _.extend({}, Utils);

    /**
     * Detects file type and creates an corresponding icon node.
     *
     * @param fileName
     *  da filename
     *
     * @returns {jQuery}
     *  the icon node as jQuery object
     */
    PortalUtils.createDocumentIcon = function (filename) {

        var moduleName = PortalUtils.getModuleName(filename),
            iconClasses = 'document-icon fa ';

        switch (moduleName) {
        case 'io.ox/office/spreadsheet':
            iconClasses += 'fa-table spreadsheet';
            break;
        case 'io.ox/office/text':
            iconClasses += 'fa-align-left text';
            break;
        case 'io.ox/office/presentation':
            iconClasses += 'fa-picture presentation';
            break;
        default:
            iconClasses += 'fa-file';
            break;
        }

        return $('<i>').addClass(iconClasses);

    };

    /**
     * Formats the timestamp representation for the recents document list:
     * - show only time, if the document is created today
     * - show year only, if the document is not modified in this year.
     *
     * @param {Number} timestamp
     * - milliseconds since Jan 1, 1970
     *
     * @returns {String}
     *  the formatted date as string
     */
    PortalUtils.formatDate = function (timestamp) {

        if (!_.isNumber(timestamp)) { return gt('unknown'); }

        var now = new CoreDate.Local(),
            timeStampDate = new CoreDate.Local(timestamp);

        function getTimeString() {
            return timeStampDate.format(CoreDate.TIME);
        }

        function getDateString() {
            // get locale country default date format
            var formatString = CoreDate.locale.date;
            // hide year if the document is modified in this year
            if (timeStampDate.getYear() === now.getYear()) {
                formatString = CoreDate.DATE_NOYEAR;
            }
            return timeStampDate.format(formatString);
        }

        function isToday(date) {
            return date.getDate() === now.getDate() &&
                date.getMonth() === now.getMonth() &&
                date.getYear() === now.getYear();
        }

        // show only time if the document is last modifed today
        return isToday(timeStampDate) ? getTimeString() : getDateString();

    };

    /**
     * Returns the application module name for the given filename.
     *
     * @param {String} filename
     *  Filename as string
     *
     * @returns {String | Null}
     *  Application module name string if it exists, otherwise null.
     */
    PortalUtils.getModuleName = function (filename) {

        var fileSettings =  ExtensionRegistry.getExtensionSettings(filename);

        if (!fileSettings) { return null; }

        return fileSettings.module ? fileSettings.module : null;
    };

    /**
     * Returns the application base name for the given filename.
     *
     * @param {String} filename
     *  Filename as string
     *
     * @returns {String | Null}
     *  Application base name string if it exists, otherwise null.
     */
    PortalUtils.getAppBaseName = function (filename) {

        var moduleName = PortalUtils.getModuleName(filename);

        if (!moduleName) { return null; }

        return moduleName.substr(moduleName.lastIndexOf('/') + 1, moduleName.length);
    };


    /**
     * Returns the given file name without the file extension.
     *
     * @param {String} filename
     *  Filename as string
     *
     * @returns {String}
     *  Specified file name as string without file extension.
     */
    PortalUtils.removeFileExtension = function (filename) {

        var shortFileName = filename,
            pos = filename.lastIndexOf('.');

        if (pos > -1) {
            shortFileName = shortFileName.substr(0, pos);
        }

        return shortFileName;
    };

    /**
     * Returns whether the given string ends with a given suffix.
     *
     * @param {String} str
     *  A specified string
     *
     * @param {String} suffix
     *  A specified suffix
     *
     * @returns {Boolean}
     *  Whether the given string ends with the specified suffix.
     */
    PortalUtils.stringEndsWith = function (str, suffix) {
        return str && suffix && str.indexOf(suffix, str.length - suffix.length) > -1;
    };

    /**
     * Returns application base names that are enabled in the Appsuite Capabilities.
     *
     * @returns {Array} appBaseNames
     *  An array of application base name strings.
     */
    PortalUtils.getEnabledApps = function () {

        var enabledApps = [];

        // get enabled office apps from Capabilities
        _.each(OFFICE_CAPABILITY_NAMES, function (name) {
            if (Capabilities.has(name)) { enabledApps.push(name); }
        });

        return enabledApps;

    };

    /**
     * Returns the name of the current active application, so that it corresponds to the
     * type of the file.
     *
     * @returns {String} appBaseName
     *  The application base name string.
     */
    PortalUtils.getActiveApp = function (app) {
        return (app.attributes.title).toLowerCase();
    };

    /**
     * Move focus to from given node to its next or previous sibling. (keyboard accessibility)
     *
     * @param {jQuery} startNode
     *  Starting node.
     *
     * @param {String = 'right'} direction
     *  Available direction strings: 'right','left', 'up', 'down'.
     */
    PortalUtils.moveFocus = function (startNode, direction) {

        if (!startNode) { return; }

        var nodeToFocus = startNode;
        direction = direction || 'right';

        /**
         * Tries to find a vertical neighbor according to cursor direction (up/down)
         *
         * @returns {jQuery} || null
         *  the vertical neighbor object or null no vertical neighbours found.
         */
        function findVerticalSibling() {
            var startNodePosition = startNode.position(),
                startNodeHeight = startNode.outerHeight(true),
                targetNodes = startNode.siblings(),
                targetNode = _.find(targetNodes, function (node) {
                    if (!$(node).attr('tabindex')) { return false; }
                    var nodePosition = $(node).position();
                    if (nodePosition.left === startNodePosition.left) {
                        switch (direction) {
                            case 'down':
                                return nodePosition.top === (startNodePosition.top + startNodeHeight);
                            case 'up':
                                return nodePosition.top === (startNodePosition.top - startNodeHeight);
                            default :
                                return false;
                        }
                    } else {
                        return false;
                    }
                });
            if (!targetNode) { return null; }
            return targetNode;
        }

        switch (direction) {
            case 'right':
                nodeToFocus = startNode.next();
                // temp workaround: jump over inserted Bootstrap tooltip element if there is any.
                if (!nodeToFocus.attr('tabindex')) { nodeToFocus = nodeToFocus.next(); }
                break;
            case 'left':
                nodeToFocus = startNode.prev();
                break;
            case 'down':
                nodeToFocus = findVerticalSibling('down');
                break;
            case 'up':
                nodeToFocus = findVerticalSibling('up');
                break;
        }

        if (!nodeToFocus) { return; }

        nodeToFocus.focus();
    };

    /**
     * Keyboard accessibility handler for recent documents and templates list.
     *
     * @param {jQuery.Event} event
     *  {Boolean} event.data.grid - Whether grid navigation (up,down,left,right) is required.
     */
    PortalUtils.keydownHandler = function (event) {

        if (!event || !event.target) { return; }

        var startNode = $(event.target),
            isGrid = event.data ? event.data.grid : false;

        if (KeyCodes.matchKeyCode(event, 'ENTER') || KeyCodes.matchKeyCode(event, 'SPACE')) { startNode.trigger('click'); }
        if (KeyCodes.matchKeyCode(event, 'LEFT_ARROW')) { PortalUtils.moveFocus(startNode, 'left'); }
        if (KeyCodes.matchKeyCode(event, 'RIGHT_ARROW')) { PortalUtils.moveFocus(startNode, 'right'); }
        if (KeyCodes.matchKeyCode(event, 'DOWN_ARROW')) { PortalUtils.moveFocus(startNode, isGrid ? 'down': 'right'); }
        if (KeyCodes.matchKeyCode(event, 'UP_ARROW')) { PortalUtils.moveFocus(startNode, isGrid ? 'up': 'left'); }
    };

    return PortalUtils;

});