/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/utils/canvaswrapper',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/object/baseobject'
    ], function (Utils, BaseObject) {

    'use strict';

    // class ContextMixin =====================================================

    /**
     * This mix-in class adds convenience methods to a 2D rendering context of
     * a canvas element.
     *
     * @constructor
     */
    function ContextMixin() {

        var // self reference
            context = this;

        // public methods -----------------------------------------------------

        /**
         * Adds a straight line to the current path.
         *
         * @param {Number} x1
         *  The X coordinate of the starting point.
         *
         * @param {Number} y1
         *  The Y coordinate of the starting point.
         *
         * @param {Number} x2
         *  The X coordinate of the end point.
         *
         * @param {Number} y2
         *  The Y coordinate of the end point.
         *
         * @param {Number[]} [pattern]
         *  The dash pattern, as pairs of numbers in a flat array. The length
         *  of the array MUST be even and MUST NOT be empty, each number in the
         *  array MUST be non-negative, and the sum of all numbers MUST be
         *  greater than zero. The first entry of each pair in this array
         *  represents the length of a visible line segment, the second entry
         *  of each pair represents the length of the gap to the next line
         *  segment. By passing multiple pairs it is possible to specify more
         *  complex dash patterns like dash-dot-dotted lines etc. If omitted, a
         *  solid line will be added to the path.
         *
         * @param {Number} [offset=0]
         *  The offset length inside the dash pattern for the starting point of
         *  the line. MUST be non-negative.
         *
         * @returns {Number}
         *  The resulting offset inside the dash pattern of the end point of
         *  the rendered line (always 0 for solid lines).
         */
        this.addLinePath = function (x1, y1, x2, y2, pattern, offset) {

            // missing pattern: solid line
            if (!pattern) {
                context.moveTo(x1, y1);
                context.lineTo(x2, y2);
                return 0;
            }

            var // signed width of the entire line box
                boxWidth = x2 - x1,
                // signed height of the entire line box
                boxHeight = y2 - y1,
                // length of a single pattern cycle
                patternLength = _.reduce(pattern, function (sum, elem) { return sum + elem; }, 0),
                // exact pixel length the entire line (including the end point)
                lineLength = Math.sqrt(boxWidth * boxWidth + boxHeight * boxHeight),
                // effective offset for the next rendered pattern
                lineOffset = _.isNumber(offset) ? (offset % lineLength) : 0,
                // the segment offsets relative to the start of the entire pattern
                segments = (function () {
                    var offset = 0, result = [], entry = null;
                    for (var index = 0; index < pattern.length; index += 2) {
                        result.push(entry = { a: offset, b: offset + pattern[index] });
                        offset = entry.b + pattern[index + 1];
                    }
                    return result;
                }());

            // process a single pattern (callbacks need to be defined outside of while loop)
            function generatePattern() {
                _.each(segments, function (segment) {

                    var // relative start offset of the line segment along the rendered line
                        segOffset = Math.max(0, lineOffset + segment.a) / lineLength,
                        // relative end offset of the line segment along the rendered line
                        segEndOffset = Math.min(lineLength, lineOffset + segment.b) / lineLength;

                    // render the line segment if it is inside the entire line box
                    if (segOffset <= segEndOffset) {
                        context.moveTo(x1 + boxWidth * segOffset, y1 + boxHeight * segOffset);
                        context.lineTo(x1 + boxWidth * segEndOffset, y1 + boxHeight * segEndOffset);
                    }
                });
            }

            // draw patterns until the end of the line is reached
            while (lineOffset < lineLength) {
                generatePattern();
                lineOffset += patternLength;
            }

            // return end offset inside the last pattern
            return lineOffset - lineLength;
        };


        /**
         * Returns the width of the passed string.
         *
         * @param {String} text
         *  The text whose width will be calculated.
         *
         * @returns {Number}
         *  The width of the passed string, as floating-point pixels.
         */
        this.getTextWidth = function (text) {
            return this.measureText(text).width;
        };

        /**
         * Returns the width of the passed single character. To increase
         * precision for different browsers (some browsers return a rounded
         * integral width for single characters), the character will be
         * repeated ten times, and a tenth of the width of the resulting
         * string will be returned.
         *
         * @param {String} char
         *  A single character.
         *
         * @returns {Number}
         *  The width of the passed character, as floating-point pixels.
         */
        this.getCharacterWidth = function (char) {
            return this.measureText(Utils.repeatString(char, 10)).width / 10;
        };

    } // class ContextMixin

    // class CanvasWrapper ====================================================

    /**
     * A wrapper class for a canvas DOM element, that provides additional
     * convenience methods to render into the canvas.
     *
     * @constructor
     *
     * @extends BaseObject
     */
    function CanvasWrapper() {

        var // the current canvas DOM element, as jQuery object
            canvasNode = $('<canvas width="1" height="1">'),

            // the rendering context of the canvas
            context = canvasNode[0].getContext('2d'),

            // the current rectangle represented by the canvas element
            contextRect = { left: 0, top: 0, width: 1, height: 1 };

        // base constructor ---------------------------------------------------

        BaseObject.call(this);

        // public methods -----------------------------------------------------

        /**
         * Returns the wrapped canvas element.
         *
         * @returns {jQuery}
         *  The wrapped canvas DOM element, as jQuery object.
         */
        this.getNode = function () {
            return canvasNode;
        };

        /**
         * Returns the extended rendering context of the canvas element. Do not
         * use this method to actually render into the canvas (see method
         * CanvasWrapper.render() for a save way to render content into the
         * canvas).
         *
         * @returns {RenderingContext2D}
         *  The rendering context of the canvas element, which has been
         *  extended with the mix-in class ContextMixin.
         */
        this.getContext = function () {
            return context;
        };

        /**
         * Changes the offset and size of the canvas element, and clears all
         * its contents.
         *
         * @param {Object} rectangle
         *  The new rectangle represented by the canvas DOM element, with the
         *  properties 'left', 'top', 'width', and 'height' in pixels. A
         *  translation according to the top-left position of the rectangle
         *  will be added to the canvas. Using the top-left position of the
         *  rectangle will effectively draw to position (0, 0) inside the
         *  canvas.
         *
         * @returns {CanvasWrapper}
         *  A reference to this instance.
         */
        this.initialize = function (rectangle) {

            // store new canvas rectangle
            contextRect = rectangle;

            // set new bitmap size and element size
            canvasNode.attr({ width: contextRect.width, height: contextRect.height }).css({ width: contextRect.width, height: contextRect.height });

            // initialize the global translation according to the rectangle
            context.setTransform(1, 0, 0, 1, -contextRect.left, -contextRect.top);

            // clear all contents (canvas may not clear itself, if its size does not change)
            return this.clear();
        };

        /**
         * Clears the entire area covered by the canvas element.
         *
         * @returns {CanvasWrapper}
         *  A reference to this instance.
         */
        this.clear = function () {
            context.clearRect(contextRect.left, contextRect.top, contextRect.width, contextRect.height);
            return this;
        };

        /**
         * Saves the current state of the canvas (colors, transformation,
         * clipping, etc.), sets up a rectangular clipping region, invokes the
         * passed callback function, and restores the canvas state afterwards.
         *
         * @param {Object} rectangle
         *  The clipping rectangle, with the numeric properties 'left', 'top',
         *  'width', and 'height'.
         *
         * @param {Function} callback
         *  The callback function. The symbol 'this' inside the callback
         *  function will be bound to this wrapper instance.
         *
         * @returns {CanvasWrapper}
         *  A reference to this instance.
         */
        this.clip = function (rectangle, callback) {
            context.save();
            context.beginPath();
            context.rect(rectangle.left, rectangle.top, rectangle.width, rectangle.height);
            context.clip();
            callback.call(this);
            context.restore();
            return this;
        };

        /**
         * Saves the current state of the canvas (colors, transformation,
         * clipping, etc.), invokes the passed callback function, and restores
         * the canvas state afterwards.
         *
         * @param {Function} callback
         *  The callback function. The rendering context of the canvas element
         *  will be passed as first parameter. The rendering context has been
         *  extended with methods defined by the mix-in class ContextMixin. The
         *  symbol 'this' inside the callback function will be bound to this
         *  wrapper instance.
         *
         * @returns {CanvasWrapper}
         *  A reference to this instance.
         */
        this.render = function (callback) {
            context.save();
            callback.call(this, context);
            context.restore();
            return this;
        };

        /**
         * Returns a data URL with the current contents of the canvas.
         *
         * @returns {String}
         *  A data URL with the current contents of the canvas.
         */
        this.getDataURL = function () {
            return canvasNode[0].toDataURL();
        };

        // initialization -----------------------------------------------------

        // extend the context with own methods
        ContextMixin.call(context);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            canvasNode.remove();
            canvasNode = context = null;
        });

    } // class CanvasWrapper

    // exports ================================================================

    // derive this class from class BaseObject
    return BaseObject.extend({ constructor: CanvasWrapper });

});
