/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/model/operationsgenerator', function () {

    'use strict';

    // class OperationsGenerator ==============================================

    /**
     * An instance of this class contains an array of operations and provides
     * methods to generate new operations.
     *
     * @constructor
     *
     * @param {DocumentStyles} documentStyles
     *  Global collection with the style sheet containers and custom formatting
     *  containers of a document.
     */
    function OperationsGenerator(documentStyles) {

        var // the operations buffer
            operations = [];

        // methods ------------------------------------------------------------

        /**
         * Returns the array with all operations that have been generated so
         * far. Does not remove the operations from this generator instance.
         *
         * @returns {Array}
         *  The array with all generated operations.
         */
        this.getOperations = function () {
            return operations;
        };

        /**
         * Reverses the entire operations array.
         *
         * @returns {OperationsGenerator}
         *  A reference to this instance.
         */
        this.reverseOperations = function () {
            operations.reverse();
            return this;
        };

        /**
         * Creates and appends a new operation to the operations array.
         *
         * @param {String} name
         *  The name of the operation.
         *
         * @param {Object} [operationOptions]
         *  Additional options that will be stored in the operation.
         *
         * @returns {Object}
         *  The created operation object.
         */
        this.generateOperation = function (name, operationOptions) {
            var operation = _.extend({ name: name }, operationOptions);
            operations.push(operation);
            return operation;
        };

        /**
         * Generates an 'insertStyleSheet' operation, if the specified style
         * sheet exists in the style sheet collection, and is marked dirty.
         *
         * @param {String} styleFamily
         *  The style family that specifies which style sheet collection will
         *  be checked for a dirty style sheet.
         *
         * @param {String} styleId
         *  The identifier of the style sheet.
         *
         * @returns {OperationsGenerator}
         *  A reference to this instance.
         */
        this.generateMissingStyleSheetOperation = function (styleFamily, styleId) {

            var // the style sheet container
                styleSheets = documentStyles.getStyleCollection(styleFamily),
                // additional properties for the operation
                operationOptions = null;

            // do not generate an operation for the default style
            if (styleSheets && (styleId !== styleSheets.getDefaultStyleId()) && styleSheets.isDirty(styleId)) {

                // create the properties for the 'insertStyleSheet' operation
                operationOptions = {
                    attrs: styleSheets.getStyleSheetAttributeMap(styleId),
                    type: styleFamily,
                    styleId: styleId,
                    styleName: styleSheets.getName(styleId),
                    parent: styleSheets.getParentId(styleId),
                    uiPriority: styleSheets.getUIPriority(styleId)
                };
                if (!_.isString(operationOptions.parent)) { delete operationOptions.parent; }
                if (!_.isNumber(operationOptions.uiPriority)) { delete operationOptions.uiPriority; }

                // generate and store the 'insertStyleSheet' operation
                this.generateOperation(OperationsGenerator.INSERT_STYLESHEET, operationOptions);

                // remove the dirty flag from the style sheet
                styleSheets.setDirty(styleId, false);
            }

            return this;
        };

    } // class OperationsGenerator

    // constants --------------------------------------------------------------

    // define names of generic operations
    _.extend(OperationsGenerator, {

        /**
         * Name of the initial operation while importing a document, sets
         * global attributes of the model, including default formatting
         * attributes.
         */
        SET_DOCUMENT_ATTRIBUTES: 'setDocumentAttributes',

        /**
         * Name of the operation that inserts a new theme into the theme
         * collection.
         */
        INSERT_THEME: 'insertTheme',

        /**
         * Name of the operation that inserts a new font description into the
         * font collection.
         */
        INSERT_FONT_DESCRIPTION: 'insertFontDescription',

        /**
         * Name of the operation that inserts a new style sheet into the style
         * sheet collection of a specific style family.
         */
        INSERT_STYLESHEET: 'insertStyleSheet',

        /**
         * Name of the operation that removes a style sheet from the style
         * sheet collection of a specific style family.
         */
        DELETE_STYLESHEET: 'deleteStyleSheet',

        /**
         * Name of the operation that does nothing (internal use).
         */
        NOOP: 'noOp'

    });

    // exports ================================================================

    return _.makeExtendable(OperationsGenerator);

});
