/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/model/editmodel',
    ['io.ox/office/tk/utils',
     'io.ox/office/baseframework/model/basemodel',
     'io.ox/office/editframework/model/operationsgenerator'
    ], function (Utils, BaseModel, OperationsGenerator) {

    'use strict';

    // class EditModel ========================================================

    /**
     * The base class for editable document models. Adds generic support for
     * operations and undo action management.
     *
     * Triggers the following events:
     * - 'operations:before'
     *      Before an array of operations will be applied via the method
     *      EditModel.applyOperations(). The event handler receives the
     *      operations array, and the external state passed to the method. If a
     *      single operation has been passed to that method, it will be
     *      converted to an array with one element before calling the event
     *      handler.
     * - 'operations:after'
     *      After an array of operations have been applied via the method
     *      EditModel.applyOperations(), regardless if successfully or not. The
     *      event handler receives the external state passed to the method
     *      EditModel.applyOperations().
     * - 'operations:success'
     *      Directly after the event 'operations:after', if the operations have
     *      been applied successfully. The event handler receives the
     *      operations array, and the external state passed to the method
     *      EditModel.applyOperations(). If a single operation has been passed
     *      to that method, it will be converted to an array with one element
     *      before calling the event handler.
     * - 'operations:error'
     *      Directly after the event 'operations:after', if applying the
     *      operations has failed. The event handler receives the external
     *      state passed to the method EditModel.applyOperations().
     * - 'change:editmode'
     *      When the edit mode of the document model has been changed. Event
     *      handlers receive the new state of the edit mode.
     *
     * @constructor
     *
     * @extends BaseModel
     *
     * @param {EditApplication} app
     *  The application containing this document model instance.
     *
     * @param {Function} OperationsGeneratorClass
     *  The constructor function of the operations generator. Must be a sub
     *  class of class OperationsGenerator.
     *
     * @param {UndoManager} undoManager
     *  The undo manager used by the document model to store undo and redo
     *  actions.
     *
     * @param {DocumentStyles} documentStyles
     *  Global collection with the style sheet containers and custom formatting
     *  containers of a document.
     */
    function EditModel(app, OperationsGeneratorClass, undoManager, documentStyles) {

        var // self reference for local functions
            self = this,

            // maps all operation names to operation handler functions
            operationHandlers = {},

            // Deferred object for current asynchronous operations, will remain
            // in resolved state as long as no operations are applied
            operationsDef = $.when(),

            // whether operations are currently applied (prevent recursive calls)
            processingOperations = false,

            // edit mode, received dynamically from server (one editing user at a time)
            editMode = null,

            // the total number of operations applied successfully
            totalOperations = 0,

            // a counter for the operations, that adds all single operations, taking care of merged
            // operations. So this number must be equal on every client, independent from merge
            // processes on client or server side.
            operationStateNumber = -1,

            // collecting the last two operations in cache
            operationsCache = { previous: null, current: null };

        // base constructors --------------------------------------------------

        BaseModel.call(this, app);

        // private methods ----------------------------------------------------

        /**
         * Extracts and returns the operation filter callback function from the
         * passed options.
         */
        function getFilterCallback(options) {
            return Utils.getFunctionOption(options, 'filter', function () { return true; });
        }

        /**
         * Called before an operations array will be applied at the document
         * model. Triggers an 'operations:before' event.
         *
         * @param {Array} operations
         *  The operations to be applied. Will be passed to the listeners of
         *  the 'operations:before' event.
         *
         * @param {Boolean} external
         *  Whether the operations have been received from the server. Will be
         *  passed to the listeners of the 'operations:before' event.
         *
         * @returns {Boolean}
         *  False if this method has been called recursively, i.e. the model
         *  already processes operations. This is considered a fatal error.
         */
        function beginProcessOperations(operations, external) {

            if (processingOperations) {
                Utils.error('EditModel.beginProcessOperations(): recursive call - currently processing operations.');
                return false;
            }

            self.trigger('operations:before', operations, external);
            processingOperations = true;
            return true;
        }

        /**
         * Called after an operations array has been applied at the document
         * model. Triggers an 'operations:after' event, and one of the
         * 'operations:success' or 'operations:error' events.
         *
         * @param {Boolean} success
         *  Whether applying the operations has succeeded.
         *
         * @param {Array} operations
         *  The operations that have been applied. Will be passed to the
         *  listeners of the operations events.
         *
         * @param {Boolean} external
         *  Whether the operations have been received from the server. Will be
         *  passed to the listeners of the operations events.
         */
        function endProcessOperations(success, operations, external) {
            processingOperations = false;
            self.trigger('operations:after', operations, external);
            if (success) {
                self.trigger('operations:success', operations, external);
            } else {
                self.setEditMode(false);
                self.trigger('operations:error', operations, external);
            }
        }

        /**
         * Handles a number for the operation state to find problems
         * in the communication between different clients earlier. This
         * is currently only used in debug mode.
         *
         * @param {Object} operation
         *  A single operation to be applied at the document model.
         *
         * @param {Boolean} external
         *  Whether the operation has been received from the server.
         */
        function handleOperationStateNumber(operation, external) {

            // checking the correct operation state number of an external operation
            // The operation state number must be identical on every client.
            // The operation state number is sent from server, when the document is loaded
            if (external) {

                // During loading the document, the initial operation state number
                // is sent by the server. If we don't have a valid value use it.
                if ((operation.osn >= 0) && (operationStateNumber === -1)) {
                    operationStateNumber = operation.osn;
                }

                // check for the correct operation state number of an external operation
                if (_.isNumber(operation.osn) && (operationStateNumber !== operation.osn)) {
                    // If this client is already synchronized -> there is a conflict in the operation state number
                    Utils.error('EditModel.handleOperationStateNumber(): Wrong document state. Operation expected state ' + operation.osn + '. But local state is: ' + operationStateNumber);
                    return false;
                }
            } else {
                // Preparing internal operations, so that remote clients can control the correct operation state number.
                // A. setting current operation state number as required base for this operation
                operation.osn = operationStateNumber;
                // B. setting length of this operation -> this might be increased by operation merge processes
                operation.opl = 1;
            }

            if (operationStateNumber >= 0) {
                // increasing the operation state number of the document
                operationStateNumber = operationStateNumber + (operation.opl || 1);
            }

            return true;
        }

        /**
         * Executes the handler function for the passed operation.
         *
         * @param {Object} operation
         *  A single operation to be applied at the document model.
         *
         * @param {Boolean} external
         *  Whether the operation has been received from the server.
         *
         * @param {Function} filter
         *  A predicate function that specifies whether to invoke the operation
         *  callback handler.
         *
         * @returns {Boolean}
         *  Whether the operation has been applied successfully.
         */
        function applyOperation(operation, external, filter) {

            var // the operation scope
                scope = Utils.getStringOption(operation, 'scope', 'all'),
                // the function that executes the operation
                operationHandler = null,
                // success value of operation
                success = false,
                // a possible log message
                message = null;

            // check operation
            if (!_.isObject(operation) || !_.isString(operation.name)) {
                Utils.error('EditModel.applyOperation(): expecting operation object.');
                return false;
            }

            // handle an operation state number. This number needs to be synchronized
            // between all clients and has then to be identical for all clients.
            if (!handleOperationStateNumber(operation, external)) {
                return false;
            }

            // get and check operation handler
            operationHandler = operationHandlers[operation.name];
            if (!_.isFunction(operationHandler)) {
                message = 'EditModel.applyOperation(): invalid operation name "' + operation.name + '".';
                Utils.error(message);
                app.sendLogMessage(message);
                return false;
            }

            // Caching the last two operations
            operationsCache.previous = operationsCache.current;
            operationsCache.current = operation;

            // execute the operation handler (unless the operation scope is
            // 'filter' for operations intended to be executed in the filter
            // component only, or the filter callback function returns false)
            try {
                success = (scope === 'filter') || (filter.call(self, operation) !== true) || (operationHandler.call(self, operation, external) !== false);
            } catch (ex) {
                Utils.exception(ex);
            }

            if (success) {
                totalOperations += 1;
            } else {
                message = 'EditModel.applyOperation(): ' + (totalOperations + 1) + (external ? ' E ' : ' I ') + JSON.stringify(operation) + ' failed';
                Utils.error(message);
                app.sendLogMessage(message);
            }
            return success;
        }

        /**
         * Executes the handler functions for all passed operations. If the
         * undo manager is active, all applied operations will be embedded into
         * a single undo group action.
         *
         * @param {Array} operations
         *  The operations to be applied at the document model.
         *
         * @param {Boolean} external
         *  Whether the operation has been received from the server.
         *
         * @param {Function} filter
         *  A predicate function that specifies whether to invoke the operation
         *  callback handler.
         *
         * @returns {Boolean}
         *  Whether applying all operations was successful.
         */
        function applyOperationsSync(operations, external, filter) {

            var // the boolean result for synchronous mode
                success = false;

            // if the operation's origin is not an external client and
            // we don't have edit rights then we skip the operations silently.
            if ((!external && !editMode) || (operations.length === 0)) {
                return true;
            }

            // enter operations mode (prevent recursive calls)
            if (beginProcessOperations(operations, external)) {
                // apply all operations at once, group into an undo action
                undoManager.enterUndoGroup(function () {
                    success = _.all(operations, function (operation) {
                        return applyOperation(operation, external, filter);
                    });
                });
            }

            // post-processing (trigger end events, leave operations mode),
            // also if beginProcessOperations() has failed (notify the error)
            endProcessOperations(success, operations, external);
            return success;
        }

        /**
         * Executes the handler functions for all passed operations in an
         * asynchronous background loop. If the undo manager is active, all
         * applied operations will be embedded into a single undo group action.
         *
         * @param {Array} operations
         *  The operations to be applied at the document model.
         *
         * @param {Boolean} external
         *  Whether the operation has been received from the server.
         *
         * @param {Function} filter
         *  A predicate function that specifies whether to invoke the operation
         *  callback handler.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved after all
         *  operations have been applied successfully, or rejected immediately
         *  after applying an operation has failed. The Promise regularly sends
         *  progress updates while applying the operations.
         */
        function applyOperationsAsync(operations, external, filter) {

            var // the resulting Promise
                promise = null,
                // the global index of the first applied operation
                startIndex = totalOperations;

            // immediate success on empty operations array
            if (operations.length === 0) { return $.when(); }

            // enter operations mode (prevent recursive calls)
            if (beginProcessOperations(operations, external)) {

                // apply all operations in a background task, group into an undo action
                promise = undoManager.enterUndoGroup(function () {

                    // return the Deferred to enterUndoGroup() to defer the open undo group
                    return app.iterateArraySliced(operations, function (operation) {

                        // if the operation's origin is not an external client and
                        // we don't have edit rights then we skip the operations silently
                        if (!external && !editMode) {
                            promise.abort();
                            return Utils.BREAK;
                        }

                        // break the loop and reject the promise if an operation has failed
                        if (!applyOperation(operation, external, filter)) {
                            return $.Deferred().reject();
                        }
                    });
                });

            } else {
                // currently applying operations (called recursively)
                promise = $.Deferred().reject();
            }

            // post-processing (trigger end events, leave operations mode)
            return promise.done(function () {
                // all passed operations applied successfully
                endProcessOperations(true, operations, external);
            }).fail(function (result) {
                if (result === 'abort') {
                    // manually aborted: notify applied operations only
                    endProcessOperations(true, operations.slice(0, totalOperations - startIndex), external);
                } else {
                    // operations failed: notify all operations
                    endProcessOperations(false, operations, external);
                }
            });
        }

        // methods ------------------------------------------------------------

        /**
         * Registers a function that will be executed when an operation with
         * the specified name will be applied.
         *
         * @param {String} name
         *  The name of the operation, as contained in the 'name' attribute of
         *  the operation object.
         *
         * @param {Function} handler
         *  The function that will be executed for the operation. All operation
         *  handler functions will be called in the context of this document
         *  model instance. Receives the operation object as first parameter,
         *  and the external state that has been passed to the method
         *  EditModel.applyOperations() as second parameter. If the operation
         *  handler returns the boolean value false, further processing of
         *  operations will be stopped immediately, and an 'operations:error'
         *  event will be triggered. Every other return value (also undefined)
         *  indicates that the operation has been applied successfully.
         *
         * @returns {EditModel}
         *  A reference to this instance.
         */
        this.registerOperationHandler = function (name, handler) {
            operationHandlers[name] = handler;
            return this;
        };

        /**
         * Creates and returns an initialized operations generator instance (an
         * instance of the class specified in the constructor parameter
         * OperationsGeneratorClass).
         *
         * @returns {OperationsGenerator}
         *  A new operations generator instance.
         */
        this.createOperationsGenerator = function () {
            return new OperationsGeneratorClass(documentStyles);
        };

        /**
         * Sends the passed operations to the server without applying them
         * locally. In special situations, a client may choose to apply the
         * operations manually before they will be sent to the server. In this
         * case, the registered operation handlers must not be called.
         *
         * @param {Object|Array} operations
         *  A single operation, or an array with operations to be sent to the
         *  server.
         *
         * @returns {Boolean}
         *  Whether registering all operations for sending was successful.
         */
        this.sendOperations = function (operations) {
            return applyOperationsSync(_.getArray(operations), false, $.noop);
        };

        /**
         * Executes the handler functions for all passed operations. If the
         * undo manager is active, all applied operations will be embedded into
         * a single undo group action.
         *
         * @param {Object|Array} operations
         *  A single operation, or an array with operations to be applied at
         *  the document model.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Function} [options.filter]
         *      A predicate function that will be invoked for each operation
         *      contained in the passed actions. Receives the operation as
         *      first parameter, and must return a Boolean specifying whether
         *      to invoke the operation handler.
         *
         * @returns {Boolean}
         *  Whether applying all operations was successful.
         */
        this.applyOperations = function (operations, options) {

            var // whether the operations have been applied successfully
                success = false;

            // create an new Deferred object which remains in pending state as
            // long as the operations will be applied
            operationsDef = $.Deferred();

            // apply the operations synchronously
            success = applyOperationsSync(_.getArray(operations), false, getFilterCallback(options));

            // resolve or reject the Deferred object, this will invoke all
            // callbacks that have been registered inside the operation handlers
            operationsDef[success ? 'resolve' : 'reject']();

            return success;
        };

        /**
         * Executes the handler functions for all operations contained in the
         * passed actions. The operations of each action will be enclosed into
         * distinct undo groups automatically.
         *
         * @param {Object|Array} actions
         *  A single action, or an array with actions containing operations to
         *  be applied at the document model.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.external=false]
         *      If set to true, the actions have been received from an external
         *      sender. Otherwise, the actions have been generated by the
         *      document model itself.
         *  @param {Boolean} [options.async=false]
         *      If set to true, the actions will be applied asynchronously in a
         *      browser timeout loop. The Promise returned by this method will
         *      be resolved or rejected after applying all actions, and will be
         *      notified about the progress of the operation.
         *  @param {Function} [options.filter]
         *      A predicate function that will be invoked for each operation
         *      contained in the passed actions. Receives the operation as
         *      first parameter, and must return a Boolean specifying whether
         *      to invoke the operation handler.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved after all
         *  actions have been applied successfully, or rejected immediately
         *  after applying an action has failed. The caller has to wait for the
         *  Promise before new actions can be applied. The Promise regularly
         *  sends progress updates while applying the actions. In synchronous
         *  mode (option 'async' not set), the returned promise is already
         *  resolved or rejected when this method finishes. In asynchronous
         *  mode, the Promise will contain an additional method 'abort()' that
         *  can be used to stop applying the operations at any time.
         */
        this.applyActions = function (actions, options) {

            var // the timer processing the array asynchronously
                timer = null,
                // whether the actions have been received from the server
                external = Utils.getBooleanOption(options, 'external', false),
                // whether to execute asynchronously
                async = Utils.getBooleanOption(options, 'async', false),
                // invocation filter callback
                filter = getFilterCallback(options),
                // total number of operations
                operationCount = 0,
                // number of operations already applied
                operationIndex = 0,
                // the promise of the inner loop processing operations of an action
                innerPromise = null,
                // resulting promise returned by this method
                resultPromise = null,
                // whether the operations have been applied successfully in synchronous mode
                success = false;

            // convert parameter to an array, filter for actions containing
            // at least one operation, get total number of operations
            actions = _.chain(actions).getArray().filter(function (action) {
                var count = _.isArray(action.operations) ? action.operations.length : 0;
                operationCount += count;
                return count > 0;
            }).value();

            // no operations: early exit
            if (operationCount === 0) { return $.when(); }

            // create an new Deferred object which remains in pending state as
            // long as the operations will be applied
            operationsDef = $.Deferred();

            // apply actions, get resulting promise
            if (async) {

                // apply the actions in a browser timeout loop
                timer = app.iterateArraySliced(actions, function (action) {

                    // apply the operations and calculate the progress
                    innerPromise = applyOperationsAsync(action.operations, external, filter)
                        .progress(function (progress) {
                            operationsDef.notify((operationIndex + action.operations.length * progress) / operationCount);
                        })
                        .done(function () {
                            operationIndex += action.operations.length;
                        });

                    // clean up after finishing the loop
                    return innerPromise.always(function () { innerPromise = null; });
                });

                // forward the result of the timer to the own Deferred object
                timer.done(function () { operationsDef.notify(1).resolve(); }).fail(_.bind(operationsDef.reject, operationsDef));

                // create an abortable promise from the Deferred object
                resultPromise = app.createAbortablePromise(operationsDef, function () {
                    // abort the outer loop processing the actions
                    timer.abort();
                    // abort the inner loop processing the operations of an action
                    if (innerPromise) { innerPromise.abort(); }
                });

            } else {

                // synchronous mode: apply all actions at once, but bail out on error
                success = _.all(actions, function (action) {
                    // applyOperationsSync() creates undo groups by itself
                    return applyOperationsSync(action.operations, external, filter);
                });

                // resolve or reject the Deferred object, this will invoke all
                // callbacks that have been registered inside the operation handlers
                resultPromise = operationsDef[success ? 'resolve' : 'reject']().promise();
            }

            return resultPromise;
        };

        /**
         * Returns the promise of a Deferred object representing the state of
         * operations currently applied. If in pending state, this model
         * currently applies operations (synchronously or asynchronously). If
         * in resolved state, the last operations have been applied
         * successfully. If in rejected state, the last operations have not
         * been applied successfully.
         *
         * @attention
         *  The promise MUST NOT be cached externally. Its solely purpose is to
         *  register done/fail callbacks from operation handlers or other code
         *  depending on operations. Whenever new operations will be applied, a
         *  new promise will be created representing the new operations (the
         *  old promises will remain in resolved/rejected state forever).
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object representing the currently or last
         *  applied operations.
         */
        this.getOperationsPromise = function () {
            return operationsDef.promise();
        };

        /**
         * Returns whether operations are currently processed.
         *
         * @returns {Boolean}
         *  Whether the model processes operations.
         */
        this.isProcessingOperations = function () {
            return processingOperations;
        };

        /**
         * Returns whether the document can be modified by the current user.
         *
         * @returns {Boolean}
         *  Whether the document can be modified.
         */
        this.getEditMode = function () {
            return editMode;
        };

        /**
         * Enables or disables the edit mode of this document model.
         *
         * @param {Boolean} state
         *  The new edit mode state. If set to true, the user can modify the
         *  document, otherwise the user can view the document contents, but
         *  cannot edit them.
         *
         * @returns {EditModel}
         *  A reference to this instance.
         */
        this.setEditMode = function (state) {
            if (editMode !== state) {
                editMode = state;
                // notify state change to listeners
                this.trigger('change:editmode', editMode);
            }
            return this;
        };

        /**
         * Returns the undo manager instance of this document containing the
         * undo action stack and the redo action stack.
         *
         * @returns {UndoManager}
         *  The undo manager instance.
         */
        this.getUndoManager = function () {
            return undoManager;
        };

        /**
         * Returns the global collection with the style sheet containers and
         * custom formatting containers of a document.
         *
         * @returns {DocumentStyles}
         *  The global document styles collection of the document.
         */
        this.getDocumentStyles = function () {
            return documentStyles;
        };

        /**
         * Returns the style collection for the passed style family.
         *
         * @param {String} family
         *  The name of the attribute family.
         *
         * @returns {StyleSheets}
         *  The style collection registered for the passed style family.
         */
        this.getStyleCollection = function (styleFamily) {
            return documentStyles.getStyleCollection(styleFamily);
        };

        /**
         * Returns the collection with all fonts registered for this document.
         *
         * @returns {FontCollection}
         *  The collection with all fonts registered for this document.
         */
        this.getFontCollection = function () {
            return documentStyles.getFontCollection();
        };

        /**
         * Returns the collection with all themes registered for this document.
         *
         * @returns {ThemeCollection}
         *  The collection with all themes registered for this document.
         */
        this.getThemeCollection = function () {
            return documentStyles.getThemeCollection();
        };

        /**
         * Returns the total number of operations applied successfully.
         */
        this.getOperationsCount = function () {
            return totalOperations;
        };

        /**
         * Returns the operation state number of the document.
         */
        this.getOperationStateNumber = function () {
            return operationStateNumber;
        };

        /**
         * Sets the operation state number of the document. Used when empty doc is fast displayed
         * @param {Number} osn - new osn number
         */
        this.setOperationStateNumber = function (osn) {
            operationStateNumber = osn;
            this.trigger('change:osn', osn); // update osn value in operations pane
        };

        /**
         * Returns the operation cache with the last two operations.
         *
         * @returns {Object}
         *  An object containing the last two operations in the properties
         *  last and current.
         */
        this.getOperationsCache = function () {
            return operationsCache;
        };

        /**
         * Returns the previous operation saved in the operation cache.
         * This is required for performance reasons. So no copy is returned
         * but the original operation object.
         *
         * @returns {Object}
         *  An object containing the previous operation.
         */
        this.getPreviousOperation = function () {
            return operationsCache.previous;
        };

        // operation handlers -------------------------------------------------

        // setting default attributes is not undoable
        this.registerOperationHandler(OperationsGenerator.SET_DOCUMENT_ATTRIBUTES, function (operation) {

            var // the attribute set from the passed operation
                attributes = _.clone(Utils.getObjectOption(operation, 'attrs', {}));

            // set the global document attributes
            if (_.isObject(attributes.document)) {
                documentStyles.setDocumentAttributes(attributes.document);
                delete attributes.document;
            }

            // set default values for other attributes
            documentStyles.setDefaultAttributes(attributes);
        });

        this.registerOperationHandler(OperationsGenerator.INSERT_THEME, function (operation) {
            this.getThemeCollection().insertTheme(operation.themeName, operation.attrs);
        });

        this.registerOperationHandler(OperationsGenerator.INSERT_FONT_DESCRIPTION, function (operation) {
            this.getFontCollection().insertFont(operation.fontName, operation.attrs);
        });

        this.registerOperationHandler(OperationsGenerator.INSERT_STYLESHEET, function (operation) {

            var // target style sheet container
                styleSheets = documentStyles.getStyleCollection(operation.type),
                // passed attributes from operation
                attributes = Utils.getObjectOption(operation, 'attrs', {});

            if (!_.isObject(styleSheets)) {
                Utils.warn('EditModel.insertStyleSheet(): invalid style family: "' + operation.type + '"');
                return false;
            }

            // TODO: undo for insertion, needs implementation of DELETE_STYLESHEET
//            undoManager.addUndo({ name: Operations.DELETE_STYLESHEET, type: operation.type, styleId: operation.styleId }, operation);

            return styleSheets.insertStyleSheet(operation.styleId, operation.styleName, operation.parent, attributes, {
                hidden: operation.hidden,
                priority: operation.uiPriority,
                defStyle: operation['default']
            });
        });

        this.registerOperationHandler(OperationsGenerator.DELETE_STYLESHEET, function (operation) {

            var // target style sheet container
                styleSheets = documentStyles.getStyleCollection(operation.type);

            if (!_.isObject(styleSheets)) {
                Utils.warn('EditModel.deleteStyleSheet(): invalid style family: "' + operation.type + '"');
                return false;
            }

            // TODO: undo for deletion
//            undoManager.addUndo({ name: Operations.INSERT_STYLESHEET, type: operation.type, styleId: operation.styleId, attrs: ..., ... }, operation);

            return styleSheets.deleteStyleSheet(operation.styleId);
        });

        this.registerOperationHandler(OperationsGenerator.NOOP, $.noop);

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            undoManager.destroy();
            documentStyles.destroy();
            undoManager = documentStyles = null;
        });

    } // class EditModel

    // exports ================================================================

    // derive this class from class BaseModel
    return BaseModel.extend({ constructor: EditModel });

});
