/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/drawinglayer/view/drawingcontrols',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/forms',
     'io.ox/office/tk/control/menumixin',
     'io.ox/office/tk/control/captionmixin',
     'io.ox/office/tk/control/widthmixin',
     'io.ox/office/baseframework/view/basecontrols',
     'io.ox/office/drawinglayer/utils/drawingutils',
     'io.ox/office/drawinglayer/view/charttypepicker',
     'io.ox/office/drawinglayer/view/chartstyleutil',
     'gettext!io.ox/office/drawing',
     'less!io.ox/office/drawinglayer/view/drawingstyle'
    ], function (Utils, Forms, MenuMixin, CaptionMixin, WidthMixin, BaseControls, DrawingUtils, ChartTypePicker, ChartStyleUtil, gt) {

    'use strict';

    var // class name shortcuts
        CheckBox = BaseControls.CheckBox,
        TextField = BaseControls.TextField,
        RadioList = BaseControls.RadioList,

        // number of color boxes per list entry in the ColorSetPicker control
        COLOR_BOX_COUNT = 3,

        // all predefined color sets
        CHART_COLOR_SETS = ChartStyleUtil.getColorSet(),

        // menu titles for axis menus
        CHART_AXIS_HEADER_OPTIONS_LABELS = {
            x: /*#. menu title: options for the X axis (categories) in a chart object */ gt.pgettext('menu-title', 'X axis options'),
            y: /*#. menu title: options for the Y axis (values) in a chart object */ gt.pgettext('menu-title', 'Y axis options'),
            z: /*#. menu title: options for the Y axis (deep axis in 3D charts) in a chart object */ gt.pgettext('menu-title', 'Z axis options')
        },

        // menu titles for axis menus
        CHART_AXIS_HEADER_TITLE_LABELS = {
            x: /*#. menu title: functions for the X axis title (categories) in a chart object */ gt.pgettext('menu-title', 'X axis title'),
            y: /*#. menu title: functions for the Y axis title (values) in a chart object */ gt.pgettext('menu-title', 'Y axis title'),
            z: /*#. menu title: functions for the Y axis title (deep axis in 3D charts) in a chart object */ gt.pgettext('menu-title', 'Z axis title')
        },

        // options for text fields for chart titles
        CHART_TITLE_TEXTFIELD_OPTIONS = {
            minWidth: 120,
            placeholder: /*#. default placeholder for editing the main title in chart objects */ gt.pgettext('chart-format', 'Type text to add a title'),
            select: true
        };

    // static class DrawingControls ===========================================

    /**
     * Additional classes defining special GUI controls for drawing objects.
     */
    var DrawingControls = {};

    // constants for header labels --------------------------------------------

    /**
     * @constant
     */
    DrawingControls.DRAWING_HEADER_LABEL =
        //#. menu title: functions for a single drawing object (image, chart, etc.)
        gt.pgettext('menu-title', 'Drawing');

    /**
     * @constant
     */
    DrawingControls.DRAWINGS_HEADER_LABEL =
        //#. menu title: functions for drawing objects (images, charts, etc.)
        gt.pgettext('menu-title', 'Drawings');

    /**
     * @constant
     */
    DrawingControls.CHART_HEADER_LABEL =
        //#. menu title: functions for a chart object (change type and source data of the chart object)
        gt.pgettext('menu-title', 'Chart');

    /**
     * @constant
     */
    DrawingControls.CHART_LEGEND_HEADER_LABEL =
        //#. menu title: settings for the legend in chart objects
        gt.pgettext('menu-title', 'Legend');

    // constants for controls -------------------------------------------------

    /**
     * Standard options for the 'Delete drawing' button.
     *
     * @constant
     */
    DrawingControls.DELETE_DRAWING_BUTTON_OPTIONS = {
        icon: 'fa-trash-o',
        // label: /*#. delete a drawing object from the document */ gt.pgettext('drawing', 'Delete'),
        tooltip: gt.pgettext('drawing', 'Delete the drawing object')
    };

    /**
     * Standard options for the 'Insert image' button.
     *
     * @constant
     */
    DrawingControls.INSERT_IMAGE_BUTTON_OPTIONS = {
        icon: 'docs-image-insert',
        label: DrawingUtils.getDrawingTypeLabel('image'),
        tooltip: gt.pgettext('drawing', 'Insert an image'),
        smallerVersion: {
            css: { width: 35 },
            hideLabel: true
        }
    };

    /**
     * Standard options for the 'Insert chart' button.
     *
     * @constant
     */
    DrawingControls.INSERT_CHART_BUTTON_OPTIONS = {
        width: null,
        icon: DrawingUtils.getDrawingTypeIcon('chart'),
        label: DrawingUtils.getDrawingTypeLabel('chart'),
        tooltip: gt.pgettext('drawing', 'Insert a chart'),
        updateCaptionMode: 'none'
    };

    /**
     * @constant
     */
    DrawingControls.CHART_DATA_POINTS_BUTTON_OPTIONS = {
        label: //#. menu title: options to format data points (bars, lines, pie segments) in chart objects
            gt.pgettext('chart-format', 'Data points')
    };

    /**
     * @constant
     */
    DrawingControls.CHART_SHOW_POINT_LABELS_BUTTON_OPTIONS = {
        label: //#. check box label: show/hide small text labels next to the single data points in charts objects
            gt.pgettext('chart-format', 'Show data point labels')
    };

    /**
     * @constant
     */
    DrawingControls.CHART_VARY_POINT_COLORS_BUTTON_OPTIONS = {
        label: //#. check box label: if active, a bar/pie chart will have different colors for each data point (default is the same color for all points)
            gt.pgettext('chart-format', 'Vary colors by point')
    };

    // class ChartTypePicker ==================================================

    DrawingControls.ChartTypePicker = ChartTypePicker;

    // class ChartColorSetPicker ==============================================

    /**
     * A drop-down menu for picking different predefined chart color sets.
     *
     * @constructor
     *
     * @extends RadioList
     */
    DrawingControls.ChartColorSetPicker = RadioList.extend({ constructor: function (app, initOptions) {

        var self = this;

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({
            label: /*#. Menu title: predefined color sets for chart objects */ gt.pgettext('chart-format', 'Color set'),
            tooltip: /*#. Menu tooltip: predefined color sets for chart objects */ gt.pgettext('chart-format', 'Predefined color set'),
            smallerVersion: {
                label: gt.pgettext('chart-format', 'Color')
            }
        }, initOptions, {
            updateCaptionMode: 'none'
        }));

        // private methods ----------------------------------------------------

        /**
         * Inserts the special mark-up for a color set into the specified list
         * entry of the drop-down menu.
         */
        function createItemHandler(event, buttonNode, value, options) {

            var // the style collection of the document
                documentStyles = app.getModel().getDocumentStyles(),
                // the color set data this list item is based on
                colorSet = options.colorSet,
                // the content node for the color boxes
                markup = null;

            // create all color boxes
            markup = '<div class="color-node">';
            _.times(COLOR_BOX_COUNT, function (index) {

                var // extract the base color from the color set
                    color = _.copy(colorSet.colors[index % colorSet.colors.length], true),
                    // extract the variation from the color set
                    variation = (colorSet.variations.length > 0) ? colorSet.variations[index % colorSet.variations.length] : null;

                if (variation) { color.transformations = variation.transformations; }
                markup += '<span class="color-node" style="background-color:' + documentStyles.getCssColor(color, 'fill') + ';"></span>';
            });
            markup += '</div>';

            // insert the color boxes into the caption of the button node
            Forms.getCaptionNode(buttonNode).prepend(markup);
        }

        /**
         * Fills the drop-down menu with all available color sets.
         */
        var initializeColorTable = app.createDebouncedMethod($.noop, function () {
            self.clearMenu();
            _.each(CHART_COLOR_SETS, function (colorSet, index) {
                self.createOptionButton('cs' + index, { tooltip: colorSet.name, colorSet: colorSet });
            });
        }, { delay: 500 });

        // initialization -----------------------------------------------------

        this.getMenuNode().addClass('color-set-picker');
        this.getMenu().on('create:item', createItemHandler);

        // insert color buttons in the drop-down menu after import and changed themes
        this.listenTo(app.getImportPromise(), 'done', function () {
            initializeColorTable();
            self.listenTo(app.getModel().getThemeCollection(), 'triggered', initializeColorTable);
        });

    }}); // class ChartColorSetPicker

    // class ChartStyleSetPicker ==============================================

    /**
     * A drop-down menu for picking different predefined chart style sets.
     *
     * @constructor
     *
     * @extends RadioList
     */
    DrawingControls.ChartStyleSetPicker = RadioList.extend({ constructor: function (app, initOptions) {

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({
            label: /*#. menu title: predefined style sets for chart objects (e.g. "Light", "Dark", "Colored") */ gt.pgettext('chart-format', 'Style set'),
            tooltip: /*#. menu tooltip: predefined style sets for chart objects (e.g. "Light", "Dark", "Colored") */ gt.pgettext('chart-format', 'Predefined style set'),
            smallerVersion: {
                label: gt.pgettext('chart-format', 'Style')
            }
        }, initOptions, {
            updateCaptionMode: 'none'
        }));

        // initialization -----------------------------------------------------

        this.createOptionButton('ss0', { label: /*#. predefined chart style set: one light color (different shades) */ gt.pgettext('chart-format', 'Light') })
            .createOptionButton('ss4', { label: /*#. predefined chart style set: different colors from color scheme */ gt.pgettext('chart-format', 'Colored') })
            .createOptionButton('ss5', { label: /*#. predefined chart style set: one dark color (different shades) */ gt.pgettext('chart-format', 'Dark') });

    }}); // class ChartStyleSetPicker

    // class ChartLayerMenuButton =============================================

    /**
     * A layer-menu with options for an axis in a chart object.
     *
     * @constructor
     *
     * @extends Group
     * @extends CaptionMixin
     * @extends WidthMixin
     * @extends MenuMixin
     */
    DrawingControls.ChartLayerMenuButton = BaseControls.LayerMenuButton.extend({ constructor: function (app, layerTitle, initOptions) {

        var // the model and view instance
            model = app.getModel(),
            view = app.getView(),

            // the drawing model this menu is currently shown for
            drawingModel = null,

            // the pop-up menu instance
            menu = null;

        // base constructor ---------------------------------------------------

        BaseControls.LayerMenuButton.call(this, app, layerTitle, Utils.extendOptions(initOptions, {closer: true, rootContainerNode: app.getWindowNode()}));

        // private methods ----------------------------------------------------

        /**
         * Add menu-entries for the axis of the chart
         *
         * @param {String} axisId
         *  The axis identifier. Must be one of 'x', 'y', or 'z'.
         */
        function addAxisMenuEntries(axisId) {

            var // the base path of axis items
                keyPath = 'drawing/chart/axis/' + axisId + '/';

            menu.addHeaderLabel(CHART_AXIS_HEADER_TITLE_LABELS[axisId])
                .addGroup(keyPath + 'title/text', new TextField(CHART_TITLE_TEXTFIELD_OPTIONS))
                .addHeaderLabel(CHART_AXIS_HEADER_OPTIONS_LABELS[axisId])
                .addGroup(keyPath + 'labels/visible', new CheckBox({
                    label: //#. (check box) show/hide the caption labels (e.g. the scaling numbers) on an axis in chart objects
                        gt.pgettext('chart-format', 'Labels'),
                    boxed: true
                }))
                .addGroup(keyPath + 'line/visible', new CheckBox({
                    label: //#. (check box) show/hide the axis line in chart objects
                        gt.pgettext('chart-format', 'Axis line'),
                    boxed: true
                }))
                .addGroup(keyPath + 'grid/visible', new CheckBox({
                    label: //#. (check box) show/hide the grid lines for a specific direction in chart objects
                        gt.pgettext('chart-format', 'Grid lines'),
                    boxed: true
                }));
        }

        /**
         * Updates the position of the menu, after the view selection has
         * changed. If no chart object is selected, the menu will be hidden.
         */
        function changeSelectionHandler() {

            // so nothing in read-only mode
            if (!model.getEditMode()) { return; }

            var // all selected drawing models
                drawingModels = _.isFunction(view.getSelectedDrawingModels) ? view.getSelectedDrawingModels() : [];

            // hide menu if selection is anything but a single chart object (TODO: support multiple charts?)
            if ((drawingModels.length !== 1) || (drawingModels[0].getType() !== 'chart')) {
                menu.hide();
                drawingModel = null;
                return;
            }

            // do nothing if the drawing model did not change
            if (drawingModel === drawingModels[0]) { return; }
            drawingModel = drawingModels[0];

            // open menu automatically
            // if (!Utils.SMALL_DEVICE) { menu.show(); }

            // update position of the menu
            if (menu.isVisible()) {
                menu.positionResetHandler();
                menu.setLayerMenuPosition();
            }
        }

        // initialization -----------------------------------------------------

        menu = this.getMenu();

        // add base menu items
        menu.addHeaderLabel(gt.pgettext('menu-title', 'Chart title'), { separator: false })
            .addGroup('drawing/chart/title/text', new TextField(CHART_TITLE_TEXTFIELD_OPTIONS));

        // add menu items for X and Y axis
        addAxisMenuEntries('x');
        addAxisMenuEntries('y');

        // adjust position and visibility of the menu according to the current selection
        this.listenTo(view, 'change:selection', changeSelectionHandler);

        // hide the menu automatically in read-only mode
        this.listenTo(model, 'change:editmode', function (event, editMode) {
            if (!editMode) { menu.hide(); }
        });

        // destroy all class members
        this.registerDestructor(function () {
            app = initOptions = model = view = drawingModel = menu = null;
        });

    }}); // class ChartLayerMenuButton

    // class ChartLegendPicker ================================================

    /**
     * A drop-down menu for picking the position of the chart legend.
     *
     * @constructor
     *
     * @extends RadioList
     */
    DrawingControls.ChartLegendPicker = RadioList.extend({ constructor: function () {

        // base constructor ---------------------------------------------------

        RadioList.call(this, {
            label: /*#. menu title: positions for the legend in a chart object */ gt.pgettext('chart-legend', 'Legend position'),
            tooltip: /*#. menu tooltip: positions for the legend in a chart object */ gt.pgettext('chart-legend', 'Visibility and position of the chart legend'),
            updateCaptionMode: 'none'
        });

        // initialization -----------------------------------------------------

        this.createOptionButton('off',      { section: 'off', label: /*#. chart legend is hidden */ gt.pgettext('chart-legend', 'Off') })
            .createOptionButton('top',      { section: 'pos', label: /*#. chart legend at top of border of the chart */ gt.pgettext('chart-legend', 'Top') })
            .createOptionButton('bottom',   { section: 'pos', label: /*#. chart legend at bottom border of the chart */ gt.pgettext('chart-legend', 'Bottom') })
            .createOptionButton('left',     { section: 'pos', label: /*#. chart legend at left border of the chart */ gt.pgettext('chart-legend', 'Left') })
            .createOptionButton('right',    { section: 'pos', label: /*#. chart legend at right border of the chart */ gt.pgettext('chart-legend', 'Right') })
            .createOptionButton('topRight', { section: 'pos', label: /*#. chart legend at top-right corner of the chart */ gt.pgettext('chart-legend', 'Top right') });

    }}); // class ChartLegendPicker

    // constants ==============================================================

    DrawingControls.CHART_TYPES = ChartTypePicker.CHART_TYPES;

    // exports ================================================================

    return DrawingControls;

});
