/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/baseframework/model/modelobject',
    ['io.ox/core/event',
     'io.ox/office/tk/utils',
     'io.ox/office/tk/object/baseobject'
    ], function (Events, Utils, BaseObject) {

    'use strict';

    // class ModelObject ======================================================

    /**
     * An abstract model object that allows to trigger change events to event
     * listeners. As long as the application still imports the document, no
     * events will be triggered to the event listeners.
     *
     * @constructor
     *
     * @extends BaseObject
     * @extends Events
     *
     * @param {EditApplication} app
     *  The application instance containing this attributed model object.
     *
     * @param {Object} [initOptions]
     *  Optional parameters:
     *  @param {Boolean} [initOptions.silent=false]
     *      If set to true, this instance will never trigger any change events.
     *      Specific sub classes may chose to be permanently silent although
     *      their base class derives from this class.
     *  @param {Boolean} [initOptions.forceTrigger=false]
     *      If set to true, this instance will trigger even during loading the
     *      document. Specific sub classes may chose to trigger all events although
     *      their base class derives from this class.
     *      This option requires, that the option 'silent' is set to  false (its
     *      default value).
     */
    function ModelObject(app, initOptions) {

        // base constructor ---------------------------------------------------

        BaseObject.call(this);

        // methods ------------------------------------------------------------

        /**
         * Executes the specified callback function. As long as the callback is
         * running, this model object will be silent, i.e. it will not notify
         * any listeners when its method ModelObject.trigger() will be called.
         *
         * @param {Function} callback
         *  The callback function to be invoked in silent mode.
         *
         * @param {Object} [context]
         *  The context to be bound to the symbol 'this' inside the callback
         *  function.
         *
         * @returns {Any}
         *  The result of the callback function.
         */
        this.executeSilently = function (callback, context) {

            var // the original trigger() method
                triggerMethod = this.trigger;

            try {
                // replace the trigger() method with a no-op version
                this.trigger = function () { return this; };
                // invoke the callback function
                return callback.call(context);
            } finally {
                // restore the trigger() method afterwards
                this.trigger = triggerMethod;
            }
        };

        /**
         * Invokes the specified method of this instance. As long as the method
         * is running, this model object will be silent, i.e. it will not
         * notify any listeners when its method ModelObject.trigger() will be
         * called.
         *
         * @param {String} methodName
         *  The name of the method of this instance to be invoked.
         *
         * @param {Any} [...]
         *  All additional parameters will be passed to the method.
         *
         * @returns {Any}
         *  The result of the invoked method.
         */
        this.invokeSilently = function (methodName) {
            var args = _.toArray(arguments).slice(1);
            return this.executeSilently(function () {
                return this[methodName].apply(this, args);
            }, this);
        };

        // initialization -----------------------------------------------------

        if (Utils.getBooleanOption(initOptions, 'silent', false)) {

            // add dummy event methods for silent objects
            this.on = this.one = this.off = this.trigger = function () { return this; };

        } else {

            // extend with Events mix-in
            Events.extend(this);

            // disable triggering until import is finished. If 'forceTrigger'
            // is set, triggering happens also in the loading phase.
            if (!Utils.getBooleanOption(initOptions, 'forceTrigger', false)) {

                // overwrite the trigger() method with a method that triggers
                // events only if the document has been imported already
                this.trigger = (function (triggerFunc) {
                    return function () {
                        if (app.isImportFinished()) {
                            this.trigger = triggerFunc; // back to original method
                            triggerFunc.apply(this, arguments);
                        }
                        return this;
                    };
                }.call(this, this.trigger));

            }

            // destroy the Events mix-in on destruction of this instance
            this.registerDestructor(_.bind(this.events.destroy, this.events));
        }

    } // class ModelObject

    // exports ================================================================

    // derive this class from class BaseObject
    return BaseObject.extend({ constructor: ModelObject });

});
