/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/baseframework/app/baseapplication',
    ['io.ox/core/extensions',
     'io.ox/core/notifications',
     'io.ox/files/api',
     'io.ox/office/tk/utils',
     'io.ox/office/tk/keycodes',
     'io.ox/office/tk/config',
     'io.ox/office/tk/io',
     'io.ox/office/tk/forms',
     'io.ox/office/tk/object/baseobject',
     'io.ox/office/baseframework/app/extensionregistry',
     'settings!io.ox/office',
     'gettext!io.ox/office/main'
    ], function (ext, Notifications, FilesAPI, Utils, KeyCodes, Config, IO, Forms, BaseObject, ExtensionRegistry, Settings, gt) {

    'use strict';

    // private static functions ===============================================

    /**
     * Returns the document type from the passed application module name.
     * The document type equals the last directory name of the module name.
     *
     * @param {String} moduleName
     *  The application module name.
     *
     * @returns {String}
     *  The document type of the passed module name.
     */
    function getDocumentType(moduleName) {
        return moduleName.substr(moduleName.lastIndexOf('/') + 1);
    }

    // class BaseApplication ==================================================

    /**
     * A mix-in class that defines common public methods for an application
     * that is based on a document file.
     *
     * Triggers the events supported by the base class ox.ui.App, and the
     * following additional events:
     * - 'docs:destroy': Before the model/view/controller instances of the
     *      application will be destroyed, after all registered before-quit
     *      handlers have been called, and none was rejected, after all
     *      registered quit handlers have been called.
     *
     * @constructor
     *
     * @extends ox.ui.App
     *
     * @param {Function} ModelClass
     *  The constructor function of the document model class. MUST derive from
     *  the class Model. Receives a reference to this application instance.
     *  MUST NOT use the methods BaseApplication.getModel(),
     *  BaseApplication.getView(), or BaseApplication.getController() during
     *  construction. For further initialization depending on valid
     *  model/view/controller instances, the constructor can use the method
     *  BaseApplication.onInit().
     *
     * @param {Function} ViewClass
     *  The constructor function of the view class. MUST derive from the class
     *  View. Receives a reference to this application instance. MUST NOT use
     *  the methods BaseApplication.getModel(), BaseApplication.getView(), or
     *  BaseApplication.getController() during construction. For further
     *  initialization depending on valid model/view/controller instances, the
     *  constructor can use the method BaseApplication.onInit().
     *
     * @param {Function} ControllerClass
     *  The constructor function of the controller class. MUST derive from the
     *  class Controller. Receives a reference to this application instance.
     *  MUST NOT use the methods BaseApplication.getModel(),
     *  BaseApplication.getView(), or BaseApplication.getController() during
     *  construction. For further initialization depending on valid
     *  model/view/controller instances, the constructor can use the method
     *  BaseApplication.onInit().
     *
     * @param {Function} importHandler
     *  A function that will be called to import the document described in the
     *  file descriptor of this application. Will be called once after
     *  launching the application. Will be called in the context of this
     *  application. Must return a Deferred object that will be resolved or
     *  rejected after the document has been loaded.
     *
     * @param {Object} launchOptions
     *  All options passed to the core launcher (the ox.launch() method) that
     *  determine the actions to perform during application launch. The
     *  supported options are dependent on the actual application type. The
     *  following standard launch options are supported:
     *  @param {String} launchOptions.action
     *      Controls how to connect the application to a document file. If the
     *      launch options map is passed at all, this is the only mandatory
     *      launch option.
     *  @param {Object} [launchOptions.file]
     *      The descriptor of the file to be imported. May be missing for
     *      specific actions, e.g. when creating a new file.
     *
     * @param {Object} [appOptions]
     *  Static application options that have been passed to the static method
     *  BaseApplication.createLauncher().
     *
     * @param {Object} [initOptions]
     *  Optional parameters:
     *  @param {Function} [initOptions.flushHandler]
     *      A function that will be called before the document will be accessed
     *      at its source location, e.g. before downloading it, printing it, or
     *      sending it as e-mail. Will be called in the context of this
     *      application. Receives a string identifier as first parameter,
     *      specifying the cause of the flush request:
     *      - 'download': The document will be prepared for downloading or
     *          printing.
     *      - 'email': The document will be prepared to be sent as e-mail.
     *      May return a Deferred object. In this case, the application will
     *      wait until this Deferred object has been resolved or rejected. In
     *      the latter case, all subsequent actions will be skipped.
     */
    function BaseApplication(ModelClass, ViewClass, ControllerClass, importHandler, launchOptions, appOptions, initOptions) {

        var // self reference
            self = this,

            // the document model instance
            model = null,

            // application view: contains panes, tool bars, etc.
            view = null,

            // the controller instance as single connection point between model and view
            controller = null,

            // all singleton values/objects for this application instance, mapped by key
            singletons = {},

            // file descriptor of the document edited by this application
            file = Utils.getObjectOption(launchOptions, 'file', null),

            // callback to flush the document before accessing its source
            flushHandler = Utils.getFunctionOption(initOptions, 'flushHandler', $.noop),

            // all registered before-quit handlers
            beforeQuitHandlers = [],

            // all registered quit handlers
            quitHandlers = [],

            // a Deferred object representing the initialization state (pending: early application construction, resolved: regular application runtime)
            initDef = $.Deferred(),

            // a Deferred object representing the import state (pending: importing, resolved/rejected: import finished with state)
            importDef = $.Deferred(),

            // Deferred object of the current call to Application.quit()
            currentQuitDef = null,

            // all (abortable) AJAX requests and browser timers currently running
            pendingPromises = [],

            // time stamp when the application launcher has started
            launchStartTime = 0,

            // time stamp when the application launcher has finished
            launchEndTime = 0;

        // private methods ----------------------------------------------------

        /**
         * Updates the application title according to the current file name. If
         * the application does not contain a file descriptor, shows the
         * localized word 'Unnamed' as title.
         */
        function updateTitle() {

            var // the root node of the application window
                windowNode = self.getWindow().nodes.outer,
                // the title for the application launcher and browser window
                title = self.getShortFileName() || gt('unnamed');

            // set title for application launcher and browser tab/window
            self.setTitle(title);
            self.getWindow().setTitle(title);

            // add data from file descriptor to application root node, for Selenium testing
            if (file) {
                if (file.folder_id) { windowNode.attr('data-file-folder', file.folder_id); }
                if (file.id) { windowNode.attr('data-file-id', file.id); }
                if (file.filename) { windowNode.attr('data-file-name', file.filename); }
                if (file.version) { windowNode.attr('data-file-version', file.version); }
                if (file.source) { windowNode.attr('data-file-source', file.source); }
                if (file.attachment) { windowNode.attr('data-file-attachment', file.attachment); }
            }
        }

        /**
         * Updates the browser URL according to the current file descriptor.
         */
        function updateUrl() {
            // only for files from OX Drive
            if (file && !('source' in file) && self.isActive()) {
                self.setState({ folder: file.folder_id, id: file.id });
            }
        }

        /**
         * Calls all handler functions contained in the passed array, and
         * returns a Deferred object that accumulates the result of all
         * handlers.
         */
        function callHandlers(handlers) {

            var // execute all handlers and store their results in an array
                results = _.map(handlers, function (handler) { return handler.call(self); });

            // accumulate all results into a single Deferred object
            return $.when.apply($, results);
        }

        /**
         * Handler for the 'unload' window events triggered when reloading or
         * closing the entire page. Executes the quit handlers, but not the
         * before-quit handlers.
         */
        function unloadHandler() {
            self.executeQuitHandlers();
        }

        // public methods -----------------------------------------------------

        /**
         * Returns the document type supported by this application. The
         * document type equals the directory name of the application.
         *
         * @returns {String}
         *  The document type of this application.
         */
        this.getDocumentType = function () {
            return getDocumentType(this.getName());
        };

        /**
         * Returns the document model instance of this application.
         *
         * @returns {DocumentModel}
         *  The document model instance of this application.
         */
        this.getModel = function () {
            return model;
        };

        /**
         * Returns the view instance of this application.
         *
         * @returns {View}
         *  The view instance of this application.
         */
        this.getView = function () {
            return view;
        };

        /**
         * Returns the controller instance of this application.
         *
         * @returns {Controller}
         *  The controller instance of this application.
         */
        this.getController = function () {
            return controller;
        };

        /**
         * Invokes the passed callback function as soon as this application has
         * finished its early construction phase. During early construction,
         * the model, view, and controller instances may not be available yet.
         * If this method is used after early construction, the passed callback
         * function will be invoked immediately.
         *
         * Should be used in all class initialization code relying on existing
         * model, view, and controller instances.
         *
         * @param {Function} callback
         *  The callback function invoked after the application has finished
         *  its early initialization.
         *
         * @param {Object} [context]
         *  The context bound to the invoked callback function.
         *
         * @returns {BaseApplication}
         *  A reference to this instance.
         */
        this.onInit = function (callback, context) {
            initDef.done(_.bind(callback, context));
            return this;
        };

        /**
         * Returns the global user settings for all applications of the same
         * type.
         *
         * @param {String} key
         *  The unique key of the user setting.
         *
         * @param defValue
         *  The default value in case the user setting does not exist yet.
         *
         * @returns
         *  The value of the global user setting.
         */
        this.getUserSettingsValue = function (key, defValue) {
            // for debugging purpose we output the whole settings object
            //Utils.log(JSON.stringify(Settings.get()));
            return Settings.get(this.getDocumentType() + '/' + key, defValue);
        };

        /**
         * Changes a global user setting for all applications of the same type.
         *
         * @param {String} key
         *  The unique key of the user setting.
         *
         * @param value
         *  The new value of the user setting.
         *
         * @returns {BaseApplication}
         *  A reference to this application instance.
         */
        this.setUserSettingsValue = function (key, value) {
            Settings.set(this.getDocumentType() + '/' + key, value).save();
            return this;
        };

        /**
         * Returns a lazy-initialized application-wide value or object. If the
         * singleton has not been created yet, calls the passed initialization
         * callback function and caches its return value internally.
         *
         * @param {String} key
         *  The unique key of the singleton.
         *
         * @param {Function} initHandler
         *  The initialization callback function that will be called when the
         *  singleton will be queried the first time for this application. Will
         *  be called in the context of this application instance. Receives the
         *  key of the singleton as first parameter (so that the same callback
         *  function can be used to initialize different singletons).
         *
         * @returns {Any}
         *  The singleton value.
         */
        this.getOrCreateSingleton = function (key, initHandler) {
            return (key in singletons) ? singletons[key] : (singletons[key] = initHandler.call(this, key));
        };

        /**
         * Returns whether this application is currently active in OX AppSuite.
         *
         * @returns {Boolean}
         *  Whether this application is currently active in OX AppSuite
         */
        this.isActive = function () {
            return !!this.getWindow().state.visible;
        };

        /**
         * Returns the unique DOM identifier of the outer root node of this
         * application.
         *
         * @returns {String}
         *  The unique window DOM identifier.
         */
        this.getWindowId = function () {
            return this.getWindow().nodes.outer.attr('id');
        };


        /**
         * Returns the absolute start time of the launcher.
         *
         * @returns {Number}
         *  The launcher start time.
         */
        this.getLaunchStartTime = function () {
            return launchStartTime;
        };

        /**
         * Returns the absolute ending time of the launcher.
         *
         * @returns {Number}
         *  The launcher end time.
         */
        this.getLaunchEndTime = function () {
            return launchEndTime;
        };

        // file descriptor ----------------------------------------------------

        /**
         * Returns whether this application contains a valid file descriptor.
         */
        this.hasFileDescriptor = function () {
            return _.isObject(file);
        };

        /**
         * Returns a clone of the file descriptor of the document edited by
         * this application.
         *
         * @returns {Object}
         *  A clone of the current file descriptor.
         */
        this.getFileDescriptor = function () {
            return _.clone(file);
        };

        /**
         * Registers the file descriptor of the new document created by this
         * application. Must not be called if the application already contains
         * a valid file descriptor.
         *
         * @param {Object} newFile
         *  The file descriptor of the new file created by the caller of this
         *  method.
         *
         * @returns {BaseApplication}
         *  A reference to this instance.
         */
        this.registerFileDescriptor = function (newFile) {
            if (_.isObject(file)) {
                Utils.error('BaseApplication.registerFileDescriptor(): file descriptor exists already');
            } else {
                file = newFile;
                FilesAPI.propagate('new', file);
                updateTitle();
                updateUrl();
            }
            return this;
        };

        /**
         * Updates the current file descriptor of the document edited by this
         * application.
         *
         * @param {Object} fileOptions
         *  All file descriptor properties to be updated.
         *
         * @returns {BaseApplication}
         *  A reference to this instance.
         */
        this.updateFileDescriptor = function (fileOptions) {
            if (_.isObject(file)) {
                _.extend(file, fileOptions);
                FilesAPI.propagate('change', file);
                updateTitle();
                updateUrl();
            }
            return this;
        };

        /**
         * Returns an object with attributes describing the file currently
         * opened by this application.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.encodeUrl=false]
         *      If set to true, special characters not allowed in URLs will be
         *      encoded.
         *
         * @returns {Object|Null}
         *  An object with file attributes, if existing; otherwise null.
         */
        this.getFileParameters = function (options) {

            var // function to encode a string to be URI conforming if specified
                encodeString = Utils.getBooleanOption(options, 'encodeUrl', false) ? encodeURIComponent : _.identity,
                // the resulting file parameters
                parameters = null;

            if (file) {
                parameters = {};
                // add the parameters to the result object, if they exist in the file descriptor
                _.each(['id', 'folder_id', 'filename', 'version', 'source', 'attached', 'module'], function (name) {
                    if (_.isString(file[name])) {
                        parameters[name] = encodeString(file[name]);
                    } else if (_.isNumber(file[name])) {
                        parameters[name] = file[name];
                    }
                });
            }

            return parameters;
        };

        /**
         * Returns the full file name of the current file, with file extension.
         *
         * @returns {String|Null}
         *  The file name of the current file descriptor; or null, if no file
         *  descriptor exists.
         */
        this.getFullFileName = function () {
            return Utils.getStringOption(file, 'filename', null);
        };

        /**
         * Returns the short file name of the current file (without file
         * extension).
         *
         * @returns {String|Null}
         *  The short file name of the current file descriptor; or null, if no
         *  file descriptor exists.
         */
        this.getShortFileName = function () {
            var fileName = this.getFullFileName();
            return _.isString(fileName) ? ExtensionRegistry.getBaseName(fileName) : null;
        };

        /**
         * Returns the file extension of the current file (without leading
         * period).
         *
         * @returns {String|Null}
         *  The file extension of the current file descriptor; or null, if no
         *  file descriptor exists.
         */
        this.getFileExtension = function () {
            var fileName = this.getFullFileName();
            return _.isString(fileName) ? ExtensionRegistry.getExtension(fileName) : null;
        };

        /**
         * Returns the import promise that is pending as long as the document
         * will be imported, and that will be resolved after successful import,
         * or rejected after import has failed.
         */
        this.getImportPromise = function () {
            return importDef.promise();
        };

        /**
         * Return whether importing the document has been completed regardless
         * whether it was successful. Will be false while the import promise is
         * pending, and true after the import promise has been resolved or
         * rejected.
         *
         * @returns {jQuery.Promise}
         *  The import promise of this document.
         */
        this.isImportFinished = function () {
            return importDef.state() !== 'pending';
        };

        /**
         * Return whether importing the document was successful. Will be false
         * while the import promise is pending or after it has been rejected,
         * and true after the import promise has been resolved.
         *
         * @returns {Boolean}
         *  Whether importing the document has been completed successfully.
         */
        this.isImportSucceeded = function () {
            return importDef.state() === 'resolved';
        };

        /**
         * Return whether importing the document has failed. Will be false
         * while the import promise is pending or after it has been resolved,
         * and true after the import promise has been rejected.
         *
         * @returns {Boolean}
         *  Whether importing the document is finished and has failed.
         */
        this.isImportFailed = function () {
            return importDef.state() === 'rejected';
        };

        /**
         * Invokes the passed callback function as soon as this application has
         * finished importing the document. If this method is used after
         * importing the document, the passed callback function will be invoked
         * immediately. This method is a convenience shortcut for using the
         * code line 'app.getImportPromise().always(callback);'.
         *
         * @param {Function} callback
         *  The callback function invoked after the application has finished
         *  importing the document.
         *
         * @param {Object} [context]
         *  The context bound to the invoked callback function.
         *
         * @returns {BaseApplication}
         *  A reference to this instance.
         */
        this.onImport = function (callback, context) {
            importDef.always(_.bind(callback, context));
            return this;
        };

        /**
         * Invokes the passed callback function as soon as this application has
         * finished importing the document successfully. If this method is used
         * after importing the document, the passed callback function will be
         * invoked immediately (unless import has failed). This method is a
         * convenience shortcut for using the code line
         * 'app.getImportPromise().done(callback);'.
         *
         * @param {Function} callback
         *  The callback function invoked after the application has finished
         *  importing the document successfully.
         *
         * @param {Object} [context]
         *  The context bound to the invoked callback function.
         *
         * @returns {BaseApplication}
         *  A reference to this instance.
         */
        this.onImportSuccess = function (callback, context) {
            importDef.done(_.bind(callback, context));
            return this;
        };

        /**
         * Invokes the passed callback function as soon as this application has
         * failed importing the document. If this method is used after
         * importing the document, the passed callback function will be invoked
         * immediately (unless import has succeeded). This method is a
         * convenience shortcut for using the code line
         * 'app.getImportPromise().fail(callback);'.
         *
         * @param {Function} callback
         *  The callback function invoked after the application has failed
         *  importing the document.
         *
         * @param {Object} [context]
         *  The context bound to the invoked callback function.
         *
         * @returns {BaseApplication}
         *  A reference to this instance.
         */
        this.onImportFailure = function (callback, context) {
            importDef.fail(_.bind(callback, context));
            return this;
        };

        /**
         * Checks if application is processing before-quit or quit handlers.
         *
         * @returns {Boolean}
         *  Whether the application is currently processing any before-quit or
         *  quit handlers.
         */
        this.isInQuit = function () {
            return currentQuitDef !== null;
        };

        // server requests ----------------------------------------------------

        /**
         * Sends a request to the server and returns the Promise of a Deferred
         * object waiting for the response. The unique identifier of this
         * application will be added to the request parameters automatically.
         * See method IO.sendRequest() for further details.
         *
         * @param {Object} options
         *  Additional options. See method IO.sendRequest() for details.
         *
         * @returns {jQuery.Promise}
         *  The Promise of the request. See method IO.sendRequest() for
         *  details.
         */
        this.sendRequest = function (options) {

            var // the AJAX request, as jQuery Deferred object
                request = null;

            // build a default map with the application UID, and add the passed options
            options = Utils.extendOptions({
                params: {
                    uid: this.get('uniqueID')
                }
            }, options);

            // send the request
            request = IO.sendRequest(options);

            // store the request internally for automatic abort on application quit
            this.registerAbortablePromise(request);

            return request;
        };

        /**
         * Sends a request to the document filter server module to determine
         * if certain features are working.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object. On success of the request, the
         *  Promise will be resolved with an array object containing all
         *  supported feature identifiers.
         */
        this.sendFeatureRequest = function () {
            return this.sendRequest({
                module: BaseApplication.FILTER_MODULE_NAME,
                params: { action: 'getfeatures' },
                resultFilter: function (data) {
                    var features = Utils.getStringOption(data, 'features');
                    return _.isString(features) ? features.split(/,/) : undefined;
                }
            });
        };

        /**
         * Sends a request to the server and returns the Promise of a Deferred
         * object waiting for the response. The unique identifier of this
         * application, and the parameters of the file currently opened by the
         * application will be added to the request parameters automatically.
         * See method IO.sendRequest() for further details.
         *
         * @param {String} module
         *  The name of the server module.
         *
         * @param {Object} options
         *  Additional options. See method IO.sendRequest() for details.
         *
         * @returns {jQuery.Promise}
         *  The Promise of the request. Will be rejected immediately, if this
         *  application is not connected to a document file. See method
         *  IO.sendRequest() for details.
         */
        this.sendFileRequest = function (module, options) {

            // reject immediately if no file is present
            if (!this.hasFileDescriptor()) {
                return $.Deferred().reject();
            }

            // build default options, and add the passed options
            options = Utils.extendOptions({
                module: module,
                params: this.getFileParameters()
            }, options);

            // send the request
            return this.sendRequest(options);
        };

        /**
         * Sends a request to the document filter server module. See method
         * BaseApplication.sendFileRequest() for further details.
         */
        this.sendFilterRequest = function (options) {
            return this.sendFileRequest(BaseApplication.FILTER_MODULE_NAME, options);
        };

        /**
         * Sends a request to the document converter server module. See method
         * BaseApplication.sendFileRequest() for further details.
         */
        this.sendConverterRequest = function (options) {
            return this.sendFileRequest(BaseApplication.CONVERTER_MODULE_NAME, options);
        };

        /**
         * Creates and returns the URL of a server request.
         *
         * @param {String} module
         *  The name of the server module.
         *
         * @param {Object} [options]
         *  Additional parameters inserted into the URL.
         *
         * @returns {String|Undefined}
         *  The final URL of the server request; or undefined, if the
         *  application is not connected to a document file, or the current
         *  session is invalid.
         */
        this.getServerModuleUrl = function (module, options) {

            // return nothing if no file is present
            if (!ox.session || !this.hasFileDescriptor()) {
                return;
            }

            // build a default options map, and add the passed options
            options = Utils.extendOptions(
                { session: ox.session, uid: this.get('uniqueID') },
                this.getFileParameters({ encodeUrl: true }),
                options
            );

            // build and return the resulting URL
            return ox.apiRoot + '/' + module + '?' + _.map(options, function (value, name) { return name + '=' + value; }).join('&');
        };

        /**
         * Creates and returns the URL of server requests used to convert a
         * document file with the document filter module.
         *
         * @param {Object} [options]
         *  Additional parameters inserted into the URL.
         *
         * @returns {String|Undefined}
         *  The final URL of the request to the document filter module; or
         *  undefined, if the application is not connected to a document file,
         *  or the current session is invalid.
         */
        this.getFilterModuleUrl = function (options) {
            return this.getServerModuleUrl(BaseApplication.FILTER_MODULE_NAME, options);
        };

        /**
         * Creates and returns the URL of server requests used to convert a
         * document file with the document converter module.
         *
         * @param {Object} [options]
         *  Additional parameters inserted into the URL.
         *
         * @returns {String|Undefined}
         *  The final URL of the request to the document converter module; or
         *  undefined, if the application is not connected to a document file,
         *  or the current session is invalid.
         */
        this.getConverterModuleUrl = function (options) {
            return this.getServerModuleUrl(BaseApplication.CONVERTER_MODULE_NAME, options);
        };

        /**
         * Creates an <img> element, sets the passed URL, and returns a
         * Deferred object waiting that the image has been loaded. If the
         * browser does not trigger the appropriate events after the specified
         * timeout, the Deferred object will be rejected automatically. If the
         * application has been shut down before the image has been loaded, the
         * pending Deferred object will neither be resolved nor rejected.
         *
         * @param {String} url
         *  The image URL.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Number} [options.timeout=10000]
         *      The time before the Deferred will be rejected without response
         *      from the image node.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved with the
         *  image element (as jQuery object), when the 'load' event has been
         *  received from the image node; or rejected, if the 'error' event has
         *  been received from the image, or after the specified timeout delay
         *  without response.
         */
        this.createImageNode = function (url, options) {

            var // the result Deferred object
                def = $.Deferred(),
                // the image node
                imgNode = $('<img>', { src: url }),
                // the duration of the failure timeout
                timeout = Utils.getIntegerOption(options, 'timeout', 10000, 0),
                // the resulting promise
                promise = this.createAbortablePromise(def, $.noop, timeout),
                // the event handlers for the image node
                handlers = {
                    load: function () { def.resolve(imgNode); },
                    error: function () { def.reject(); }
                };

            // wait that the image is loaded
            imgNode.one(handlers);
            return promise.always(function () { imgNode.off(handlers); });
        };

        /**
         * Safely destroys the passed image nodes. On platforms with restricted
         * memory (especially iPad and iPhone), allocating too many images may
         * result in an immediate browser crash. Simply removing image nodes
         * from the DOM does not free the image resource data allocated by the
         * browser. The suggested workaround is to replace the 'src' attribute
         * of the image node with a local small image which will result in
         * releasing the original image data by the internal resource manager.
         * To be really sure that the garbage collector does not destroy the
         * DOM node instance before the resource manager releases the image
         * data, a reference to the node will be kept in memory for additional
         * 60 seconds.
         *
         * @param {HTMLElement|jQuery} nodes
         *  The image elements, or any other container elements with descendant
         *  image nodes, either as single DOM node, or as jQuery collection
         *  with one or multiple elements.
         */
        this.destroyImageNodes = function (nodes) {

            var // filter passed image nodes, find descendant image nodes
                imgNodes = $(nodes).filter('img').add($(nodes).find('img'));

            if (imgNodes.length > 0) {
                Utils.insertHiddenNodes(imgNodes);
                $(imgNodes).off().attr('src', 'data:image/gif;base64,R0lGODdhAgACAIAAAAAAAP///ywAAAAAAgACAAACAoRRADs=');
                _.delay(function () { imgNodes.remove(); imgNodes = null; }, 60000);
            }
        };

        // application setup --------------------------------------------------

        /**
         * Registers a quit handler function that will be executed before the
         * application will be closed.
         *
         * @param {Function} beforeQuitHandler
         *  A function that will be called before the application will be
         *  closed. Will be called in the context of this application instance.
         *  May return a Deferred object, which must be resolved or rejected by
         *  the quit handler function. If the Deferred object will be rejected,
         *  the application remains alive.
         *
         * @returns {BaseApplication}
         *  A reference to this application instance.
         */
        this.registerBeforeQuitHandler = function (beforeQuitHandler) {
            beforeQuitHandlers.push(beforeQuitHandler);
            return this;
        };

        /**
         * Registers a quit handler function that will be executed before the
         * application will be destroyed. This may happen if the application
         * has been closed normally, if the user logs out from the entire
         * AppSuite, or before the browser window or browser tab will be closed
         * or refreshed. If the quit handlers return a Deferred object, closing
         * the application, and logging out will be deferred until the Deferred
         * objects have been resolved or rejected.
         *
         * @param {Function} quitHandler
         *  A function that will be called before the application will be
         *  closed. Will be called in the context of this application instance.
         *  May return a Deferred object, which must be resolved or rejected by
         *  the quit handler function.
         *
         * @returns {BaseApplication}
         *  A reference to this application instance.
         */
        this.registerQuitHandler = function (quitHandler) {
            quitHandlers.push(quitHandler);
            return this;
        };

        /**
         * Executes all registered quit handlers and returns a Deferred object
         * that will be resolved or rejected according to the results of the
         * handlers.
         *
         * @internal
         *  Not for external use. Only used from the implementation of the
         *  'io.ox/core/logout' extension point and the window's 'unload' event
         *  handler.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved/rejected when all quit handlers
         *  have been executed.
         */
        this.executeQuitHandlers = function () {
            // Bug 28044: unbind the 'unload' event handler immediately, otherwise
            // logging out may trigger quit handlers twice (from logout and unload)
            $(window).off('unload', unloadHandler);
            return callHandlers(quitHandlers);
        };

        /**
         * Registers the promise of a Deferred object representing any action
         * running asynchronously, e.g. an AJAX request to the server, or a
         * background timer executing custom asynchronous code. The promise
         * must provide a method abort(). When the application shuts down, all
         * pending promises will be aborted automatically.
         *
         * @param {jQuery.Deferred|jQuery.Promise} promise
         *  Any Deferred object or a promise that provides an additional method
         *  abort().
         *
         * @param {Number} [timeout]
         *  If specified and a positive number, the delay time in milliseconds
         *  after the specified promise will be rejected with the result string
         *  'timeout'.
         *
         * @returns {BaseApplication}
         *  A reference to this instance.
         */
        this.registerAbortablePromise = function (promise, timeout) {

            // do not register promise that is not pending anymore
            if (promise.state() !== 'pending') { return; }

            // do not register promises when applications shuts down already
            if (pendingPromises) {
                pendingPromises.push(promise.always(function () {
                    pendingPromises = _.without(pendingPromises, promise);
                }));
            } else {
                promise.abort();
            }

            // abort automatically after the specified timeout
            if (_.isNumber(timeout) && (timeout > 0)) {
                var timer = window.setTimeout(function () {
                    if (promise.state() === 'pending') { promise.abort('timeout'); }
                }, timeout);
                promise.always(function () {
                    if (timer) { window.clearTimeout(timer); timer = null; }
                });
            }

            return this;
        };

        /**
         * Creates a promise for the passed Deferred object representing any
         * code running asynchronously. The promise will contain an additional
         * method abort() that when called before the Deferred object has been
         * resolved or rejected, invokes the specified callback function, and
         * rejects the passed Deferred object. Additionally, the promise will
         * be stored internally as long as it is in pending state. When the
         * application shuts down, all pending promises will be aborted
         * automatically.
         *
         * @param {jQuery.Deferred} deferred
         *  The Deferred object to create an abortable promise for.
         *
         * @param {Function} [callback]
         *  An optional callback function that will be invoked when the promise
         *  has been aborted, and the Deferred object is still pending.
         *
         * @param {Number} [timeout]
         *  If specified and a positive number, the delay time in milliseconds
         *  after the promise returned by this method will be rejected, if the
         *  passed original Deferred object is still pending.
         *
         * @returns {jQuery.Promise}
         *  The promise for the passed Deferred object, with an additional
         *  method abort().
         */
        this.createAbortablePromise = function (deferred, callback, timeout) {

            var // the promise of the passed Deferred object
                promise = deferred.promise();

            // add a custom abort() method to the promise
            promise.abort = function (cause) {
                if (deferred.state() === 'pending') {
                    if (_.isFunction(callback)) { callback.call(this); }
                    deferred.reject(cause || 'abort');
                }
                return this;
            };

            // replace the abort method after resolving/rejecting the promise (release callback closure)
            promise.always(function () {
                promise.abort = function () { return this; };
                callback = null;
            });

            // store the promise internally for automatic abort on application quit
            this.registerAbortablePromise(promise, timeout);

            return promise;
        };

        // timeouts -----------------------------------------------------------

        /**
         * Invokes the passed callback function once in a browser timeout. If
         * the application will be closed before the callback function has been
         * started, it will not be called anymore.
         *
         * @param {Function} callback
         *  The callback function that will be executed in a browser timeout
         *  after the delay time. Does not receive any parameters.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Object} [options.context]
         *      The context that will be bound to 'this' in the passed callback
         *      function.
         *  @param {Number} [options.delay=0]
         *      The time (in milliseconds) the execution of the passed callback
         *      function will be delayed.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved after the
         *  callback function has been executed. If the callback function
         *  returns a simple value or object, the Deferred object will be
         *  resolved with that value. If the callback function returns a
         *  Deferred object or a Promise by itself, its state and result value
         *  will be forwarded to the Promise returned by this method. The
         *  Promise contains an additional method 'abort()' that can be called
         *  before the timeout has been fired to cancel the pending execution
         *  of the callback function. In that case, the Promise will be
         *  rejected. When the application will be closed, it aborts all
         *  pending callback functions automatically.
         */
        this.executeDelayed = function (callback, options) {

            var // the context for the callback function
                context = Utils.getOption(options, 'context'),
                // the delay time for the next execution of the callback
                delay = Utils.getIntegerOption(options, 'delay', 0, 0),
                // the current browser timeout identifier
                timeout = null,
                // the resulting Deferred object
                def = $.Deferred(),
                // the abortable promise of the Deferred object
                promise = this.createAbortablePromise(def, function () {
                    if (timeout) { window.clearTimeout(timeout); timeout = null; }
                });

            // do not start new timer when application already shuts down
            if (promise.state() === 'pending') {

                // create a new browser timeout
                timeout = window.setTimeout(function () {
                    timeout = null;
                    // execute the callback function, forward its result to the promise
                    $.when(callback.call(context)).done(_.bind(def.resolve, def)).fail(_.bind(def.reject, def));
                }, delay);
            }

            return promise;
        };

        /**
         * Invokes the passed callback function repeatedly in a browser timeout
         * loop. If the application will be closed before the callback function
         * has been started, or while the callback function will be repeated,
         * it will not be called anymore.
         *
         * @param {Function} callback
         *  The callback function that will be invoked repeatedly in a browser
         *  timeout loop after the initial delay time. Receives the zero-based
         *  index of the execution cycle as first parameter. The return value
         *  of the function will be used to decide whether to continue the
         *  repeated execution. If the function returns Utils.BREAK, execution
         *  will be stopped. If the function returns a Deferred object, or the
         *  Promise of a Deferred object, looping will be deferred until the
         *  Deferred object is resolved or rejected. After resolving the
         *  Deferred object, the delay time will start, and the callback will
         *  be called again. Otherwise, after the Deferred object has been
         *  rejected, execution will be stopped, and the Promise returned by
         *  this method will be rejected too. All other return values will be
         *  ignored, and the callback loop will continue.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Object} [options.context]
         *      The context that will be bound to 'this' in the passed callback
         *      function.
         *  @param {Number} [options.delay=0]
         *      The time (in milliseconds) the initial invocation of the passed
         *      callback function will be delayed.
         *  @param {Number} [options.repeatDelay=options.delay]
         *      The time (in milliseconds) the repeated invocation of the
         *      passed callback function will be delayed. If omitted, the
         *      specified initial delay time (option 'delay') will be used.
         *  @param {Number} [options.cycles]
         *      If specified, the maximum number of cycles to be executed.
         *  @param {Boolean} [options.fastAsync=false]
         *      If set to true, and the callback function returns a pending
         *      promise, there will not be any further delay before the
         *      callback function will be invoked again.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved or rejected
         *  after the callback function has been invoked the last time. The
         *  Promise will be resolved, if the callback function has returned the
         *  Utils.BREAK object in the last iteration, or if the maximum number
         *  of iterations (see option 'cycles') has been reached. It will be
         *  rejected, if the callback function has returned a rejected Deferred
         *  object or Promise. The returned Promise contains an additional
         *  method 'abort()' that can be called before or while the callback
         *  loop is executed to stop the loop immediately. In that case, the
         *  Promise will be rejected. When the application will be closed, it
         *  aborts all running callback loops automatically.
         */
        this.repeatDelayed = function (callback, options) {

            var // the context for the callback function
                context = Utils.getOption(options, 'context'),
                // the delay time for the first invocation of the callback
                initDelay = Utils.getIntegerOption(options, 'delay', 0, 0),
                // the delay time for the next invocations of the callback
                repeatDelay = Utils.getIntegerOption(options, 'repeatDelay', initDelay, 0),
                // the number of cycles to be executed
                cycles = Utils.getIntegerOption(options, 'cycles'),
                // whether to continue immediately after asynchronous callbacks
                fastAsync = Utils.getBooleanOption(options, 'fastAsync'),
                // the index of the next cycle
                index = 0,
                // the current timeout before invoking the callback
                timer = null,
                // the resulting Deferred object
                def = $.Deferred(),
                // the abortable promise of the Deferred object
                promise = this.createAbortablePromise(def, function () {
                    if (timer) { timer.abort(); timer = null; }
                });

            // creates and registers a browser timeout that executes the callback
            function createTimer(delay) {

                // create a new browser timeout
                timer = self.executeDelayed(function () {

                    var // the result of the callback
                        result = callback.call(context, index),
                        // whether the result was a pending promise
                        pending = false;

                    // immediately break the loop if callback returns Utils.BREAK
                    if (result === Utils.BREAK) {
                        def.resolve(Utils.BREAK);
                        return;
                    }

                    // convert to a promise, check whether it is still pending
                    result = $.when(result);
                    pending = result.state() === 'pending';

                    // wait for the result
                    result.done(function () {
                        // do not create a new timeout if execution has been aborted while the asynchronous code was running
                        if (def.state() !== 'pending') { return; }
                        // decide whether to start the next iteration
                        index += 1;
                        if (!_.isNumber(cycles) || (index < cycles)) {
                            createTimer((fastAsync && pending) ? 0 : repeatDelay);
                        } else {
                            def.resolve();
                        }
                    });

                    // reject the own Deferred object on failure
                    result.fail(_.bind(def.reject, def));

                }, Utils.extendOptions(options, { delay: delay }));
            }

            // start the initial timer
            createTimer(initDelay);

            return promise;
        };

        /**
         * Invokes the passed callback function repeatedly until it returns a
         * specific value. To prevent performance problems, or a frozen browser
         * user interface, the callback function may be invoked from several
         * time slices of a browser timeout loop. In difference to the method
         * BaseApplication.repeatDelayed() that uses an own browser timeout for
         * each invocation of the callback function, this method tries to pack
         * multiple invocations into a single time slice until the total
         * execution time of the time slice is exceeded, and continues with a
         * new time slice in another browser timeout afterwards. If the
         * application will be closed while the loop is running, it will be
         * aborted immediately.
         *
         * @param {Function} callback
         *  The callback function that will be invoked repeatedly. Receives the
         *  zero-based index of the execution cycle as first parameter. The
         *  return value of the function will be used to decide whether to
         *  continue the repeated execution. If the function returns
         *  Utils.BREAK, execution will be stopped. If the function returns a
         *  Deferred object, or the Promise of a Deferred object, looping will
         *  be deferred until the Deferred object is resolved or rejected.
         *  After resolving the Deferred object, the delay time will start, and
         *  the callback will be called again. Otherwise, after the Deferred
         *  object has been rejected, execution will be stopped, and the
         *  Promise returned by this method will be rejected too. All other
         *  return values will be ignored, and the callback loop will continue.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Object} [options.context]
         *      The calling context for the iterator callback function.
         *  @param {Number} [options.delay]
         *      The initial delay time (in milliseconds) before invoking the
         *      callback function the first time. If omitted, the callback
         *      function will be executed immediately with the invocation of
         *      this method.
         *  @param {Number} [options.slice=200]
         *      The time (in milliseconds) for a single synchronous time slice.
         *      The callback function will be invoked repeatedly until the time
         *      has elapsed; or until the iterator function returns
         *      Utils.BREAK, a pending promise, or a rejected promise (see
         *      parameter 'callback' above for more details).
         *  @param {Number} [options.interval=10]
         *      The delay time (in milliseconds) between two synchronous time
         *      slices. Will not be used if the iterator returns a pending
         *      promise. In this case, the implementation will wait for the
         *      promise to resolve and will continue the loop afterwards
         *      without further delay.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved after the
         *  callback function returns Utils.BREAK; or that will be rejected, if
         *  the iterator callback function returns a rejected promise. The
         *  promise will be notified each time a new time slice starts (with
         *  the zero-based index of the time slice). The promise contains an
         *  additional method 'abort()' that can be called to cancel the loop
         *  immediately. In that case, the promise will be rejected. When
         *  closing the application, it will abort all running loops
         *  automatically.
         */
        this.repeatSliced = function (callback, options) {

            var // the context for the callback function
                context = Utils.getOption(options, 'context'),
                // initial delay time
                delay = Utils.getIntegerOption(options, 'delay', null, 0),
                // slice time for synchronous processing
                slice = Utils.getIntegerOption(options, 'slice', 200, 10),
                // delay time between time slices
                interval = Utils.getIntegerOption(options, 'interval', 10, 0),
                // index of the next time slice
                sliceIndex = 0,
                // index of the next cycle
                index = 0,
                // the interval timer for the time slices
                timer = null,
                // the resulting Deferred object
                def = $.Deferred(),
                // the abortable promise of the resulting Deferred object
                promise = this.createAbortablePromise(def, function () {
                    if (timer) { timer.abort(); timer = null; }
                });

            /*
             * Invokes the callback synchronously multiple times. Stops and
             * returns the first promise received from the iterator, otherwise
             * stops after the time specified in the option 'slice'. Used as
             * callback for the method BaseApplication.repeatDelayed().
             */
            function processTimeSlice() {

                var // start time of this iteration
                    start = _.now(),
                    // current return value of the iterator
                    result = null;

                // notify the own Deferred object about the progress
                def.notify(sliceIndex);
                sliceIndex += 1;

                // invoke callback repeatedly (loop body returns on any exit conditions)
                while (true) {

                    // invoke the callback function
                    result = callback.call(context, index);
                    index += 1;

                    // iterator has returned Utils.BREAK: resolve the own Deferred
                    // object immediately, and break the background loop
                    if (result === Utils.BREAK) {
                        def.resolve(Utils.BREAK);
                        return Utils.BREAK;
                    }

                    // iterator has returned a pending or rejected promise: return it to defer
                    // the loop (ignore resolved promises and continue with next invocation)
                    if (_.isObject(result) && _.isFunction(result.promise) && (result.state() !== 'resolved')) {
                        return result;
                    }

                    // check elapsed time at the end of the loop body (this ensures
                    // that the callback will be executed at least once)
                    if (_.now() - start >= slice) {
                        return;
                    }
                }
            }

            /*
             * Starts an interval timer that invokes the processElementsSync()
             * method as often as needed.
             */
            function startIntervalTimer(initDelay, repeatDelay) {
                if (def.state() === 'pending') {
                    // create the interval timer with the passed delay options
                    timer = self.repeatDelayed(processTimeSlice, { delay: initDelay, repeatDelay: repeatDelay, fastAsync: true });
                    // resolve/reject the own Deferred object when the timer has finished
                    timer.done(_.bind(def.resolve, def)).fail(_.bind(def.reject, def));
                }
            }

            // start the interval timer according to the passed delay options
            if (_.isNumber(delay)) {
                // first time slice will be processed deferred
                startIntervalTimer(delay, interval);
            } else {
                // first time slice will be processed immediately
                $.when(processTimeSlice())
                    .done(_.partial(startIntervalTimer, interval, interval))
                    .fail(_.bind(def.reject, def));
            }

            return promise;
        };

        /**
         * Invokes the passed callback function for all elements contained in
         * the passed data array. To prevent performance problems, or a frozen
         * browser user interface, the callback function will be invoked from
         * several time slices of a browser timeout loop. If the application
         * will be closed while the loop is running, it will be aborted
         * immediately.
         *
         * @param {Array|jQuery} array
         *  A JavaScript array, or another array-like object that provides a
         *  'length'property, and element access via bracket notation.
         *
         * @param {Function} iterator
         *  The iterator callback function that will be invoked for each array
         *  element. Receives the following parameters:
         *  (1) {Any} element
         *      The current array element.
         *  (2) {Number} index
         *      The array index of the current element.
         *  (3) {Array|jQuery} array
         *      The entire array as passed in the 'array' parameter.
         *  The return value of the iterator will be used to decide whether to
         *  continue to process the next array elements. If the iterator
         *  returns Utils.BREAK, the loop will be stopped immediately. If the
         *  iterator returns a promise of a Deferred object, the loop will be
         *  deferred until the promise has been resolved or rejected. After
         *  resolving the promise, the loop will continue to process the next
         *  array element. Otherwise, after rejecting the promise, the loop
         *  will be aborted immediately (the promise returned by this method
         *  will be rejected too, see below). All other return values will be
         *  ignored.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Object} [options.context]
         *      The calling context for the iterator callback function.
         *  @param {Number} [options.delay]
         *      The initial delay time (in milliseconds) before processing the
         *      first array elements. If omitted, the first elements will be
         *      processed synchronously with the invocation of this method.
         *  @param {Number} [options.slice=200]
         *      The time (in milliseconds) for a single synchronous time slice.
         *      The iterator function will be invoked for several array
         *      elements until the time has elapsed; or until the iterator
         *      function returns Utils.BREAK, a pending promise, or a rejected
         *      promise (see parameter 'iterator' above for more details).
         *  @param {Number} [options.interval=10]
         *      The delay time (in milliseconds) between two synchronous time
         *      slices. Will not be used if the iterator returns a pending
         *      promise for an array element. In this case, the implementation
         *      will wait for the promise to resolve and will continue the loop
         *      afterwards without further delay.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved after all
         *  array elements have been processed successfully, or the iterator
         *  callback function returns Utils.BREAK; or that will be rejected, if
         *  the iterator callback function returns a rejected promise. The
         *  promise will be notified about the progress (as a floating-point
         *  value between 0.0 and 1.0). The promise contains an additional
         *  method 'abort()' that can be called to cancel the loop immediately
         *  while processing the array. In that case, the promise will be
         *  rejected. When closing the application, it will abort all running
         *  loops automatically.
         */
        this.iterateArraySliced = function (array, iterator, options) {

            var // index of next processed array element
                index = 0,
                // promise representing the sliced loop
                timer = null,
                // the resulting Deferred object
                def = $.Deferred(),
                // the abortable promise of the resulting Deferred object
                promise = this.createAbortablePromise(def, function () {
                    if (timer) { timer.abort(); timer = null; }
                });

            // do nothing if passed array is empty
            if (array.length === 0) {
                def.notify(1).resolve();
                return promise;
            }

            // start the sliced loop
            var timer = this.repeatSliced(function () {
                // end of array reached
                if (array.length <= index) {
                    def.notify(1).resolve();
                    return Utils.BREAK;
                }
                // invoke the iterator for a single array element
                var result = iterator.call(this, array[index], index, array);
                index += 1;
                return result;
            }, options);

            // resolve/reject the own Deferred object according to the timer result
            timer.done(_.bind(def.resolve, def)).fail(_.bind(def.reject, def));

            // notify the own Deferred object about the progress (once per time slice)
            timer.progress(function () { def.notify(index / array.length); });
            promise.done(function () { def.notify(1); });

            return promise;
        };

        /**
         * Creates a synchronized method wrapping a callback function that
         * executes asynchronous code. The synchronized method buffers multiple
         * fast invocations and executes the callback function successively,
         * always waiting for the asynchronous code. In difference to debounced
         * methods, invocations of a synchronized method will never be skipped,
         * and each call of the asynchronous callback function receives its
         * original arguments passed to the synchronized method.
         *
         * @param {Function} callback
         *  A function that will be called every time the synchronized method
         *  has been called. Will be called in the context the synchronized
         *  method has been called with, and receives all parameters that have
         *  been passed to the synchronized method. If this function returns a
         *  pending Deferred object or Promise, subsequent invocations of the
         *  synchronized method will be postponed until the Deferred object or
         *  Promise will be resolved or rejected. All other return values will
         *  be interpreted as synchronous invocations of the callback function.
         *
         * @returns {Function}
         *  The synchronized method that can be called multiple times, and that
         *  executes the asynchronous callback function sequentially. Returns
         *  the Promise of a Deferred object that will be resolved or rejected
         *  after the callback function has been invoked. If the callback
         *  function returns a Deferred object or Promise, the synchronized
         *  method will wait for it, and will forward its state and response to
         *  its Promise. Otherwise, the Promise will be resolved with the
         *  return value of the callback function. When the application will be
         *  closed, pending callbacks will not be executed anymore.
         */
        this.createSynchronizedMethod = function (callback) {

            var // arguments and returned Promise of pending calls of the method
                pendingInvocations = [],
                // Promise representing the callback function currently running
                runningPromise = null,
                // the background loop processing all pending invocations
                timer = null;

            // invokes the callback once with the passed set of arguments
            function invokeCallback(invocationData) {
                // create the Promise
                runningPromise = $.when(callback.apply(invocationData.ctxt, invocationData.args));
                // register callbacks after assignment to handle synchronous callbacks correctly
                runningPromise
                    .always(function () { runningPromise = null; })
                    .done(_.bind(invocationData.def.resolve, invocationData.def))
                    .fail(_.bind(invocationData.def.reject, invocationData.def));
            }

            // create and return the synchronized method
            return function () {

                var // all data about the current invocation (arguments and returned Deferred object)
                    invocationData = { ctxt: this, args: arguments, def: $.Deferred() };

                // cache invocation data, if a callback function is currently running
                // Bug 28593: also, if pending callbacks exist without running callback
                // (race condition: running callback resolves -> 'runningPromise' reset
                // -> this method called -> background loop starts the next callback)
                if (runningPromise || (pendingInvocations.length > 0)) {
                    pendingInvocations.push(invocationData);
                    // start timer that processes the array
                    if (!timer) {
                        timer = self.repeatDelayed(function () {
                            if (runningPromise) { return runningPromise; }
                            if (pendingInvocations.length === 0) { return Utils.BREAK; }
                            invokeCallback(pendingInvocations.shift());
                            return runningPromise;
                        });
                        // forget the timer after the last callback invocation
                        timer.always(function () { timer = null; });
                    }
                } else {
                    // invoke the callback function directly on first call
                    invokeCallback(invocationData);
                }

                // return a Promise that will be resolved/rejected after invocation
                return invocationData.def.promise();
            };
        };

        /**
         * Creates a debounced method that can be called multiple times during
         * the current script execution. The passed callback will be executed
         * once in a browser timeout.
         *
         * @param {Function} directCallback
         *  A function that will be called every time the debounced method has
         *  been called. Will be called in the context the debounced method has
         *  been called with, and receives all parameters that have been passed
         *  to the debounced method.
         *
         * @param {Function} deferredCallback
         *  A function that will be called in a browser timeout after the
         *  debounced method has been called at least once during the execution
         *  of the current script. Will be called in the context the debounced
         *  method has been called with, and does not receive any parameters.
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all options also supported by the
         *  method BaseApplication.executeDelayed(), except the option
         *  'context'. Note that the delay time will restart after each call of
         *  the debounced method, causing the execution of the deferred
         *  callback to be postponed until the debounced method has not been
         *  called again during the delay (this is the behavior of the method
         *  _.debounce()). Additionally, supports the following options:
         *  @param {Number} [options.maxDelay]
         *      If specified, a delay time used as a hard limit to execute the
         *      deferred callback after the first call of the debounced method,
         *      even if it has been called repeatedly afterwards and the normal
         *      delay time is still running.
         *
         * @returns {Function}
         *  The debounced method that can be called multiple times, and that
         *  executes the deferred callback function once after execution of the
         *  current script ends. Passes all arguments to the direct callback
         *  function, and returns its result.
         */
        this.createDebouncedMethod = function (directCallback, deferredCallback, options) {

            var // whether to not restart the timer on repeated calls with delay time
                maxDelay = Utils.getIntegerOption(options, 'maxDelay', 0),
                // the current timer used to execute the callback
                minTimer = null,
                // timer used for the maxDelay option
                maxTimer = null,
                // first call in this stack frame
                firstCall = true;

            // aborts and clears all timers
            function clearTimers() {
                if (minTimer) { minTimer.abort(); }
                if (maxTimer) { maxTimer.abort(); }
                minTimer = maxTimer = null;
                firstCall = true;
            }

            // creates the timers that execute the deferred callback
            function createTimers(context) {

                var // timer callback invoking the deferred callback, bound to the current context (but ignoring the return
                    // value, especially Deferred objects returned by the callback will not be forwarded to the timer)
                    timerCallback = function () {
                        clearTimers();
                        deferredCallback.call(context);
                    };

                // abort running timer on first call
                if (firstCall && minTimer) {
                    minTimer.abort();
                    minTimer = null;
                }

                // create a new timeout executing the callback function
                if (!minTimer) {
                    minTimer = self.executeDelayed(timerCallback, options);
                }

                // reset the first-call flag, but set it back in a direct
                // timeout, this helps to prevent recreation of the browser
                // timeout on every call of the debounced method
                firstCall = false;
                _.defer(function () { firstCall = true; });

                // on first call, create a timer for the maximum delay
                if (!maxTimer && (maxDelay > 0)) {
                    maxTimer = self.executeDelayed(timerCallback, Utils.extendOptions(options, { delay: maxDelay }));
                }
            }

            // create and return the debounced method
            return function () {

                // create a new timeout executing the callback function
                createTimers(this);

                // call the direct callback with the passed arguments
                return directCallback.apply(this, arguments);
            };
        };

        // application runtime ------------------------------------------------

        /**
         * Downloads a file from the server, specified by the passed URL. If
         * the passed object is a Deferred object, the method waits for it to
         * resolve to the file URL, but immediately opens a pop-up window in
         * the background to prevent that the browser pop-up blocker will be
         * triggered.
         *
         * @param {String|jQuery.Deferred|jQuery.Promise} fileUrl
         *  The URL of the document. If this parameter is a Deferred object or
         *  a Promise, waits for it to be resolved with the file URL.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved if the file
         *  has been downloaded successfully, or rejected otherwise (e.g. when
         *  a pop-up blocker prevents the download).
         */
        this.downloadFile = function (fileUrl) {

            var // the pop-up window used to download the file
                popupWindow = null,
                // the result Deferred object
                def = $.Deferred();

            // synchronous mode: open the file directly in a new window
            if (_.isString(fileUrl)) {
                // browser may prevent opening pop-up windows
                return window.open(fileUrl) ? def.resolve(fileUrl) : def.reject({ cause: 'popupblocker' });
            }

            // passed Deferred already rejected: do not open pop-up window
            if (fileUrl.state() === 'rejected') {
                // TODO: show warning?
                return def.reject();
            }

            // open pop-up window early to prevent browser pop-up blockers
            popupWindow = window.open('about:blank', '_blank');
            window.focus();

            // browser plug-ins may still prevent opening pop-up windows
            if (!popupWindow) {
                return def.reject({ cause: 'popupblocker' });
            }

            // block application window while waiting for the file URL
            view.enterBusy();

            // initialize or hide the pop-up window, leave busy mode
            fileUrl
                .always(function () {
                    view.leaveBusy();
                })
                .done(function (url) {
                    popupWindow.location = url;
                    popupWindow.focus();
                    def.resolve(url);
                })
                .fail(function (response) {
                    popupWindow.close();
                    view.grabFocus();
                    def.reject(response);
                });

            return def.promise();
        };

        /**
         * Downloads the document file currently opened by this application.
         *
         * @param {String} [format='native']
         *  The requested file format. The default format 'native' will return
         *  the document in its native file format.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Array} [options.features]
         *      An array with the names of all server features required to
         *      successfully download the document. The current availability of
         *      the features will be checked before the file will actually be
         *      downloaded.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved if the
         *  document file has been downloaded successfully, or rejected
         *  otherwise.
         */
        this.download = function (format, options) {

            // callback for Deferred.then() to check the availability of required server features
            function resolveFeatures() {

                var // the required features
                    requiredFeatures = Utils.getArrayOption(options, 'features', []);

                // do not send server request, if no features are required
                if (requiredFeatures.length === 0) { return; }

                // ask for all available server features, check that all required features currently exist
                return self.sendFeatureRequest().then(function (availableFeatures) {
                    var isFeatureMissing = _.any(requiredFeatures, function (feature) {
                        return !_.contains(availableFeatures, feature);
                    });
                    if (isFeatureMissing) {
                        return $.Deferred().reject({ cause: 'missingfeatures' });
                    }
                });
            }

            // callback for Deferred.then() to propagate changed file to the files API
            // (bug 30246: not for mail/task attachments!)
            function propagateChangedFile() {
                if (!('source' in file)) {
                    return FilesAPI.propagate('change', file);
                }
            }

            // callback for Deferred.then() to resolve the Deferred object with the download URL
            function resolveDownloadUrl() {

                var urlOptions = {
                        action: 'getdocument',
                        documentformat: format || 'native',
                        mimetype: file.file_mimetype ? encodeURIComponent(file.file_mimetype) : '',
                        nocache: _.uniqueId() // needed to trick the browser cache (is not evaluated by the backend)
                    };

                return ExtensionRegistry.isEditable(urlOptions.filename) ? self.getFilterModuleUrl(urlOptions) : self.getConverterModuleUrl(urlOptions);
            }

            var // the Promise that will be resolved with the download URL
                downloadPromise = null;

            if (this.hasFileDescriptor()) {
                downloadPromise =
                    $.when(flushHandler.call(this, 'download'))
                    .then(resolveFeatures)
                    .then(propagateChangedFile)
                    .then(resolveDownloadUrl);
            } else {
                downloadPromise = $.Deferred().reject();
            }

            // Bug 28251: call downloadFile() with a Promise that will be resolved
            // with the download URL to prevent the pop-up blocker of the browser
            return this.downloadFile(downloadPromise);
        };

        /**
         * Downloads the document file currently opened by this application.
         * This method should be used by the UI as the method will provide
         * error messages to the user in case of an error.
         *
         * @param {String} [format='native']
         *  The requested file format. The default format 'native' will return
         *  the document in its native file format.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Array} [options.features]
         *      An array with the names of all server features required to
         *      successfully download the document. The current availability of
         *      the features will be checked before the file will actually be
         *      downloaded.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved if the
         *  document file has been downloaded successfully, or rejected
         *  otherwise.
         */
        this.downloadDocument = function (format, options) {

            if (this.isImportSucceeded()) {
                return this.download(format, options);
            }

            view.yell({
                type: 'error',
                headline: gt('Download Error'),
                message: gt('The document was not loaded successfully. Downloading is not possible.')
            });
            return $.Deferred().reject();
        };

        /**
         * Prints the document file currently opened by this application.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved if the
         *  document file has been printed successfully, or rejected otherwise.
         */
        this.print = function () {

            var // the Promise for the downloaded document
                downloadPromise = null;

            if (this.isImportSucceeded()) {
                downloadPromise = this.download('pdf', { features: ['documentconverter'] });
            } else {
                downloadPromise = $.Deferred().reject({ cause: 'importfailed' });
            }

            // show error message on failure
            downloadPromise.fail(function (response) {

                var // the alert type
                    type = 'error',
                    // the message text
                    message = null;

                switch (Utils.getStringOption(response, 'cause')) {
                case 'missingfeatures':
                    message = gt('The document converter is currently not available.');
                    break;
                case 'importfailed':
                    message = gt('The document was not loaded successfully. Printing is not possible.');
                    break;
                case 'popupblocker':
                    type = 'warning';
                    message = gt('The browser has blocked a pop-up window. Please enable pop-ups for printing.');
                    break;
                default:
                    message = gt('The server does not answer the request.');
                }

                view.yell({ type: type, headline: gt('Printing Error'), message: message });
            });

            return downloadPromise;
        };

        /**
         * Creates a new mail message with the current document attached
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved if the mail
         *  application has been launched successfully, or rejected otherwise.
         */
        this.sendMail = function () {

            var // the Promise with the result of the flush callback handler
                flushPromise = (this.isImportSucceeded() && this.hasFileDescriptor()) ? $.when(flushHandler.call(this, 'email')) : $.Deferred().reject();

            // launch Mail application if flushing was successful
            flushPromise = flushPromise.then(function () {
                return ox.launch('io.ox/mail/write/main').then(function () {

                    var // the mail writer application
                        mailApp = this;

                    // initializes the mail writer application responding to the specified mail
                    function replyToMail(mail) {
                        return mailApp.replyall({ folder: mail.folder_id, id: mail.id });
                    }

                    // appends the current document (must be a Drive file) to the mail writer application
                    function appendDriveFile() {
                        mailApp.addFiles([{
                            filename: file.filename,
                            folder_id: file.folder_id,
                            id: file.id,
                            size: file.file_size,
                            file_size: file.file_size
                        }], 'infostore');
                    }

                    // bug 32760: special handling, if file is an attachment, e.g. of a mail or task
                    if ('source' in file) {
                        switch (file.source) {

                        case 'mail':
                            return replyToMail(file).done(function () {
                                mailApp.setAttachments({ attachments: [{
                                    atmsgref: file.folder_id + '/' + file.id,
                                    content: null,
                                    content_type: file.file_mimetype,
                                    disp: 'attachment',
                                    filename: file.filename,
                                    group: 'attachment',
                                    id: file.attached,
                                    size: file.file_size,
                                    type: self.getFileExtension()
                                }] });
                            });

                        case 'task':
                            Utils.warn('BaseApplication.sendMail(): attaching a task attachment to a mail not implemented');
                            return mailApp.compose();

                        default:
                            Utils.warn('BaseApplication.sendMail(): unknown file source: "' + file.source + '"');
                            return mailApp.compose();
                        }
                    }

                    // bug 34300: mail attachments copied to Drive
                    if (_.isObject(file.origin)) {
                        if (file.origin.source === 'mail') {
                            return replyToMail(file.origin).done(appendDriveFile);
                        }
                        Utils.warn('BaseApplication.sendMail(): unknown file origin: "' + file.origin.source + '"');
                        return mailApp.compose();
                    }

                    // default case: attach a file from OX Drive
                    return mailApp.compose().done(appendDriveFile);
                });
            }, function () {

                var // the message text
                    message = null,
                    // cause
                    cause = self.isImportSucceeded() ? 'unknown' : 'importfailed';

                switch (cause) {
                    case 'importfailed':
                        message = gt('The document was not loaded successfully. Send as mail is not possible.');
                        break;
                    default:
                        message = gt('The server does not answer the request. Send as mail is not possible.');
                }

                view.yell({ type: 'error', headline: gt('Send as mail'), message: message });
            });

            return flushPromise;
        };

        /**
         * Returns whether the application contains pending document changes
         * not yet sent to the server. Intended to be overwritten on demand by
         * sub classes.
         *
         * @attention
         *  Do not rename this method. The OX core uses it to decide whether to
         *  show a warning before the browser refreshes or closes the page.
         *
         * @returns {Boolean}
         *  Whether the application contains pending document changes.
         */
        this.hasUnsavedChanges = function () {
            return false;
        };

        // initialization -----------------------------------------------------

        // set launcher callback invoked from base class
        this.setLauncher(function () {

            var // the application window
                win = null,
                // the value of the 'spellcheck' attribute of the <body> element
                spellState = null,
                // duration of a single step between two log messages
                lapTime = _.now(),
                // launch source
                actionSource = Utils.getStringOption(launchOptions, 'actionSource', null);

            function logLaunchDuration(message) {
                var time = _.now();
                Utils.info('BaseApplication.setLauncher(): ' + message + ' finished in ' + (time - lapTime) + 'ms (total ' + (time - launchStartTime) + 'ms)');
                lapTime = time;
            }

            Utils.info('BaseApplication.setLauncher(): starting construction');
            launchStartTime = lapTime;

            // check that launch options have been passed
            if (!_.isObject(launchOptions) || !_.isString(launchOptions.action)) {
                return $.Deferred().reject();
            }

            // create the application window
            win = ox.ui.createWindow({
                name: self.getName(),
                search: Utils.getBooleanOption(appOptions, 'search', false),
                chromeless: Utils.getBooleanOption(appOptions, 'chromeless', false)
            });

            // set the window at the application instance
            self.setWindow(win);
            updateTitle();

            // wait for unload events and execute quit handlers
            $(window).on('unload', unloadHandler);

            // create the MVC instances
            model = new ModelClass(self);
            view = new ViewClass(self);
            controller = new ControllerClass(self);

            // disable global spell checking while this application is active
            win.on('show', function () {
                spellState = $(document.body).attr('spellcheck');
                $(document.body).attr('spellcheck', false);
            });
            win.on('beforehide', function () {
                if (spellState) {
                    $(document.body).attr('spellcheck', spellState);
                } else {
                    $(document.body).removeAttr('spellcheck');
                }
            });

            // bug 29711: update URL when applications becomes active
            win.on('show', updateUrl);

            // launch application per URL
            if (actionSource === 'url') {
                FilesAPI.get(launchOptions.file).done(function (fileDescriptor) {
                    // get complete file descriptor asynchronously and save to local file descriptor
                    file = fileDescriptor;
                    // resolve app initialization and continue app launching
                    initDef.resolve();
                }).fail(function (message) {
                    var cause = Utils.getStringOption(message, 'error', 'unknown');
                    Utils.error('BaseApplication.setLauncher(): Failed loading file descriptor from Files API. Error code: ' + cause);
                    initDef.reject(message);
                });
            } else { // normal launching with the given file descriptor from launchOptions
                initDef.resolve();
            }

            // call initialization handlers before showing the application window
            // in order to get the 'open' event of the window at all, it must be shown (also without file).
            // in case of URL launching, wait until the complete file descriptor is loaded successfully from FilesAPI
            initDef.done(function () {

                win.show(function () {

                    var // the promise returned by the import handler
                        handlerPromise = null;

                    // log the time needed to construct the application window and MVC instances
                    logLaunchDuration('application construction');

                    // prepare view for importing
                    view.hideBeforeImport();
                    logLaunchDuration('GUI preparation');

                    // invoke the import handler
                    handlerPromise = importHandler.call(self);

                    // return view to idle state
                    handlerPromise.always(function () {
                        logLaunchDuration('document import');
                        view.showAfterImport();
                        self.getWindow().nodes.outer.attr('data-doc-loaded', true);
                        updateTitle();
                        logLaunchDuration('GUI initialization');
                    });

                    // resolve the global import Deferred object on success, unless the
                    // application is already closing (import canceled)
                    handlerPromise.done(function (result) {
                        if (!self.isInQuit()) { importDef.resolve(result); }
                    });

                    // handle import failures, unless the  application is already closing (import canceled)
                    handlerPromise.fail(function (result) {
                        var cause = Utils.getStringOption(result, 'cause', 'unknown');
                        Utils.error('BaseApplication.launch(): Importing document "' + self.getFullFileName() + '" failed. Error code: "' + cause + '".');
                        // do not show any messages nor trigger events, if application is shutting down
                        if (!self.isInQuit()) { importDef.reject(result); }
                    });

                    // log import duration
                    handlerPromise.always(function () {
                        logLaunchDuration('postprocessing');
                        launchEndTime = _.now();
                    });
                });

            });

            // handle errors while initilizaling the application
            initDef.fail(function (message) {
                var cause = Utils.getStringOption(message, 'error', 'unknown');
                Notifications.yell('error', cause);
                Utils.error('BaseApplication.setLauncher(): initialization failed. Error code: "' + cause + '".');
                self.quit();
            });

            // Do not return a Deferred object that waits for the import process, application
            // must be registered synchronously, otherwise it is possible to open multiple
            // application instances for the same document.
        });

        // set quit callback invoked from base class (app.quit())
        this.setQuit(function () {

            var // store original 'quit()' method (needs to be restored after resuming)
                origQuitMethod = self.quit;

            // create the result deferred (rejecting means resume application)
            currentQuitDef = $.Deferred();

            // override 'quit()' method to prevent repeated/recursive calls while in quit mode
            self.quit = function () { return currentQuitDef.promise(); };

            // call all before-quit handlers, rejecting one will resume application
            callHandlers(beforeQuitHandlers)
            .done(function () {

                // Abort all running AJAX requests and other asynchronous code to prevent JS
                // errors from callbacks when the Deferred objects resolve/reject normally.
                // Will be done before invoking the quit handlers, to prevent race conditions
                // such as sending an AJAX request right after closing the document on the
                // server, which would lead to server exceptions, timeouts, etc. Each aborted
                // promise removes itself from the array. A single abort() call inside the
                // while-loop may abort other dependent promises as well.
                while (pendingPromises.length > 0) { pendingPromises[0].abort(); }

                // Bug 32989: Create a local copy of the quit handler array, and clear the
                // original array, before executing the quit handlers. This allows a quit
                // handler to register other special quit handlers that will be executed e.g.
                // when calling app.executeQuitHandlers() from logout while the quit handlers are
                // still running. Execute all quit handlers, do not care about the result.
                callHandlers(quitHandlers.splice(0))
                .always(function () {

                    // fail-safety: quit handlers should not leave pending requests
                    if (pendingPromises.length > 0) {
                        Utils.warn('BaseApplication.quit(): found pending promises after quit handlers');
                        while (pendingPromises.length > 0) { pendingPromises[0].abort(); }
                    }
                    pendingPromises = null;

                    // always resolve (really close the application), regardless
                    // of the result of the quit handlers
                    currentQuitDef.resolve();
                });
            })
            .fail(function () {
                // resume to running state of application
                self.quit = origQuitMethod;
                currentQuitDef.reject();
                currentQuitDef = null;
            });

            return currentQuitDef.promise();
        });

        // prevent usage of these methods in derived classes
        delete this.setLauncher;
        delete this.setQuit;

        // destroy MVC instances after core 'quit' event (after window has been hidden)
        this.on('quit', function () {

            $(window).off('unload', unloadHandler);
            // trigger listeners before destroying the MVC instances
            self.trigger('docs:destroy');

            // destroy singletons
            _.each(singletons, function (singleton) {
                if ((singleton instanceof BaseObject) && !singleton.destroyed) {
                    singleton.destroy();
                }
            });

            // destroy MVC instances
            controller.destroy();
            view.destroy();
            model.destroy();

            // bug 32948: 'currentQuitDef' must not be cleared here to keep the application in 'quitting' state, needed
            // in case another app.quit() will be called, e.g. from 'Reload document' after internal error while quitting
            self = model = view = controller = importHandler = flushHandler = beforeQuitHandlers = quitHandlers = null;
        });

    } // class BaseApplication

    // constants --------------------------------------------------------------

    /**
     * The name of the document filter server module.
     */
    BaseApplication.FILTER_MODULE_NAME = 'oxodocumentfilter';

    /**
     * The name of the document converter server module.
     */
    BaseApplication.CONVERTER_MODULE_NAME = 'oxodocumentconverter';

    // static methods ---------------------------------------------------------

    /**
     * Creates a static launcher that has to be returned by the main module of
     * each application type. The launcher tries to find a running application
     * which is working on a file described in the launch options passed to the
     * launcher. If no such application exists, it creates and returns a new
     * application object.
     *
     * @param {String} moduleName
     *  The application type identifier.
     *
     * @param {Function} ApplicationClass
     *  The constructor function of the application mix-in class that will
     *  extend the core application object. Receives the launch options passed
     *  to the launcher as first parameter.
     *
     * @param {Object} [appOptions]
     *  Optional parameters controlling the creation of the ox.ui.App base
     *  class instance:
     *  @param {String} [appOptions.icon]
     *      If specified, the CSS class name of a Bootstrap icon that will be
     *      shown in the top level launcher tab next to the application title.
     *  @param {Boolean} [options.search=false]
     *      Whether the application window will contain and support the global
     *      search tool bar.
     *  @param {Boolean} [options.chromeless=false]
     *      Whether to hide the main window tool bar attached to the left or
     *      bottom border.
     *
     * @returns {Object}
     *  The launcher object expected by the ox.launch() method.
     */
    BaseApplication.createLauncher = function (moduleName, ApplicationClass, appOptions) {

        var // the icon shown in the top bar launcher
            icon = Utils.getStringOption(appOptions, 'icon'),

            urlFileDescriptor = { id: _.url.hash('id'), folder_id: _.url.hash('folder') };

        // executed when a new application will be launched via ox.launch()
        function launchApp(launchOptions) {

            // Load from URL: if no launch options are given, check if we have 'simple' file descriptor (file id, folder id)
            // in the URL hash params and build new simple launch options from it
            if (!launchOptions && $.isNumeric(urlFileDescriptor.id) && $.isNumeric(urlFileDescriptor.folder_id)) {
                launchOptions = { action: 'load', file: urlFileDescriptor, actionSource: 'url'};
            }

            var // get file descriptor from options
                file = Utils.getObjectOption(launchOptions, 'file', null),
                // find running application for the specified document
                runningApps = file ? ox.ui.App.get(moduleName).filter(isRunningApplication) : [],
                // the new application instance
                app = null;

            // returns whether the passed application instance works on the exact
            // document file contained in the launch options
            function isRunningApplication(app) {
                var appFile = app.getFileDescriptor();
                return _.isObject(appFile) && Utils.hasEqualProperties(file, appFile, ['source', 'folder_id', 'id', 'attached']);
            }

            // return running application if existing
            if (runningApps.length > 1) {
                Utils.warn('BaseApplication.launchApp(): found multiple applications for the same file.');
            }
            if (runningApps.length > 0) {
                return runningApps[0];
            }

            // no running application: create and initialize a new application object
            app = ox.ui.createApp({
                name: moduleName,
                closable: true,
                userContent: icon.length > 0,
                userContentIcon: icon,
                userContentClass: getDocumentType(moduleName) + '-launcher'
            });

            // call mix-in constructor
            ApplicationClass.call(app, launchOptions, appOptions);

            return app;
        }

        // build the complete CSS classes for the application icon
        icon = _.isString(icon) ? Forms.getIconClasses(icon) : '';

        // Bug 28664: remove all save points before they are checked to decide
        // whether to show the 'unsaved documents' dialog
        ext.point('io.ox/core/logout').extend({
            id: moduleName + '/logout/before',
            index: 'first', // run before the save points are checked by the core
            logout: function () {
                _.each(ox.ui.App.get(moduleName), function (app) {
                    if (!app.hasUnsavedChanges()) { app.removeRestorePoint(); }
                });
            }
        });

        // listen to user logout and notify all running applications
        ext.point('io.ox/core/logout').extend({
            id: moduleName + '/logout/quit',
            index: 'last', // run after the dialog (not called if logout was rejected)
            logout: function () {
                var promises = _.map(ox.ui.App.get(moduleName), function (app) {
                    // bug 33152: ignore the result of the quit handlers (rejected promise cancels logout)
                    var def = $.Deferred();
                    app.executeQuitHandlers().always(function () { def.resolve(); });
                    return def.promise();
                });
                return $.when.apply($, promises);
            }
        });

        // ox.launch() expects an object with the method getApp()
        return { getApp: launchApp };
    };

    // exports ================================================================

    return _.makeExtendable(BaseApplication);

});
