/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2014 Open-Xchange Inc., Tarrytown, NY, USA. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define(['io.ox/office/spreadsheet/utils/sheetutils', 'io.ox/office/tk/utils'], function (SheetUtils, Utils) {

    'use strict';

    describe('OX Spreadsheet', function () {
        describe('SheetUtils module', function () {

            it('should exist', function () {
                expect(SheetUtils).to.be.an('object');
            });

            // private helpers ================================================

            var // convenience shortcuts
                c = SheetUtils.parseCellName,
                cl = function (str) { return _(str.split(/\s+/)).map(c); },
                r = SheetUtils.parseRangeName,
                rl = function (str) { return _(str.split(/\s+/)).map(r); },
                i = function (f, l) { return { first: f, last: l }; };

            function unorderedRangesMatcher(expectedRanges) {
                expectedRanges = _.chain(expectedRanges).map(SheetUtils.getRangeName).sort().join().value();
                return function (resultRanges) {
                    resultRanges = _.chain(resultRanges).map(SheetUtils.getRangeName).sort().join().value();
                    return expectedRanges === resultRanges;
                };
            }

            // constants ======================================================

            describe('constant "MAX_FILL_CELL_COUNT"', function () {
                it('should exist', function () {
                    expect(SheetUtils.MAX_FILL_CELL_COUNT).to.be.a('number');
                    expect(SheetUtils.MAX_FILL_CELL_COUNT).to.be.above(0);
                });
            });

            describe('constant "MAX_AUTOFILL_COL_ROW_COUNT"', function () {
                it('should exist', function () {
                    expect(SheetUtils.MAX_AUTOFILL_COL_ROW_COUNT).to.be.a('number');
                    expect(SheetUtils.MAX_AUTOFILL_COL_ROW_COUNT).to.be.above(0);
                });
            });

            describe('constant "MAX_MERGED_RANGES_COUNT"', function () {
                it('should exist', function () {
                    expect(SheetUtils.MAX_MERGED_RANGES_COUNT).to.be.a('number');
                    expect(SheetUtils.MAX_MERGED_RANGES_COUNT).to.be.above(0);
                });
            });

            describe('constant "MAX_UNMERGE_CELL_COUNT"', function () {
                it('should exist', function () {
                    expect(SheetUtils.MAX_UNMERGE_CELL_COUNT).to.be.a('number');
                    expect(SheetUtils.MAX_UNMERGE_CELL_COUNT).to.be.above(0);
                });
            });

            describe('constant "MAX_CHANGE_ROWS_COUNT"', function () {
                it('should exist', function () {
                    expect(SheetUtils.MAX_CHANGE_ROWS_COUNT).to.be.a('number');
                    expect(SheetUtils.MAX_CHANGE_ROWS_COUNT).to.be.above(0);
                });
            });

            describe('constant "MAX_SELECTION_CELL_COUNT"', function () {
                it('should exist', function () {
                    expect(SheetUtils.MAX_SELECTION_CELL_COUNT).to.be.a('number');
                    expect(SheetUtils.MAX_SELECTION_CELL_COUNT).to.be.above(0);
                });
            });

            describe('constant "MAX_LENGTH_STANDARD_CELL"', function () {
                it('should exist', function () {
                    expect(SheetUtils.MAX_LENGTH_STANDARD_CELL).to.be.a('number');
                    expect(SheetUtils.MAX_LENGTH_STANDARD_CELL).to.be.above(0);
                });
            });

            describe('constant "MAX_LENGTH_STANDARD_EDIT"', function () {
                it('should exist', function () {
                    expect(SheetUtils.MAX_LENGTH_STANDARD_EDIT).to.be.a('number');
                    expect(SheetUtils.MAX_LENGTH_STANDARD_EDIT).to.be.above(0);
                });
            });

            describe('constant "MIN_CELL_SIZE"', function () {
                it('should exist', function () {
                    expect(SheetUtils.MIN_CELL_SIZE).to.be.a('number');
                    expect(SheetUtils.MIN_CELL_SIZE).to.be.above(0);
                });
            });

            describe('constant "MIN_ZOOM"', function () {
                it('should exist', function () {
                    expect(SheetUtils.MIN_ZOOM).to.be.a('number');
                    expect(SheetUtils.MIN_ZOOM).to.be.above(0);
                    expect(SheetUtils.MIN_ZOOM).to.be.below(1);
                });
            });

            describe('constant "MAX_ZOOM"', function () {
                it('should exist', function () {
                    expect(SheetUtils.MAX_ZOOM).to.be.a('number');
                    expect(SheetUtils.MAX_ZOOM).to.be.above(1);
                });
            });

            describe('constant "MULTI_SELECTION"', function () {
                it('should exist', function () {
                    expect(SheetUtils.MULTI_SELECTION).to.be.a('boolean');
                });
            });

            // parser methods =================================================

            // Parser methods must be checked first; they are used by all
            // remaining tests in this file for convenience, for example
            // "c('A2')" instead of "[0,1]"; or "r('A2:C4')" instead of
            // "{start:[0,1],end:[2,3]}".

            describe('method "parseColName"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('parseColName');
                });
                it('should return the correct column index', function () {
                    expect(SheetUtils.parseColName('A')).to.equal(0);
                    expect(SheetUtils.parseColName('B')).to.equal(1);
                    expect(SheetUtils.parseColName('C')).to.equal(2);
                    expect(SheetUtils.parseColName('Z')).to.equal(25);
                    expect(SheetUtils.parseColName('AA')).to.equal(26);
                    expect(SheetUtils.parseColName('AB')).to.equal(27);
                    expect(SheetUtils.parseColName('AC')).to.equal(28);
                    expect(SheetUtils.parseColName('AZ')).to.equal(51);
                    expect(SheetUtils.parseColName('BA')).to.equal(52);
                    expect(SheetUtils.parseColName('CA')).to.equal(78);
                    expect(SheetUtils.parseColName('ZZ')).to.equal(701);
                    expect(SheetUtils.parseColName('AAA')).to.equal(702);
                    expect(SheetUtils.parseColName('ZZZ')).to.equal(18277);
                    expect(SheetUtils.parseColName('AAAA')).to.equal(18278);
                    expect(SheetUtils.parseColName('ZZZZ')).to.equal(475253);
                });
                it('should accept lower-case and mixed-case column names', function () {
                    expect(SheetUtils.parseColName('a')).to.equal(0);
                    expect(SheetUtils.parseColName('z')).to.equal(25);
                    expect(SheetUtils.parseColName('aa')).to.equal(26);
                    expect(SheetUtils.parseColName('aA')).to.equal(26);
                    expect(SheetUtils.parseColName('Aa')).to.equal(26);
                });
                it('should return -1 for invalid column names', function () {
                    expect(SheetUtils.parseColName('')).to.equal(-1);
                    expect(SheetUtils.parseColName('0')).to.equal(-1);
                    expect(SheetUtils.parseColName('$')).to.equal(-1);
                    expect(SheetUtils.parseColName(' A ')).to.equal(-1);
                    expect(SheetUtils.parseColName('AAAAA')).to.equal(-1);
                });
            });

            // ----------------------------------------------------------------

            describe('method "parseRowName"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('parseRowName');
                });
                it('should return the correct row index', function () {
                    expect(SheetUtils.parseRowName('1')).to.equal(0);
                    expect(SheetUtils.parseRowName('2')).to.equal(1);
                    expect(SheetUtils.parseRowName('3')).to.equal(2);
                    expect(SheetUtils.parseRowName('9')).to.equal(8);
                    expect(SheetUtils.parseRowName('10')).to.equal(9);
                    expect(SheetUtils.parseRowName('11')).to.equal(10);
                    expect(SheetUtils.parseRowName('99999999')).to.equal(99999998);
                });
                it('should accept leading zero characters', function () {
                    expect(SheetUtils.parseRowName('01')).to.equal(0);
                    expect(SheetUtils.parseRowName('0002')).to.equal(1);
                    expect(SheetUtils.parseRowName('099999999')).to.equal(99999998);
                });
                it('should return -1 for invalid row names', function () {
                    expect(SheetUtils.parseRowName('')).to.equal(-1);
                    expect(SheetUtils.parseRowName('0')).to.equal(-1);
                    expect(SheetUtils.parseRowName('-1')).to.equal(-1);
                    expect(SheetUtils.parseRowName('A')).to.equal(-1);
                    expect(SheetUtils.parseRowName('$')).to.equal(-1);
                    expect(SheetUtils.parseRowName(' 1 ')).to.equal(-1);
                    expect(SheetUtils.parseRowName('100000000')).to.equal(-1);
                });
            });

            // ----------------------------------------------------------------

            describe('method "parseCellName"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('parseCellName');
                });
                it('should return the correct cell address', function () {
                    expect(SheetUtils.parseCellName('A1')).to.deep.equal([0, 0]);
                    expect(SheetUtils.parseCellName('B1')).to.deep.equal([1, 0]);
                    expect(SheetUtils.parseCellName('C1')).to.deep.equal([2, 0]);
                    expect(SheetUtils.parseCellName('A2')).to.deep.equal([0, 1]);
                    expect(SheetUtils.parseCellName('A3')).to.deep.equal([0, 2]);
                    expect(SheetUtils.parseCellName('ZZZZ99999999')).to.deep.equal([475253, 99999998]);
                });
                it('should accept lower-case columns and leading zero characters', function () {
                    expect(SheetUtils.parseCellName('aa010')).to.deep.equal([26, 9]);
                    expect(SheetUtils.parseCellName('aA0010')).to.deep.equal([26, 9]);
                    expect(SheetUtils.parseCellName('Aa00010')).to.deep.equal([26, 9]);
                });
                it('should return null for invalid cell names', function () {
                    expect(SheetUtils.parseCellName('')).to.be['null'];
                    expect(SheetUtils.parseCellName('A')).to.be['null'];
                    expect(SheetUtils.parseCellName('1')).to.be['null'];
                    expect(SheetUtils.parseCellName('$')).to.be['null'];
                    expect(SheetUtils.parseCellName('$A$1')).to.be['null'];
                    expect(SheetUtils.parseCellName('A0')).to.be['null'];
                    expect(SheetUtils.parseCellName('AAAAA1')).to.be['null'];
                    expect(SheetUtils.parseCellName('A100000000')).to.be['null'];
                });
            });

            // ----------------------------------------------------------------

            describe('method "parseRangeName"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('parseRangeName');
                });
                it('should return the correct range address', function () {
                    expect(SheetUtils.parseRangeName('A2:C4')).to.deep.equal({ start: [0, 1], end: [2, 3] });
                    expect(SheetUtils.parseRangeName('A1:ZZZZ99999999')).to.deep.equal({ start: [0, 0], end: [475253, 99999998] });
                });
                it('should return the adjusted range address', function () {
                    expect(SheetUtils.parseRangeName('C2:A4')).to.deep.equal({ start: [0, 1], end: [2, 3] });
                    expect(SheetUtils.parseRangeName('A4:C2')).to.deep.equal({ start: [0, 1], end: [2, 3] });
                    expect(SheetUtils.parseRangeName('C4:A2')).to.deep.equal({ start: [0, 1], end: [2, 3] });
                });
                it('should accept lower-case columns and leading zero characters', function () {
                    expect(SheetUtils.parseRangeName('a01:aa010')).to.deep.equal({ start: [0, 0], end: [26, 9] });
                    expect(SheetUtils.parseRangeName('A001:aA0010')).to.deep.equal({ start: [0, 0], end: [26, 9] });
                });
                it('should return null for invalid range names', function () {
                    expect(SheetUtils.parseRangeName('')).to.be['null'];
                    expect(SheetUtils.parseRangeName('A')).to.be['null'];
                    expect(SheetUtils.parseRangeName('1')).to.be['null'];
                    expect(SheetUtils.parseRangeName('A1')).to.be['null'];
                    expect(SheetUtils.parseRangeName('$')).to.be['null'];
                    expect(SheetUtils.parseRangeName('$A$1:$A$1')).to.be['null'];
                    expect(SheetUtils.parseRangeName('A0:A1')).to.be['null'];
                    expect(SheetUtils.parseRangeName('AAAAA1:AAAAA1')).to.be['null'];
                    expect(SheetUtils.parseRangeName('A100000000:A100000000')).to.be['null'];
                });
            });

            // methods ========================================================

            describe('method "adjustRange"', function () {

                var range1 = { start: [0, 1], end: [2, 3] },
                    range2 = { start: [2, 1], end: [0, 3] },
                    range3 = { start: [0, 3], end: [2, 1] },
                    range4 = { start: [2, 3], end: [0, 1] };

                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('adjustRange');
                });
                it('should return correct result', function () {
                    expect(SheetUtils.adjustRange(range1)).to.deep.equal({ start: [0, 1], end: [2, 3] });
                    expect(SheetUtils.adjustRange(range2)).to.deep.equal({ start: [0, 1], end: [2, 3] });
                    expect(SheetUtils.adjustRange(range3)).to.deep.equal({ start: [0, 1], end: [2, 3] });
                    expect(SheetUtils.adjustRange(range4)).to.deep.equal({ start: [0, 1], end: [2, 3] });
                });
                it('should modify argument in-place', function () {
                    expect(range1).to.deep.equal({ start: [0, 1], end: [2, 3] });
                    expect(range2).to.deep.equal({ start: [0, 1], end: [2, 3] });
                    expect(range3).to.deep.equal({ start: [0, 1], end: [2, 3] });
                    expect(range4).to.deep.equal({ start: [0, 1], end: [2, 3] });
                });
            });

            // ----------------------------------------------------------------

            describe('method "getAdjustedRange"', function () {

                var range1 = { start: [0, 1], end: [2, 3] },
                    range2 = { start: [2, 1], end: [0, 3] },
                    range3 = { start: [0, 3], end: [2, 1] },
                    range4 = { start: [2, 3], end: [0, 1] };

                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('getAdjustedRange');
                });
                it('should return correct result', function () {
                    expect(SheetUtils.getAdjustedRange(range1)).to.deep.equal({ start: [0, 1], end: [2, 3] });
                    expect(SheetUtils.getAdjustedRange(range2)).to.deep.equal({ start: [0, 1], end: [2, 3] });
                    expect(SheetUtils.getAdjustedRange(range3)).to.deep.equal({ start: [0, 1], end: [2, 3] });
                    expect(SheetUtils.getAdjustedRange(range4)).to.deep.equal({ start: [0, 1], end: [2, 3] });
                });
                it('should not modify argument in-place', function () {
                    expect(range1).to.deep.equal({ start: [0, 1], end: [2, 3] });
                    expect(range2).to.deep.equal({ start: [2, 1], end: [0, 3] });
                    expect(range3).to.deep.equal({ start: [0, 3], end: [2, 1] });
                    expect(range4).to.deep.equal({ start: [2, 3], end: [0, 1] });
                });
            });

            // ----------------------------------------------------------------

            describe('method "getColCount"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('getColCount');
                });
                it('should return the correct count', function () {
                    expect(SheetUtils.getColCount(r('A2:A3'))).to.equal(1);
                    expect(SheetUtils.getColCount(r('C2:C3'))).to.equal(1);
                    expect(SheetUtils.getColCount(r('C2:D3'))).to.equal(2);
                    expect(SheetUtils.getColCount(r('C2:E3'))).to.equal(3);
                });
            });

            // ----------------------------------------------------------------

            describe('method "getRowCount"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('getRowCount');
                });
                it('should return the correct count', function () {
                    expect(SheetUtils.getRowCount(r('B1:C1'))).to.equal(1);
                    expect(SheetUtils.getRowCount(r('B3:C3'))).to.equal(1);
                    expect(SheetUtils.getRowCount(r('B3:C4'))).to.equal(2);
                    expect(SheetUtils.getRowCount(r('B3:C5'))).to.equal(3);
                });
            });

            // ----------------------------------------------------------------

            describe('method "getIndexCount"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('getIndexCount');
                });
                it('should return the correct coumn count', function () {
                    expect(SheetUtils.getIndexCount(r('B1:C1'), true)).to.equal(2);
                });
                it('should return the correct row count', function () {
                    expect(SheetUtils.getIndexCount(r('B1:C1'), false)).to.equal(1);
                });
            });

            // ----------------------------------------------------------------

            describe('method "getCellCount"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('getCellCount');
                });
                it('should return the correct count', function () {
                    expect(SheetUtils.getCellCount(r('A1:A1'))).to.equal(1);
                    expect(SheetUtils.getCellCount(r('C3:C3'))).to.equal(1);
                    expect(SheetUtils.getCellCount(r('C3:D3'))).to.equal(2);
                    expect(SheetUtils.getCellCount(r('C3:E3'))).to.equal(3);
                    expect(SheetUtils.getCellCount(r('C3:C4'))).to.equal(2);
                    expect(SheetUtils.getCellCount(r('C3:C5'))).to.equal(3);
                    expect(SheetUtils.getCellCount(r('C3:D4'))).to.equal(4);
                    expect(SheetUtils.getCellCount(r('C3:E5'))).to.equal(9);
                });
            });

            // ----------------------------------------------------------------

            describe('method "rangeContainsCol"', function () {
                var range = r('C2:D9');
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('rangeContainsCol');
                });
                it('should return false for columns left of the range', function () {
                    expect(SheetUtils.rangeContainsCol(range, 0)).to.be['false'];
                    expect(SheetUtils.rangeContainsCol(range, 1)).to.be['false'];
                });
                it('should return true for columns inside the range', function () {
                    expect(SheetUtils.rangeContainsCol(range, 2)).to.be['true'];
                    expect(SheetUtils.rangeContainsCol(range, 3)).to.be['true'];
                });
                it('should return false for columns right of the range', function () {
                    expect(SheetUtils.rangeContainsCol(range, 4)).to.be['false'];
                    expect(SheetUtils.rangeContainsCol(range, 9)).to.be['false'];
                });
            });

            // ----------------------------------------------------------------

            describe('method "rangeContainsRow"', function () {
                var range = r('B3:I4');
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('rangeContainsRow');
                });
                it('should return false for rows above the range', function () {
                    expect(SheetUtils.rangeContainsRow(range, 0)).to.be['false'];
                    expect(SheetUtils.rangeContainsRow(range, 1)).to.be['false'];
                });
                it('should return true for rows inside the range', function () {
                    expect(SheetUtils.rangeContainsRow(range, 2)).to.be['true'];
                    expect(SheetUtils.rangeContainsRow(range, 3)).to.be['true'];
                });
                it('should return false for rows below the range', function () {
                    expect(SheetUtils.rangeContainsRow(range, 4)).to.be['false'];
                    expect(SheetUtils.rangeContainsRow(range, 9)).to.be['false'];
                });
            });

            // ----------------------------------------------------------------

            describe('method "rangeContainsIndex"', function () {
                var range = r('C2:C2');
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('rangeContainsIndex');
                });
                it('should check column indexes', function () {
                    expect(SheetUtils.rangeContainsIndex(range, 1, true)).to.be['false'];
                    expect(SheetUtils.rangeContainsIndex(range, 2, true)).to.be['true'];
                    expect(SheetUtils.rangeContainsIndex(range, 3, true)).to.be['false'];
                });
                it('should check row indexes', function () {
                    expect(SheetUtils.rangeContainsIndex(range, 0, false)).to.be['false'];
                    expect(SheetUtils.rangeContainsIndex(range, 1, false)).to.be['true'];
                    expect(SheetUtils.rangeContainsIndex(range, 2, false)).to.be['false'];
                });
            });

            // ----------------------------------------------------------------

            describe('method "rangeContainsCell"', function () {

                var boundRange = r('B4:D6');

                function runTestsForAddresses(addresses, result) {
                    _(cl(addresses)).each(function (address) {
                        expect(SheetUtils.rangeContainsCell(boundRange, address)).to.be[result];
                    });
                }

                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('rangeContainsCell');
                });
                it('should return false for cells outside the range', function () {
                    runTestsForAddresses('A3 B3 C3 D3 E3 A4 E4 A5 E5 A6 E6 A7 B7 C7 D7 E7', false);
                });
                it('should return true for cells inside the range', function () {
                    runTestsForAddresses('B4 C4 D4 B5 C5 D5 B6 C6 D6', true);
                });
            });

            // ----------------------------------------------------------------

            describe('method "rangeContainsOuterCell"', function () {

                var boundRange = r('B4:D6');

                function runTestsForAddresses(addresses, result) {
                    _(cl(addresses)).each(function (address) {
                        expect(SheetUtils.rangeContainsOuterCell(boundRange, address)).to.be[result];
                    });
                }

                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('rangeContainsOuterCell');
                });
                it('should return false for cells outside the range', function () {
                    runTestsForAddresses('A3 B3 C3 D3 E3 A4 E4 A5 E5 A6 E6 A7 B7 C7 D7 E7', false);
                });
                it('should return true for cells at the range border', function () {
                    runTestsForAddresses('B4 C4 D4 B5 D5 B6 C6 D6', true);
                });
                it('should return true for cells inside the range', function () {
                    runTestsForAddresses('C5', false);
                });
            });

            // ----------------------------------------------------------------

            describe('method "rangeContainsRange"', function () {

                var boundRange = r('C3:F6');

                function runTestsForRanges(ranges, result) {
                    _(rl(ranges)).each(function (range) {
                        expect(SheetUtils.rangeContainsRange(boundRange, range)).to.be[result];
                    });
                }

                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('rangeContainsRange');
                });
                it('should return false for ranges outside the range', function () {
                    runTestsForRanges('A1:B8 G1:H8 A1:H2 A7:H8', false);
                });
                it('should return false for ranges partly overlapping the range', function () {
                    runTestsForRanges('A1:C8 F1:H8 A1:H3 A6:H8 D1:E3 D6:E8 A4:C5 F5:H6', false);
                });
                it('should return true for ranges inside the range', function () {
                    runTestsForRanges('C3:C3 C3:F3 F3:F3 F3:F6 F6:F6 C6:F6 C6:C6 C3:C6 C3:F6 D4:E5', true);
                });
            });

            // ----------------------------------------------------------------

            describe('method "rangesContainCol"', function () {

                var ranges = rl('B4:D6 G9:H10 D6:E7');

                function runTestsForIndexes(indexes, result) {
                    _(indexes).each(function (col) {
                        expect(SheetUtils.rangesContainCol(ranges, col)).to.be[result];
                    });
                }

                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('rangesContainCol');
                });
                it('should return false for columns outside the ranges', function () {
                    runTestsForIndexes([0, 5, 8, 9, 10], false);
                });
                it('should return true for columns inside the ranges', function () {
                    runTestsForIndexes([1, 2, 3, 4, 6, 7], true);
                });
            });

            // ----------------------------------------------------------------

            describe('method "rangesContainRow"', function () {

                var ranges = rl('B4:D6 G9:H10 D6:E7');

                function runTestsForIndexes(indexes, result) {
                    _(indexes).each(function (row) {
                        expect(SheetUtils.rangesContainRow(ranges, row)).to.be[result];
                    });
                }

                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('rangesContainRow');
                });
                it('should return false for rows outside the ranges', function () {
                    runTestsForIndexes([0, 1, 2, 7, 10, 11, 12], false);
                });
                it('should return true for rows inside the ranges', function () {
                    runTestsForIndexes([3, 4, 5, 6, 8, 9], true);
                });
            });

            // ----------------------------------------------------------------

            describe('method "rangesContainIndex"', function () {
                var ranges = rl('C2:C2 E4:E4');
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('rangesContainIndex');
                });
                it('should check column indexes', function () {
                    expect(SheetUtils.rangesContainIndex(ranges, 1, true)).to.be['false'];
                    expect(SheetUtils.rangesContainIndex(ranges, 2, true)).to.be['true'];
                    expect(SheetUtils.rangesContainIndex(ranges, 3, true)).to.be['false'];
                    expect(SheetUtils.rangesContainIndex(ranges, 4, true)).to.be['true'];
                    expect(SheetUtils.rangesContainIndex(ranges, 5, true)).to.be['false'];
                });
                it('should check row indexes', function () {
                    expect(SheetUtils.rangesContainIndex(ranges, 0, false)).to.be['false'];
                    expect(SheetUtils.rangesContainIndex(ranges, 1, false)).to.be['true'];
                    expect(SheetUtils.rangesContainIndex(ranges, 2, false)).to.be['false'];
                    expect(SheetUtils.rangesContainIndex(ranges, 3, false)).to.be['true'];
                    expect(SheetUtils.rangesContainIndex(ranges, 4, false)).to.be['false'];
                });
            });

            // ----------------------------------------------------------------

            describe('method "rangesContainCell"', function () {

                var ranges = rl('B4:D6 D6:E7');

                function runTestsForAddresses(addresses, result) {
                    _(cl(addresses)).each(function (address) {
                        expect(SheetUtils.rangesContainCell(ranges, address)).to.be[result];
                    });
                }

                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('rangesContainCell');
                });
                it('should return false for cells outside the ranges', function () {
                    runTestsForAddresses('A3 B3 C3 D3 E3 A4 E4 A5 E5 A6 F6 A7 B7 C7 F7 C8 D8 E8 F8', false);
                });
                it('should return true for cells inside the ranges', function () {
                    runTestsForAddresses('B4 C4 D4 B5 C5 D5 B6 C6 D6 E6 D7 E7', true);
                });
            });

            // ----------------------------------------------------------------

            describe('method "findFirstRange"', function () {

                var ranges = rl('B4:D6 D6:E7');

                function runTestsForAddresses(addresses, index) {
                    _(cl(addresses)).each(function (address) {
                        var result = SheetUtils.findFirstRange(ranges, address);
                        if (_.isNumber(index)) {
                            expect(result).to.deep.equal(ranges[index]);
                        } else {
                            expect(result).to.be['null'];
                        }
                    });
                }

                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('findFirstRange');
                });
                it('should return null for cells outside the ranges', function () {
                    runTestsForAddresses('A3 B3 C3 D3 E3 A4 E4 A5 E5 A6 F6 A7 B7 C7 F7 C8 D8 E8 F8', null);
                });
                it('should return first range in the list', function () {
                    runTestsForAddresses('B4 C4 D4 B5 C5 D5 B6 C6 D6', 0);
                });
                it('should return second range in the list', function () {
                    runTestsForAddresses('D7 E6 E7', 1);
                });
            });

            // ----------------------------------------------------------------

            describe('method "compareCells"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('compareCells');
                });
                it('should return 0 for equal addresses', function () {
                    expect(SheetUtils.compareCells(c('A3'), c('A3'))).to.equal(0);
                    expect(SheetUtils.compareCells(c('C1'), c('C1'))).to.equal(0);
                });
                it('should return negative value for addresses in order', function () {
                    expect(SheetUtils.compareCells(c('C3'), c('D3'))).to.be.below(0);
                    expect(SheetUtils.compareCells(c('C3'), c('C4'))).to.be.below(0);
                    expect(SheetUtils.compareCells(c('C3'), c('A4'))).to.be.below(0);
                });
                it('should return positive value for addresses in reversed order', function () {
                    expect(SheetUtils.compareCells(c('C3'), c('B3'))).to.be.above(0);
                    expect(SheetUtils.compareCells(c('C3'), c('C2'))).to.be.above(0);
                    expect(SheetUtils.compareCells(c('C3'), c('E2'))).to.be.above(0);
                });
            });

            // ----------------------------------------------------------------

            describe('method "getUniqueRanges"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('getUniqueRanges');
                });
                it('should not modify unique ranges', function () {
                    expect(SheetUtils.getUniqueRanges(rl('B3:C4 B3:C5 B2:C4'))).to.deep.equal(rl('B3:C4 B3:C5 B2:C4'));
                });
                it('should remove duplicate ranges', function () {
                    expect(SheetUtils.getUniqueRanges(rl('B3:C4 B3:C4 B3:C5 B2:C4 B3:C5'))).to.deep.equal(rl('B3:C4 B3:C5 B2:C4'));
                });
                it('should accept single range objects', function () {
                    expect(SheetUtils.getUniqueRanges(r('B3:C4'))).to.deep.equal(rl('B3:C4'));
                });
                it('should accept an empty array', function () {
                    expect(SheetUtils.getUniqueRanges([])).to.deep.equal([]);
                });
            });

            // ----------------------------------------------------------------

            describe('method "rangeOverlapsRange"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('rangeOverlapsRange');
                });
                it('should return false for distinct ranges', function () {
                    expect(SheetUtils.rangeOverlapsRange(r('B3:C4'), r('D3:E4'))).to.be['false'];
                    expect(SheetUtils.rangeOverlapsRange(r('D3:E4'), r('B3:C4'))).to.be['false'];
                    expect(SheetUtils.rangeOverlapsRange(r('B3:C4'), r('B5:C6'))).to.be['false'];
                    expect(SheetUtils.rangeOverlapsRange(r('B5:C6'), r('B3:C4'))).to.be['false'];
                });
                it('should return true for partly overlapping ranges', function () {
                    expect(SheetUtils.rangeOverlapsRange(r('B3:C4'), r('C4:D5'))).to.be['true'];
                    expect(SheetUtils.rangeOverlapsRange(r('B3:C4'), r('C2:D3'))).to.be['true'];
                    expect(SheetUtils.rangeOverlapsRange(r('B3:C4'), r('A2:B3'))).to.be['true'];
                    expect(SheetUtils.rangeOverlapsRange(r('B3:C4'), r('A4:B5'))).to.be['true'];
                });
                it('should return true for ranges containing each other', function () {
                    expect(SheetUtils.rangeOverlapsRange(r('B3:C4'), r('B3:B3'))).to.be['true'];
                    expect(SheetUtils.rangeOverlapsRange(r('B3:C4'), r('B3:B4'))).to.be['true'];
                    expect(SheetUtils.rangeOverlapsRange(r('B3:C4'), r('B3:C4'))).to.be['true'];
                    expect(SheetUtils.rangeOverlapsRange(r('B3:B4'), r('B3:C4'))).to.be['true'];
                    expect(SheetUtils.rangeOverlapsRange(r('B3:B3'), r('B3:C4'))).to.be['true'];
                });
            });

            // ----------------------------------------------------------------

            describe('method "rangesOverlapRanges"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('rangesOverlapRanges');
                });
                it('should return false for distinct ranges', function () {
                    expect(SheetUtils.rangesOverlapRanges(rl('A1:B2 B2:C3 C3:D4'), rl('C1:D1 A3:A4 D1:D2 A4:B4'))).to.be['false'];
                });
                it('should return true for overlapping ranges', function () {
                    expect(SheetUtils.rangesOverlapRanges(rl('A1:B2 B2:C3 C3:D4'), rl('C1:D1 A3:B4 D1:D2'))).to.be['true'];
                });
                it('should accept empty array', function () {
                    expect(SheetUtils.rangesOverlapRanges([], rl('C1:D1 A3:B4 D1:D2'))).to.be['false'];
                    expect(SheetUtils.rangesOverlapRanges(rl('A1:B2 B2:C3 C3:D4'), [])).to.be['false'];
                    expect(SheetUtils.rangesOverlapRanges([], [])).to.be['false'];
                });
                it('should accept single range address', function () {
                    expect(SheetUtils.rangesOverlapRanges(rl('A1:B2 B2:C3 C3:D4'), r('A3:B4'))).to.be['true'];
                    expect(SheetUtils.rangesOverlapRanges(r('A3:B4'), rl('A1:B2 B2:C3 C3:D4'))).to.be['true'];
                    expect(SheetUtils.rangesOverlapRanges(r('A1:B2'), r('B2:C3'))).to.be['true'];
                });
            });

            // ----------------------------------------------------------------

            describe('method "anyRangesOverlap"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('anyRangesOverlap');
                });
                it('should return false for distinct ranges', function () {
                    expect(SheetUtils.anyRangesOverlap(rl('B3:C4 D3:E4 B5:C6 D5:E6'))).to.be['false'];
                });
                it('should return true for overlapping ranges', function () {
                    expect(SheetUtils.anyRangesOverlap(rl('B3:C4 D3:E4 B5:D6 D5:E6 B3:C4'))).to.be['true'];
                    expect(SheetUtils.anyRangesOverlap(rl('B3:C4 D3:E4 B5:D6 D5:E6 C4:D5'))).to.be['true'];
                });
                it('should return false for an empty array', function () {
                    expect(SheetUtils.anyRangesOverlap([])).to.be['false'];
                });
            });

            // ----------------------------------------------------------------

            describe('method "getIntersectionRange"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('getIntersectionRange');
                });
                it('should return null for distinct ranges', function () {
                    expect(SheetUtils.getIntersectionRange(r('B3:C4'), r('D3:E4'))).to.be['null'];
                    expect(SheetUtils.getIntersectionRange(r('D3:E4'), r('B3:C4'))).to.be['null'];
                    expect(SheetUtils.getIntersectionRange(r('B3:C4'), r('B5:C6'))).to.be['null'];
                    expect(SheetUtils.getIntersectionRange(r('B5:C6'), r('B3:C4'))).to.be['null'];
                });
                it('should return intersection range for overlapping ranges', function () {
                    expect(SheetUtils.getIntersectionRange(r('B3:E5'), r('A2:B3'))).to.deep.equal(r('B3:B3'));
                    expect(SheetUtils.getIntersectionRange(r('B3:E5'), r('A2:B6'))).to.deep.equal(r('B3:B5'));
                    expect(SheetUtils.getIntersectionRange(r('B3:E5'), r('A4:C6'))).to.deep.equal(r('B4:C5'));
                    expect(SheetUtils.getIntersectionRange(r('B3:E5'), r('A4:F4'))).to.deep.equal(r('B4:E4'));
                    expect(SheetUtils.getIntersectionRange(r('B3:E5'), r('C1:D8'))).to.deep.equal(r('C3:D5'));
                    expect(SheetUtils.getIntersectionRange(r('B3:E5'), r('A1:F8'))).to.deep.equal(r('B3:E5'));
                    expect(SheetUtils.getIntersectionRange(r('B3:E5'), r('B3:E5'))).to.deep.equal(r('B3:E5'));
                    expect(SheetUtils.getIntersectionRange(r('B3:E5'), r('C3:D5'))).to.deep.equal(r('C3:D5'));
                    expect(SheetUtils.getIntersectionRange(r('B3:E5'), r('C4:C4'))).to.deep.equal(r('C4:C4'));
                });
            });

            // ----------------------------------------------------------------

            describe('method "getIntersectionRanges"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('getIntersectionRanges');
                });
                it('should return empty array for ranges outside the bounding range', function () {
                    expect(SheetUtils.getIntersectionRanges(rl('A1:B2 A2:A3 F1:G8'), r('B3:E5'))).to.be.empty;
                });
                it('should return intersection ranges', function () {
                    expect(SheetUtils.getIntersectionRanges(rl('A1:B2 C1:D8 F1:G8 A4:G4'), r('B3:E5'))).to.deep.equal(rl('C3:D5 B4:E4'));
                });
                it('should accept empty array', function () {
                    expect(SheetUtils.getIntersectionRanges([], r('B3:E5'))).to.be.empty;
                });
                it('should accept single range address', function () {
                    expect(SheetUtils.getIntersectionRanges(r('C1:D8'), r('B3:E5'))).to.deep.equal(rl('C3:D5'));
                });
            });

            // ----------------------------------------------------------------

            describe('method "iterateIntersectionRanges"', function () {

                function run(ranges, boundRange, options, breaker) {
                    return chai.runIterator(SheetUtils.iterateIntersectionRanges, SheetUtils, [ranges, boundRange, chai.ITERATOR, options], breaker);
                }

                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('iterateIntersectionRanges');
                });
                it('should not visit ranges outside the bounding range', function () {
                    var result = run(rl('A1:B2 A2:A3 F1:G8'), r('B3:E5'));
                    expect(result.runs).to.be.empty;
                });
                it('should visit all intersection ranges', function () {
                    var result = run(rl('A1:B2 C1:D8 F1:G8 A4:G4'), r('B3:E5'));
                    expect(result.args).to.deep.equal([[r('C3:D5'), r('C1:D8'), 1], [r('B4:E4'), r('A4:G4'), 3]]);
                });
                it('should accept empty array', function () {
                    var result = run([], r('B3:E5'));
                    expect(result.runs).to.be.empty;
                });
                it('should accept single range address', function () {
                    var result = run(r('C1:D8'), r('B3:E5'));
                    expect(result.args).to.deep.equal([[r('C3:D5'), r('C1:D8'), 0]]);
                });
                it('should visit all intersection ranges in reversed order', function () {
                    var result = run(rl('A1:B2 C1:D8 F1:G8 A4:G4'), r('B3:E5'), { reverse: true });
                    expect(result.args).to.deep.equal([[r('B4:E4'), r('A4:G4'), 3], [r('C3:D5'), r('C1:D8'), 1]]);
                });
                it('should bind specified context to the iterator function', function () {
                    var context = {},
                        result = run(rl('A1:B2 C1:D8 F1:G8 A4:G4'), r('B3:E5'), { context: context });
                    expect(result.runs[0].context).to.equal(context);
                    expect(result.runs[1].context).to.equal(context);
                });
                it('should return Utils.BREAK from iterator function', function () {
                    var result = run(rl('A1:B2 C1:D8 F1:G8 A4:G4'), r('B3:E5'), undefined, function () { return Utils.BREAK; });
                    expect(result.runs).to.have.length(1);
                    expect(result.ret).to.equal(Utils.BREAK);
                });
            });

            // ----------------------------------------------------------------

            describe('method "getBoundingRange"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('getBoundingRange');
                });
                it('should return the bounding range for a single range', function () {
                    expect(SheetUtils.getBoundingRange(r('C3:E5'))).to.deep.equal(r('C3:E5'));
                });
                it('should return the bounding range for a range list', function () {
                    expect(SheetUtils.getBoundingRange(rl('C3:E5 B7:C8 F2:G3'))).to.deep.equal(r('B2:G8'));
                });
                it('should accept multiple parameters', function () {
                    expect(SheetUtils.getBoundingRange(r('C3:E5'), r('B7:C8'), r('F2:G3'))).to.deep.equal(r('B2:G8'));
                    expect(SheetUtils.getBoundingRange(rl('C3:E5 B7:C8'), r('F2:G3'))).to.deep.equal(r('B2:G8'));
                });
                it('should accept missing parameters', function () {
                    expect(SheetUtils.getBoundingRange()).to.be['null'];
                    expect(SheetUtils.getBoundingRange([])).to.be['null'];
                    expect(SheetUtils.getBoundingRange([], [])).to.be['null'];
                });
            });

            // ----------------------------------------------------------------

            describe('method "getIntervalSize"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('getIntervalSize');
                });
                it('should return the correct interval size', function () {
                    expect(SheetUtils.getIntervalSize(i(0, 0))).to.be.equal(1);
                    expect(SheetUtils.getIntervalSize(i(0, 1))).to.be.equal(2);
                    expect(SheetUtils.getIntervalSize(i(0, 6))).to.be.equal(7);
                    expect(SheetUtils.getIntervalSize(i(0, 7))).to.be.equal(8);
                    expect(SheetUtils.getIntervalSize(i(1, 7))).to.be.equal(7);
                    expect(SheetUtils.getIntervalSize(i(6, 7))).to.be.equal(2);
                    expect(SheetUtils.getIntervalSize(i(7, 7))).to.be.equal(1);
                });
            });

            // ----------------------------------------------------------------

            describe('method "intervalContainsIndex"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('intervalContainsIndex');
                });
                it('should return false for indexes outside the interval', function () {
                    expect(SheetUtils.intervalContainsIndex(i(2, 4), 0)).to.be['false'];
                    expect(SheetUtils.intervalContainsIndex(i(2, 4), 1)).to.be['false'];
                    expect(SheetUtils.intervalContainsIndex(i(2, 4), 5)).to.be['false'];
                    expect(SheetUtils.intervalContainsIndex(i(2, 4), 6)).to.be['false'];
                });
                it('should return true for indexes inside the interval', function () {
                    expect(SheetUtils.intervalContainsIndex(i(2, 4), 2)).to.be['true'];
                    expect(SheetUtils.intervalContainsIndex(i(2, 4), 3)).to.be['true'];
                    expect(SheetUtils.intervalContainsIndex(i(2, 4), 4)).to.be['true'];
                });
            });

            // ----------------------------------------------------------------

            describe('method "intervalContainsInterval"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('intervalContainsInterval');
                });
                it('should return false for distinct intervals', function () {
                    expect(SheetUtils.intervalContainsInterval(i(2, 4), i(0, 1))).to.be['false'];
                    expect(SheetUtils.intervalContainsInterval(i(2, 4), i(5, 6))).to.be['false'];
                    expect(SheetUtils.intervalContainsInterval(i(2, 4), i(7, 8))).to.be['false'];
                    expect(SheetUtils.intervalContainsInterval(i(0, 1), i(2, 4))).to.be['false'];
                    expect(SheetUtils.intervalContainsInterval(i(5, 6), i(2, 4))).to.be['false'];
                    expect(SheetUtils.intervalContainsInterval(i(7, 8), i(2, 4))).to.be['false'];
                });
                it('should return false for partly overlapping intervals', function () {
                    expect(SheetUtils.intervalContainsInterval(i(2, 4), i(0, 2))).to.be['false'];
                    expect(SheetUtils.intervalContainsInterval(i(2, 4), i(1, 3))).to.be['false'];
                    expect(SheetUtils.intervalContainsInterval(i(2, 4), i(3, 5))).to.be['false'];
                    expect(SheetUtils.intervalContainsInterval(i(2, 4), i(4, 6))).to.be['false'];
                });
                it('should return true for intervals containing the second interval', function () {
                    expect(SheetUtils.intervalContainsInterval(i(2, 4), i(2, 2))).to.be['true'];
                    expect(SheetUtils.intervalContainsInterval(i(2, 4), i(2, 3))).to.be['true'];
                    expect(SheetUtils.intervalContainsInterval(i(2, 4), i(2, 4))).to.be['true'];
                    expect(SheetUtils.intervalContainsInterval(i(2, 4), i(3, 4))).to.be['true'];
                    expect(SheetUtils.intervalContainsInterval(i(2, 4), i(4, 4))).to.be['true'];
                });
                it('should return false for intervals contained by the second interval', function () {
                    expect(SheetUtils.intervalContainsInterval(i(2, 2), i(2, 4))).to.be['false'];
                    expect(SheetUtils.intervalContainsInterval(i(2, 3), i(2, 4))).to.be['false'];
                    expect(SheetUtils.intervalContainsInterval(i(3, 4), i(2, 4))).to.be['false'];
                    expect(SheetUtils.intervalContainsInterval(i(4, 4), i(2, 4))).to.be['false'];
                });
            });

            // ----------------------------------------------------------------

            describe('method "intervalOverlapsInterval"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('intervalOverlapsInterval');
                });
                it('should return false for distinct intervals', function () {
                    expect(SheetUtils.intervalOverlapsInterval(i(2, 4), i(0, 1))).to.be['false'];
                    expect(SheetUtils.intervalOverlapsInterval(i(2, 4), i(5, 6))).to.be['false'];
                    expect(SheetUtils.intervalOverlapsInterval(i(2, 4), i(7, 8))).to.be['false'];
                    expect(SheetUtils.intervalOverlapsInterval(i(0, 1), i(2, 4))).to.be['false'];
                    expect(SheetUtils.intervalOverlapsInterval(i(5, 6), i(2, 4))).to.be['false'];
                    expect(SheetUtils.intervalOverlapsInterval(i(7, 8), i(2, 4))).to.be['false'];
                });
                it('should return true for partly overlapping intervals', function () {
                    expect(SheetUtils.intervalOverlapsInterval(i(2, 4), i(0, 2))).to.be['true'];
                    expect(SheetUtils.intervalOverlapsInterval(i(2, 4), i(1, 3))).to.be['true'];
                    expect(SheetUtils.intervalOverlapsInterval(i(2, 4), i(3, 5))).to.be['true'];
                    expect(SheetUtils.intervalOverlapsInterval(i(2, 4), i(4, 6))).to.be['true'];
                });
                it('should return true for intervals containing each other', function () {
                    expect(SheetUtils.intervalOverlapsInterval(i(2, 4), i(2, 2))).to.be['true'];
                    expect(SheetUtils.intervalOverlapsInterval(i(2, 2), i(2, 4))).to.be['true'];
                    expect(SheetUtils.intervalOverlapsInterval(i(2, 4), i(2, 3))).to.be['true'];
                    expect(SheetUtils.intervalOverlapsInterval(i(2, 3), i(2, 4))).to.be['true'];
                    expect(SheetUtils.intervalOverlapsInterval(i(2, 4), i(2, 4))).to.be['true'];
                    expect(SheetUtils.intervalOverlapsInterval(i(2, 4), i(3, 4))).to.be['true'];
                    expect(SheetUtils.intervalOverlapsInterval(i(3, 4), i(2, 4))).to.be['true'];
                    expect(SheetUtils.intervalOverlapsInterval(i(2, 4), i(4, 4))).to.be['true'];
                    expect(SheetUtils.intervalOverlapsInterval(i(4, 4), i(2, 4))).to.be['true'];
                });
            });

            // ----------------------------------------------------------------

            describe('method "getUniqueIntervals"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('getUniqueIntervals');
                });
                it('should not modify unique intervals', function () {
                    expect(SheetUtils.getUniqueIntervals([i(3, 4), i(3, 5), i(2, 4)])).to.deep.equal([i(3, 4), i(3, 5), i(2, 4)]);
                });
                it('should remove duplicate intervals', function () {
                    expect(SheetUtils.getUniqueIntervals([i(3, 4), i(3, 4), i(3, 5), i(2, 4), i(3, 5)])).to.deep.equal([i(3, 4), i(3, 5), i(2, 4)]);
                });
                it('should accept single interval objects', function () {
                    expect(SheetUtils.getUniqueIntervals(i(3, 4))).to.deep.equal([i(3, 4)]);
                });
                it('should accept an empty array', function () {
                    expect(SheetUtils.getUniqueIntervals([])).to.deep.equal([]);
                });
            });

            // ----------------------------------------------------------------

            describe('method "getIntersectionInterval"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('getIntersectionInterval');
                });
                it('should return null for distinct intervals', function () {
                    expect(SheetUtils.getIntersectionInterval(i(2, 4), i(0, 1))).to.be['null'];
                    expect(SheetUtils.getIntersectionInterval(i(2, 4), i(5, 6))).to.be['null'];
                    expect(SheetUtils.getIntersectionInterval(i(2, 4), i(7, 8))).to.be['null'];
                    expect(SheetUtils.getIntersectionInterval(i(0, 1), i(2, 4))).to.be['null'];
                    expect(SheetUtils.getIntersectionInterval(i(5, 6), i(2, 4))).to.be['null'];
                    expect(SheetUtils.getIntersectionInterval(i(7, 8), i(2, 4))).to.be['null'];
                });
                it('should return intersection interval for overlapping intervals', function () {
                    expect(SheetUtils.getIntersectionInterval(i(2, 4), i(0, 2))).to.deep.equal(i(2, 2));
                    expect(SheetUtils.getIntersectionInterval(i(2, 4), i(1, 3))).to.deep.equal(i(2, 3));
                    expect(SheetUtils.getIntersectionInterval(i(2, 4), i(3, 5))).to.deep.equal(i(3, 4));
                    expect(SheetUtils.getIntersectionInterval(i(2, 4), i(4, 6))).to.deep.equal(i(4, 4));
                });
                it('should return intersection interval for intervals containing each other', function () {
                    expect(SheetUtils.getIntersectionInterval(i(2, 4), i(2, 2))).to.deep.equal(i(2, 2));
                    expect(SheetUtils.getIntersectionInterval(i(2, 2), i(2, 4))).to.deep.equal(i(2, 2));
                    expect(SheetUtils.getIntersectionInterval(i(2, 4), i(2, 3))).to.deep.equal(i(2, 3));
                    expect(SheetUtils.getIntersectionInterval(i(2, 3), i(2, 4))).to.deep.equal(i(2, 3));
                    expect(SheetUtils.getIntersectionInterval(i(2, 4), i(2, 4))).to.deep.equal(i(2, 4));
                    expect(SheetUtils.getIntersectionInterval(i(2, 4), i(3, 4))).to.deep.equal(i(3, 4));
                    expect(SheetUtils.getIntersectionInterval(i(3, 4), i(2, 4))).to.deep.equal(i(3, 4));
                    expect(SheetUtils.getIntersectionInterval(i(2, 4), i(4, 4))).to.deep.equal(i(4, 4));
                    expect(SheetUtils.getIntersectionInterval(i(4, 4), i(2, 4))).to.deep.equal(i(4, 4));
                });
            });

            // ----------------------------------------------------------------

            describe('method "getIntersectionIntervals"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('getIntersectionIntervals');
                });
                it('should return empty array for intervals outside the bounding interval', function () {
                    expect(SheetUtils.getIntersectionIntervals([i(0, 0), i(0, 1), i(1, 1), i(7, 7), i(7, 8), i(8, 8)], i(2, 6))).to.be.empty;
                });
                it('should return intersection intervals', function () {
                    expect(SheetUtils.getIntersectionIntervals([i(0, 1), i(1, 3), i(2, 4), i(5, 8)], i(2, 6))).to.deep.equal([i(2, 3), i(2, 4), i(5, 6)]);
                });
                it('should accept empty array', function () {
                    expect(SheetUtils.getIntersectionIntervals([], i(2, 6))).to.be.empty;
                });
                it('should accept single interval object instead of array', function () {
                    expect(SheetUtils.getIntersectionIntervals(i(1, 2), i(2, 6))).to.deep.equal([i(2, 2)]);
                });
            });

            // ----------------------------------------------------------------

            describe('method "getBoundingInterval"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('getBoundingInterval');
                });
                it('should return the bounding interval for a single interval', function () {
                    expect(SheetUtils.getBoundingInterval(i(3, 5))).to.deep.equal(i(3, 5));
                });
                it('should return the bounding interval for an interval list', function () {
                    expect(SheetUtils.getBoundingInterval([i(3, 5), i(7, 8), i(0, 0)])).to.deep.equal(i(0, 8));
                });
                it('should accept multiple parameters', function () {
                    expect(SheetUtils.getBoundingInterval(i(3, 5), i(7, 8), i(0, 0))).to.deep.equal(i(0, 8));
                    expect(SheetUtils.getBoundingInterval([i(3, 5), i(7, 8)], i(0, 0))).to.deep.equal(i(0, 8));
                });
                it('should accept missing parameters', function () {
                    expect(SheetUtils.getBoundingInterval()).to.be['null'];
                    expect(SheetUtils.getBoundingInterval([])).to.be['null'];
                    expect(SheetUtils.getBoundingInterval([], [])).to.be['null'];
                });
            });

            // ----------------------------------------------------------------

            describe('method "getUnifiedIntervals"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('getUnifiedIntervals');
                });
                it('should sort but not merge distinct intervals', function () {
                    expect(SheetUtils.getUnifiedIntervals([i(4, 5), i(1, 2), i(7, 8)])).to.deep.equal([i(1, 2), i(4, 5), i(7, 8)]);
                });
                it('should merge adjacent intervals', function () {
                    expect(SheetUtils.getUnifiedIntervals([i(3, 4), i(1, 2), i(5, 6), i(7, 8)])).to.deep.equal([i(1, 8)]);
                });
                it('should merge overlapping intervals', function () {
                    expect(SheetUtils.getUnifiedIntervals([i(3, 6), i(1, 3), i(3, 6), i(6, 8)])).to.deep.equal([i(1, 8)]);
                });
                it('should accept empty array', function () {
                    expect(SheetUtils.getUnifiedIntervals([])).to.be.empty;
                });
                it('should accept single interval object instead of array', function () {
                    expect(SheetUtils.getUnifiedIntervals(i(1, 2))).to.deep.equal([i(1, 2)]);
                });
                it('should not modify the passed parameter', function () {
                    var intervals = [i(4, 5), i(1, 2)];
                    SheetUtils.getUnifiedIntervals(intervals);
                    expect(intervals).to.deep.equal([i(4, 5), i(1, 2)]);
                });
            });

            // ----------------------------------------------------------------

            describe('method "getColInterval"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('getColInterval');
                });
                it('should return the column interval of a range', function () {
                    expect(SheetUtils.getColInterval(r('B3:B3'))).to.deep.equal(i(1, 1));
                    expect(SheetUtils.getColInterval(r('B3:C4'))).to.deep.equal(i(1, 2));
                    expect(SheetUtils.getColInterval(r('B3:D5'))).to.deep.equal(i(1, 3));
                    expect(SheetUtils.getColInterval(r('C4:D5'))).to.deep.equal(i(2, 3));
                    expect(SheetUtils.getColInterval(r('D5:D5'))).to.deep.equal(i(3, 3));
                });
            });

            // ----------------------------------------------------------------

            describe('method "getColIntervals"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('getColIntervals');
                });
                it('should sort but not merge distinct column intervals', function () {
                    expect(SheetUtils.getColIntervals(rl('E6:F7 B3:C4 H9:I10'))).to.deep.equal([i(1, 2), i(4, 5), i(7, 8)]);
                });
                it('should merge adjacent column intervals', function () {
                    expect(SheetUtils.getColIntervals(rl('D5:E6 B3:C4 F7:G8 H9:I10'))).to.deep.equal([i(1, 8)]);
                });
                it('should merge overlapping column intervals', function () {
                    expect(SheetUtils.getColIntervals(rl('D5:G8 B3:D5 D5:G8 G8:I10'))).to.deep.equal([i(1, 8)]);
                });
                it('should accept empty array', function () {
                    expect(SheetUtils.getColIntervals([])).to.be.empty;
                });
                it('should accept single range object instead of array', function () {
                    expect(SheetUtils.getColIntervals(r('B3:C4'))).to.deep.equal([i(1, 2)]);
                });
            });

            // ----------------------------------------------------------------

            describe('method "getRowInterval"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('getRowInterval');
                });
                it('should return the column interval of a range', function () {
                    expect(SheetUtils.getRowInterval(r('B3:B3'))).to.deep.equal(i(2, 2));
                    expect(SheetUtils.getRowInterval(r('B3:C4'))).to.deep.equal(i(2, 3));
                    expect(SheetUtils.getRowInterval(r('B3:D5'))).to.deep.equal(i(2, 4));
                    expect(SheetUtils.getRowInterval(r('C4:D5'))).to.deep.equal(i(3, 4));
                    expect(SheetUtils.getRowInterval(r('D5:D5'))).to.deep.equal(i(4, 4));
                });
            });

            // ----------------------------------------------------------------

            describe('method "getRowIntervals"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('getRowIntervals');
                });
                it('should sort but not merge distinct row intervals', function () {
                    expect(SheetUtils.getRowIntervals(rl('E6:F7 B3:C4 H9:I10'))).to.deep.equal([i(2, 3), i(5, 6), i(8, 9)]);
                });
                it('should merge adjacent row intervals', function () {
                    expect(SheetUtils.getRowIntervals(rl('D5:E6 B3:C4 F7:G8 H9:I10'))).to.deep.equal([i(2, 9)]);
                });
                it('should merge overlapping row intervals', function () {
                    expect(SheetUtils.getRowIntervals(rl('D5:G8 B3:D5 D5:G8 G8:I10'))).to.deep.equal([i(2, 9)]);
                });
                it('should accept empty array', function () {
                    expect(SheetUtils.getRowIntervals([])).to.be.empty;
                });
                it('should accept single range object instead of array', function () {
                    expect(SheetUtils.getRowIntervals(r('B3:C4'))).to.deep.equal([i(2, 3)]);
                });
            });

            // ----------------------------------------------------------------

            describe('method "getInterval"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('getInterval');
                });
                it('should return the column interval of a range', function () {
                    expect(SheetUtils.getInterval(r('B3:C4'), true)).to.deep.equal(i(1, 2));
                });
                it('should return the row interval of a range', function () {
                    expect(SheetUtils.getInterval(r('B3:C4'), false)).to.deep.equal(i(2, 3));
                });
            });

            // ----------------------------------------------------------------

            describe('method "getIntervals"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('getIntervals');
                });
                it('should return column intervals', function () {
                    expect(SheetUtils.getIntervals(rl('D5:E6 B3:C4 F7:G8 H9:I10'), true)).to.deep.equal([i(1, 8)]);
                });
                it('should return row intervals', function () {
                    expect(SheetUtils.getIntervals(rl('D5:E6 B3:C4 F7:G8 H9:I10'), false)).to.deep.equal([i(2, 9)]);
                });
            });

            // ----------------------------------------------------------------

            describe('method "makeRangeFromIntervals"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('makeRangeFromIntervals');
                });
                it('should build range address from intervals', function () {
                    expect(SheetUtils.makeRangeFromIntervals(i(1, 2), i(2, 3))).to.deep.equal(r('B3:C4'));
                });
                it('should accept single indexes instead of intervals', function () {
                    expect(SheetUtils.makeRangeFromIntervals(1, i(2, 3))).to.deep.equal(r('B3:B4'));
                    expect(SheetUtils.makeRangeFromIntervals(i(1, 2), 2)).to.deep.equal(r('B3:C3'));
                    expect(SheetUtils.makeRangeFromIntervals(1, 2)).to.deep.equal(r('B3:B3'));
                });
            });

            // ----------------------------------------------------------------

            describe('method "makeRangesFromIntervals"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('makeRangesFromIntervals');
                });
                it('should build range addresses from intervals', function () {
                    expect(SheetUtils.makeRangesFromIntervals([i(1, 2), 2], [i(2, 3), 3])).to.satisfy(unorderedRangesMatcher(rl('B3:C4 B4:C4 C3:C4 C4:C4')));
                });
                it('should accept single objects instead of arrays', function () {
                    expect(SheetUtils.makeRangesFromIntervals(i(1, 2), [i(2, 3), 3])).to.satisfy(unorderedRangesMatcher(rl('B3:C4 B4:C4')));
                    expect(SheetUtils.makeRangesFromIntervals(2, [i(2, 3), 3])).to.satisfy(unorderedRangesMatcher(rl('C3:C4 C4:C4')));
                    expect(SheetUtils.makeRangesFromIntervals([i(1, 2), 2], i(2, 3))).to.satisfy(unorderedRangesMatcher(rl('B3:C4 C3:C4')));
                    expect(SheetUtils.makeRangesFromIntervals([i(1, 2), 2], 3)).to.satisfy(unorderedRangesMatcher(rl('B4:C4 C4:C4')));
                    expect(SheetUtils.makeRangesFromIntervals(i(1, 2), i(2, 3))).to.deep.equal([r('B3:C4')]);
                    expect(SheetUtils.makeRangesFromIntervals(2, 3)).to.deep.equal([r('C4:C4')]);
                });
                it('should accept empty arrays', function () {
                    expect(SheetUtils.makeRangesFromIntervals([], [i(2, 3), 3])).to.deep.equal([]);
                    expect(SheetUtils.makeRangesFromIntervals([i(1, 2), 2], [])).to.deep.equal([]);
                    expect(SheetUtils.makeRangesFromIntervals([], [])).to.deep.equal([]);
                });
            });

            // ----------------------------------------------------------------

            describe('method "getUnifiedRanges"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('getUnifiedRanges');
                });
                it('should not merge distinct ranges', function () {
                    expect(SheetUtils.getUnifiedRanges(rl('C6:D7 B3:C4 E4:F5'))).to.satisfy(unorderedRangesMatcher(rl('B3:C4 E4:F5 C6:D7')));
                });
                it('should merge adjacent ranges', function () {
                    expect(SheetUtils.getUnifiedRanges(rl('B5:C6 B3:C4'))).to.deep.equal(rl('B3:C6'));
                    expect(SheetUtils.getUnifiedRanges(rl('D3:E4 B3:C4'))).to.deep.equal(rl('B3:E4'));
                    expect(SheetUtils.getUnifiedRanges(rl('D3:E4 B5:C6 B3:C4'))).to.satisfy(unorderedRangesMatcher(rl('B3:E4 B5:C6')));
                    expect(SheetUtils.getUnifiedRanges(rl('D3:E4 B5:C6 B3:C4 D5:E6'))).to.deep.equal(rl('B3:E6'));
                    expect(SheetUtils.getUnifiedRanges(rl('B3:B5 B6:D6 E4:E6 C3:E3 C4:D5'))).to.deep.equal(rl('B3:E6'));
                });
                it('should merge overlapping ranges', function () {
                    expect(SheetUtils.getUnifiedRanges(rl('B4:C6 B3:C5'))).to.deep.equal(rl('B3:C6'));
                    expect(SheetUtils.getUnifiedRanges(rl('C3:E4 B3:D4'))).to.deep.equal(rl('B3:E4'));
                    expect(SheetUtils.getUnifiedRanges(rl('C3:E5 E5:G7'))).to.satisfy(unorderedRangesMatcher(rl('C3:E4 C5:G5 E6:G7')));
                    expect(SheetUtils.getUnifiedRanges(rl('C3:E4 B4:C6 B3:D4'))).to.satisfy(unorderedRangesMatcher(rl('B3:E4 B5:C6')));
                    expect(SheetUtils.getUnifiedRanges(rl('C3:E4 B5:D6 B3:D4 D4:E6'))).to.deep.equal(rl('B3:E6'));
                    expect(SheetUtils.getUnifiedRanges(rl('B3:B6 B6:E6 E3:E6 B3:E3 C3:C5 C4:E4 D4:D6 B5:D5'))).to.deep.equal(rl('B3:E6'));
                });
                it('should split ranges to prevent covering a cell multiple times', function () {
                    expect(SheetUtils.getUnifiedRanges(rl('C3:C5 B4:D4'))).to.satisfy(unorderedRangesMatcher(rl('C3:C3 B4:D4 C5:C5')));
                });
                it('should accept empty array', function () {
                    expect(SheetUtils.getUnifiedRanges([])).to.be.empty;
                });
                it('should accept single range object instead of array', function () {
                    expect(SheetUtils.getUnifiedRanges(r('B3:C4'))).to.deep.equal(rl('B3:C4'));
                });
            });

            // ----------------------------------------------------------------

            describe('method "getRemainingRanges"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('getRemainingRanges');
                });
                it('should return remaining ranges', function () {
                    expect(SheetUtils.getRemainingRanges(rl('B3:D5 D5:F7'), rl('E3:F4 B6:C7'))).to.satisfy(unorderedRangesMatcher(rl('B3:D4 B5:F5 D6:F7')));
                    expect(SheetUtils.getRemainingRanges(rl('B3:D5 D5:F7'), rl('D3:F5 B5:D7'))).to.satisfy(unorderedRangesMatcher(rl('B3:C4 E6:F7')));
                    expect(SheetUtils.getRemainingRanges(rl('B3:D5 D5:F7'), rl('C4:E6'))).to.satisfy(unorderedRangesMatcher(rl('B3:D3 B4:B5 F5:F6 D7:F7')));
                    expect(SheetUtils.getRemainingRanges(rl('B3:D5 D5:F7'), rl('C4:C4 E6:E6'))).to.satisfy(unorderedRangesMatcher(rl('B3:D3 B4:B4 D4:D4 B5:F5 D6:D6 F6:F6 D7:F7')));
                    expect(SheetUtils.getRemainingRanges(rl('B3:D5 D5:F7'), rl('B3:F7'))).to.be.empty;
                });
                it('should accept empty array', function () {
                    expect(SheetUtils.getRemainingRanges(rl('B3:D5 D5:F7'), [])).to.satisfy(unorderedRangesMatcher(rl('B3:D4 B5:F5 D6:F7')));
                    expect(SheetUtils.getRemainingRanges([], rl('B3:D5 D5:F7'))).to.be.empty;
                    expect(SheetUtils.getRemainingRanges([], [])).to.be.empty;
                });
                it('should accept single range object instead of array', function () {
                    expect(SheetUtils.getRemainingRanges(rl('B3:D5 D5:F7'), r('C4:E6'))).to.satisfy(unorderedRangesMatcher(rl('B3:D3 B4:B5 F5:F6 D7:F7')));
                    expect(SheetUtils.getRemainingRanges(r('C4:E6'), rl('B3:D5 D5:F7'))).to.satisfy(unorderedRangesMatcher(rl('E4:E4 C6:C6')));
                    expect(SheetUtils.getRemainingRanges(r('B3:D5'), r('D5:F7'))).to.satisfy(unorderedRangesMatcher(rl('B3:D4 B5:C5')));
                });
            });

            // ----------------------------------------------------------------

            describe('method "shortenRangesByCellCount"', function () {
                var ranges = rl('A1:C3 C1:E3 E1:G3');
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('shortenRangesByCellCount');
                });
                it('should shorten ranges', function () {
                    expect(SheetUtils.shortenRangesByCellCount(ranges, -1)).to.deep.equal([]);
                    expect(SheetUtils.shortenRangesByCellCount(ranges, 0)).to.deep.equal([]);
                    expect(SheetUtils.shortenRangesByCellCount(ranges, 1)).to.deep.equal(rl('A1:A1'));
                    expect(SheetUtils.shortenRangesByCellCount(ranges, 2)).to.deep.equal(rl('A1:B1'));
                    expect(SheetUtils.shortenRangesByCellCount(ranges, 3)).to.deep.equal(rl('A1:C1'));
                    expect(SheetUtils.shortenRangesByCellCount(ranges, 4)).to.deep.equal(rl('A1:C1 A2:A2'));
                    expect(SheetUtils.shortenRangesByCellCount(ranges, 5)).to.deep.equal(rl('A1:C1 A2:B2'));
                    expect(SheetUtils.shortenRangesByCellCount(ranges, 6)).to.deep.equal(rl('A1:C2'));
                    expect(SheetUtils.shortenRangesByCellCount(ranges, 7)).to.deep.equal(rl('A1:C2 A3:A3'));
                    expect(SheetUtils.shortenRangesByCellCount(ranges, 8)).to.deep.equal(rl('A1:C2 A3:B3'));
                    expect(SheetUtils.shortenRangesByCellCount(ranges, 9)).to.deep.equal(rl('A1:C3'));
                    expect(SheetUtils.shortenRangesByCellCount(ranges, 10)).to.deep.equal(rl('A1:C3 C1:C1'));
                    expect(SheetUtils.shortenRangesByCellCount(ranges, 11)).to.deep.equal(rl('A1:C3 C1:D1'));
                    expect(SheetUtils.shortenRangesByCellCount(ranges, 12)).to.deep.equal(rl('A1:C3 C1:E1'));
                    expect(SheetUtils.shortenRangesByCellCount(ranges, 13)).to.deep.equal(rl('A1:C3 C1:E1 C2:C2'));
                    expect(SheetUtils.shortenRangesByCellCount(ranges, 14)).to.deep.equal(rl('A1:C3 C1:E1 C2:D2'));
                    expect(SheetUtils.shortenRangesByCellCount(ranges, 15)).to.deep.equal(rl('A1:C3 C1:E2'));
                    expect(SheetUtils.shortenRangesByCellCount(ranges, 16)).to.deep.equal(rl('A1:C3 C1:E2 C3:C3'));
                    expect(SheetUtils.shortenRangesByCellCount(ranges, 17)).to.deep.equal(rl('A1:C3 C1:E2 C3:D3'));
                    expect(SheetUtils.shortenRangesByCellCount(ranges, 18)).to.deep.equal(rl('A1:C3 C1:E3'));
                    expect(SheetUtils.shortenRangesByCellCount(ranges, 19)).to.deep.equal(rl('A1:C3 C1:E3 E1:E1'));
                    expect(SheetUtils.shortenRangesByCellCount(ranges, 20)).to.deep.equal(rl('A1:C3 C1:E3 E1:F1'));
                    expect(SheetUtils.shortenRangesByCellCount(ranges, 21)).to.deep.equal(rl('A1:C3 C1:E3 E1:G1'));
                    expect(SheetUtils.shortenRangesByCellCount(ranges, 22)).to.deep.equal(rl('A1:C3 C1:E3 E1:G1 E2:E2'));
                    expect(SheetUtils.shortenRangesByCellCount(ranges, 23)).to.deep.equal(rl('A1:C3 C1:E3 E1:G1 E2:F2'));
                    expect(SheetUtils.shortenRangesByCellCount(ranges, 24)).to.deep.equal(rl('A1:C3 C1:E3 E1:G2'));
                    expect(SheetUtils.shortenRangesByCellCount(ranges, 25)).to.deep.equal(rl('A1:C3 C1:E3 E1:G2 E3:E3'));
                    expect(SheetUtils.shortenRangesByCellCount(ranges, 26)).to.deep.equal(rl('A1:C3 C1:E3 E1:G2 E3:F3'));
                });
                it('should not shorten ranges', function () {
                    expect(SheetUtils.shortenRangesByCellCount(ranges, 27)).to.deep.equal(ranges);
                    expect(SheetUtils.shortenRangesByCellCount(ranges, 28)).to.deep.equal(ranges);
                });
                it('should accept empty array', function () {
                    expect(SheetUtils.shortenRangesByCellCount([], 0)).to.deep.equal([]);
                    expect(SheetUtils.shortenRangesByCellCount([], 9)).to.deep.equal([]);
                });
                it('should accept single range object instead of array', function () {
                    expect(SheetUtils.shortenRangesByCellCount(r('A1:C3'), 0)).to.deep.equal([]);
                    expect(SheetUtils.shortenRangesByCellCount(r('A1:C3'), 1)).to.deep.equal(rl('A1:A1'));
                    expect(SheetUtils.shortenRangesByCellCount(r('A1:C3'), 5)).to.deep.equal(rl('A1:C1 A2:B2'));
                });
            });

            // ----------------------------------------------------------------

            describe('method "getCellKey"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('getCellKey');
                });
                it('should return cell key for cell address', function () {
                    expect(SheetUtils.getCellKey(c('A2'))).to.equal('0,1');
                    expect(SheetUtils.getCellKey(c('B3'))).to.equal('1,2');
                    expect(SheetUtils.getCellKey(c('ZZ1000'))).to.equal('701,999');
                });
            });

            // ----------------------------------------------------------------

            describe('method "parseCellKey"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('parseCellKey');
                });
                it('should return cell address for cell key', function () {
                    expect(SheetUtils.parseCellKey('0,1')).to.deep.equal(c('A2'));
                    expect(SheetUtils.parseCellKey('1,2')).to.deep.equal(c('B3'));
                    expect(SheetUtils.parseCellKey('701,999')).to.deep.equal(c('ZZ1000'));
                });
            });

            // ----------------------------------------------------------------

            describe('method "getRangeKey"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('getRangeKey');
                });
                it('should return range key for range address', function () {
                    expect(SheetUtils.getRangeKey(r('A2:A2'))).to.equal('0,1:0,1');
                    expect(SheetUtils.getRangeKey(r('B3:D5'))).to.equal('1,2:3,4');
                    expect(SheetUtils.getRangeKey(r('A1:ZZ1000'))).to.equal('0,0:701,999');
                });
            });

            // ----------------------------------------------------------------

            describe('method "parseRangeKey"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('parseRangeKey');
                });
                it('should return range address for range key', function () {
                    expect(SheetUtils.parseRangeKey('0,1:0,1')).to.deep.equal(r('A2:A2'));
                    expect(SheetUtils.parseRangeKey('1,2:3,4')).to.deep.equal(r('B3:D5'));
                    expect(SheetUtils.parseRangeKey('0,0:701,999')).to.deep.equal(r('A1:ZZ1000'));
                });
            });

            // ----------------------------------------------------------------

            describe('method "getColName"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('getColName');
                });
                it('should return the correct column name', function () {
                    expect(SheetUtils.getColName(0)).to.equal('A');
                    expect(SheetUtils.getColName(1)).to.equal('B');
                    expect(SheetUtils.getColName(2)).to.equal('C');
                    expect(SheetUtils.getColName(25)).to.equal('Z');
                    expect(SheetUtils.getColName(26)).to.equal('AA');
                    expect(SheetUtils.getColName(27)).to.equal('AB');
                    expect(SheetUtils.getColName(28)).to.equal('AC');
                    expect(SheetUtils.getColName(51)).to.equal('AZ');
                    expect(SheetUtils.getColName(52)).to.equal('BA');
                    expect(SheetUtils.getColName(78)).to.equal('CA');
                    expect(SheetUtils.getColName(701)).to.equal('ZZ');
                    expect(SheetUtils.getColName(702)).to.equal('AAA');
                    expect(SheetUtils.getColName(18277)).to.equal('ZZZ');
                    expect(SheetUtils.getColName(18278)).to.equal('AAAA');
                    expect(SheetUtils.getColName(475253)).to.equal('ZZZZ');
                });
            });

            // ----------------------------------------------------------------

            describe('method "getColIntervalName"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('getColIntervalName');
                });
                it('should return the correct column interval name', function () {
                    expect(SheetUtils.getColIntervalName(i(0, 0))).to.equal('A:A');
                    expect(SheetUtils.getColIntervalName(i(1, 2))).to.equal('B:C');
                    expect(SheetUtils.getColIntervalName(i(0, 701))).to.equal('A:ZZ');
                });
                it('should accept single index instead of interval', function () {
                    expect(SheetUtils.getColIntervalName(1)).to.equal('B:B');
                });
            });

            // ----------------------------------------------------------------

            describe('method "getColIntervalsName"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('getColIntervalsName');
                });
                it('should return the correct column intervals name', function () {
                    expect(SheetUtils.getColIntervalsName([0, i(1, 2), i(0, 701)])).to.equal('A:A,B:C,A:ZZ');
                });
                it('should use the passed separator string', function () {
                    expect(SheetUtils.getColIntervalsName([0, i(1, 2), i(0, 701)], ' + ')).to.equal('A:A + B:C + A:ZZ');
                });
                it('should accept empty array', function () {
                    expect(SheetUtils.getColIntervalsName([])).to.equal('');
                });
                it('should accept single interval instead of array', function () {
                    expect(SheetUtils.getColIntervalsName(i(1, 2))).to.equal('B:C');
                });
                it('should accept single index instead of array', function () {
                    expect(SheetUtils.getColIntervalsName(1)).to.equal('B:B');
                });
            });

            // ----------------------------------------------------------------

            describe('method "getRowName"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('getRowName');
                });
                it('should return the correct row name', function () {
                    expect(SheetUtils.getRowName(0)).to.equal('1');
                    expect(SheetUtils.getRowName(1)).to.equal('2');
                    expect(SheetUtils.getRowName(2)).to.equal('3');
                    expect(SheetUtils.getRowName(8)).to.equal('9');
                    expect(SheetUtils.getRowName(9)).to.equal('10');
                    expect(SheetUtils.getRowName(10)).to.equal('11');
                    expect(SheetUtils.getRowName(99999998)).to.equal('99999999');
                });
            });

            // ----------------------------------------------------------------

            describe('method "getRowIntervalName"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('getRowIntervalName');
                });
                it('should return the correct row interval name', function () {
                    expect(SheetUtils.getRowIntervalName(i(0, 0))).to.equal('1:1');
                    expect(SheetUtils.getRowIntervalName(i(1, 2))).to.equal('2:3');
                    expect(SheetUtils.getRowIntervalName(i(0, 999))).to.equal('1:1000');
                });
                it('should accept single index instead of interval', function () {
                    expect(SheetUtils.getRowIntervalName(1)).to.equal('2:2');
                });
            });

            // ----------------------------------------------------------------

            describe('method "getRowIntervalsName"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('getRowIntervalsName');
                });
                it('should return the correct row intervals name', function () {
                    expect(SheetUtils.getRowIntervalsName([0, i(1, 2), i(0, 999)])).to.equal('1:1,2:3,1:1000');
                });
                it('should use the passed separator string', function () {
                    expect(SheetUtils.getRowIntervalsName([0, i(1, 2), i(0, 999)], ' + ')).to.equal('1:1 + 2:3 + 1:1000');
                });
                it('should accept empty array', function () {
                    expect(SheetUtils.getRowIntervalsName([])).to.equal('');
                });
                it('should accept single interval instead of array', function () {
                    expect(SheetUtils.getRowIntervalsName(i(1, 2))).to.equal('2:3');
                });
                it('should accept single index instead of array', function () {
                    expect(SheetUtils.getRowIntervalsName(1)).to.equal('2:2');
                });
            });

            // ----------------------------------------------------------------

            describe('method "getIndexName"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('getIndexName');
                });
                it('should return the column name', function () {
                    expect(SheetUtils.getIndexName(0, true)).to.equal('A');
                });
                it('should return the row name', function () {
                    expect(SheetUtils.getIndexName(0, false)).to.equal('1');
                });
            });

            // ----------------------------------------------------------------

            describe('method "getIntervalName"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('getIntervalName');
                });
                it('should return the column interval name', function () {
                    expect(SheetUtils.getIntervalName(i(1, 2), true)).to.equal('B:C');
                    expect(SheetUtils.getIntervalName(1, true)).to.equal('B:B');
                });
                it('should return the row interval name', function () {
                    expect(SheetUtils.getIntervalName(i(1, 2), false)).to.equal('2:3');
                    expect(SheetUtils.getIntervalName(1, false)).to.equal('2:2');
                });
            });

            // ----------------------------------------------------------------

            describe('method "getIntervalsName"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('getIntervalsName');
                });
                it('should return the column intervals name', function () {
                    expect(SheetUtils.getIntervalsName([0, i(1, 2), i(0, 701)], true)).to.equal('A:A,B:C,A:ZZ');
                    expect(SheetUtils.getIntervalsName(i(1, 2), true)).to.equal('B:C');
                    expect(SheetUtils.getIntervalsName(1, true)).to.equal('B:B');
                });
                it('should return the row intervals name', function () {
                    expect(SheetUtils.getIntervalsName([0, i(1, 2), i(0, 999)], false)).to.equal('1:1,2:3,1:1000');
                    expect(SheetUtils.getIntervalsName(i(1, 2), false)).to.equal('2:3');
                    expect(SheetUtils.getIntervalsName(1, false)).to.equal('2:2');
                });
            });

            // ----------------------------------------------------------------

            describe('method "getCellName"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('getCellName');
                });
                it('should return the correct cell name', function () {
                    expect(SheetUtils.getCellName(c('A1'))).to.equal('A1');
                    expect(SheetUtils.getCellName(c('B1'))).to.equal('B1');
                    expect(SheetUtils.getCellName(c('C1'))).to.equal('C1');
                    expect(SheetUtils.getCellName(c('A2'))).to.equal('A2');
                    expect(SheetUtils.getCellName(c('A3'))).to.equal('A3');
                    expect(SheetUtils.getCellName(c('ZZZZ99999999'))).to.equal('ZZZZ99999999');
                });
            });

            // ----------------------------------------------------------------

            describe('method "getRangeName"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('getRangeName');
                });
                it('should return the correct range name', function () {
                    expect(SheetUtils.getRangeName(r('A2:C4'))).to.equal('A2:C4');
                    expect(SheetUtils.getRangeName(r('A1:ZZZZ99999999'))).to.equal('A1:ZZZZ99999999');
                });
            });

            // ----------------------------------------------------------------

            describe('method "getRangesName"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('getRangesName');
                });
                it('should return the correct range list name', function () {
                    expect(SheetUtils.getRangesName(rl('B3:C4 C4:D5 D5:E6'))).to.equal('B3:C4,C4:D5,D5:E6');
                });
                it('should use the passed separator string', function () {
                    expect(SheetUtils.getRangesName(rl('B3:C4 C4:D5 D5:E6'), ' + ')).to.equal('B3:C4 + C4:D5 + D5:E6');
                });
                it('should accept empty array', function () {
                    expect(SheetUtils.getRangesName([])).to.equal('');
                });
                it('should accept single range object instead of array', function () {
                    expect(SheetUtils.getRangesName(r('B3:C4'))).to.equal('B3:C4');
                });
            });

            // ----------------------------------------------------------------

            describe('method "generateSheetName"', function () {
                it('should exist', function () {
                    expect(SheetUtils).itself.to.respondTo('generateSheetName');
                });
                it('should return the English sheet name', function () {
                    expect(SheetUtils.generateSheetName(0)).to.equal('Sheet1');
                    expect(SheetUtils.generateSheetName(1)).to.equal('Sheet2');
                    expect(SheetUtils.generateSheetName(9)).to.equal('Sheet10');
                });
            });

            // ----------------------------------------------------------------

        });
    });
});
