/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/popup/listmenu',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/keycodes',
     'io.ox/office/tk/forms',
     'io.ox/office/tk/popup/basemenu'
    ], function (Utils, KeyCodes, Forms, BaseMenu) {

    'use strict';

    var // the CSS class for menu section nodes
        SECTION_CLASS = 'section-container';

    // class ListMenu =========================================================

    /**
     * Wraps a pop-up menu element, shown on top of the application window, and
     * relative to an arbitrary DOM node. An item list contains a set of button
     * control elements (the list items) that can be stacked vertically, or can
     * be arranged in a tabular layout. The list items can be highlighted. The
     * items will be grouped into vertically stacked section nodes.
     *
     * Instances of this class trigger the following events (additionally to
     * the events triggered by the base class BaseMenu):
     * - 'create:section'
     *      After a new section node has been created with the method
     *      ListMenu.createSectionNode(). Event handlers receive the following
     *      parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {jQuery} sectionNode
     *          The root node of the new section, as jQuery object.
     *      (3) {String} sectionId
     *          The section identifier.
     *      (4) {Object} [options]
     *          The options passed to the method ListMenu.createSectionNode().
     * - 'create:item'
     *      After a button node for a new list item has been created with the
     *      method ListMenu.createItemNode(). Event handlers receive the
     *      following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {jQuery} buttonNode
     *          The button element representing the new list item, as jQuery
     *          object.
     *      (3) {Any} value
     *          The value of the new list item.
     *      (4) {Object} [options]
     *          The options passed to the method ListMenu.createItemNode().
     *
     * @constructor
     *
     * @extends BaseMenu
     *
     * @param {Object} [initOptions]
     *  Initial options for the item list menu element. Supports all options
     *  also supported by the base class BaseMenu. Additionally, the following
     *  options are supported:
     *  @param {Function} [initOptions.itemMatcher=_.isEqual]
     *      A comparison function that returns whether a list item should be
     *      selected, deselected, or kept unmodified according to a specific
     *      value. Receives an arbitrary value as first parameter, and the
     *      value of the current list item element as second parameter. The
     *      function must return the Boolean value true to select the
     *      respective list item element, the Boolean value false to deselect
     *      the list item element, or any other value to leave the list item
     *      element unmodified. If omitted, uses _.isEqual() which compares
     *      arrays and objects deeply, and does not skip any list item.
     *  @param {String} [initOptions.itemDesign='list']
     *      The design mode of the list item elements:
     *      - 'list': All item elements will be stacked vertically, and will
     *          occupy the complete width of the pop-up menu. Highlighted item
     *          elements will be drawn with a changed background color.
     *      - 'grid': All item elements will be arranged in a tabular layout
     *          per menu section. Highlighted items will be drawn with a thick
     *          border, leaving the background untouched.
     *  @param {Boolean} [initOptions.sortItems=false]
     *      If set to true, the list items will be sorted according to the sort
     *      order that has been specified via the 'sortIndex' options of each
     *      inserted list item (see method ListMenu.createItemNode() for
     *      details).
     *  @param {Number} [options.gridColumns=10]
     *      The default number of columns in each menu section in 'grid' design
     *      mode (see option 'itemDesign' above). This value can be overridden
     *      for single menu sections by passing this option to the method
     *      ListMenu.createSectionNode().
     */
    function ListMenu(initOptions) {

        var // self reference
            self = this,

            // comparator for list item values
            itemMatcher = Utils.getFunctionOption(initOptions, 'itemMatcher', _.isEqual),

            // the design mode
            design = Utils.getStringOption(initOptions, 'itemDesign', 'list'),

            // whether the list items will be sorted
            sorted = Utils.getBooleanOption(initOptions, 'sortItems', false),

            // default number of columns in each section node ('grid' design mode)
            defColumns = Utils.getIntegerOption(initOptions, 'gridColumns', 10, 1);

        // base constructor ---------------------------------------------------

        BaseMenu.call(this, Utils.extendOptions({ preferFocusFilter: Forms.SELECTED_SELECTOR }, initOptions));

        // private methods ----------------------------------------------------

        /**
         * Sorts all list items in the specified section node.
         */
        function sortSectionNode(sectionNode) {

            var // all existing item buttons in the section, as plain array (for sorting)
                buttonNodes = sectionNode.find(Forms.BUTTON_SELECTOR).get(),
                // table cell nodes (grid design)
                cellNodes = null;

            // sort by fixed index, default to label text of the button
            buttonNodes = _.sortBy(buttonNodes, function (btnNode) {
                var index = $(btnNode).data('index');
                return _.isNull(index) ? $(btnNode).text().toLowerCase() : index;
            });

            // insert button nodes depending on design mode
            switch (design) {
            case 'list':
                sectionNode.append.apply(sectionNode, buttonNodes);
                break;
            case 'grid':
                cellNodes = sectionNode.find('>table>tbody>tr>td');
                _.each(buttonNodes, function (btnNode, index) {
                    cellNodes.eq(index).append(btnNode);
                });
                break;
            }
        }

        /**
         * Sorts all list items in the specified section node debounced.
         */
        var sortSectionNodeDebounced = (function () {

            var // all section nodes to be sorted
                sectionNodes = $();

            // sorts all registered sections debounced
            var sortAllSectionNodes = _.debounce(function () {

                // do not run debounced code in a destroyed instance
                if (!self || self.destroyed) { return; }

                // protect the focused list item
                self.guardFocusedListItem(function () {
                    // process all registered section nodes
                    sectionNodes.each(function () { sortSectionNode($(this)); });
                    // reset the cache of dirty section nodes
                    sectionNodes = $();
                });
            }, 10);

            // the actual sortSectionNodeDebounced() method to be returned from the local scope
            function sortSectionNodeDebounced(sectionNode) {
                // jQuery keeps the collection unique
                sectionNodes = sectionNodes.add(sectionNode);
                // sort all registered sections debounced
                sortAllSectionNodes();
            }

            return sortSectionNodeDebounced;
        }());

        /**
         * Handles 'keydown' events in the root node of the pop-up menu.
         */
        function keyDownHandler(event) {

            var // the list item to be focused
                targetNode = self.getItemNodeForKeyEvent(event, $(event.target));

            // stop event propagation for all supported keys
            if (targetNode.length > 0) {
                targetNode.focus();
                self.scrollToChildNode(targetNode);
                return false;
            }
        }

        // public methods -----------------------------------------------------

        /**
         * Invokes the passed callback function which wants to modify the
         * contents of this drop-down list. The previously focused list item
         * will be focused again afterwards.
         *
         * @param {Function} callback
         *  The callback function that will be invoked while remembering the
         *  list item currently focused.
         *
         * @returns {ListMenu}
         *  A reference to this instance.
         */
        this.guardFocusedListItem = function (callback, context) {

            var // the focused list item
                focusNode = this.getItemNodes().filter(':focus'),
                // value of the focused list item (bug 33737, 33747: preserve browser focus)
                value = Forms.getButtonValue(focusNode.first());

            // invoke passed callback function
            callback.call(context);

            // restore browser focus by looking up the original focus node, or
            // by selecting a new focus node via the value of the old node
            if (focusNode.length > 0) {
                // first try to find the original node, resolve by value if this fails
                if (this.getItemNodes().filter(focusNode).length === 0) {
                    focusNode = this.findItemNodes(value);
                }
                focusNode.focus();
            }

            return this;
        };

        /**
         * Adds a new section to this pop-up menu, if it does not exist yet.
         *
         * @param {String} sectionId
         *  The unique identifier of the menu section.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.label]
         *      If specified, a heading label will be created for the section.
         *  @param {Number} [options.gridColumns]
         *      The number of columns in the menu section, if this menu is in
         *      'grid' design mode (see option 'itemDesign' passed to the
         *      constructor). If omitted, the default value for this option as
         *      specified in the constructor options will be used.
         *
         * @returns {jQuery}
         *  The root node of the menu section, as jQuery object.
         */
        this.createSectionNode = function (sectionId, options) {

            var // try to find an existing section node
                sectionNode = this.getSectionNode(sectionId);

            if (sectionNode.length === 0) {

                // create the header label mark-up
                var headerMarkup = Utils.getStringOption(options, 'label', '');
                if (headerMarkup.length > 0) {
                    headerMarkup = '<div class="header-label" aria-hidden="true">' + Utils.escapeHTML(headerMarkup) + '</div>';
                }

                // create the section root node
                sectionNode = $('<div class="' + SECTION_CLASS + ' group ' + Forms.HIDDEN_CLASS + '" data-section="' + sectionId + '">' + headerMarkup + '</div>');

                // add the number of grid columns as jQuery data attribute
                sectionNode.data('grid-columns', Utils.getIntegerOption(options, 'gridColumns', defColumns, 1));

                // insert the new section into the drop-down menu, notify listeners
                this.appendContentNodes(sectionNode);
                this.trigger('create:section', sectionNode, sectionId, options);
            }
            return sectionNode;
        };

        /**
         * Returns the DOM nodes of all menu sections.
         *
         * @returns {jQuery}
         *  The DOM nodes representing all menu sections, as jQuery collection.
         */
        this.getSectionNodes = function () {
            return this.getNode().find('.' + SECTION_CLASS);
        };

        /**
         * Returns the DOM node of the specified menu section.
         *
         * @param {String} sectionId
         *  The unique identifier of the menu section.
         *
         * @returns {jQuery}
         *  The DOM node representing the specified menu section. If the menu
         *  section does not exist, returns an empty jQuery collection.
         */
        this.getSectionNode = function (sectionId) {
            return this.getSectionNodes().filter('[data-section="' + sectionId  + '"]');
        };

        /**
         * Adds a new item element to this list. If the items will be sorted
         * (see the sort options passed to the constructor), the item element
         * will be inserted according to these settings.
         *
         * @param {Any} value
         *  The unique value associated to the list item. MUST NOT be null.
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all formatting options for button
         *  elements supported by the method Forms.createButtonMarkup().
         *  Additionally, the following options are supported:
         *  @param {String} [options.section='']
         *      The unique identifier of the menu section the new list item
         *      will be inserted into. If a section with this identifier does
         *      not exist yet, it will be created without a header label. Can
         *      be omitted for example for simple list menus without sections.
         *  @param {String} [options.dataValue]
         *      A string that will be inserted into the 'data-value' attribute
         *      of the button node. If omitted, the JSON string representation
         *      of the 'value' parameter will be used instead (all double-quote
         *      characters will be removed from the string though), unless the
         *      value is a function.
         *
         * @returns {jQuery}
         *  The button element representing the new list item, as jQuery
         *  object.
         */
        this.createItemNode = function (value, options) {

            var // the root node of the menu section
                sectionNode = this.createSectionNode(Utils.getStringOption(options, 'section', '')),
                // create the button element representing the item
                buttonNode = $(Forms.createButtonMarkup(options));

            // make section node visible (initially hidden)
            Forms.showNodes(sectionNode, true);

            // initialize button node
            Forms.setButtonValue(buttonNode, value, options);
            buttonNode.data('index', Utils.getOption(options, 'sortIndex', null));
            buttonNode.attr('role', 'option');

            // insert the new button according to the design mode
            switch (design) {
            case 'list':
                sectionNode.append(buttonNode);
                break;
            case 'grid':
                // create a new table element for the button if required
                var tableNode = sectionNode.find('>table');
                if (tableNode.length === 0) {
                    tableNode = $('<table>').attr({ role: 'grid', 'aria-label': Utils.getStringOption(initOptions, 'tooltip', null, true) }).appendTo(sectionNode);
                }
                // create a new table row element for the button if required
                var rowNode = tableNode.find('>tbody>tr').last();
                if ((rowNode.length === 0) || (rowNode.children().last().children().length > 0)) {
                    // IE9 does not allow to write to the 'innerHTML' property for <TR> nodes :-O
                    rowNode = $('<tr role="row">' + Utils.repeatString('<td></td>', sectionNode.data('grid-columns')) + '</tr>').appendTo(tableNode);
                }
                // insert the button into the first empty cell
                var cellNode = rowNode.find('>td:empty').first();
                cellNode.attr({ role: 'gridcell', 'aria-label': buttonNode.attr('aria-label') || '' }).append(buttonNode);
                break;
            }

            // sort the button nodes in the section
            if (sorted) { sortSectionNodeDebounced(sectionNode); }

            // notify listeners
            this.trigger('create:item', buttonNode, value, options);
            return buttonNode;
        };

        /**
         * Removes an item element from this list.
         *
         * @param {Any} value
         *  The unique value associated to the list item. MUST NOT be null.
         *
         * @returns {ListMenu}
         *  A reference to this instance.
         */
        this.deleteItemNodes = function (value) {

            var // all item buttons to be deleted
                buttonNodes = this.findItemNodes(value),
                // all modified section nodes
                sectionNodes = $();

            // collect modified section nodes, notify listeners
            buttonNodes.each(function () {
                var buttonNode = $(this);
                sectionNodes = sectionNodes.add(buttonNode.closest('.' + SECTION_CLASS));
                self.trigger('delete:item', buttonNode, value);
            });

            // remove button nodes from DOM
            buttonNodes.remove();

            // update tables in grid design mode, hide empty sections
            sectionNodes.each(function () {

                var // current section node
                    sectionNode = $(this);

                // all existing buttons in the section
                buttonNodes = sectionNode.find(Forms.BUTTON_SELECTOR);

                // reinsert buttons in grid design mode, update or remove table
                if (design === 'grid') {
                    var tableNode = sectionNode.find('>table');
                    if (buttonNodes.length === 0) {
                        tableNode.remove();
                    } else {
                        // redistribute all buttons
                        var cellNodes = tableNode.find('>tbody>tr>td');
                        buttonNodes.each(function (index) { cellNodes.eq(index).append(this); });
                        // delete last empty row
                        var rowNode = cellNodes.last().parent();
                        if (rowNode.find(Forms.BUTTON_SELECTOR).length === 0) {
                            rowNode.remove();
                        }
                    }
                }

                // hide empty sections
                Forms.showNodes(sectionNode, buttonNodes.length > 0);
            });

            return this;
        };

        /**
         * Returns all button elements representing the menu items of this
         * list.
         *
         * @returns {jQuery}
         *  All button elements representing the menu items in this item list,
         *  as jQuery collection.
         */
        this.getItemNodes = function () {
            return this.getNode().find(Forms.BUTTON_SELECTOR);
        };

        /**
         * Returns the button elements representing the list item with the
         * specified value.
         *
         * @param {Any} value
         *  The value whose list item buttons will be returned.
         *
         * @returns {jQuery}
         *  The button elements representing the list items with the specified
         *  value, as jQuery collection.
         */
        this.findItemNodes = function (value) {
            return Forms.filterButtonNodes(this.getItemNodes(), value, itemMatcher);
        };

        /**
         * Returns the button elements representing all selected list items in
         * this list menu.
         *
         * @returns {jQuery}
         *  The button elements representing all selected list items, as jQuery
         *  collection.
         */
        this.getSelectedItemNodes = function () {
            return Forms.filterSelectedNodes(this.getItemNodes());
        };

        /**
         * Selects the specified list item in this menu, and deselects all
         * other list items.
         *
         * @param {Number} index
         *  The zero-based index of the list item to be selected.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.scroll=false]
         *      If set to true, the menu will scroll to the selected item node
         *      automatically.
         *
         * @returns {jQuery}
         *  The button element representing the selected list item, as jQuery
         *  object. If the passed index is invalid, an empty jQuery collection
         *  will be returned.
         */
        this.selectItem = function (index, options) {
            return this.selectItemNode(this.getItemNodes().eq(index), options);
        };

        /**
         * Selects the specified button elements representing one or more list
         * items in this menu, and deselects all other list items.
         *
         * @param {HTMLElement|jQuery} buttonNodes
         *  The list item buttons to be selected, as plain DOM node, or as
         *  jQuery collection that may contain multiple list items.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.scroll=false]
         *      If set to true, the menu will scroll to the selected item node
         *      automatically.
         *
         * @returns {jQuery}
         *  The button elements representing the selected list items, as jQuery
         *  collection.
         */
        this.selectItemNode = function (buttonNode, options) {
            buttonNode = $(buttonNode);
            Forms.selectButtonNodes(this.getItemNodes(), false);
            Forms.selectButtonNodes(buttonNode, true);
            if ((buttonNode.length > 0) && Utils.getBooleanOption(options, 'scroll', false)) {
                this.scrollToChildNode(buttonNode);
            }
            return buttonNode;
        };

        /**
         * Selects the button elements representing the list item with the
         * passed value, and deselects all other list items.
         *
         * @param {Any} value
         *  The value whose list item buttons will be selected. List items will
         *  be selected or deselected according to the item matcher callback
         *  function passed to the constructor (option 'itemMatcher').
         *
         * @returns {jQuery}
         *  The button elements representing the selected list items with the
         *  specified value, as jQuery collection.
         */
        this.selectMatchingItemNodes = function (value) {
            Forms.selectMatchingButtonNodes(this.getItemNodes(), value, itemMatcher);
            return this.getSelectedItemNodes();
        };

        /**
         * Returns the target list item to be focused or selected after moving
         * the cursor according to the passed keyboard event.
         *
         * @param {jQuery.Event} keyEvent
         *  The event object caught for a 'keydown' event.
         *
         * @param {HTMLElement|jQuery|Null} buttonNode
         *  The button element representing the source list item the cursor
         *  movement originates from. If omitted, the first or last list item
         *  will be selected instead according to the key code. All keys in
         *  'down' orientation (CURSOR_DOWN, PAGE_DOWN) and the HOME key will
         *  select the first list item, all keys in 'up' orientation
         *  (CURSOR_UP, PAGE_UP) and the END key will select the last list
         *  item.
         *
         * @returns {jQuery}
         *  The target list item to be focused or selected, as jQuery object.
         *  If the passed key event does not describe any key supported for
         *  cursor movement, or the passed button node is invalid, an empty
         *  jQuery collection will be returned instead.
         */
        this.getItemNodeForKeyEvent = function (keyEvent, buttonNode) {

            var // all visible button nodes
                buttonNodes = Forms.filterVisibleNodes(self.getItemNodes());

            // returns the number of list items currently visible
            function getListPageSize() {
                var itemHeight = buttonNodes.first().outerHeight();
                return (itemHeight > 0) ? Math.max(1, Math.floor(self.getNode()[0].clientHeight / itemHeight) - 1) : 1;
            }

            // returns the previous or next item node according to the passed distance
            function getNextItemNode(diff, wrap) {

                var // index of the focused button
                    index = buttonNodes.index(buttonNode);

                if (index < 0) {
                    index = (diff < 0) ? (buttonNodes.length - 1) : 0;
                } else if (wrap) {
                    index += (diff % buttonNodes.length) + buttonNodes.length;
                    index %= buttonNodes.length;
                } else {
                    index = Utils.minMax(index + diff, 0, buttonNodes.length - 1);
                }
                return buttonNodes.eq(index);
            }

            // returns the target item node for 'list' design
            function getItemNodeForList() {

                switch (keyEvent.keyCode) {
                case KeyCodes.UP_ARROW:
                    return getNextItemNode(-1, true);
                case KeyCodes.DOWN_ARROW:
                    return getNextItemNode(1, true);
                case KeyCodes.PAGE_UP:
                    return getNextItemNode(-getListPageSize());
                case KeyCodes.PAGE_DOWN:
                    return getNextItemNode(getListPageSize());
                }

                return $();
            }

            // returns the item node in the previous or next table row in 'grid' design
            function getItemNodeInNextRow(diff) {

                var // the table cell node containing the focused button
                    cellNode = $(Utils.findFarthest(self.getNode(), buttonNode, 'td')),
                    // the table row node containing the focused button
                    rowNode = cellNode.parent(),
                    // all button elements in the focused row
                    rowButtonNodes = rowNode.children().children(),
                    // all row nodes in all tables
                    rowNodes = self.getSectionNodes().find('>table>tbody>tr'),
                    // the column index of the cell in the row
                    colIndex = cellNode.index(),
                    // the row index of the cell in the table
                    rowIndex = rowNodes.index(rowNode),
                    // the column index of the cell in the table
                    colRatio = (colIndex >= 0) ? (colIndex / rowButtonNodes.length) : 0;

                if ((rowNodes.length === 0) || (rowIndex < 0)) {
                    return (diff < 0) ? buttonNodes.last() : buttonNodes.first();
                }

                // move focus to the specified row, try to keep column position constant inside row
                rowIndex += (diff % rowNodes.length) + rowNodes.length;
                rowNode = rowNodes.eq(rowIndex % rowNodes.length);
                rowButtonNodes = rowNode.children().children();
                colIndex = Math.min(Math.floor(colRatio * rowNode.children().length), rowButtonNodes.length - 1);
                return rowButtonNodes.eq(colIndex);
            }

            // returns the target item node for 'grid' design
            function getItemNodeForGrid() {

                switch (keyEvent.keyCode) {
                case KeyCodes.LEFT_ARROW:
                    return getNextItemNode(-1, true);
                case KeyCodes.RIGHT_ARROW:
                    return getNextItemNode(1, true);
                case KeyCodes.UP_ARROW:
                    return getItemNodeInNextRow(-1);
                case KeyCodes.DOWN_ARROW:
                    return getItemNodeInNextRow(1);
                }

                return $();
            }

            // convert to jQuery collection
            buttonNode = $(buttonNode);

            // generic key shortcuts
            switch (keyEvent.keyCode) {
            case KeyCodes.HOME:
                return buttonNodes.first();
            case KeyCodes.END:
                return buttonNodes.last();
            }

            // special shortcuts dependent on design mode
            switch (design) {
            case 'list':
                return getItemNodeForList();
            case 'grid':
                return getItemNodeForGrid();
            }

            return $();
        };

        /**
         * Scrolls the passed list item node into the visible area of the menu.
         *
         * @param {HTMLElement|jQuery} buttonNode
         *  The list item DOM element that will be made visible by scrolling
         *  the menu root node. If this object is a jQuery collection, uses the
         *  first node it contains.
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all options that are supported by the
         *  method Utils.scrollToChildNode(). The option 'padding' is defaulted
         *  to the value 5.
         *
         * @returns {ListMenu}
         *  A reference to this instance.
         */
        this.scrollToChildNode = (function () {
            var baseMethod = _.bind(self.scrollToChildNode, self);
            return function (buttonNode, options) {
                return baseMethod(buttonNode, Utils.extendOptions({ padding: 5 }, options));
            };
        }());

        // initialization -----------------------------------------------------

        // add CSS marker for the item design
        this.getNode().addClass('list-menu').attr({ 'data-design': design, role: 'listbox' });

        // register event handlers
        this.getNode().on('keydown', keyDownHandler);

        // convert ENTER, SPACE, and TAB keys in the menu to click events
        Forms.setButtonKeyHandler(this.getNode(), { tab: true });

        // special handling for tables in 'grid' design
        if (design === 'grid') {
            this.on({
                'popup:beforeshow popup:beforelayout': function () { self.getSectionNodes().find('>table').css('width', 'auto'); },
                'popup:show popup:layout': function () { self.getSectionNodes().find('>table').css('width', ''); }
            });
        }

        // destroy all class members on destruction
        this.registerDestructor(function () {
            initOptions = self = itemMatcher = null;
        });

    } // class ListMenu

    // exports ================================================================

    // derive this class from class BaseMenu
    return BaseMenu.extend({ constructor: ListMenu });

});
