/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2011
 * Mail: info@open-xchange.com
 *
 * @author carsten-driesner
 */

define('io.ox/office/tk/errorcode',
    [
     'io.ox/office/tk/utils',
    ], function (Utils) {

    'use strict';

    // class ErrorCode ===========================================================

    /**
     * Provides methods to check a server result for possible errors.
     *
     * @constructor
     *
     * @param {Object} [resultData]
     *  The result object directly provided by the server which contains error
     *  information that are used to initialize ErrorCode instance.
     */
    function ErrorCode(resultData) {
        var // the error code
            code = ErrorCode.NO_ERROR,
            // the error description
            description = '',
            // the error code as string constant
            codeAsStringConstant = 'NO_ERROR',
            // error class
            errorClass = ErrorCode.ERRORCLASS_NO_ERROR;

        /**
         * Checks if this instance represents an error state or not.
         *
         * @returns {Boolean}
         *  TRUE if the instance represents an error state, FALSE if not.
         */
        this.isError = function () {
            return code !== ErrorCode.NO_ERROR;
        };

        /**
         * Provides the error code.
         *
         * @returns {Number}
         *  The unique error code which describes what error has happened.
         */
        this.getCode = function () {
            return code;
        };

        /**
         * Provides a textual description what error has happened.
         *
         * @returns {String}
         *  A description of the error.
         */
        this.getDescription = function () {
            return description;
        };

        /**
         * Provides a unique string constant that specifies the error.
         *
         * @returns
         *  A unique string constant for the error.
         */
        this.getCodeAsConstant = function () {
            return codeAsStringConstant;
        };

        /**
         * Provides the error class of the error, which describes the
         *
         * @returns {Number}
         */
        this.getErrorClass = function () {
            return errorClass;
        };

        /**
         * Provides the textual representation of the error code object.
         *
         * @returns
         *  The textual representation of the error code object.
         */
        this.getErrorText = function () {
            return 'Error: code=' + code + ', constant=' + codeAsStringConstant + ', description=' + description;
        };

        // check the provided resultData object for error information
        if (_.isObject(resultData) && resultData.error) {
            code = Utils.getNumberOption(resultData.error, ErrorCode.PROPERTY_ERRORCODE, ErrorCode.NO_ERROR);
            description = Utils.getStringOption(resultData.error, ErrorCode.PROPERTY_ERRORDESCRIPTION, '');
            codeAsStringConstant = Utils.getStringOption(resultData.error, ErrorCode.PROPERTY_ERRORCODEASSTRING, 'NO_ERROR');
            errorClass = Utils.getNumberOption(resultData.error, ErrorCode.PROPERTY_ERRORCLASS, ErrorCode.ERRORCLASS_NO_ERROR);
        }
    }

    // static methods ---------------------------------------------------------

    /**
     * Extracts the error information from result data received from the server
     * side.
     *
     * @param serverResult {Object}
     *  The result data sent by the server.
     *
     * @return
     *  The error information extracted from the server data or null if no
     *  error information could be extracted.
     */
    ErrorCode.extractErrorInformation = function (serverResult) {
        return (_.isObject(serverResult) && serverResult.error) ? _.extend({}, {error: serverResult.error}) : null;
    };

    /**
     * Extracts the error information from a real-time result received from the
     * server side.
     *
     * @param rtResult {Object}
     *  The result data sent by the server.
     *
     * @returns
     *  An error object containing the error code of the real-time answer or
     *  null if no error information could be extracted.
     */
    ErrorCode.extractRealtimeError = function (rtResult) {
        return (_.isObject(rtResult) && rtResult.code) ? {code: rtResult.code} : null;
    };

    // constants --------------------------------------------------------------

    // the constant error code which specifies no error
    ErrorCode.NO_ERROR = 0;

    // the constant error code as string for no error
    ErrorCode.CONSTANT_NO_ERROR = 'NO_ERROR';

    // the property names
    ErrorCode.PROPERTY_ERRORCODE = 'errorCode';
    ErrorCode.PROPERTY_ERRORDESCRIPTION = 'errorDescription';
    ErrorCode.PROPERTY_ERRORCODEASSTRING = 'error';
    ErrorCode.PROPERTY_ERRORCLASS = 'errorClass';

    // the error classes describing the severity of an error
    ErrorCode.ERRORCLASS_NO_ERROR = 0;
    ErrorCode.ERRORCLASS_WARNING = 1;
    ErrorCode.ERRORCLASS_ERROR = 2;
    ErrorCode.ERRORCLASS_FATAL_ERROR = 3;

    // exports ================================================================

    return ErrorCode;
});
