/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/control/checkbox',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/forms',
     'io.ox/office/tk/control/button'
    ], function (Utils, Forms, Button) {

    'use strict';

    // class CheckBox =========================================================

    /**
     * Creates a container element used to hold a toggle button, styled like a
     * regular check box.
     *
     * @constructor
     *
     * @extends Button
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options of the Button base class,
     *  except for the 'toggle' option which will always be set to true, and
     *  'highlight' (check boxes will never be highlighted). Additionally, the
     *  following options are supported:
     *  @param {Boolean} [initOptions.boxed=false]
     *      If set to true, the icons to visualize the unchecked and checked
     *      state will contain rectangular boxes. Otherwise, unchecked state is
     *      simply empty space, and checked state is a simple check mark.
     */
    function CheckBox(initOptions) {

        var // boxed icon mode
            boxed = Utils.getBooleanOption(initOptions, 'boxed', false),

            // container for the check mark icon
            checkNode = $('<span class="check-mark">' + createIconMarkup(false) + '</span>');

        // base constructor ---------------------------------------------------

        Button.call(this, Utils.extendOptions({attributes: { role: 'checkbox' }} /* just default aria role */ , initOptions, {
            toggle: true,
            highlight: function () { return false; }, // never highlight the check box
        }));

        // private methods ----------------------------------------------------

        /**
         * Returns the HTML mark-up for the check mark.
         */
        function createIconMarkup(state) {
            return Forms.createIconMarkup(state ? (boxed ? 'fa-check-square-o' : 'fa-check') : (boxed ? 'fa-square-o' : 'fa-none'));
        }

        /**
         * The update handler for this button.
         */
        function updateHandler(state, options, oldVal) {
        	if (state !== oldVal) {
        		checkNode[0].innerHTML = createIconMarkup(state);
            	this.getButtonNode().attr({'aria-pressed': state, 'aria-checked': state});
        	}
        }

        // initialization -----------------------------------------------------

        // add special marker classes used to adjust formatting
        this.getNode().addClass('check-box');
        this.getButtonNode().prepend(checkNode);

        // register an update handler that updates the check icon
        this.registerUpdateHandler(updateHandler);

    } // class CheckBox

    // exports ================================================================

    // derive this class from class Button
    return Button.extend({ constructor: CheckBox });

});
