/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/statuspane',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/forms',
     'io.ox/office/baseframework/view/toolpane',
     'io.ox/office/baseframework/view/toolbar',
     'io.ox/office/spreadsheet/view/controls',
     'gettext!io.ox/office/spreadsheet'
    ], function (Utils, Forms, ToolPane, ToolBar, Controls, gt) {

    'use strict';

    var // class name shortcuts
        Button = Controls.Button,
        DynamicLabel = Controls.DynamicLabel,
        ComponentMenuButton = Controls.ComponentMenuButton;

    // class StatusPane =======================================================

    /**
     * The status pane in spreadsheet documents containing the sheet tabs and
     * the subtotal results of the current selection.
     *
     * @constructor
     *
     * @extends ToolPane
     */
    function StatusPane(app) {

        var // self reference
            self = this,

            // get application model and view
            view = app.getView(),

            // the tool bar for all sheet controls
            sheetToolBar = null,

            // the tool bar for additional infos
            infoToolBar = null,

            // the sheet selector controls
            sheetGroup = null,
            largeSheetList = null,
            smallSheetList = null,

            // label for raw formula-view
            formulaText = null;

        // base constructor ---------------------------------------------------

        ToolPane.call(this, app, { position: 'bottom', classes: 'status-pane standard-design' });

        // private methods ----------------------------------------------------

        /**
         * Refreshes the layout of the sheet selectors according to the current
         * width of the sheet tool bar.
         */
        function refreshSheetToolBar() {

            var // the node of the pane
                paneNode = self.getNode(),
                // total width of pane
                paneWidth = paneNode.width(),

                // the node of the sheet tool bar
                sheetNode = sheetToolBar.getNode(),
                // the width of the center node
                avaSheetWidth = sheetNode.closest('.float-wrapper').width(),

                // max width of the info node in percent
                minSheetWidth = 240;

            function getCurrentWidth(node) {
                return Math.ceil(node.outerWidth());
            }

            // calculate and set remaining space for the specified sheet selector
            function setRemainingWidth(group) {
                var remainingWidth = group.getNode().outerWidth() + avaSheetWidth - getCurrentWidth(sheetNode);
                group.setWidth(remainingWidth);
            }

            largeSheetList.hide();
            smallSheetList.hide();
            sheetGroup.show();
            sheetGroup.setFullWidth();

            // hide the sheet list selectors, show the sheet tab selector with automatic width
            sheetNode.css({ width: 'auto', paddingLeft: view.getHeaderWidth() + 'px' });

            if (avaSheetWidth < getCurrentWidth(sheetNode)) {
                sheetNode.css({ paddingLeft: '' });

                if (avaSheetWidth < 380) {
                    // check min width for sheets
                    if (avaSheetWidth < minSheetWidth) {
                        // set fix min width for sheets
                        avaSheetWidth = minSheetWidth;
                        // set remaining width (incl. 20px padding) to formula
                        formulaText.setWidth(paneWidth - minSheetWidth - 20);
                    } else {
                        // reset width of formula
                        formulaText.setWidth('auto');
                    }
                    // show large sheet drop-down list for small devices
                    sheetGroup.hide();
                    largeSheetList.show();
                    // calculate and set remaining space for the sheet drop-down button
                    largeSheetList.setWidth(avaSheetWidth);
                    setRemainingWidth(largeSheetList);
                } else {
                    // show additional sheet selector controls
                    smallSheetList.show();
                    // calculate and set remaining space for the sheet tabs
                    setRemainingWidth(sheetGroup);
                }
            } else {
                // reset width of formula
                formulaText.setWidth('auto');
            }

            // reset sheet component to its current width
            sheetNode.css({ width: avaSheetWidth });
        }

        // initialization -----------------------------------------------------

        this.addViewComponent(sheetToolBar = new ToolBar(app)
            .addGroup('view/sheet/active', largeSheetList = new Controls.ActiveSheetList(app, { showNames: true }))
            .addGap()
            .addGroup('view/sheet/active', smallSheetList = new Controls.ActiveSheetList(app))
            .addGap()
            .addGroup('view/sheet/active', sheetGroup = new Controls.ActiveSheetGroup(app))
            .addGap()
            .addGroup('sheet/insert', new Button({ icon: 'fa-plus', tooltip: gt('Insert sheet') }))
            .addGroup(null, new ComponentMenuButton(app, { label: Controls.MORE_LABEL, tooltip: gt('More sheet operations'), position: 'top' })
                .addGroup('sheet/rename/dialog', new Button({ label: gt('Rename sheet') }))
                .addGroup('sheet/copy/dialog', new Button({ label: gt('Copy sheet') }))
                .addGroup('sheet/delete', new Button({ label: gt('Delete sheet') }))
                .addGroup('sheet/visible', new Button({ label: gt('Hide sheet'), value: false }))
                .addSeparator()
                .addGroup('sheet/reorder/dialog', new Button({ label: gt('Reorder sheets') }))
                .addGroup('sheet/showall', new Button({ label: gt('Show all hidden sheets') }))
            )
        );

        this.addViewComponent(infoToolBar = new ToolBar(app)
            .addGroup('view/status', new Controls.DynamicLabel(), { visibleKey: 'view/status' })
            .addGroup('view/subtotals/type', new Controls.SubtotalList(app, { position: 'top' }), { visibleKey: 'view/subtotals/type' })
            .addGroup('view/formula', formulaText = new DynamicLabel({ classes: 'formula-text', setToolTip: true }), { visibleKey: 'view/formula'} ),
        { targetArea: 'trailing' });

        // refresh sheet tool bar after sheet tabs have been inserted, deleted, or changed
        this.listenTo(sheetGroup, 'change:sheets', refreshSheetToolBar);
        this.on('pane:layout', refreshSheetToolBar);

        // always hide the status pane when an internal application error occurs
        this.listenTo(app, 'docs:state:error', function () { self.hide(); });

        // destroy class members on destruction
        this.registerDestructor(function () {
            app = self = view = sheetToolBar = sheetGroup = largeSheetList = smallSheetList = null;
        });

    } // class StatusPane

    // exports ================================================================

    // derive this class from class ToolPane
    return ToolPane.extend({ constructor: StatusPane });

});
