/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/cornerpane',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/object/triggerobject',
     'io.ox/office/spreadsheet/utils/sheetutils'
    ], function (Utils, TriggerObject, SheetUtils) {

    'use strict';

    // class CornerPane =======================================================

    /**
     * Represents the top-left corner in the spreadsheet view.
     *
     * Triggers the following events:
     * - 'select:all'
     *      After this corner pane has been clicked which will select the
     *      entire sheet. Event handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {String} mode
     *          The selection mode according to the keyboard modifier keys:
     *          - 'select': Standard selection without modifier keys. This
     *              should move the current active cell in the selection to the
     *              first visible cell in the selection.
     *          - 'extend': Extend current active range (SHIFT). This should
     *              keep the current active cell in the selection unmodified.
     *
     * @constructor
     *
     * @extends TriggerObject
     *
     * @param {SpreadsheetApplication} app
     *  The application that contains this corner pane.
     */
    function CornerPane(app) {

        var // self reference
            self = this,

            // the spreadsheet model and view
            view = null,

            // the container node of this corner pane
            rootNode = $('<div class="corner-pane">'),

            // the cell DOM node used to calculate the size
            cellNode = $('<div class="cell noI18n">');

        // base constructor ---------------------------------------------------

        TriggerObject.call(this);

        // private methods ----------------------------------------------------

        /**
         * Handles mouse clicks and selects the entire sheet.
         */
        function clickHandler(event) {

            if ((event.type === 'touchstart') || (event.button === 0)) {
                self.trigger('select:all', event.shiftKey ? 'extend' : 'select');
            }

            // return focus to active grid pane, after the event has been processed
            _.defer(function () { view.grabFocus(); });
        }

        // methods ------------------------------------------------------------

        /**
         * Returns the root DOM node of this header pane.
         *
         * @returns {jQuery}
         *  The root node of this header pane, as jQuery object.
         */
        this.getNode = function () {
            return rootNode;
        };

        /**
         * Initializes the size of this corner pane.
         *
         * @param {Number} maxRow
         *  The maximum index of the row to be displayed in row header panes.
         *
         * @returns {CornerPane}
         *  A reference to this instance.
         */
        this.initializePaneLayout = function (maxRow) {

            var // the exact node size (fractional pixels)
                nodeSize = null;

            // prepare the cell node text and font size
            cellNode
                .text(SheetUtils.getRowName(maxRow))
                .css({ fontSize: view.getHeaderFontSize() + 'pt', lineHeight: 'normal' })
                .appendTo(rootNode);

            // copy size of cell node to root node, and remove the cell node
            nodeSize = Utils.getExactNodeSize(cellNode);
            rootNode.css({ width: Math.ceil(nodeSize.width) + 2, height: Math.ceil(nodeSize.height) + 2 });
            cellNode.remove();

            return this;
        };

        // initialization -----------------------------------------------------

        // initialize class members
        app.onInit(function () {
            // the spreadsheet view
            view = app.getView();
            // immediately react on mousedown (do not wait for click)
            rootNode.on('mousedown touchstart', clickHandler);
        });

    } // class CornerPane

    // exports ================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: CornerPane });

});
