/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/controls',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/forms',
     'io.ox/office/editframework/utils/color',
     'io.ox/office/editframework/utils/border',
     'io.ox/office/editframework/view/editcontrols',
     'io.ox/office/spreadsheet/utils/sheetutils',
     'io.ox/office/spreadsheet/utils/paneutils',
     'io.ox/office/spreadsheet/view/mixin/orderdndmixin',
     'gettext!io.ox/office/spreadsheet'
    ], function (Utils, Forms, Color, Border, EditControls, SheetUtils, PaneUtils, OrderDnDMixin, gt) {

    'use strict';

    var // class name shortcuts
        RadioGroup = EditControls.RadioGroup,
        RadioList = EditControls.RadioList,

        // all available border styles
        BORDER_PRESET_STYLES = [
            { value: 'single:hair',       label: /*#. border line style (width 0.5 pixels, solid) */ gt.pgettext('borders', 'Hair line'),                         icon: { style: 'solid',      width: 0 } },
            { value: 'single:thin',       label: /*#. border line style (width 1 pixel, solid) */ gt.pgettext('borders', 'Thin line'),                            icon: { style: 'solid',      width: 1 } },
            { value: 'dashed:thin',       label: /*#. border line style (width 1 pixel, dashed) */ gt.pgettext('borders', 'Dashed thin line'),                    icon: { style: 'dashed',     width: 1 } },
            { value: 'dotted:thin',       label: /*#. border line style (width 1 pixel, dotted) */ gt.pgettext('borders', 'Dotted thin line'),                    icon: { style: 'dotted',     width: 1 } },
            { value: 'dashDot:thin',      label: /*#. border line style (width 1 pixel, dash-dot) */ gt.pgettext('borders', 'Dot-and-dash thin line'),            icon: { style: 'dashDot',    width: 1 } },
            { value: 'dashDotDot:thin',   label: /*#. border line style (width 1 pixel, dash-dot-dot) */ gt.pgettext('borders', 'Two-dots-and-dash thin line'),   icon: { style: 'dashDotDot', width: 1 } },
            { value: 'single:medium',     label: /*#. border line style (width 2 pixels, solid) */ gt.pgettext('borders', 'Medium line'),                         icon: { style: 'solid',      width: 2 } },
            { value: 'dashed:medium',     label: /*#. border line style (width 2 pixels, dashed) */ gt.pgettext('borders', 'Dashed medium line'),                 icon: { style: 'dashed',     width: 2 } },
            { value: 'dashDot:medium',    label: /*#. border line style (width 2 pixel, dash-dot) */ gt.pgettext('borders', 'Dot-and-dash medium line'),          icon: { style: 'dashDot',    width: 2 } },
            { value: 'dashDotDot:medium', label: /*#. border line style (width 2 pixel, dash-dot-dot) */ gt.pgettext('borders', 'Two-dots-and-dash medium line'), icon: { style: 'dashDotDot', width: 2 } },
            { value: 'single:thick',      label: /*#. border line style (width 3 pixels, solid) */ gt.pgettext('borders', 'Thick line'),                          icon: { style: 'solid',      width: 3 } },
            { value: 'double:thick',      label: /*#. border line style (width 3 pixels, double line) */ gt.pgettext('borders', 'Double line'),                   icon: { style: 'solid',      width: 1, count: 2 } }
        ],

        // label for a custom number format list entry
        CUSTOM_LABEL = /*#. number format category in spreadsheets: all user-defined number formats */ gt.pgettext('number-format', 'Custom'),

        // predefined number format categories
        NUMBER_FORMAT_CATEGORIES = [
            { value: 'standard',   label: /*#. number format category in spreadsheets: no special format, show numbers as they are */ gt.pgettext('number-format', 'General') },
            { value: 'number',     label: /*#. number format category in spreadsheets: numbers with specific count of decimal places */ gt.pgettext('number-format', 'Number') },
            { value: 'currency',   label: /*#. number format category in spreadsheets: number with currency symbol */ gt.pgettext('number-format', 'Currency') },
            { value: 'date',       label: /*#. number format category in spreadsheets: date formats */ gt.pgettext('number-format', 'Date') },
            { value: 'time',       label: /*#. number format category in spreadsheets: time formats */ gt.pgettext('number-format', 'Time') },
            { value: 'datetime',   label: /*#. number format category in spreadsheets: combined date/time formats */ gt.pgettext('number-format', 'Date and time') },
            { value: 'percent',    label: /*#. number format category in spreadsheets: numbers with percent sign */ gt.pgettext('number-format', 'Percentage') },
            { value: 'text',       label: /*#. number format category in spreadsheets: text only */ gt.pgettext('number-format', 'Text')},
            { value: 'scientific', label: /*#. number format category in spreadsheets: scientific notation (e.g. 1.23E+10) */ gt.pgettext('number-format', 'Scientific') },
            { value: 'fraction',   label: /*#. number format category in spreadsheets: fractional numbers (e.g. 3 1/4) */ gt.pgettext('number-format', 'Fraction') },
            { value: 'custom',     label: CUSTOM_LABEL, hidden: true }
        ],

        // all number format GUI labels, mapped by category name
        NUMBER_FORMAT_CATEGORY_LABELS = _.chain(NUMBER_FORMAT_CATEGORIES).map(function (entry) { return [entry.value, entry.label]; }).object().value();

    // static class SpreadsheetControls =======================================

    /**
     * Additional classes defining specialized GUI controls for the OX
     * Spreadsheet application.
     *
     * @extends EditControls
     */
    var SpreadsheetControls = _.clone(EditControls);

    // constants ==============================================================

    /**
     * A text label or tool tip for a 'Format Painter' control.
     *
     * @constant
     */
    SpreadsheetControls.FORMAT_PAINTER_LABEL =
        //#. copy formatting from one location and apply it to other cells
        gt('Format painter');

    /**
     * A special value for the 'Custom' entry in the 'Number Format Code'
     * drop-down list, which must be different from any valid number format.
     *
     * @constant
     */
    SpreadsheetControls.CUSTOM_FORMAT_VALUE = '\x00';

    // class ActiveSheetGroup =================================================

    /**
     * The selector for the current active sheet, as radio group (all sheet
     * tabs are visible at the same time).
     *
     * @constructor
     *
     * @extends RadioGroup
     */
    SpreadsheetControls.ActiveSheetGroup = RadioGroup.extend({ constructor: function (app) {

        var // self reference
            self = this,

            // get application model and view
            model = null,
            view = null,

            // the scroll button nodes
            prevButton = $(Forms.createButtonMarkup({ icon: 'fa-angle-left', tooltip: gt('Scroll to the previous sheet') })).addClass('scroll-button'),
            nextButton = $(Forms.createButtonMarkup({ icon: 'fa-angle-right', tooltip: gt('Scroll to the next sheet') })).addClass('scroll-button'),

            // index of the first and last sheet tab button currently visible
            firstIndex = 0,
            lastIndex = 0,

            // whether scroll mode is enabled
            scrollable = false;

        // base constructor ---------------------------------------------------

        RadioGroup.call(this, { classes: 'active-sheet-group', role: 'tablist' });
        OrderDnDMixin.call(this, app, { cursor: 'ew-resize', icon: 'fa-caret-down' });

        // private methods ----------------------------------------------------

        /**
         * Shows all sheet buttons and removes explicit width, and hides the
         * scroll buttons.
         */
        function showAllSheetButtons() {
            self.getOptionButtons().removeClass('hidden').css('width', '');
            prevButton.addClass('hidden');
            nextButton.addClass('hidden').css({ position: '', right: '' });
            scrollable = false;
        }

        /**
         * Shows the sheet buttons starting at the start sheet index currently
         * cached.
         */
        function renderSheetButtons() {

            var // the root node of this group
                groupNode = self.getNode(),
                // the inner width of the group (space available for buttons)
                groupWidth = groupNode.width(),
                // all sheet tab buttons
                sheetButtons = self.getOptionButtons(),
                // the width of all sheet buttons
                buttonsWidth = 0,
                // relative start/end position of all sheet buttons
                buttonPositions = null,
                // total width of all buttons, including scroll buttons
                totalWidth = 0,
                // remaining width available in the group node for sheet buttons
                remainingWidth = 0,
                // maximum index of button that can be shown at the left border (without gap to right border)
                maxIndex = 0;

            // do nothing if there are no sheets available (yet)
            if (sheetButtons.length === 0) { return; }

            // initialize all sheet buttons, hide scroll buttons
            showAllSheetButtons();
            buttonsWidth = sheetButtons.last().offset().left + sheetButtons.last().outerWidth() - sheetButtons.first().offset().left;
            if (buttonsWidth <= groupWidth) { return; }

            // collect position information of all buttons (performance: once for all buttons, before manipulating DOM again)
            var leadingOffset = sheetButtons.first().offset().left;
            buttonPositions = sheetButtons.map(function () {
                var buttonOffset = $(this).offset().left - leadingOffset;
                return {
                    offset: buttonOffset,
                    leadingWidth: buttonOffset + $(this).outerWidth(),
                    trailingWidth: buttonsWidth - buttonOffset
                };
            });

            // do not show scroll buttons, if there is only one visible sheet
            if (sheetButtons.length > 1) {

                // show scroll buttons
                prevButton.removeClass('hidden');
                nextButton.removeClass('hidden');
                scrollable = true;

                // get total width of all buttons, including scroll buttons, and inner remaining width
                totalWidth = nextButton.offset().left + nextButton.outerWidth() - prevButton.offset().left;
                remainingWidth = groupWidth - (totalWidth - buttonsWidth);

                // hide leading sheet tabs
                maxIndex = Utils.findFirstIndex(buttonPositions, function (position) { return position.trailingWidth <= remainingWidth; }, { sorted: true });
                firstIndex = Utils.minMax(firstIndex, 0, Math.max(0, maxIndex));
                sheetButtons.slice(0, firstIndex).addClass('hidden');

                // adjust button positions relative to first visible button
                var relOffset = buttonPositions[firstIndex].offset;
                _.each(buttonPositions, function (position) {
                    position.offset -= relOffset;
                    position.leadingWidth -= relOffset;
                });

                // hide trailing sheet tabs
                maxIndex = Utils.findFirstIndex(buttonPositions, function (position) { return position.leadingWidth > remainingWidth; }, { sorted: true });
                lastIndex = ((maxIndex < 0) ? sheetButtons.length : maxIndex) - 1;
                sheetButtons.slice(lastIndex + 1).addClass('hidden');

                // enable/disable scroll buttons
                Forms.enableNodes(prevButton, firstIndex > 0);
                Forms.enableNodes(nextButton, lastIndex + 1 < sheetButtons.length);

            } else {
                // single sheet: initialize indexes
                firstIndex = lastIndex = 0;
                totalWidth = buttonsWidth;
                remainingWidth = groupWidth;
            }

            // set fixed position for the right scroll button
            nextButton.css({ position: 'absolute', right: 0 });

            // shrink single visible sheet button, if still oversized
            var sheetButton = sheetButtons.eq(firstIndex);
            if ((firstIndex === lastIndex) && (remainingWidth < sheetButton.outerWidth())) {
                sheetButton.width(remainingWidth - (sheetButton.outerWidth() - sheetButton.width()));
                return;
            }

            // add the first hidden sheet button after the last visible, if there is some space left
            if (lastIndex + 1 === buttonPositions.length) { return; }
            remainingWidth -= buttonPositions[lastIndex + 1].offset;
            if (remainingWidth >= 30) {
                sheetButton = sheetButtons.eq(lastIndex + 1);
                sheetButton.removeClass('hidden');
                sheetButton.width(remainingWidth - (sheetButton.outerWidth() - sheetButton.width()));
            }
        }

        /**
         * Scrolls to the left to make at least one preceding sheet tab button
         * visible.
         */
        function scrollToPrev() {

            var // index of the old last visible sheet
                oldLastIndex = lastIndex;

            // move to previous sheets until the index of the last sheet has changed
            while ((firstIndex > 0) && (oldLastIndex === lastIndex)) {
                firstIndex -= 1;
                renderSheetButtons();
            }
        }

        /**
         * Scrolls to the right to make at least one following sheet tab button
         * visible.
         */
        function scrollToNext() {

            var // all sheet tab buttons
                sheetButtons = self.getOptionButtons(),
                // index of the old last visible sheet
                oldLastIndex = lastIndex;

            // move to next sheets until the index of the last sheet has changed
            while ((lastIndex + 1 < sheetButtons.length) && (oldLastIndex === lastIndex)) {
                firstIndex += 1;
                renderSheetButtons();
            }
        }

        /**
         * Handles tracking on the scroll-back button. Stops event propagation,
         * to not interfere with sheet move tracking.
         */
        function prevTrackingHandler(event) {
            scrollToPrev();
            event.stopPropagation();
        }

        /**
         * Handles tracking on the scroll-forward button. Stops event
         * propagation, to not interfere with sheet move tracking.
         */
        function nextTrackingHandler(event) {
            scrollToNext();
            event.stopPropagation();
        }

        /**
         * Handles end of tracking on the scroll buttons. Moves the browser
         * focus back to the application, and stops event propagation, to not
         * interfere with sheet move tracking.
         */
        function trackingEndHandler(event) {
            view.grabFocus();
            event.stopPropagation();
        }

        /**
         * Called every time the value of the control (the index of the active
         * sheet) has been changed.
         */
        function updateHandler(newIndex, options, oldIndex) {

            // do not change anything, if the sheet index did not change
            if (!scrollable || (newIndex === oldIndex)) {
                return;
            }

            // scroll backwards, if the active sheet is located before the first visible
            if (newIndex < firstIndex) {
                firstIndex = newIndex;
                renderSheetButtons();
                return;
            }

            // scroll forwards, if the active sheet is located after the last visible
            if (newIndex > lastIndex) {
                firstIndex = newIndex;
                renderSheetButtons();
            }
        }

        /**
         * Handles double click events on sheet tab buttons and initiates
         * renaming the sheet.
         */
        function doubleClickHandler(event) {

            var // the zero-based index of the sheet to be renamed
                sheet = Forms.getButtonValue(event.currentTarget);

            // ignore double clicks on scroll buttons
            if (_.isNumber(sheet)) {
                app.getController().executeItem('sheet/rename/dialog');
            }
        }

        /**
         * Recreates all sheet tabs in this radio group.
         */
        var fillList = app.createDebouncedMethod($.noop, function () {

            // create buttons for all visible sheets
            self.clearOptionButtons();
            view.iterateVisibleSheets(function (sheetModel, index, sheetName) {
                self.createOptionButton(index, {
                    icon: sheetModel.isLocked() ? 'fa-lock' : null,
                    iconClasses: 'small-icon',
                    label: _.noI18n(sheetName),
                    tooltip: _.noI18n(sheetName),
                    attributes: { role: 'tab' }
                });
            });

            // move trailing scroll button to the end
            self.addChildNodes(nextButton);

            // explicitly notify listeners, needed for reliable layout
            self.trigger('change:sheets');
        });

        // methods ------------------------------------------------------------

        /**
         * Sets this control to automatic width, shows all sheet tab buttons,
         * and hides the scroll buttons.
         *
         * @returns {ActiveSheetGroup}
         *  A reference to this instance.
         */
        this.setFullWidth = function () {
            this.getNode().css('width', '');
            showAllSheetButtons();
            return this;
        };

        /**
         * Sets this control to a fixed width. Performs further initialization
         * according to the available space: shows/hides the scroll buttons,
         * and draws the available sheet tabs according to the last cached
         * scroll position.
         *
         * @param {Number} width
         *  The new outer width of this control, in pixels.
         *
         * @returns {ActiveSheetGroup}
         *  A reference to this instance.
         */
        this.setWidth = function (width) {
            this.getNode().width(width);
            renderSheetButtons();
            return this;
        };

        // initialization -----------------------------------------------------

        // get application model and view
        app.onInit(function () {
            model = app.getModel();
            view = app.getView();
            self.listenTo(view, 'change:sheets', fillList); // registering handler early for fast preview
        });

        // the update handler scrolls to the active sheet, after it has been changed
        this.registerUpdateHandler(updateHandler);

        // rebuild the list of available sheets after it has been changed
        this.listenTo(app.getImportPromise(), 'done', function () {
            fillList();
            self.listenTo(model, 'change:sheet:attributes', fillList); // e.g. sheet protection
        });

        // add scroll buttons to the group node
        this.addChildNodes(prevButton, nextButton);

        // handle tracking events of scroll buttons
        prevButton.enableTracking({ autoRepeat: true })
            .on('tracking:start tracking:repeat', prevTrackingHandler)
            .on('tracking:move', function (event) { event.stopPropagation(); })
            .on('tracking:end tracking:cancel', trackingEndHandler);
        nextButton.enableTracking({ autoRepeat: true })
            .on('tracking:start tracking:repeat', nextTrackingHandler)
            .on('tracking:move', function (event) { event.stopPropagation(); })
            .on('tracking:end tracking:cancel', trackingEndHandler);

        // double click shows rename dialog (TODO: in-place editing)
        this.getNode().on('dblclick', Forms.BUTTON_SELECTOR, doubleClickHandler);

        // change sheet order when dragging sheet buttons around
        this.on('change:order', function (evt, from, to) {
            if (view.leaveCellEditMode('auto', { validate: true })) {
                model.moveSheet(from, to);
            }
        });

    }}); // class ActiveSheetGroup

    // class ActiveSheetList ==================================================

    /**
     * The selector for the current active sheet, as radio drop-down list.
     *
     * @constructor
     *
     * @extends RadioList
     */
    SpreadsheetControls.ActiveSheetList = RadioList.extend({ constructor: function (app, initOptions) {

        var // self reference
            self = this,

            // whether to show the sheet name as label text
            showNames = Utils.getBooleanOption(initOptions, 'showNames', false);

        // base constructor ---------------------------------------------------

        RadioList.call(this, {
            icon: showNames ? null : 'fa-bars',
            iconClasses: showNames ? 'small-icon' : null,
            tooltip: gt('Select sheet'),
            position: 'top',
            caret: showNames,
            updateCaptionMode: showNames ? 'label icon' : 'none'
        });

        // private methods ----------------------------------------------------

        /**
         * Recreates all sheet entries in the drop-down menu.
         */
        var fillList = app.createDebouncedMethod($.noop, function () {
            self.clearMenu();
            app.getView().iterateVisibleSheets(function (sheetModel, index, sheetName) {
                self.createOptionButton(index, {
                    icon: sheetModel.isLocked() ? 'fa-lock' : null,
                    iconClasses: 'small-icon',
                    label: _.noI18n(sheetName),
                    tooltip: _.noI18n(sheetName)
                });
            });
        });

        // methods ------------------------------------------------------------

        /**
         * Sets this control to fixed width mode.
         *
         * @param {Number} width
         *  The new outer width of this control.
         *
         * @returns {ActiveSheetList}
         *  A reference to this instance.
         */
        this.setWidth = function (width) {
            this.getNode().css({ width: width });
            // prevent wrong position/size of the button element due to padding/margin
            this.getMenuButton().css({ position: 'absolute', left: 0, right: 0 });
            return this;
        };

        // initialization -----------------------------------------------------

        // registering handler at the view
        app.onInit(function () {
            self.listenTo(app.getView(), 'change:sheets', fillList);
        });

        this.listenTo(app.getImportPromise(), 'done', function () {
            fillList();
            self.listenTo(app.getModel(), 'change:sheet:attributes', fillList); // e.g. sheet protection
        });

    }}); // class ActiveSheetList

    // class CellHAlignmentPicker =============================================

    /**
     * A picker control for horizontal cell alignment.
     *
     * @constructor
     *
     * @extends RadioList
     */
    SpreadsheetControls.CellHAlignmentPicker = RadioList.extend({ constructor: function (initOptions) {

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({
            icon: 'docs-para-align-left',
            tooltip: /*#. text alignment in cells */ gt('Horizontal alignment'),
            updateCaptionMode: 'icon',
            dropDownVersion: {
                label: gt('Horizontal alignment')
            }
        }, initOptions));

        // initialization -----------------------------------------------------

        this.createOptionButton('auto',    { icon: 'docs-cell-h-align-auto',    label: EditControls.HOR_ALIGNMENT_LABELS.auto })
            .createOptionButton('left',    { icon: 'docs-cell-h-align-left',    label: EditControls.HOR_ALIGNMENT_LABELS.left })
            .createOptionButton('center',  { icon: 'docs-cell-h-align-center',  label: EditControls.HOR_ALIGNMENT_LABELS.center })
            .createOptionButton('right',   { icon: 'docs-cell-h-align-right',   label: EditControls.HOR_ALIGNMENT_LABELS.right })
            .createOptionButton('justify', { icon: 'docs-cell-h-align-justify', label: EditControls.HOR_ALIGNMENT_LABELS.justify });

    }}); // class CellHAlignmentPicker

    // class CellVAlignmentPicker =============================================

    /**
     * A picker control for vertical cell alignment.
     *
     * @constructor
     *
     * @extends RadioList
     */
    SpreadsheetControls.CellVAlignmentPicker = RadioList.extend({ constructor: function (initOptions) {

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({
            icon: 'docs-cell-vertical-bottom',
            tooltip: /*#. text alignment in cells */ gt('Vertical alignment'),
            updateCaptionMode: 'icon',
            dropDownVersion: {
                label: gt('Vertical alignment')
            }
        }, initOptions));

        // initialization -----------------------------------------------------

        this.createOptionButton('top',     { icon: 'docs-cell-v-align-top',     label: EditControls.VERT_ALIGNMENT_LABELS.top })
            .createOptionButton('middle',  { icon: 'docs-cell-v-align-middle',  label: EditControls.VERT_ALIGNMENT_LABELS.middle })
            .createOptionButton('bottom',  { icon: 'docs-cell-v-align-bottom',  label: EditControls.VERT_ALIGNMENT_LABELS.bottom })
            .createOptionButton('justify', { icon: 'docs-cell-v-align-justify', label: EditControls.VERT_ALIGNMENT_LABELS.justify });

    }}); // class CellVAlignmentPicker

    // class CellBorderPicker =================================================

    /**
     * The selector for cell borders. Hides specific items in the drop-down
     * menu, according to the current cell selection.
     *
     * @constructor
     *
     * @extends EditControls.BorderPicker
     */
    SpreadsheetControls.CellBorderPicker = EditControls.BorderPicker.extend({ constructor: function (app, initOptions) {

        var // the spreadsheet view instance
            view = app.getView();

        // base constructor ---------------------------------------------------

        EditControls.BorderPicker.call(this, Utils.extendOptions({
            tooltip: EditControls.CELL_BORDERS_LABEL
        }, initOptions, {
            showInsideHor: function () { return view.hasMultipleRowsSelected(); },
            showInsideVert: function () { return view.hasMultipleColumnsSelected(); }
        }));

    }}); // class CellBorderPicker

    // class BorderPresetStylePicker ==========================================

    /**
     * A drop-down list control for border line styles used in OOXML files.
     *
     * @constructor
     *
     * @extends EditControls.BorderStylePicker
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options supported by the base class
     *  EditControls.BorderStylePicker.
     */
    SpreadsheetControls.BorderPresetStylePicker = EditControls.BorderStylePicker.extend({ constructor: function (app, initOptions) {

        // base constructor ---------------------------------------------------

        EditControls.BorderStylePicker.call(this, BORDER_PRESET_STYLES, initOptions);

    }}); // class BorderPresetStylePicker

    // class CellBorderColorPicker ============================================

    /**
     * The selector for the cell border color.
     *
     * @constructor
     *
     * @extends EditControls.ColorPicker
     */
    SpreadsheetControls.CellBorderColorPicker = EditControls.ColorPicker.extend({ constructor: function (app, initOptions) {

        var // self reference
            self = this;

        // base constructor ---------------------------------------------------

        EditControls.ColorPicker.call(this, app, getGridColor(), Utils.extendOptions({
            icon: 'docs-cell-fill-color',
            tooltip: EditControls.BORDER_COLOR_LABEL
        }, initOptions));

        // private methods ----------------------------------------------------

        function getGridColor() {
            return app.getView().getEffectiveGridColor(Color.BLACK);
        }

        // initialization -----------------------------------------------------

        // refresh the 'automatic' border color (depends on sheet grid color)
        this.listenTo(app.getView(), 'change:sheet:viewattributes', function () {
            self.setAutoColor(getGridColor());
        });

    }}); // class CellBorderColorPicker

    // class MergePicker ======================================================

    /**
     * The selector for merged ranges, as drop-down list with split button.
     *
     * @constructor
     *
     * @extends RadioList
     */
    SpreadsheetControls.MergePicker = RadioList.extend({ constructor: function (app, initOptions) {

        var // self reference
            self = this,

            // whether to show the 'merge' menu item
            showMergeItem = false,
            // whether to show the 'merge horizontal' menu item
            showHMergeItem = false,
            // whether to show the 'merge vertical' menu item
            showVMergeItem = false,
            // whether to show the 'unmerge' menu item
            showUnmergeItem = false;

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({
            icon: 'docs-merged-cells-on',
            tooltip: gt('Merge or unmerge cells')
        }, initOptions, {
            highlight: _.identity,
            splitValue: 'toggle',
            updateCaptionMode: 'none'
        }));

        // private methods ----------------------------------------------------

        /**
         * Updates the icon according to the merged state of the selection.
         */
        function updateHandler() {
            self.setIcon(showUnmergeItem ? 'docs-merged-cells-off' : 'docs-merged-cells-on');
        }

        /**
         * Updates the visibility flags of the list items after the selection
         * in the active sheet has been changed, or after the merge collection
         * of the active sheet has changed.
         */
        function updateVisibilityFlags() {

            var // the collection of merged ranges in the active sheet
                mergeCollection = app.getView().getMergeCollection(),
                // the selected cell ranges
                ranges = app.getView().getSelectedRanges();

            // show the 'merge' list item, if at least one range consists of more than a cell or a merged range
            showMergeItem = _.any(ranges, function (range) {
                var mergedRange = (SheetUtils.getCellCount(range) > 1) ? mergeCollection.getMergedRange(range.start) : null;
                return !mergedRange || !SheetUtils.rangeContainsRange(mergedRange, range);
            });

            // show the 'horizontal' list item, if at least one range does not consist entirely of horizontally merged cells
            showHMergeItem = _.any(ranges, function (range) {
                return (range.start[0] < range.end[0]) && (range.start[1] < range.end[1]) && !mergeCollection.isHorizontallyMerged(range);
            });

            // show the 'vertical' list item, if at least one range does not consist entirely of vertically merged cells
            showVMergeItem = _.any(ranges, function (range) {
                return (range.start[0] < range.end[0]) && (range.start[1] < range.end[1]) && !mergeCollection.isVerticallyMerged(range);
            });

            // show the 'unmerge' list item, if at least one range overlaps with a merged range
            showUnmergeItem = mergeCollection.rangesOverlapAnyMergedRange(ranges);
        }

        // initialization -----------------------------------------------------

        // create all drop-down menu items
        this.createOptionButton('merge',      { icon: 'docs-merged-cells-on',         label: gt('Merge cells'),              visible: function () { return showMergeItem; } })
            .createOptionButton('horizontal', { icon: 'docs-merged-cells-horizontal', label: gt('Merge cells horizontally'), visible: function () { return showHMergeItem; } })
            .createOptionButton('vertical',   { icon: 'docs-merged-cells-vertical',   label: gt('Merge cells vertically'),   visible: function () { return showVMergeItem; } })
            .createOptionButton('unmerge',    { icon: 'docs-merged-cells-off',        label: gt('Unmerge cells'),            visible: function () { return showUnmergeItem; } });

        // update the icon in the drop-down button
        this.registerUpdateHandler(updateHandler);

        // update state flags used for visibility of list items (to keep the 'visible' callback functions simple and fast)
        this.listenTo(app.getImportPromise(), 'done', function () {
            updateVisibilityFlags();
            self.listenTo(app.getView(), 'change:selection insert:merged delete:merged', updateVisibilityFlags);
        });

    }}); // class MergePicker

    // class FormatCategoryGroup ==============================================

    /**
     * A button group with selected number format categories.
     *
     * @constructor
     *
     * @extends RadioGroup
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options of the base class RadioGroup.
     */
    SpreadsheetControls.FormatCategoryGroup = RadioList.extend({ constructor: function (app, initOptions) {

        var // self reference
            self = this,
            // accessible name for the widget
            ariaName =  /*#. number format categories */ gt('Number format categories');

        // base constructor ---------------------------------------------------

        RadioGroup.call(this, Utils.extendOptions({ toggleValue: 'standard', role: 'radiogroup' }, initOptions));

        // initialization -----------------------------------------------------

        // set accessible name - WCAG 2.0 Success Criterion 4.1.2
        this.getNode().addClass('format-category-group').attr({ title: ariaName, 'aria-label': ariaName, 'aria-labelledby': 'dummy' });

        // initialize the option buttons
        this.listenTo(app.getImportPromise(), 'done', function () {

            var // the number formatter providing the currency symbol for the currency button
                numberFormatter = app.getModel().getNumberFormatter(),
                // the currency symbol of the current language
                currencySymbol = numberFormatter.getCurrencySymbol();

            self.createOptionButton('currency', {
                    label: _.noI18n(currencySymbol),
                    tooltip: NUMBER_FORMAT_CATEGORY_LABELS.currency,
                    attributes: { role: 'radio' },
                    dropDownVersion: {
                        label: gt('Currency')
                    }
                })
                .createOptionButton('percent', {
                    label: _.noI18n('%'),
                    tooltip: NUMBER_FORMAT_CATEGORY_LABELS.percent,
                    attributes: { role: 'radio' },
                    dropDownVersion: {
                        label: gt('Percent')
                    }
                });
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            app = initOptions = self = null;
        });

    }}); // class FormatCategoryGroup

    // class FormatCategoryPicker =============================================

    /**
     * A drop-down list control for choosing number format categories, that
     * will load predefined format codes in the FormatCodePicker.
     *
     * @constructor
     *
     * @extends RadioList
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options of the base class RadioList.
     */
    SpreadsheetControls.FormatCategoryPicker = RadioList.extend({ constructor: function (app, initOptions) {

        var // self reference
            self = this;

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({
            label: _.noI18n('123'),
            tooltip: gt('Number format')
        }, initOptions, {
            updateCaptionMode: 'none'
        }));

        // initialization -----------------------------------------------------

        this.getNode().addClass('format-category-group');

        // initialize the drop-down list items
        this.listenTo(app.getImportPromise(), 'done', function () {
            // insert all supported list items
            _.each(NUMBER_FORMAT_CATEGORIES, function (entry) {
                if (!entry.hidden) {
                    self.createOptionButton(entry.value, { label: entry.label, dataValue: entry.value });
                }
            });
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            app = initOptions = self = null;
        });

    }}); // class FormatCategoryPicker

    // class FormatCodePicker =================================================

    /**
     * A drop-down list control for selection of predefined number formats
     * depending to currently selected category in the FormatCategoryPicker.
     *
     * @constructor
     *
     * @extends RadioList
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options of the base class RadioList.
     */
    SpreadsheetControls.FormatCodePicker = RadioList.extend({ constructor: function (app, initOptions) {

        var // self reference
            self = this,

            // the model and view
            model = app.getModel(),
            view = app.getView(),

            // the category of the format codes currently shown in the list
            listCategory = null;

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({
            label: EditControls.MORE_LABEL,
            tooltip: /*#. special number format codes for different categories of number formats */ gt('Format codes')
        }, initOptions, {
            updateCaptionMode: 'none'
        }));

        // private methods ----------------------------------------------------

        /**
         * Updates the label of the drop-down menu button according to the
         * current value of this control.
         */
        function updateHandler(value) {

            var // the current number format category
                currentCategory = view.getNumberFormatCategory(),
                // all codes for the current number format
                categoryCodes = model.getNumberFormatter().getCategoryCodes(currentCategory),
                // the active format code
                formatCodeEntry = _.findWhere(categoryCodes, { value: value }),
                //  menu instance of the radio list
                menu = self.getMenu();


            // refill the list, if the category has changed
            if (listCategory !== currentCategory) {
                self.clearMenu();
                _.each(categoryCodes, function (entry) {
                    self.createOptionButton(entry.value, {
                        label: _.noI18n(entry.label),
                        labelStyle: entry.red ? 'color:#F00000;' : null
                    });
                });
                listCategory = currentCategory;
            }

            // handle format codes which are not supported by injecting a 'Custom' button on the fly.
            if (!formatCodeEntry) {
                var customNode = menu.findItemNodes(SpreadsheetControls.CUSTOM_FORMAT_VALUE);
                // if custom button already exist, just select it and quit
                if (customNode.length !== 0) {
                    menu.selectItemNode(customNode);
                    return;
                }
                // create custom button and select it
                self.createOptionButton(SpreadsheetControls.CUSTOM_FORMAT_VALUE, { label: CUSTOM_LABEL });
                menu.selectMatchingItemNodes(SpreadsheetControls.CUSTOM_FORMAT_VALUE);
            } else {
                // remove old, invalid custom node
                menu.deleteItemNodes(SpreadsheetControls.CUSTOM_FORMAT_VALUE);
            }

        }

        // initialization -----------------------------------------------------

        // register custom update handler to update button caption
        this.registerUpdateHandler(updateHandler);

    }}); // class FormatCodePicker

    // class CellStylePicker ==================================================

    /**
     * A drop-down menu for cell style sheets.
     *
     * @constructor
     *
     * @extends EditControls.StyleSheetPicker
     */
    SpreadsheetControls.CellStylePicker = EditControls.StyleSheetPicker.extend({ constructor: function (app) {

        var // the collection of cell styles in the document
            styleCollection = app.getModel().getStyleCollection('cell');

        // base constructor ---------------------------------------------------

        EditControls.StyleSheetPicker.call(this, app, 'cell', {
            width: 150,
            icon: 'docs-table-style',
            tooltip: /*#. tool tip: predefined styles for spreadsheet cells */ gt('Cell style'),
            gridColumns: 6,
            i18nModulePath: 'io.ox/office/spreadsheet/resource/cellstylenames',
            sections: ['markup', 'headings', 'themes', 'hidden'],
            smallerVersion: {
            	css: {
            		width: 56
            	},
            	hideLabel: true
            }
        });

        // private methods ----------------------------------------------------

        /**
         * Updates the passed button element in the drop-down menu.
         */
        function createItemHandler(event, buttonNode, styleId) {

            var // the merged attributes of the passed cell style sheet
                attributes = styleCollection.getStyleSheetAttributes(styleId),
                // the character attributes
                charAttrs = attributes.character,
                // the cell attributes
                cellAttrs = attributes.cell,
                // the caption element to be formatted
                captionNode = Forms.getCaptionNode(buttonNode),
                // an additional border node as overlay (to prevent different line height in caption)
                borderMarkup = null;

            // add relevant character and cell formatting
            captionNode.css({
                fontFamily: styleCollection.getCssFontFamily(charAttrs.fontName),
                fontSize: Utils.minMax(Utils.round(10 + (charAttrs.fontSize - 10) / 1.5, 0.1), 6, 22) + 'pt',
                fontWeight: charAttrs.bold ? 'bold' : 'normal',
                fontStyle: charAttrs.italic ? 'italic' : 'normal',
                textDecoration: styleCollection.getCssTextDecoration(charAttrs),
                color: styleCollection.getCssTextColor(charAttrs.color, [cellAttrs.fillColor]),
                textAlign: PaneUtils.getCssTextAlignment({ result: 'a', attributes: attributes }),
                backgroundColor: styleCollection.getCssColor(cellAttrs.fillColor, 'fill')
            });

            // TODO: canvas for complex border styles
            borderMarkup = '<div class="borders" style="';
            borderMarkup += 'border-top:' + styleCollection.getCssBorder(cellAttrs.borderTop, { preview: true }) + ';';
            borderMarkup += 'border-bottom:' + styleCollection.getCssBorder(cellAttrs.borderBottom, { preview: true }) + ';';
            borderMarkup += 'border-left:' + styleCollection.getCssBorder(cellAttrs.borderLeft, { preview: true }) + ';';
            borderMarkup += 'border-right:' + styleCollection.getCssBorder(cellAttrs.borderRight, { preview: true }) + ';';
            borderMarkup += '"></div>';
            buttonNode.append(borderMarkup);
        }

        // initialization -----------------------------------------------------

        // add visible formatting of the cell styles to the list item buttons
        this.getMenu().on('create:item', createItemHandler);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            app = styleCollection = null;
        });

    }}); // class CellStylePicker

    // class SubtotalList =====================================================

    /**
     * The subtotal 'customizer'. A yet another drop down list for displaying
     * desired subtotal value.
     *
     * @constructor
     *
     * @extends RadioList
     */
    SpreadsheetControls.SubtotalList = RadioList.extend({ constructor: function (app, initOptions) {

        var // self reference
            self = this,

            // the number formatter of the document
            numberFormatter = app.getModel().getNumberFormatter();

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({
            tooltip: gt('Type of the subtotal value')
        }, initOptions));

        // private methods ----------------------------------------------------

        /**
         * Returns caption options of a list entry for the specified subtotal
         * type and value.
         */
        function getCaptionOptions(type, value) {

            var // contents and formatting of the active cell
                activeCell = app.getView().getCellContents(),
                // whether the active cell is formatted as date or time
                dateTime = /^(date|time|datetime)$/.test(activeCell.format.cat),
                // the resulting label text
                label = '';

            // format finite numbers with the number format of the active cell
            if (_.isNumber(value) && _.isFinite(value)) {

                // different formatting and special handling according to subtotal type
                if ((type === 'cells') || (type === 'numbers')) {
                    // subtotals of type 'cells' and 'numbers' are integers
                    label = String(value);
                } else if ((type === 'sum') && dateTime) {
                    // type 'sum' with date/time: show localized #N/A error
                    label = app.convertErrorCodeToString(SheetUtils.ErrorCodes.NA);
                } else {
                    // format number according to the active cell's number format
                    label = numberFormatter.formatCellContent(value, activeCell.attributes);
                }
            }

            switch (type) {
            case 'sum':
                return {
                    label: /*#. Subtotal result in OX Spreadsheet's status bar: Sum of all selected numeric cells */ gt('Sum: %1$s', _.noI18n(label)),
                    tooltip: gt('Sum of selected cells')
                };
            case 'min':
                return {
                    label: /*#. Subtotal result in OX Spreadsheet's status bar: Minimum of all selected numeric cells */ gt('Min: %1$s', _.noI18n(label)),
                    tooltip: gt('Minimum value in selection')
                };
            case 'max':
                return {
                    label: /*#. Subtotal result in OX Spreadsheet's status bar: Maximum of all selected numeric cells */ gt('Max: %1$s', _.noI18n(label)),
                    tooltip: gt('Maximum value in selection')
                };
            case 'cells':
                return {
                    label: /*#. Subtotal result in OX Spreadsheet's status bar: Count of all selected numeric cells AND text cells */ gt('Count: %1$s', _.noI18n(label)),
                    tooltip: gt('Number of selected cells that contain data')
                };
            case 'numbers':
                return {
                    label: /*#. Subtotal result in OX Spreadsheet's status bar: Count of all selected numeric cells only (NOT text cells) */ gt('Numerical count: %1$s', _.noI18n(label)),
                    tooltip: gt('Number of selected cells that contain numerical data')
                };
            case 'average':
                return {
                    label: /*#. Subtotal result in OX Spreadsheet's status bar: Average of all selected numeric cells */ gt('Average: %1$s', _.noI18n(label)),
                    tooltip: gt('Average of selected cells')
                };
            }
        }

        /**
         * Updates the visible label of this control.
         */
        function updateHandler(type) {

            var // get fresh subtotals
                subtotals = app.getView().getSubtotals();

            // if selection only consists of string values, show cell count
            if (('numbers' in subtotals) && (subtotals.numbers === 0)) { type = 'cells'; }

            // update the caption of the menu button
            if (type in subtotals) {
                self.setLabel(getCaptionOptions(type, subtotals[type]).label);
            }
        }

        /**
         * Recreates all sheet entries in the drop-down menu.
         */
        var fillList = app.createDebouncedMethod($.noop, function () {

            var // get fresh subtotals
                subtotals = app.getView().getSubtotals();

            function createOptionButton(section, type) {
                self.createOptionButton(type, _.extend({ section: section }, getCaptionOptions(type, subtotals[type])));
            }

            // always create the cell count entry
            self.clearMenu().createMenuSection('cells');
            createOptionButton('cells', 'cells');

            // create numerical entries on demand
            if (subtotals.numbers > 0) {
                self.createMenuSection('numbers');
                createOptionButton('numbers', 'numbers');
                createOptionButton('numbers', 'sum');
                createOptionButton('numbers', 'min');
                createOptionButton('numbers', 'max');
                createOptionButton('numbers', 'average');
            }
        }, { delay: 200 });

        // initialization -----------------------------------------------------

        this.registerUpdateHandler(updateHandler);

        this.listenTo(app.getImportPromise(), 'done', function () {
            fillList();
            self.listenTo(app.getView(), 'change:layoutdata', fillList);
        });

    }}); // class SubtotalList

    // exports ================================================================

    return SpreadsheetControls;

});
