/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Mario Schroeder <mario.schroeder@open-xchange.com>
 */
define('io.ox/office/spreadsheet/utils/clipboard',
        ['io.ox/office/tk/utils',
         'io.ox/office/editframework/utils/color',
         'io.ox/office/editframework/view/hyperlinkutil',
         'io.ox/office/spreadsheet/utils/sheetutils',
         'io.ox/office/spreadsheet/model/formula/tokenarray'
        ], function (Utils, Color, HyperlinkUtil, SheetUtils, TokenArray) {

    'use strict';


    var // XML name spaces for Office and Excel custom tags
        EXCEL_NAMESPACES = 'xmlns:o="urn:schemas-microsoft-com:office:office" xmlns:x="urn:schemas-microsoft-com:office:excel" xmlns="http://www.w3.org/TR/REC-html40"',

        // the HTML header
        HTML_HEADER = '<head><meta charset="utf-8"></head>',

        // the localized names of the standard number format
        STANDARD_NUMBER_FORMATS = [
            'Standard',
            'General',
            'Estandarra',
            'Skoueriek',
            'Standaard',
            'Yleinen',
            'Bendras',
            '\u041E\u043F\u0448\u0442\u043E',
            '\u0639\u0627\u062F\u06CC',
            'Geral',
            '\u0050\u0061\u0064\u0072\u00E3\u006F',
            'Estandar'
        ],

        // regex to check for error codes
        ERROR_CODES_REGEX = /#(DIV\/0|N\/A|NAME|NULL|NUM|REF|VALUE)(\!|\?)/i,

        // predefined number formats
        PREDEFINED_NUMBER_FORMATS = (function () {

            var result = {},
                formats = {
                id_0:  'General',
                id_1:  '0',
                id_2:  '0.00',
                id_3:  '#,##0',
                id_4:  '#,##0.00',
                id_5:  '$#,##0_);($#,##0)',
                id_6:  '$#,##0_);[Red]($#,##0)',
                id_7:  '$#,##0.00_);($#,##0.00)',
                id_8:  '$#,##0.00_);[Red]($#,##0.00)',
                id_9:  '0%',
                id_10: '0.00%',
                id_11: '0.00E+00',
                id_12: '# ?/?',
                id_13: '# ??/??',
                id_14: 'MM-DD-YY',
                id_15: 'D-MMM-YY',
                id_16: 'D-MMM',
                id_17: 'MMM-YY',
                id_18: 'h:mm AM/PM',
                id_19: 'h:mm:ss AM/PM',
                id_20: 'h:mm',
                id_21: 'h:mm:ss',
                id_22: 'M/D/YY h:mm',
                id_37: '#,##0 ;(#,##0)',
                id_38: '#,##0 ;[Red](#,##0)',
                id_39: '#,##0.00;(#,##0.00)',
                id_40: '#,##0.00;[Red](#,##0.00)',
                id_45: 'mm:ss',
                id_46: '[h]:mm:ss',
                id_47: 'mmss.0',
                id_48: '##0.0E+0',
                id_49: '@'
            };

            for (var formatId in formats) {
                result[formatId] = {
                        excel: convertNumberFormatToExcel(formats[formatId]),
                        calc: convertNumberFormatToCalc(formats[formatId])
                    };
            }
            return result;
        })();


    // private static functions ===============================================

    /**
     * Converts a number format code to be used with the mso-number-format style of Excel.
     *
     * @param {String} format
     *  The generic number format code.
     *
     * @return {String}
     *  The number format code prepared for Excel.
     */
    function convertNumberFormatToExcel(format) {
        var result;

        if (format.toLowerCase() === 'standard') {
            return 'General';
        }

        result = Utils.cleanString(format);

        if (_.browser.IE) {
            // use ISO Latin-1 code
            return result
                // replace the ampersand with the text &amp; (must be done first!)
                .replace(/&/g, '&amp;')
                // replace space or backslash space with the text &#32;
                .replace(/(\\\s|\s)/g, '&#32;')
                // replace backslash double quote with the text &#92;&#34;
                .replace(/\\"/g, '&#92;&#34;')
                // replace backslash apostrophe with the text &#92;&#39;
                .replace(/\\'/g, '&#92;&#39;')
                // replace backslash percent with the text &#92;&#37;
                .replace(/\\%/g, '&#92;&#37;')
                // replace backslash exclamation mark with the text &#92;&#33;
                .replace(/\\!/g, '&#92;&#33;')
                // replace double backslash the text &#92;&#92;
                .replace(/\\\\/g, '&#92;&#92;')
                // replace double quote, Euro symbol, double quote with the text &#92;&#8364; as work around for IE problems with the double quote
                .replace(/"€"/g, '&#92;&#8364;')
                // replace the left angle bracket with the text &lt;
                .replace(/</g, '&lt;')
                // replace the right angle bracket with the text &gt;
                .replace(/>/g, '&gt;')
                // replace the double quote character with the text &#34;
                .replace(/"/g, '&#34;')
                // replace the apostrophe with the text &#39; (&apos; is not an HTML entity!)
                .replace(/'/g, '&#39;');

        } else {
            // use UTF-8 representation
            return result
                // replace the ampersand with the text &amp; (must be done first!)
                .replace(/&/g, '&amp;')
                // replace space or backslash space with the text \\0020;
                .replace(/(\\\s|\s)/g, '\\0020')
                // replace backslash double quote with the text \\005c&#34;
                .replace(/\\"/g, '\\005c&#34;')
                // replace backslash percent with the text \\005c&#37;
                .replace(/\\%/g, '\\005c&#37;')
                // replace backslash exclamation mark with the text \\005c&#33;
                .replace(/\\!/g, '\\005c&#33;')
                // replace double backslash the text \\005c;&#92;
                .replace(/\\\\/g, '\\005c\\005c')
                // replace the left angle bracket with the text &lt;
                .replace(/</g, '&lt;')
                // replace the right angle bracket with the text &gt;
                .replace(/>/g, '&gt;')
                // replace the double quote character with the text &#34;
                .replace(/"/g, '&#34;')
                // replace the apostrophe with the text &#39; (&apos; is not an HTML entity!)
                .replace(/'/g, '&#39;');
        }
    }

    /**
     * Converts a number format code to be used with the sdnum attribute of Calc.
     *
     * @param {String} format
     *  The generic number format code.
     *
     * @return {String}
     *  The number format code prepared for Calc.
     */
    function convertNumberFormatToCalc(format) {
        if (format.toLowerCase() === 'standard') {
            return 'General';
        }

        return Utils.escapeHTML(format);
    }


    // static class Clipboard =================================================

    var Clipboard = {};

    // constants --------------------------------------------------------------

    // methods ----------------------------------------------------------------

    /**
     * Creates a unique client clipboard id.
     *
     * @return {String}
     *  The client clipboard id.
     */
    Clipboard.createClientClipboardId = function () {
        return ox.session + '-' + Date.now();
    };

    /**
     * Returns the predefined number format code for the given number format id.
     *
     * @param {Number} numFmtId
     *  The number format id.
     *
     * @param {String} representation
     *  The representation of the number format ('excel', 'calc').
     *
     * @return {String|undefined}
     *  The number format code.
     */
    Clipboard.getPredefinedNumberFormat = function (numFmtId, representation) {
        var format = PREDEFINED_NUMBER_FORMATS['id_' + numFmtId];
        if (!format) { return undefined; }
        return (representation.toLowerCase() === 'excel') ? format.excel : format.calc;
    };

    /**
     * Returns the number format code for the given cell data to be exported to Excel.
     *
     * @param {Object} cellData
     *  The contents (display string, result, formula) and formatting of the cell.
     *
     * @return {String|undefined}
     *  The number format code.
     */
    Clipboard.getExcelNumberFormat = function (cellData) {

        var // the number format of the cell
            numberFormat = cellData && cellData.attributes && cellData.attributes.cell && cellData.attributes.cell.numberFormat,
            // the number format code
            numberFormatCode;

        if (!numberFormat) { return undefined; }

        if (numberFormat.code && numberFormat.code.length > 0) {
            // handle number format code
            numberFormatCode = convertNumberFormatToExcel(numberFormat.code);

        } else {
            // handle number format id
            numberFormatCode = Clipboard.getPredefinedNumberFormat(numberFormat.id, 'excel');
        }

        Utils.log('clipboard - numberformat - id: ' + numberFormat.id + ', code: ' + numberFormat.code + ', mso-number-format: ' + numberFormatCode);

        return numberFormatCode;
    };

    /**
     * Returns the number format code for the given cell data to be exported to Calc.
     *
     * @param {Object} cellData
     *  The contents (display string, result, formula) and formatting of the cell.
     *
     * @return {String|undefined}
     *  The number format code.
     */
    Clipboard.getCalcNumberFormat = function (cellData) {

        var // the number format of the cell
            numberFormat = cellData && cellData.attributes && cellData.attributes.cell && cellData.attributes.cell.numberFormat,
            // the number format code
            numberFormatCode;

        if (!numberFormat) { return undefined; }

        if (numberFormat.code && numberFormat.code.length > 0) {
            // handle number format code
            numberFormatCode = convertNumberFormatToCalc(numberFormat.code);

        } else {
            // handle number format id
            numberFormatCode = Clipboard.getPredefinedNumberFormat(numberFormat.id, 'calc');
        }

        Utils.log('clipboard - numberformat - id: ' + numberFormat.id + ', code: ' + numberFormat.code + ', calc-number-format: ' + numberFormatCode);

        return numberFormatCode;
    };

    /**
     * Returns true for the standard number format.
     *
     * @param {String} format
     *  The number format code as String.
     *
     * @return {Boolean}
     *  Returns true for the standard number format.
     */
    Clipboard.isStandardNumberFormat = function (format) {

        if (!_.isString(format)) { return false; }

        var foo = _.some(STANDARD_NUMBER_FORMATS, function (item) {
            return (_.isString(this) && this.indexOf(item.toLowerCase()) > -1);
        }, format.toLowerCase());

        return foo;
    };

    /**
     * Creates a number format cell style object from a Calc numberformat String
     *
     * @param {String} format
     *  The number format code as String.
     *
     *  @return {Object|null}
     *   The number format object
     */
    Clipboard.parseCalcNumberFormat = function (format) {

        var numberFormat, language;

        if (!_.isString(format)) { return null; }

        if (Clipboard.isStandardNumberFormat(format)) {
            return { id: 0 };
        }

        numberFormat = format.match(/(?:(\d{4,5}|0);){0,1}(?:(\d{4,5}|0);){0,1}(.*)/);
        if (_.isArray(numberFormat) && numberFormat[3] && numberFormat[3].length > 0) {
            numberFormat[2] = parseInt(numberFormat[2], 10);
            if (numberFormat[2] > 0) {
                language = numberFormat[2];
            } else {
                numberFormat[1] = parseInt(numberFormat[1], 10);
                if (numberFormat[1] > 0) {
                    language = numberFormat[1];
                } else {
                    language = 1033;
                }
            }

            return { code: numberFormat[3], language: language };
        }

        return null;
    };

    /**
     * Returns the direct child element that matches the given jQuery selector.
     * Is limited to work inside HTML tables, e.g. find a 'strong' as child of a 'td' element.
     *
     * @param {jQuery} element
     *  The jQuery root element to start looking for the child.
     *
     * @return {jQuery}
     *  The matching child element.
     */
    Clipboard.findMatchingChild = function (element, selector) {
        return element.find(selector).filter(function () {
            var parent = $(this).parents('span, a, th, td, tr, tfoot, thead, tbody, colgroup, table').first();
            return (parent[0] === element[0]);
        });
    };

    /**
     * Returns true if the the given jQuery selector matches a direct child element.
     * Is limited to work inside HTML tables, e.g. find a 'strong' as child of a 'td' element.
     *
     * @param {jQuery} element
     *  The jQuery root element to start looking for the child.
     *
     * @return {Boolean}
     *  Returns true if a child of the given 'element' matches the given 'selector'.
     */
    Clipboard.hasMatchingChild = function (element, selector) {
        return (Clipboard.findMatchingChild(element, selector).length > 0);
    };

    /**
     * Converts a CSS font size with measurement unit into a HTML font size.
     *
     * @param {String} cssSize
     *  The CSS font size value with its measurement unit to be converted, as String.
     *
     * @returns {Number}
     *  The converted CSS size.
     */
    Clipboard.convertToFontSize = function (cssSize) {
        var size = Utils.convertCssLength(cssSize, 'pt');
        if (size > 30) { return 7; }    // 7 -> 36pt
        if (size > 21) { return 6; }    // 6 -> 24pt
        if (size > 16) { return 5; }    // 5 -> 18pt
        if (size > 13) { return 4; }    // 4 -> 14pt
        if (size > 11) { return 3; }    // 3 -> 12pt
        if (size > 9)  { return 2; }    // 2 -> 10pt
        if (size > 0)  { return 1; }    // 1 -> 8pt
        return 0;
    };

    /**
     * Converts a HTML font size String (used as size attribute inside a font element)
     * into a size in point.
     *
     * @param {String} size
     *  The font size to be converted, as String.
     *
     * @returns {Number}
     *  The converted size, in point.
     */
    Clipboard.convertFontSizeToPoint = function (size) {
        switch (size) {
        case '1':
            return 8;
        case '2':
            return 10;
        case '3':
            return 12;
        case '4':
            return 14;
        case '5':
            return 18;
        case '6':
            return 24;
        case '7':
            return 36;
        default:
            return 0;
        }
    };

    /**
     * Returns whether the passed value is an error code.
     *
     * @param {String} value
     *  Any literal value that can be used in formulas.
     *
     * @returns {Boolean}
     *  Whether the passed value is an error code.
     */
    Clipboard.isErrorCode = function (value) {
        return ERROR_CODES_REGEX.test(value);
    };

    /**
     * Creates a HTML table from the active range of the current selection.
     *
     * @param {SpreadsheetApplication} app
     *  The application requesting the HTML table.
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing the cells.
     *
     * @param {Object} range
     *  The range address.
     *
     * @param {String} clientClipboardId
     *  The client clipboard id to identify the copy data when pasting.
     *
     * @return {String}
     *  The resulting HTML mark-up of the <table> element.
     */
    Clipboard.getHTMLStringFromRange = function (app, sheetModel, range, clientClipboardId) {

        var // the current table row
            row = range.start[1],
            // the style sheet container of the document
            documentStyles = app.getModel().getDocumentStyles(),
            // the formula parser
            tokenArray = new TokenArray(app, sheetModel, { silent: true, grammar: 'ui' }),
            // the table string
            table = '<html ' + EXCEL_NAMESPACES + '>' + HTML_HEADER + '<body><table id="ox-clipboard-data" data-ox-clipboard-id="' + clientClipboardId + '"><tbody><tr>';

        sheetModel.getCellCollection().iterateCellsInRanges(range, function (cellData, origRange, mergedRange) {

            var address = cellData.address,
                isReferenceCell,
                colspan,
                rowspan,
                style,
                excelNumberFormat,
                calcNumberFormat,
                formula;

            // check if the cell is the reference cell of a merged collection
            isReferenceCell = mergedRange && _.isEqual(address, mergedRange.start);

            // check if we need to add a new row to the table
            if (row !== address[1]) {
                table += '</tr><tr>';
                row = address[1];
            }

            // render cell if it's the reference cell or if it's not a merged cell
            if (isReferenceCell || !mergedRange) {

                table += '<td ';

                if (isReferenceCell) {
                    // add cell with colspan and rowspan
                    colspan = mergedRange.end[0] - mergedRange.start[0] + 1;
                    rowspan = mergedRange.end[1] - mergedRange.start[1] + 1;

                    table += (colspan > 1) ? 'colspan="' + colspan + '" ' : '';
                    table += (rowspan > 1) ? 'rowspan="' + rowspan + '" ' : '';
                }

                // add cell and character style attributes
                style = Utils.escapeHTML(Clipboard.getCssCharacterStyles(cellData, documentStyles) + Clipboard.getCssCellStyles(cellData, documentStyles));

                // handle Excel number format
                excelNumberFormat = Clipboard.getExcelNumberFormat(cellData);
                if (excelNumberFormat) {
                    style += 'mso-number-format:\'' + excelNumberFormat + '\';';
                }

                if (style.length > 0) {
                    table += ' style="' + style + '"';
                }

                // set the cell value type
                if (_.isNumber(cellData.result)) {
                    table += ' x:num="' + cellData.result + '"';
                    table += ' sdval="' + cellData.result + '"';
                } else if (_.isBoolean(cellData.result)) {
                    table += ' x:bool="' + cellData.result + '"';
                } else if (_.isString(cellData.result) && cellData.result[0] === '#') {
                    table += ' x:err="' + Utils.escapeHTML(cellData.result) + '"';
                } else if (_.isString(cellData.result)) {
                    table += ' x:str="' + Utils.escapeHTML(cellData.result) + '"';
                }

                // handle Calc number format
                calcNumberFormat = Clipboard.getCalcNumberFormat(cellData);
                if (calcNumberFormat) {
                    table += ' sdnum="1033;1033;' + calcNumberFormat + '"';
                }

                // handle formula
                if (_.isString(cellData.formula)) {

                    // relocate all formulas relative to A1
                    tokenArray.parseFormula(cellData.formula);
                    formula = tokenArray.getRelocatedFormula(range.start, [0, 0]);
                    Utils.log('clipboard - formula - range: ' + JSON.stringify(range) + ', original: ' + cellData.formula + ', relocated: ' + formula);

                    if (!Clipboard.isErrorCode(formula)) {
                        table += ' x:fmla="' + Utils.escapeHTML(formula) + '"';
                    }
                }

                // add HTML attributes
                table += Clipboard.getHtmlCellStyles(cellData, documentStyles);

                table += '>';

                // add HTML elements and cell content
                table += Clipboard.getHtmlCharacterStyles(cellData, documentStyles);

                table += '</td>';
            }
        }, { merged: true, ordered: true });

        tokenArray.destroy();
        table += '</tr></tbody></table></body></html>';
        return table;
    };

    /**
     * Creates a plain text representation from the active range of the current
     * selection.
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing the cells.
     *
     * @param {Object} range
     *  The range address.
     *
     * @returns {String}
     *  The resulting plain text string.
     */
    Clipboard.getPlainTextFromRange = function (sheetModel, range) {

        var // the address of the first visible cell
            startAddress = null,
            // the resulting TAB-separated display strings
            result = '';

        sheetModel.getCellCollection().iterateCellsInRanges(range, function (cellData) {
            if (!startAddress) {
                startAddress = cellData.address;
            } else if (cellData.address[0] > startAddress[0]) {
                result += '\t';
            } else if (cellData.address[1] > startAddress[1]) {
                result += '\r\n';
            }
            result += cellData.display;
        }, { ordered: true });

        return result;
    };

    /**
     * Returns the character styles as CSS style String.
     *
     * @param {Object} cellData
     *  The contents (display string, result, formula) and formatting of the cell.
     *
     * @param {DocumentStyles} documentStyles
     *  The document styles collection of the document.
     *
     *  @return {String}
     *   The CSS style String.
     */
    Clipboard.getCssCharacterStyles = function (cellData, documentStyles) {

        var // attribute map for the 'character' attribute family
            charAttributes = cellData && cellData.attributes && cellData.attributes.character,
            // the fill color cell attribute
            fillColor = cellData && cellData.attributes && cellData.attributes.cell && cellData.attributes.cell.fillColor,
            // the CSS text decoration
            textDecoration = 'none',
            // the character style attribute data
            characterStyle = '';

        if (!charAttributes) { return ''; }

        if (charAttributes.fontName) { characterStyle += 'font-family:' + documentStyles.getCssFontFamily(charAttributes.fontName) + ';'; }
        if (charAttributes.fontSize) { characterStyle += 'font-size:' + charAttributes.fontSize + 'pt;'; }

        characterStyle += 'font-weight:' + ((charAttributes.bold) ? 'bold;' : 'normal;');
        characterStyle += 'font-style:' + ((charAttributes.italic) ? 'italic;' : 'normal;');

        if (charAttributes.underline) { textDecoration = Utils.addToken(textDecoration, 'underline', 'none'); }
        if (charAttributes.strike !== 'none') { textDecoration = Utils.addToken(textDecoration, 'line-through', 'none'); }
        characterStyle += 'text-decoration:' + textDecoration + ';';

        if (charAttributes.color) { characterStyle += 'color:' + documentStyles.getCssTextColor(charAttributes.color, [fillColor]) + ';'; }

        return characterStyle;
    };

    /**
     * Returns the cell styles as CSS style String.
     *
     * @param {Object} cellData
     *  The contents (display string, result, formula) and formatting of the cell.
     *
     * @param {DocumentStyles} documentStyles
     *  The document styles collection of the document.
     *
     *  @return {String}
     *   The CSS style String.
     */
    Clipboard.getCssCellStyles = function (cellData, documentStyles) {

        var // attribute map for the 'cell' attribute family
            cellAttributes = cellData && cellData.attributes && cellData.attributes.cell,
            // the cell style attribute data
            cellStyle = '';

        if (!cellAttributes) { return ''; }

        if (_.contains(['left', 'center', 'right', 'justify'], cellAttributes.alignHor)) { cellStyle += 'text-align:' + cellAttributes.alignHor + ';'; }
        if (_.contains(['top', 'middle', 'bottom', 'justify'], cellAttributes.alignVert)) { cellStyle += 'vertical-align:' + cellAttributes.alignVert + ';'; }

        cellStyle += 'white-space:' + ((cellAttributes.wrapText) ? 'normal;' : 'nowrap;');
        if (cellAttributes.fillColor) { cellStyle += 'background-color:' + documentStyles.getCssColor(cellAttributes.fillColor, 'fill') + ';'; }

        _.each(['Top', 'Bottom', 'Left', 'Right'], function (attrPart) {

            var // the cell style attribute name
                oxAttrName = 'border' + attrPart,
                // the CSS attribute name
                cssAttrName = 'border-' + attrPart.toLowerCase() + ':',
                // the border cell style attribute value
                border = cellAttributes[oxAttrName],
                // the effective CSS attributes
                cssAttrs = documentStyles.getCssBorderAttributes(border);

            cellStyle += cssAttrName;
            cellStyle += ((cssAttrs && cssAttrs.width > 0) ? cssAttrs.style + ' ' + cssAttrs.width + 'px ' + cssAttrs.color : 'none');
            cellStyle += ';';
        });

        return cellStyle;
    };

    /**
     * Returns the character styles as a String of the HTML elements tree.
     *
     * @param {Object} cellData
     *  The contents (display string, result, formula) and formatting of the cell.
     *
     * @param {DocumentStyles} documentStyles
     *  The document styles collection of the document.
     *
     *  @return {String}
     *   The HTML elements String.
     */
    Clipboard.getHtmlCharacterStyles = function (cellData, documentStyles) {

        var // attribute map for the 'character' attribute family
            charAttributes = cellData && cellData.attributes && cellData.attributes.character,
            // the fill color cell attribute
            fillColor = cellData && cellData.attributes && cellData.attributes.cell && cellData.attributes.cell.fillColor,
            // the character style HTML elements
            characterStyle = '',
            // the hyperlink URL
            url;


        if (!charAttributes) { return cellData.display; }

        // create opening tags
        if (charAttributes.bold) { characterStyle += '<b>'; }
        if (charAttributes.italic) { characterStyle += '<i>'; }
        if (charAttributes.underline) { characterStyle += '<u>'; }
        if (charAttributes.strike !== 'none') { characterStyle += '<s>'; }

        url = (charAttributes.url) ? Utils.escapeHTML(HyperlinkUtil.checkForHyperlink(charAttributes.url)) : null;
        if (url && url.length) {
            characterStyle += '<a href="' + url + '">';
        }

        if (charAttributes.fontName || charAttributes.fontSize || charAttributes.color) {
            characterStyle += '<font ';
            // To make it work in Calc we need to use just the font name instead of the entire CSS font family
            if (charAttributes.fontName) { characterStyle += 'face="' + Utils.escapeHTML(charAttributes.fontName) + '" '; }
            // We need to use rgb colors, like #RRGGBB
            if (charAttributes.color) { characterStyle += 'color="' + documentStyles.getCssTextColor(charAttributes.color, [fillColor]) + '" '; }
            // Add the font-size CSS also here to support precise font sizes in Excel
            if (charAttributes.fontSize) { characterStyle += 'size="' + Clipboard.convertToFontSize(charAttributes.fontSize) + '" style="font-size:' + charAttributes.fontSize + 'pt;"'; }
            characterStyle += '>';
        }

        // add cell value
        characterStyle += Utils.escapeHTML(cellData.display);

        // create closing tags
        if (charAttributes.fontName || charAttributes.fontSize || charAttributes.color) { characterStyle += '</font>'; }
        if (url && url.length) { characterStyle += '</a>'; }
        if (charAttributes.strike !== 'none') { characterStyle += '</s>'; }
        if (charAttributes.underline) { characterStyle += '</u>'; }
        if (charAttributes.italic) { characterStyle += '</i>'; }
        if (charAttributes.bold) { characterStyle += '</b>'; }

        return characterStyle;
    };

    /**
     * Returns the cell styles as HTML attributes String.
     *
     * @param {Object} cellData
     *  The contents (display string, result, formula) and formatting of the cell.
     *
     * @param {DocumentStyles} documentStyles
     *  The document styles collection of the document.
     *
     *  @return {String}
     *   The HTML attributes String.
     */
    Clipboard.getHtmlCellStyles = function (cellData, documentStyles) {

        var // attribute map for the 'cell' attribute family
            cellAttributes = cellData && cellData.attributes && cellData.attributes.cell,
            // the cell style attribute data
            cellStyle = '';

        if (!cellAttributes) { return ''; }

        if (_.contains(['left', 'center', 'right', 'justify'], cellAttributes.alignHor)) { cellStyle += ' align="' + cellAttributes.alignHor + '"'; }
        if (_.contains(['top', 'middle', 'bottom'], cellAttributes.alignVert)) { cellStyle += ' valign="' + cellAttributes.alignVert + '"'; }
        if (cellAttributes.fillColor && !Color.isAutoColor(cellAttributes.fillColor)) { cellStyle += ' bgcolor="' + documentStyles.getCssColor(cellAttributes.fillColor, 'fill') + '"'; }

        return cellStyle;
    };

    /**
     * Retrieves the client clipboard ID from the HTML data.
     *
     * @param {String|jQuery} html
     *  The HTML data to check.
     *
     * @returns {String|undefined}
     *  The client clipboard ID or undefined.
     */
    Clipboard.getClientClipboardId = function (html) {
        return $(html).find('[id=ox-clipboard-data]').attr('data-ox-clipboard-id') || $(html).filter('[id=ox-clipboard-data]').attr('data-ox-clipboard-id');
    };

    /**
     * Returns true if the origin of the HTML data is MS Excel.
     *
     * @param {String|jQuery} html
     *  The HTML data to check.
     *
     * @returns {Boolean}
     */
    Clipboard.isExcelHtml = function (html) {
        return (($(html).find('meta[content*=Excel]').length > 0) || ($(html).filter('meta[content*=Excel]').length > 0));
    };

    /**
     * Returns true if the HTML contains exactly one table element.
     *
     * @param {String|jQuery} html
     *  The HTML data to check.
     *
     * @returns {Boolean}
     *  Returns true if a html table is found.
     */
    Clipboard.containsHtmlTable = function (html) {
        var table;

        // look for table elements in the jQuery set $(html)
        table = $(html).filter('table');
        if (table.length === 0) {
            // look for table elements that are descendants of the jQuery set $(html)
            table = $(html).find('table');
        }
        if (table.length === 0) {
            return false;
        } else {
            // check if the table has nested tables
            return (table.find('table').length === 0);
        }
    };

    /**
     * Parses the HTML from the clipboard data for a table and
     * creates the cell contents and merged cell collection.
     *
     * @param {String|jQuery} html
     *  The HTML input to parse.
     *
     * @return {Object}
     *  An object with the following attributes:
     *      - {Array} contents
     *          A two-dimensional array containing the values and attribute sets
     *          for the cells.
     *      - {Array} mergeCollection
     *          An array of merged cell ranges.
     */
    Clipboard.parseHTMLData = function (html, documentStyles) {

        var // a two dimensional array containing the cell contents
            contents = [],
            // an array of ranges of merged cells
            mergeCollection = [],
            // the table node
            table,
            // the row nodes
            rows,
            // the number of columns
            colCount,
            // the default character and cell attributes
            defaultAttrs = documentStyles.getDefaultAttributes(['character', 'cell']);


        table = $(html).find('table');
        if (table.length === 0) {
            table = $(html).filter('table');
        }

        // merge in the character and cell attributes of the HTML table
        delete defaultAttrs.cell.numberFormat;
        delete defaultAttrs.character.url;
        documentStyles.extendAttributes(defaultAttrs, Clipboard.getCharacterAttributes(table));
        documentStyles.extendAttributes(defaultAttrs, Clipboard.getCellAttributes(table));

        // make sure that table header, body and footer are in the correct order
        rows = $(_.union(table.find('thead tr').toArray(), table.find('tbody tr').toArray(), table.find('tfoot tr').toArray()));

        colCount = _.max(_.map(rows, function (row) { return $(row).find('td,th').length; }));


        rows.each(function (rowId) {

            var colId,
                rowData = [],
                cells,
                cell,
                cellId = 0,
                cellData,
                cellValue,
                colspan,
                rowspan,
                isMergedCell,
                rowParent,
                rowAttrs,
                linkChild;


            rowParent = $(this).parents('thead, tbody, tfoot, table').first();

            rowAttrs = Clipboard.getCharacterAttributes(rowParent);
            documentStyles.extendAttributes(rowAttrs, Clipboard.getCellAttributes(rowParent));
            documentStyles.extendAttributes(rowAttrs, Clipboard.getCharacterAttributes($(this)));
            documentStyles.extendAttributes(rowAttrs, Clipboard.getCellAttributes($(this)));


            cells = $(this).find('td,th');

            for (colId = 0; colId < colCount; colId++) {

                isMergedCell = SheetUtils.rangesContainCell(mergeCollection, [colId, rowId]);

                if (isMergedCell) {
                    // the cell is part of a merged collection, but not the reference cell
                    cellData = {};

                } else {
                    // the cell is not merged or it is the reference cell
                    cell = $(cells.get(cellId));
                    colspan = cell.attr('colspan');
                    rowspan = cell.attr('rowspan');

                    if (colspan || rowspan) {
                        // the cell is merged and it's a reference cell
                        colspan = (colspan && colspan.length > 0) ? parseInt(colspan, 10) : 1;
                        rowspan = (rowspan && rowspan.length > 0) ? parseInt(rowspan, 10) : 1;

                        mergeCollection.push({start: [colId, rowId], end: [colId + colspan - 1, rowId + rowspan - 1]});
                    }

                    // look for Calc cell value
                    cellValue = cell.attr('sdval');
                    if (_.isString(cellValue) && cellValue.length > 0) {
                        cellValue = parseFloat(cellValue, 10);
                    }
                    if (!_.isNumber(cellValue) || _.isNaN(cellValue)) {
                        cellValue = cell.text();
                        // if the cell value contains '\n' we need to make sure that we keep only the line feeds
                        // the user wants and not those Excel added by surprise.
                        if ((cellValue.indexOf('\n') > -1) || (cell.find('br').length > 0)) {
                            // remove line feeds from HTML and leave only the <br> elements
                            cell.html(cell.html().replace(/<br\s*\/?>([^\n]*)\n\u0020+/gi, '<br>$1').replace(/\n\u0020/g, ''));
                            // make '\n' out of the <br> elements
                            cell.find('br').replaceWith('\n');
                            // finally the cell value contains only the user line feeds
                            cellValue = cell.text();
                        }
                    }


                    cellData = {
                        value : cellValue,
                        attrs : _.copy(defaultAttrs, true)
                    };

                    // merge in the character and cell attributes of the HTML row and cell
                    documentStyles.extendAttributes(cellData.attrs, rowAttrs);
                    documentStyles.extendAttributes(cellData.attrs, Clipboard.getCharacterAttributes(cell));
                    documentStyles.extendAttributes(cellData.attrs, Clipboard.getCellAttributes(cell));

                    // merge in the hyperlink's character attributes that are attached to it's child span
                    linkChild = Clipboard.findMatchingChild(cell, 'a').children('span').first();
                    if (linkChild.length > 0) {
                        documentStyles.extendAttributes(cellData.attrs, Clipboard.getCharacterAttributes(linkChild));
                    }

                    cellId++;
                }

                rowData.push(cellData);
            }

            contents.push(rowData);
        });

        return {contents: contents, mergeCollection: mergeCollection};
    };

    /**
     * Parses the plain text from the clipboard data and creates the cell contents.
     * Rows should be separated by line endings ('\r\n', '\r' or '\n')
     * and cells should be separated by tab or semicolon.
     *
     * @param {String} text
     *  The text input to parse.
     *
     * @return {Object}
     *  An object with the following attributes:
     *      - {Array} contents
     *          A two-dimensional array containing the values and attribute sets
     *          for the cells.
     */
    Clipboard.parseTextData = function (text) {

        var // a two dimensional array containing the cell contents
            contents = [],
            // the row nodes
            rows,
            // the cell separator
            sep;

        if (!_.isString(text)) { return {contents: []}; }

        // remove \r,\n, \t and ; to remove empty rows at the end of the text
        text = text.replace(/[;\s]+$/, '');

        // parse rows according to the line ending type
        rows = (text.indexOf('\r\n') > -1) ? text.split(/\r\n/) : text.split(/[\r\n]/);
        // look for the cell separator
        sep = (text.indexOf('\t') > -1) ? '\t' : ';';

        _.each(rows, function (row) {
            var rowData = [],
                cells;

            cells = row.split(sep);
            _.each(cells, function (cell) {
                rowData.push({ value: Utils.cleanString(cell) });
            });
            contents.push(rowData);
        });

        return {contents: contents};
    };


    /**
     * Creates a new range from the active cell and the parser result.
     *
     * @param {Number[]} activeCell
     *  The logical address of the active cell.
     *
     * @param {Object} result
     *  The parser result object which needs to cain the following attributes:
     *      - {Array} contents
     *          A two-dimensional array containing the values and attribute sets
     *          for the cells.
     *
     * @returns {Object}
     *  The address of the cell range covered by the active cell and the parser result.
     */
    Clipboard.createRangeFromParserResult = function (activeCell, result) {
        var range = { start: _.copy(activeCell), end: _.copy(activeCell) };
        range.end[0] += (result.contents[0].length > 0) ? result.contents[0].length - 1 : 0;
        range.end[1] += (result.contents.length > 0) ? result.contents.length - 1 : 0;

        return range;
    };

    /**
     * Returns the attribute map for the 'character' attribute family of the HTML cell.
     *
     * @param {jQuery} element
     *  The HTML element.
     *
     *  @return {Object}
     *   The character attribute map.
     */
    Clipboard.getCharacterAttributes = function (element) {

        var // attribute map for the 'character' attribute family
            attrs = { character: {} },
            // the CSS attributes
            fontName, fontSize, fontWeight, fontStyle, textDecoration, color, url,
            // the <font> element, if HTML elements are used instead of CSS
            fontElement = Clipboard.findMatchingChild(element, 'font');


        fontName = fontElement.attr('face') || element.css('font-family');
        if (_.isString(fontName) && fontName.length > 0) {
            fontName = (fontName.split(',')[0]).replace(/["']/g, '');
            attrs.character.fontName = fontName;
        }

        fontSize = Clipboard.convertFontSizeToPoint(fontElement.attr('size')) || Utils.convertCssLength(element.css('font-size'), 'pt');
        if (fontSize > 0) {
            attrs.character.fontSize = fontSize;
        }

        if (Clipboard.hasMatchingChild(element, 'em, i')) {
            attrs.character.italic = true;
        } else {
            fontStyle = element.css('font-style');
            if (_.isString(fontStyle) && fontStyle.length > 0) {
                attrs.character.italic = (fontStyle === 'italic');
            }
        }

        if (Clipboard.hasMatchingChild(element, 'strong, b')) {
            attrs.character.bold = true;
        } else {
            fontWeight = element.css('font-weight');
            if (_.isString(fontWeight) && fontWeight.length > 0) {
                attrs.character.bold = (fontWeight === 'bold' || fontWeight === 'bolder' || fontWeight === '600' || fontWeight === '700' || fontWeight === '800' || fontWeight === '900');
            }
        }

        textDecoration = element.css('text-decoration');

        if (Clipboard.hasMatchingChild(element, 'u')) {
            attrs.character.underline = true;
        } else if (_.isString(textDecoration) && textDecoration.length > 0) {
            attrs.character.underline = (textDecoration.indexOf('underline') >= 0);
        }

        if (Clipboard.hasMatchingChild(element, 's')) {
            attrs.character.strike = 'single';
        } else if (_.isString(textDecoration) && textDecoration.length > 0) {
            attrs.character.strike = ((textDecoration.indexOf('line-through') >= 0) ? 'single' : 'none');
        }

        color = fontElement.attr('color') || element.css('color');
        color = Color.convertCssColorToRgbColor(color);
        if (color) {
            attrs.character.color = color;
        }

        url = Clipboard.findMatchingChild(element, 'a').attr('href');
        if (_.isString(url) && url.length > 0) {
            attrs.character.url = Utils.escapeHTML(url);
        }

        return attrs;
    };

    /**
     * Returns the attribute map for the 'cell' attribute family of the HTML element.
     *
     * @param {jQuery} element
     *  The HTML element.
     *
     *  @return {Object}
     *   The cell attribute map.
     */
    Clipboard.getCellAttributes = function (element) {

        var // attribute map for the 'cell' attribute family
            attrs = { cell: {} },
            // the CSS attributes
            alignHor, alignVert, wrapText, fillColor,
            // the number format
            numberFormat;

        // map css border style to operation border line style
        function mapCssBorderStyle(style) {
            switch (style) {
            case 'solid':
            case 'groove':
            case 'ridge':
            case 'inset':
            case 'outset':
                return 'single';
            case 'double':
                return 'double';
            case 'dotted':
                return 'dotted';
            case 'dashed':
                return 'dashed';
            default:
                return 'none';
            }
        }


        alignHor = element.attr('align') || element.css('text-align');
        // check value and filter prefix (webkit-, moz-, ms-)
        alignHor = _.find(['left', 'center', 'right', 'justify'], function (item) { return (_.isString(this) && this.toLowerCase().indexOf(item) > -1);  }, alignHor);
        if (alignHor) {
            attrs.cell.alignHor = alignHor;
        }

        alignVert = element.attr('valign') || element.css('vertical-align');
        // check value and filter prefix (webkit-, moz-, ms-)
        alignVert = _.find(['top', 'middle', 'bottom', 'justify'], function (item) { return (_.isString(this) && this.toLowerCase().indexOf(item) > -1);  }, alignVert);
        if (alignVert) {
            attrs.cell.alignVert = alignVert;
        }

        wrapText = element.css('white-space');
        if (_.isString(wrapText) && wrapText.length > 0) {
            attrs.cell.wrapText = (wrapText !== 'nowrap') ? true : false;
        }

        fillColor = element.attr('bgcolor') || element.css('background-color');
        fillColor = Color.convertCssColorToRgbColor(fillColor);
        if (fillColor) {
            attrs.cell.fillColor = fillColor;
        }

        _.each(['Top', 'Bottom', 'Left', 'Right'], function (attrPart) {

            var // the element style attribute name
                oxAttrName = 'border' + attrPart,
                // the CSS attribute name
                cssAttrName = 'border-' + attrPart.toLowerCase(),
                // the border style
                borderStyle = mapCssBorderStyle(element.css(cssAttrName + '-style')),
                // the border width
                borderWidth,
                // the border color
                borderColor;


            if (borderStyle !== 'none') {

                borderWidth = element.css(cssAttrName + '-width');
                borderColor = element.css(cssAttrName + '-color');
                borderColor = Color.convertCssColorToRgbColor(borderColor);

                attrs.cell[oxAttrName] = {
                    style : borderStyle,
                    width : Utils.convertCssLengthToHmm(borderWidth),
                    color : (borderColor ? borderColor : Color.AUTO)
                };
            }

        });

        numberFormat = Clipboard.parseCalcNumberFormat(element.attr('sdnum'));
        if (numberFormat) {
            attrs.cell.numberFormat = numberFormat;
        }

        return attrs;
    };

    // exports ================================================================

    return Clipboard;

});
