/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/tokenarray',
    ['io.ox/office/tk/utils',
     'io.ox/office/baseframework/model/modelobject',
     'io.ox/office/spreadsheet/utils/sheetutils',
     'io.ox/office/spreadsheet/model/formula/tokenutils',
     'io.ox/office/spreadsheet/model/formula/tokenizer'
    ], function (Utils, ModelObject, SheetUtils, TokenUtils, Tokenizer) {

    'use strict';

    // private global functions ===============================================

    /**
     * Invokes the passed callback function, after initializing or updating the
     * global cache of visited names.
     */
    var guardVisitedNames = (function () {

        var // the cache of all visited names while recursively processing token arrays
            // (prevent endless loop in cyclic name references)
            visitedNamesCache = null;

        // a new callback function that will be passed to the passed callback function
        // returns whether the name referred by the passed name token has not been visited
        // yet, and marks it as visited
        function visitNameTokenOnce(token) {

            var // build a unique key for the passed name token
                nameKey = token.getSheet() + '!' + token.getValue().toUpperCase(),
                // whether the name token has been visited already
                visited = visitedNamesCache[nameKey];

            // mark the name as 'visited'
            visitedNamesCache[nameKey] = true;
            return !visited;
        }

        // return the actual guardVisitedNames() function
        return function (callback, context) {

            var // whether this is the initial call (initialize cache of visited names)
                initialCall = !_.isObject(visitedNamesCache);

            // Initialize cache of visited names. Using a real module-global variable
            // allows to use the same cache for invocations of this method on multiple
            // instances of the TokenArray class (for each visited defined name).
            if (initialCall) { visitedNamesCache = {}; }

            try {
                // invoke the passed callback function
                return callback.call(context, visitNameTokenOnce);
            } finally {
                // deinitialize cache of visited names
                if (initialCall) { visitedNamesCache = null; }
            }
        };
    }());

    // class TokenArray =======================================================

    /**
     * Providing support for parsing localized formula strings and handling of
     * the resulting formula token arrays.
     *
     * Instances of this class trigger the following events:
     * - 'change:tokens'
     *      After the token array has been changed (manually, or by parsing a
     *      formula), or cleared.
     * - 'change:token'
     *      After a single token in the token array has been changed. The event
     *      handlers receive a reference to the changed token, and the array
     *      index of the token.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {SpreadsheetApplication} app
     *  The application containing this formula parser.
     *
     * @param {SheetModel} [sheetModel]
     *  The model of the sheet that contains the object with this token array.
     *  If omitted, the token array is not associated to a specific sheet.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options supported by the base class
     *  ModelObject. Additionally, the following options are supported:
     *  @param {String} [initOptions.grammar='op']
     *      The formula grammar to be used for parsing formula expressions. See
     *      the Tokenizer class constructor for details.
     */
    function TokenArray(app, sheetModel, initOptions) {

        var // self reference
            self = this,

            // the spreadsheet document model
            model = app.getModel(),

            // the formula grammar for the tokenizer
            grammar = Utils.getStringOption(initOptions, 'grammar', 'op'),

            // the formula expression tokenizer
            tokenizer = new Tokenizer(app, grammar, sheetModel),

            // the array of all formula tokens
            tokens = [],

            // cached tokens compiled to prefix notation
            compiled = null;

        // base constructor ---------------------------------------------------

        ModelObject.call(this, app, initOptions);

        // private methods ----------------------------------------------------

        /**
         * Appends the passed new token to the token array, and triggers a
         * 'change:token' event for this token.
         *
         * @param {Token} token
         *  The new formula token. This token array will take ownership.
         */
        function appendToken(token) {
            tokens.push(token);
            compiled = null;
            self.trigger('change:token', token, tokens.length - 1);
        }

        /**
         * Invokes the specified method at all tokens, and triggers a
         * 'change:tokens' event, if any of the methods has returned true.
         */
        function invokeForAllTokensAndTrigger(methodName) {

            var // whether any token has returned true
                changed = false,
                // the arguments to be passed to the token method
                args = _.toArray(arguments).slice(1);

            // invoke method for all tokens, and collect changed state
            _.each(tokens, function (token) {
                if (_.isFunction(token[methodName]) && token[methodName].apply(token, args)) { changed = true; }
            });

            // notify listeners
            if (changed) { self.trigger('change:tokens'); }
            return changed;
        }

        // public methods -----------------------------------------------------

        /**
         * Removes all tokens from this token array.
         *
         * @returns {TokenArray}
         *  A reference to this instance.
         */
        this.clear = function () {
            if (tokens.length > 0) {
                tokens = [];
                compiled = null;
                this.trigger('change:tokens');
            }
            return this;
        };

        /**
         * Returns the identifier of the current formula grammar.
         *
         * @returns {String}
         *  The current formula grammar. See the Tokenizer class constructor
         *  for details.
         */
        this.getGrammar = function () {
            return tokenizer.getGrammar();
        };

        /**
         * Appends a new reference token to this token array.
         *
         * @param {Object} cell1Ref
         *  The first cell reference. Must contain the following properties:
         *  @param {Number[]} cell1Ref.address
         *      The address for the cell reference.
         *  @param {Boolean} [cell1Ref.absCol=false]
         *      Whether the column reference will be marked as absolute
         *      (leading dollar sign).
         *  @param {Boolean} [cell1Ref.absRow=false]
         *      Whether the row reference will be marked as absolute (leading
         *      dollar sign).
         *
         * @param {Object} [cell2Ref]
         *  The second cell reference. Must contain the same properties as the
         *  parameter 'cell1Ref'. If omitted, the reference token will refer to
         *  a single cell specified by 'cell1Ref'.
         *
         * @param {Object} [sheet1Ref]
         *  The first sheet reference. Must contain the following properties:
         *  @param {Number|String} sheet1Ref.sheet
         *      The zero-based index of the sheet the reference token will
         *      refer to, or the explicit name of a non-existing sheet. If
         *      omitted, the reference token will not contain a sheet name.
         *  @param {Boolean} [sheet1Ref.abs=false]
         *      Whether the sheet reference will be marked as absolute.
         *
         * @param {Object} [sheet2Ref]
         *  The second sheet reference, if the reference token refers to a
         *  range of sheets. Must contain the same properties as the parameter
         *  'sheet1Ref'. If omitted, the reference token will refer to a single
         *  sheet specified by 'sheet1Ref'.
         *
         * @returns {TokenArray}
         *  A reference to this instance.
         */
        this.appendReference = function (cell1Ref, cell2Ref, sheet1Ref, sheet2Ref) {
            appendToken(tokenizer.createReferenceToken(cell1Ref, cell2Ref, sheet1Ref, sheet2Ref));
            return this;
        };

        /**
         * Appends a new reference token to this token array, that will be
         * built from the passed cell range address.
         *
         * @param {Object} range
         *  The address of the cell range to be added as reference token to
         *  this token array.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Number} [options.abs=false]
         *      If set to true, all column and row references will be marked as
         *      absolute (leading dollar signs).
         *  @param {Number} [options.sheet]
         *      The zero-based index of the sheet the reference token will
         *      point to. If omitted, no sheet reference will be inserted into
         *      the reference token (sheet-local reference).
         *
         * @returns {TokenArray}
         *  A reference to this instance.
         */
        this.appendRange = function (range, options) {

            var // whether the range spans entire columns or rows
                colRange = model.isColRange(range),
                rowRange = model.isRowRange(range),
                // the cell references
                abs = Utils.getBooleanOption(options, 'abs', false),
                cell1Ref = { address: range.start, absCol: abs || rowRange, absRow: abs || (colRange && !rowRange) },
                cell2Ref = _.isEqual(range.start, range.end) ? null : { address: range.end, absCol: abs || rowRange, absRow: abs || (colRange && !rowRange) },
                // the sheet reference
                sheet = Utils.getIntegerOption(options, 'sheet'),
                sheetRef = _.isNumber(sheet) ? { sheet: sheet, abs: true } : null;

            return this.appendReference(cell1Ref, cell2Ref, sheetRef);
        };

        /**
         * Appends new reference tokens to this token array, that will be built
         * from the passed cell range addresses. If the passed range array
         * contains more than one range address, the reference tokens will be
         * separated by list operator tokens.
         *
         * @param {Array} ranges
         *  The addresses of the cell ranges to be added as reference tokens to
         *  this token array.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Number} [options.abs=false]
         *      If set to true, all column and row references will be marked as
         *      absolute (leading dollar signs).
         *  @param {Number} [options.sheet]
         *      The zero-based index of the sheet the reference tokens will
         *      point to. If omitted, no sheet references will be inserted into
         *      the reference tokens (sheet-local references).
         *  @param {Boolean} [options.parentheses=false]
         *      If set to true, and the passed array contains more than one
         *      range, the entire range list will be enclosed into parentheses.
         *
         * @returns {TokenArray}
         *  A reference to this instance.
         */
        this.appendRangeList = function (ranges, options) {

            var // whether to enclose the range list into parentheses
                parentheses = (ranges.length > 1) && Utils.getBooleanOption(options, 'parentheses', false);

            if (parentheses) {
                appendToken(tokenizer.createParenthesisToken(true));
            }

            _.each(ranges, function (range, index) {
                if (index > 0) { appendToken(tokenizer.createSeparatorToken()); }
                this.appendRange(range, options);
            }, this);

            if (parentheses) {
                appendToken(tokenizer.createParenthesisToken(false));
            }

            return this;
        };

        /**
         * Parses the specified formula string.
         *
         * @param {String} formula
         *  The formula string, without the leading equality sign.
         *
         * @returns {TokenArray}
         *  A reference to this instance.
         */
        this.parseFormula = function (formula) {

            // do not parse if document is still importing (referenced sheets may be missing)
            // store formula text in a bad token which will be reparsed after import is done
            if (!app.isImportFinished()) {
                TokenUtils.log('TokenArray.parseFormula(): caching formula: ' + formula);
                tokens = [tokenizer.createToken('cache', formula)];
                return this;
            }

            // extract all supported tokens from start of formula string
            tokens = tokenizer.parseFormula(formula);
            compiled = null;

            // notify listeners
            this.trigger('change:tokens');
            return this;
        };

        /**
         * Transforms the passed formula expression according to the specified
         * column or row operation.
         *
         * @param {Number} sheet
         *  The zero-based index of the changed sheet.
         *
         * @param {Object} interval
         *  The column/row interval of the operation, with the index properties
         *  'first' and 'last'.
         *
         * @param {Boolean} insert
         *  Whether the specified column/row interval has been inserted into
         *  the sheet (true), or deleted from the sheet (false).
         *
         * @param {Boolean} columns
         *  Whether the specified interval is a column interval (true), or a
         *  row interval (false).
         *
         * @returns {Boolean}
         *  Whether any token in this token array has been changed.
         */
        this.transformFormula = function (sheet, interval, insert, columns) {
            return invokeForAllTokensAndTrigger('transformToken', sheet, interval, insert, columns);
        };

        /**
         * Relocates this formula to a new reference address by adjusting the
         * relative column and row components in all reference tokens. The
         * string representation of all reference tokens will be adjusted
         * accordingly.
         *
         * @param {Number[]} refAddress
         *  The address of the original reference cell the formula is related
         *  to.
         *
         * @param {Number[]} targetAddress
         *  The address of the new reference cell.
         *
         * @param {Boolean} [wrapReferences=false]
         *  If set to true, relocated range addresses that are located outside
         *  the sheet range will be wrapped at the sheet borders.
         *
         * @returns {Boolean}
         *  Whether any token in this token array has been changed.
         */
        this.relocateFormula = function (refAddress, targetAddress, wrapReferences) {
            return !_.isEqual(refAddress, targetAddress) && invokeForAllTokensAndTrigger('relocateToken', refAddress, targetAddress, wrapReferences);
        };

        /**
         * Changes all tokens with sheet references containing the specified
         * old sheet index to the new sheet index. Used for example when
         * copying existing sheets, and adjusting all references and defined
         * names that point to the original sheet.
         *
         * @param {Number} oldSheet
         *  The zero-based index of the original sheet.
         *
         * @param {Number} newSheet
         *  The zero-based index of the target sheet.
         *
         * @returns {Boolean}
         *  Whether any token in this token array has been changed.
         */
        this.relocateSheet = function (oldSheet, newSheet) {
            return (oldSheet !== newSheet) && invokeForAllTokensAndTrigger('relocateSheet', oldSheet, newSheet);
        };

        /**
         * Returns the string representation of the formula represented by this
         * token array.
         *
         * @returns {String}
         *  The string representation of the formula.
         */
        this.getFormula = function () {
            return _.reduce(tokens, function (formula, token) { return formula + token.getText(); }, '');
        };

        /**
         * Returns the string representation of the formula represented by this
         * token array, if it would have been relocated according to the passed
         * settings. This token array will not be modified.
         *
         * @param {Number[]} refAddress
         *  The address of the original reference cell the formula is related
         *  to.
         *
         * @param {Number[]} targetAddress
         *  The address of the new reference cell.
         *
         * @param {Boolean} [wrapReferences=false]
         *  If set to true, relocated range addresses that are located outside
         *  the sheet range will be wrapped at the sheet borders.
         *
         * @returns {String}
         *  The string representation of the relocated formula.
         */
        this.getRelocatedFormula = function (refAddress, targetAddress, wrapReferences) {
            return _.reduce(tokens, function (formula, token) {
                return formula + (_.isFunction(token.generateReference) ? token.generateReference(refAddress, targetAddress, wrapReferences) : token.getText());
            }, '');
        };

        /**
         * Returns the token at the specified array index.
         *
         * @param {Number} index
         *  The array index of the token to be returned.
         *
         * @param {String|RegExp} [typeSpec]
         *  Either a string with a single type identifier that must match the
         *  type of the token exactly, or a regular expression that will be
         *  tested against the type of the token. If omitted, the type of the
         *  token will not be tested.
         *
         * @returns {Token|Null}
         *  The token instance at the specified index; or null if the index is
         *  invalid, or the token does not match the specified type.
         */
        this.getToken = function (index, typeSpec) {
            var token = tokens[index];
            return (token && (!typeSpec || token.isType(typeSpec))) ? token : null;
        };

        /**
         * Invokes a callback function used to modify the specified token in
         * this token array. Afterwards, a 'change:token' event will be
         * triggered.
         *
         * @param {Number} index
         *  The array index of the token to be modified.
         *
         * @param {String|RegExp} [typeSpec]
         *  Either a string with a single type identifier that must match the
         *  type of the token exactly, or a regular expression that will be
         *  tested against the type of the token. If the token at the passed
         *  index does not match the type, the callback function will not be
         *  invoked, and no event will be triggered. If omitted, the type of
         *  the token will not be tested.
         *
         * @param {Function} callback
         *  The callback function invoked for the specified token. Receives the
         *  following parameters:
         *  (1) {Token} token
         *      The token instance.
         *  (2) {Number} index
         *      The token index, as passed to the method modifyToken().
         *  Must return a Boolean value specifying whether the token has been
         *  modified. If the callback returns true, this token array will
         *  trigger a 'change:token' event.
         *
         * @returns {TokenArray}
         *  A reference to this instance.
         */
        this.modifyToken = function (index, typeSpec, callback, context) {

            // shift parameters, if 'typeSpec' has been omitted
            if (_.isFunction(typeSpec)) {
                context = callback;
                callback = typeSpec;
                typeSpec = null;
            }

            var // the token to be modified
                token = this.getToken(index, typeSpec);

            // invoke the callback, trigger change event if callback returns true
            if (token && callback.call(context, token, index)) {
                this.trigger('change:token', token, index);
            }

            return this;
        };

        /**
         * Invokes the passed iterator function for all tokens contained in
         * this token array.
         *
         * @param {Function} iterator
         *  The iterator function invoked for all tokens. Receives the
         *  following parameters:
         *  (1) {Token} token
         *      The current token.
         *  (2) {Number} index
         *      The zero-based token index.
         *  If the iterator returns the Utils.BREAK object, the iteration
         *  process will be stopped immediately.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Object} [options.context]
         *      The calling context for the iterator callback function.
         *  @param {Boolean} [options.reverse=false]
         *      If set to true, the tokens will be visited in reversed order.
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateTokens = function (iterator, options) {
            return Utils.iterateArray(tokens, function (token, index) {
                return iterator.call(this, token, index);
            }, options);
        };

        /**
         * Returns the text cursor position of the specified token.
         *
         * @param {Number} index
         *  The array index of the formula token.
         *
         * @returns {Object|Null}
         *  The text cursor position of the token, in the integer properties
         *  'start' and 'end'; or null, if the passed index is invalid.
         */
        this.getTokenPosition = function (index) {

            var // the text cursor position of the token
                position = { start: 0, end: 0 };

            // validate index
            if ((index < 0) || (index >= tokens.length)) { return null; }

            // iterate to the specified token and calculate text position
            for (var currIndex = 0; currIndex <= index; currIndex += 1) {
                position.start = position.end;
                position.end = position.start + tokens[currIndex].getText().length;
            }

            return position;
        };

        /**
         * Returns information about the token containing or preceding the
         * passed text cursor position in the formula string.
         *
         * @param {Number} pos
         *  The text cursor position in the formula string.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.preferOperand=true]
         *      Specifies how to resolve the situation that the text cursor
         *      position is located between two tokens. If set to true, and one
         *      of the tokens is an operand token (constant value tokens, array
         *      literal tokens, reference tokens, name tokens, or function name
         *      tokens), this token will be returned. Otherwise, always returns
         *      the token preceding the text position.
         *
         * @returns {Object|Null}
         *  A descriptor for the token whose text representation contains the
         *  specified text cursor position, in the following properties:
         *  - {Token} token
         *      The token instance located at the text cursor position.
         *  - {Number} index
         *      The array index of the token.
         *  - {Number} start
         *      The text position of the first character of the token in the
         *      string representation of this token array.
         *  - {Number} end
         *      The text position behind (!) the last character of the token in
         *      the string representation of this token array.
         *  If the text cursor position is located outside of the formula text,
         *  or at its beginning (without the option 'preferOperand'), this
         *  method will return the value null.
         */
        this.getTokenAtPosition = function (pos, options) {

            var // whether to prefer operand tokens
                preferOperand = Utils.getBooleanOption(options, 'preferOperand', false),
                // the length of the formula text already consumed (including the resulting token)
                length = 0,
                // the array index of the token containing or preceding the text cursor position
                index = 0;

            // returns whether the passed token is an operand token
            function isOperand(token) {
                return (/^(value|array|ref|name|func)$/).test(token.getType());
            }

            // cursor is located at the beginning of the formula, and operands are preferred:
            // return the first token (following the text cursor position)
            if (preferOperand && (pos === 0) && (tokens.length > 0) && isOperand(tokens[0])) {
                return { token: tokens[0], index: 0, start: 0, end: tokens[0].getText().length };
            }

            // invalid text position passed
            if (pos <= 0) { return null; }

            // find the token containing or preceding the passed text position
            index = Utils.findFirstIndex(tokens, function (token) {
                length += token.getText().length;
                return pos <= length;
            });
            if (index < 0) { return null; }

            // check if the cursor is located between two tokens, and the trailing is the preferred operand
            if (preferOperand && (pos === length) && (index + 1 < tokens.length) && !isOperand(tokens[index]) && isOperand(tokens[index + 1])) {
                return { token: tokens[index + 1], index: index + 1, start: length, end: length  + tokens[index + 1].getText().length };
            }

            // build and return the complete descriptor for the token
            return { token: tokens[index], index: index, start: length - tokens[index].getText().length, end: length };
        };

        /**
         * Returns information about the innermost function containing the
         * passed text cursor position in the formula string.
         *
         * @param {Number} pos
         *  The text cursor position in the formula string.
         *
         * @returns {Object|Null}
         *  A descriptor for the innermost function containing the specified
         *  text cursor position, in the following properties:
         *  - {String} name
         *      The translated function name.
         *  - {Number} paramIndex
         *      The zero-based index of the function parameter covering the
         *      specified text cursor position.
         *  - {Number} paramCount
         *      The total number of parameters found in the function parameter
         *      list. If the formula is incomplete or contains errors, this
         *      number may be wrong, but it will always be greater than the
         *      value returned in the property 'paramIndex'.
         *  If the text cursor position is outside of any function, this method
         *  will return the value null.
         */
        this.getFunctionAtPosition = function (pos) {

            var // information about the token at the passed position
                tokenInfo = this.getTokenAtPosition(pos, { preferOperand: true }),
                // current index, current token
                index = 0, token = null,
                // the parameter index and count
                param = 0, count = 0;

            // seek to the preceding opening parenthesis, skip embedded pairs of
            // parentheses, return number of skipped separator characters
            function seekToOpeningParenthesis() {
                var seps = 0;
                while (index >= 0) {
                    switch (tokens[index].getType()) {
                    case 'open':
                        return seps;
                    case 'close':
                        index -= 1;
                        seekToOpeningParenthesis();
                        break;
                    case 'sep':
                        seps += 1;
                        break;
                    }
                    index -= 1;
                }
                return 0;
            }

            // seek to the following closing parenthesis, skip embedded pairs of
            // parentheses, return number of skipped separator characters
            function seekToClosingParenthesis() {
                var seps = 0;
                while (index < tokens.length) {
                    switch (tokens[index].getType()) {
                    case 'open':
                        index += 1;
                        seekToClosingParenthesis();
                        break;
                    case 'close':
                        return seps;
                    case 'sep':
                        seps += 1;
                        break;
                    }
                    index += 1;
                }
                return seps;
            }

            // invalid text cursor position passed
            if (!tokenInfo) { return null; }

            // try as long as a function has been found, or the beginning of the formula is reached
            index = tokenInfo.index;
            while (index >= 0) {

                // seek to the preceding opening parenthesis
                param = seekToOpeningParenthesis();

                // check if a function name precedes the parenthesis
                if ((token = this.getToken(index - 1, 'func'))) {
                    index += 1;
                    count = seekToClosingParenthesis() + 1;
                    return { name: token.getText(), paramIndex: param, paramCount: count };
                }

                // continue seeking to preceding opening parenthesis
                index -= 1;
            }

            // cursor is not located inside a function, but it may point to a top-level
            // function name, in that case return this function with invalid parameter index
            if ((tokenInfo.token.getType() === 'func') && (tokens[tokenInfo.index + 1].getType() === 'open')) {
                index = tokenInfo.index + 2;
                count = seekToClosingParenthesis() + 1;
                return { name: tokenInfo.token.getText(), paramIndex: -1, paramCount: count };
            }

            return null;
        };

        /**
         * Resolves the token array to a list of constant strings. The formula
         * represented by this token array must consist of one or more literal
         * string tokens, separated by list operators. Whitespace tokens will
         * be ignored.
         *
         * @returns {Array|Null}
         *  The string literals contained in this token array. If the formula
         *  cannot be resolved successfully to a list of strings, this method
         *  returns null.
         */
        this.resolveStringList = function () {

            var // the resulting strings
                result = [],
                // a string with a single character reflecting the token types in the token array
                tokenTypes = '';

            // process all tokens as long as they are considered valid, collect token types
            _.all(tokens, function (token) {
                switch (token.getType()) {
                case 'lit':
                    tokenTypes += 'L';
                    result.push(token.getValue());
                    return _.isString(_.last(result));
                case 'sep':
                    tokenTypes += ',';
                    return true;
                case 'ws':
                    return true;
                default:
                    tokenTypes += 'I'; // invalid token
                    return false;
                }
            });

            // on success, check the formula structure
            return (/^L(,L)*$/).test(tokenTypes) ? result : null;
        };

        /**
         * Resolves the token array to a list of range addresses. The formula
         * represented by this token array must be completely resolvable to a
         * range list (single reference, a list of references separated by the
         * list operator, optionally enclosed in parentheses, a defined name
         * that resolves to a range list, also recursively, or a mixed list of
         * defined names and references). To extract the range addresses
         * contained in an arbitrary formula, the method
         * TokenArray.extractRanges() can be used.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Number[]} [options.refAddress]
         *      The source reference address used to relocate reference tokens
         *      with relative column/row components. If omitted, cell A1 will
         *      be used as source reference cell.
         *  @param {Number[]} [options.targetAddress]
         *      The target reference address used to relocate reference tokens
         *      with relative column/row components. If omitted, cell A1 will
         *      be used as target reference cell.
         *  @param {Boolean} [options.wrapReferences=false]
         *      If set to true, relocated range addresses that are located
         *      outside the sheet range will be wrapped at the sheet borders.
         *
         * @returns {Array|Null}
         *  The addresses of the resulting cell ranges. Each range address
         *  object contains the additional property 'sheet' with the zero-based
         *  sheet index of the range. If the formula cannot be resolved
         *  successfully to a range list, this method returns null.
         */
        this.resolveRangeList = function (options) {

            var // the resulting range addresses
                ranges = [];

            // helper guard to prevent endless loops for circular references in defined names
            guardVisitedNames(function (visitNameTokenOnceFunc) {

                var // source reference address for relative references
                    refAddress = Utils.getArrayOption(options, 'refAddress', [0, 0]),
                    // target reference address for relative references
                    targetAddress = Utils.getArrayOption(options, 'targetAddress', [0, 0]),
                    // whether to wrap range addresses at sheet borders
                    wrapReferences = Utils.getBooleanOption(options, 'wrapReferences', false),
                    // a string with a single character reflecting the token types in the token array
                    tokenTypes = '';

                // extracts the cell range address from the passed reference token
                function resolveReferenceRange(refToken) {

                    var // the sheet range (different start/end for multi-sheet references)
                        sheetRange = refToken.getSheetRange(),
                        // the cell range address, will be null for invalid ranges (#REF! errors)
                        range = refToken.getRelocatedRange(refAddress, targetAddress, wrapReferences),
                        // whether the token contains a valid range (formula must not contain multi-sheet references)
                        valid = _.isObject(sheetRange) && (sheetRange.sheet1 === sheetRange.sheet2) && _.isObject(range);

                    if (valid) {
                        range.sheet = sheetRange.sheet1;
                        ranges.push(range);
                    }
                    return valid;
                }

                // extracts all cell range addresses from the passed name token
                function resolveNamedRanges(nameToken) {

                    var // the token array with the definition of the name
                        tokenArray = visitNameTokenOnceFunc(nameToken) ? nameToken.getTokenArray() : null,
                        // the resulting ranges referred by the name
                        namedRanges = null;

                    // names are always defined relative to cell A1, and relative references will warp at sheet borders
                    if (tokenArray) {
                        namedRanges = tokenArray.resolveRangeList(Utils.extendOptions(options, { refAddress: [0, 0], wrapReferences: true }));
                    }

                    if (namedRanges) {
                        ranges = ranges.concat(namedRanges);
                        return true;
                    }
                    return false;
                }

                // process all tokens as long as they are considered valid, collect token types
                _.all(tokens, function (token) {
                    switch (token.getType()) {
                    case 'ref':
                        tokenTypes += 'R';
                        return resolveReferenceRange(token);
                    case 'name':
                        tokenTypes += 'R';
                        return resolveNamedRanges(token);
                    case 'open':
                        tokenTypes += '<';
                        return true;
                    case 'close':
                        tokenTypes += '>';
                        return true;
                    case 'sep':
                        tokenTypes += ',';
                        return true;
                    case 'ws':
                        return true;
                    default:
                        tokenTypes += 'I'; // invalid token
                        return false;
                    }
                });

                // reset resulting ranges if formula structure is invalid
                if (!(/^(R(,R)*|<R(,R)*>)$/).test(tokenTypes)) {
                    ranges = null;
                }
            }, this);

            return ranges;
        };

        /**
         * Returns the addresses of all cell ranges this token array refers to.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Number} [options.filterSheet]
         *      If specified, the result will contain only ranges contained in
         *      the sheet with this index.
         *  @param {Number[]} [options.refAddress]
         *      The source reference address used to relocate reference tokens
         *      with relative column/row components. If omitted, cell A1 will
         *      be used as source reference cell.
         *  @param {Number[]} [options.targetAddress]
         *      The target reference address used to relocate reference tokens
         *      with relative column/row components. If omitted, cell A1 will
         *      be used as target reference cell.
         *  @param {Boolean} [options.wrapReferences=false]
         *      If set to true, relocated range addresses that are located
         *      outside the sheet range will be wrapped at the sheet borders.
         *  @param {Boolean} [options.resolveNames=false]
         *      If set to true, will return the range addresses referred by all
         *      defined names contained in this token array.
         *
         * @returns {Array}
         *  The addresses of all cell ranges contained in this token array with
         *  additional information, in an array of objects with the following
         *  properties:
         *  - {Object} sheet1
         *      The zero-based index of the first sheet in the sheet range
         *      contained by the token.
         *  - {Object} sheet2
         *      The zero-based index of the last sheet in the sheet range
         *      contained by the token.
         *  - {Object} range
         *      The range address represented by a formula token.
         *  - {Number} index
         *      The internal array index of the token containing the range
         *      (may be equal for different objects in the array, for example
         *      for a single defined name referring to multiple cell ranges).
         *  - {String} type
         *      The type of the token containing the range. Will be either
         *      'ref' for a regular reference token, or 'name' for a defined
         *      name that evaluates to one or more ranges in the specified
         *      sheet (only if option 'resolveNames' is set, see above).
         */
        this.extractRanges = function (options) {

            var // the resulting ranges
                ranges = [];

            // helper guard to prevent endless loops for circular references in defined names
            guardVisitedNames(function (visitNameTokenOnceFunc) {

                var // the index of the sheet to filter the result
                    filterSheet = Utils.getIntegerOption(options, 'filterSheet'),
                    // source reference address for relative references
                    refAddress = Utils.getArrayOption(options, 'refAddress', [0, 0]),
                    // target reference address for relative references
                    targetAddress = Utils.getArrayOption(options, 'targetAddress', [0, 0]),
                    // whether to wrap range addresses at sheet borders
                    wrapReferences = Utils.getBooleanOption(options, 'wrapReferences', false),
                    // whether to resolve defined name tokens
                    resolveNames = Utils.getBooleanOption(options, 'resolveNames', false);

                // extracts the cell range address from the passed reference token
                function extractReferenceRange(refToken, tokenIndex) {
                    if (!_.isNumber(filterSheet) || refToken.containsSheet(filterSheet)) {
                        var range = refToken.getRelocatedRange(refAddress, targetAddress, wrapReferences); // returns null for invalid ranges (#REF! errors)
                        if (range) { ranges.push(_.extend({ range: range, index: tokenIndex, type: 'ref' }, refToken.getSheetRange())); }
                    }
                }

                // extracts all cell range addresses from the passed name token
                function extractNamedRanges(nameToken, tokenIndex) {
                    var tokenArray = (resolveNames && visitNameTokenOnceFunc(nameToken)) ? nameToken.getTokenArray() : null,
                        rangeInfos = null;
                    if (tokenArray) {
                        // defined names are always defined relative to cell A1, and relative references will warp at sheet borders
                        rangeInfos = tokenArray.extractRanges(Utils.extendOptions(options, { refAddress: [0, 0], wrapReferences: true }));
                        _.each(rangeInfos, function (rangeInfo) { _.extend(rangeInfo, { index: tokenIndex, type: 'name' }); });
                        ranges = ranges.concat(rangeInfos);
                    }
                }

                _.each(tokens, function (token, index) {
                    switch (token.getType()) {
                    case 'ref':
                        extractReferenceRange(token, index);
                        break;
                    case 'name':
                        extractNamedRanges(token, index);
                        break;
                    }
                });
            }, this);

            return ranges;
        };

        /**
         * Calculates the result of the formula represented by this token
         * array.
         *
         * @param {Number[]} refAddress
         *  The address of the reference cell the formula is related to.
         *
         * @returns {Object}
         *  The result descriptor for the formula. See method
         *  Interpreter.interpretTokens() for details.
         */
        this.interpretFormula = function (refAddress) {

            // compile the infix tokens to prefix notation
            if (!compiled) {
                compiled = model.getFormulaCompiler().compileTokens(tokens, tokenizer);
            }

            // check compiler result
            if (compiled.error) {
                return { type: 'error', value: compiled.error };
            }

            // interpret the formula
            return model.getFormulaInterpreter().interpretTokens(compiled.tokens, tokenizer, sheetModel, refAddress);
        };

        // initialization -----------------------------------------------------

        // reparse formula after the document has been imported and all sheets exist
        if (!app.isImportFinished()) {
            // the token array may be destroyed before import finishes (operations that delete the object containing this token array)
            this.listenTo(app.getImportPromise(), 'done', function () {
                if ((tokens.length === 1) && (tokens[0].getType() === 'cache')) {
                    self.parseFormula(tokens[0].getText());
                }
            });
        }

        // refresh all sheet references after the collection of sheets has been changed
        this.listenTo(model, 'insert:sheet:after', function (event, sheet) { invokeForAllTokensAndTrigger('transformSheet', sheet, null); });
        this.listenTo(model, 'delete:sheet:after', function (event, sheet) { invokeForAllTokensAndTrigger('transformSheet', null, sheet); });
        this.listenTo(model, 'move:sheet:after', function (event, to, from) { invokeForAllTokensAndTrigger('transformSheet', to, from); });
        this.listenTo(model, 'rename:sheet', function (event, sheet) { invokeForAllTokensAndTrigger('renameSheet', sheet); });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = model = sheetModel = tokens = compiled = null;
        });

    } // class TokenArray

    // exports ================================================================

    // derive this class from class ModelObject
    return ModelObject.extend({ constructor: TokenArray });

});
