/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/operators',
    ['io.ox/office/tk/utils',
     'io.ox/office/spreadsheet/utils/sheetutils',
     'io.ox/office/spreadsheet/model/formula/tokenutils',
     'io.ox/office/spreadsheet/model/formula/matrix',
     'io.ox/office/spreadsheet/model/formula/reference'
    ], function (Utils, SheetUtils, TokenUtils, Matrix, Reference) {

    'use strict';

    /**************************************************************************
     *
     * This module exports a map with descriptors of all operators and
     * functions supported by the client-side formula interpreter.
     *
     * Each operator descriptor is mapped by the name of the operator.
     *
     * Each function descriptor is mapped by the native function name as used
     * in operations (not the translated function names).
     *
     * Each descriptor (operators and functions) supports the following
     * properties:
     *
     * @property {Number} minParams
     *  Minimum number of parameters supported by the operator or function.
     *  Must be 1 or 2 for operators.
     *
     * @property {Number} [maxParams]
     *  Maximum number of parameters supported by the operator or function.
     *  Must not be less than 'minParams'. Must be 1 or 2 for operators. For
     *  functions, can be omitted to specify that the function accepts the
     *  maximum number of function parameters supported by the current file
     *  format.
     *
     * @property {String} type
     *  The default type of the return value of the operator or function. The
     *  following return types are supported:
     *  - 'val'
     *      A single value (number, string, Boolean value, or error code). May
     *      cause repeated invocation of the operator or function, if called in
     *      a matrix type context.
     *  - 'mat':
     *      A matrix constant, even if it has size 1x1.
     *  - 'ref':
     *      An unresolved cell reference (instance of class Reference). Error
     *      code literals will be accepted too.
     *  - 'any':
     *      The operator or function can return values of any of the previous
     *      type. The outer context of the operator or function will cause the
     *      appropriate type conversion.
     *
     * @property {Array} [signature=[]]
     *  The type signature of the operator or function, as array of strings
     *  specifying the expected data type of the respective operands. Can only
     *  be omitted for functions without parameters. The following data types
     *  are supported:
     *  - 'val':
     *      A single literal value of type number, string, or Boolean. The
     *      special value null represents an empty function parameter, e.g. in
     *      the formula =SUM(1,,2). If the parameter is a constant matrix, its
     *      top-left element will be used. If the parameter is a cell range
     *      reference, it will be resolved to the content value of the cell
     *      related to the reference cell of the formula. If the resulting
     *      value is an error code, it will be thrown as exception value,
     *      without resolving the operator or function.
     *  - 'val:num':
     *      Similar to type 'val', but converts the parameter to a floating-
     *      point number (see method FormulaContext.convertToNumber() for
     *      details). If this conversion fails, the operator or function will
     *      result in a #VALUE! error.
     *  - 'val:int':
     *      Similar to type 'val:num', but converts the parameter to an integer
     *      by rounding down after conversion to a floating-point number (see
     *      method FormulaContext.convertToNumber() for details). Negative
     *      numbers will be rounded down too (NOT towards zero). If this
     *      conversion fails, the operator or function will result in a #VALUE!
     *      error.
     *  - 'val:date':
     *      Similar to type 'val', but converts the parameter to a Date object
     *      (see method FormulaContext.convertToDate() for details). If this
     *      conversion fails, the operator or function will result in a #VALUE!
     *      error.
     *  - 'val:str':
     *      Similar to type 'val', but converts the parameter to a string (see
     *      method FormulaContext.convertToString() for details). If this
     *      conversion fails, the operator or function will result in a #VALUE!
     *      error.
     *  - 'val:bool':
     *      Similar to type 'val', but converts the parameter to a Boolean
     *      value (see method FormulaContext.convertToBoolean() for details).
     *      If this conversion fails, the operator or function will result in a
     *      #VALUE! error.
     *  - 'val:any':
     *      Similar to type 'val', but accepts error codes, and passes them as
     *      value to the resolver function.
     *  - 'mat':
     *      A two-dimensional matrix of literal values available in formulas
     *      (numbers, strings, Boolean values), as instance of the class
     *      Matrix. If the matrix contains one or more error codes, the first
     *      will be thrown as exception value, without resolving the operator
     *      or function. The maximum size of a constant matrix is restricted.
     *      Too large matrixes will cause to throw an UNSUPPORTED_ERROR error
     *      code. If the parameter is a single literal value, it will be
     *      converted to a 1x1 matrix (except for an error code which will be
     *      thrown). If the parameter is a cell range reference, it will be
     *      resolved to a matrix with the content values of all cell (unless
     *      the matrix would be too large, or one of the cells contains an
     *      error code).
     *  - 'mat:num':
     *      Similar to type 'mat', but checks that all matrix elements are
     *      valid floating-point numbers. In difference to parameters of type
     *      'val', other data types will NOT be converted to numbers, but
     *      result in throwing the #VALUE! error code immediately.
     *  - 'mat:str':
     *      Similar to type 'mat', but checks that all matrix elements are
     *      strings. In difference to parameters of type 'val', other data
     *      types will NOT be converted to strings, but result in throwing the
     *      #VALUE! error code immediately.
     *  - 'mat:bool':
     *      Similar to type 'mat', but checks that all matrix elements are
     *      Boolean values. In difference to parameters of type 'val', other
     *      data types will NOT be converted to Boolean values, but result in
     *      throwing the #VALUE! error code immediately.
     *  - 'mat:any':
     *      Similar to type 'mat', but accepts error codes as matrix elements,
     *      and passes them to the resolver function.
     *  - 'ref':
     *      A reference structure, as instance of the class Reference. If the
     *      parameter is not a reference, a special error code will be thrown
     *      indicating that the structure of the formula is considered invalid.
     *  - 'any':
     *      The original operand (instance of class Operand): single value
     *      literals (including error codes), constant matrixes, or unresolved
     *      cell references.
     *
     * @property {Function} resolve
     *  The implementation of the operator or function.
     *  (1) Calling context:
     *      The resolver function will be called with an instance of the class
     *      FormulaContext as calling context. Thus, the symbol 'this' provides
     *      useful helper methods to be used in the implementation of the
     *      operator or function.
     *  (2) Parameters:
     *      The resolver function receives the converted operands according to
     *      the type signature specified in the property 'signature'. If the
     *      operands cannot be resolved to the specified types, the operator or
     *      function will result in the respective error code according to the
     *      error, or a fatal exception will be thrown specifying that the
     *      formula structure is invalid. The resolver function will not be
     *      invoked in this case.
     *  (3) Return value:
     *      The resolver function must return either a literal value (number,
     *      string, Boolean value, UTC Date object, or error code as instance
     *      of ErrorCode), a constant two-dimensional matrix containing values
     *      of any of the types mentioned before, a cell reference literal
     *      (instance of the class Reference), or an operand as received by an
     *      'any' parameter. If the returned number is infinite or NaN, it will
     *      be converted to a #NUM! error. If the returned (absolute) number is
     *      less than the least supported normalized positive number (see
     *      constant TokenUtils.MIN_NUMBER), it will be converted to zero. All
     *      these rules apply to all numbers in a matrix too.
     *  (4) Exceptions:
     *      Error code literals can be thrown as exception value, instead of
     *      being returned. This may happen implicitly, e.g. when using the
     *      data type conversion methods provided by the calling context (see
     *      class FormulaContext). Additionally, the resolver function may
     *      throw specific exception codes:
     *      - 'fatal': An unrecoverable internal error occurred. Interpretation
     *          of the formula will be canceled.
     *
     *************************************************************************/

    var // shortcut for the map of error code literals
        ErrorCodes = SheetUtils.ErrorCodes;

    // private global functions ===============================================

    /**
     * Returns the quotient of the passed numbers. If the divisor is zero, the
     * error code #DIV/0! will be thrown instead.
     */
    function divide(number1, number2) {
        if (number2 === 0) { throw ErrorCodes.DIV0; }
        return number1 / number2;
    }

    /**
     * Returns the power of the passed numbers. If both numbers are zero, the
     * error code #NUM! will be thrown instead of returning 1.
     */
    function power(number1, number2) {
        // Math.pow() returns 1 for 0^0; Excel returns #NUM! error
        if ((number1 === 0) && (number2 === 0)) { throw ErrorCodes.NUM; }
        return Math.pow(number1, number2);
    }

    /**
     * Rounds the passed number to the next integer towards zero (positive
     * numbers will be rounded down, negative numbers will be rounded up).
     */
    function trunc(number) {
        return (number < 0) ? Math.ceil(number) : Math.floor(number);
    }

    /**
     * Compares the passed literal values. The following rules apply for
     * comparison:
     * (1) Two null values are equal.
     * (2) A null value is converted to 0, if compared to a number.
     * (3) A null value is converted to '', if compared to a string.
     * (4) A null value is converted to FALSE, if compared to a Boolean.
     * (5) Numbers are always less than strings and Boolean values.
     * (6) Strings are always less than Boolean values.
     * (7) Two numbers are simply compared by value.
     * (8) Two strings are compared lexicographically and case-insensitive.
     * (9) FALSE is less than TRUE.
     *
     * @param {Number|String|Boolean|ErrorCode|Null} value1
     *  The first value to be compared.
     *
     * @param {Number|String|Boolean|ErrorCode|Null} value2
     *  The second value to be compared.
     *
     * @returns {Number}
     *  - A negative value, if value1 is less than value2.
     *  - A positive value, if value1 is greater than value2.
     *  - Zero, if both values are of the same type and are equal.
     */
    function compareValues(value1, value2) {

        // converts special value null (empty cell) according to the other value type
        function convertNull(value) {
            if (_.isNumber(value)) { return 0; }
            if (_.isString(value)) { return ''; }
            if (_.isBoolean(value)) { return false; }
            throw 'fatal';
        }

        // returns a numeric identifier according to the value type
        function getTypeId(value) {
            if (_.isNumber(value)) { return 1; }
            if (_.isString(value)) { return 2; }
            if (_.isBoolean(value)) { return 3; }
            throw 'fatal';
        }

        // convert null values to typed values according to the other value
        if (_.isNull(value1) && _.isNull(value2)) { return 0; }
        if (_.isNull(value1)) { value1 = convertNull(value2); }
        if (_.isNull(value2)) { value2 = convertNull(value1); }

        // get type identifiers, compare different types
        var type1 = getTypeId(value1), type2 = getTypeId(value2);
        if (type1 !== type2) { return type1 - type2; }

        // compare values of equal types
        switch (type1) {
        case 1:
            var diff = value1 - value2;
            return (Math.abs(diff) < TokenUtils.MIN_NUMBER) ? 0 : diff;
        case 2:
            value1 = value1.toUpperCase();
            value2 = value2.toUpperCase();
            return (value1 < value2) ? -1 : (value1 > value2) ? 1 : 0;
        case 3:
            // Booleans will be converted to numbers implicitly
            return value1 - value2;
        }

        throw 'fatal';
    }

    /**
     * Returns the greatest common denominator of the passed values.
     *
     * @param {Number} value1
     *  The first number.
     *
     * @param {Number} value2
     *  The second number.
     *
     * @returns {Number}
     *  The greatest common denominator of the passed values.
     */
    function getGCD(value1, value2) {

        function fmod(v1, v2) {
            return v1 - Math.floor(v1 / v2) * v2;
        }

        var f = fmod(value1, value2);
        while (f > 0) {
            value1 = value2;
            value2 = f;
            f = fmod(value1, value2);
        }

        return value2;
    }

    /**
     * Returns the binomial coefficient of the passed input values.
     *
     * @param {Number} n
     *  The upper value of the binomial coefficient (size of the set to be
     *  chosen from).
     *
     * @param {Number} k
     *  The lower value of the binomial coefficient (number of choices from the
     *  set).
     *
     * @returns {Number}
     *  The binomial coefficient, if both values are non-negative, and 'n' is
     *  greater than or equal to 'k'.
     *
     * @throws {ErrorCode}
     *  The #NUM! error code, if the passed values are invalid, or if the
     *  resulting binomial coefficient is too large.
     */
    function getBinomialCoefficient(n, k) {

        // check input values
        if ((n < 0) || (k < 0) || (n < k)) { throw ErrorCodes.NUM; }

        // special cases (n 0) and (n n)
        if ((k === 0) || (k === n)) { return 1; }

        // binomial coefficient is symmetric: (n k) = (n n-k); use lower k for performance
        if (k * 2 > n) { k = n - k; }

        var // the resulting binomial coefficient
            result = n - k + 1;

        // The function isFinite() is quite expensive, by using two nested loops (the inner
        // loop runs up to 10 iterations without checking the result) the entire algorithm
        // becomes about 30% to 40% faster.
        for (var i = 2; (i <= k) && isFinite(result); ) {
            for (var j = 0; (j < 10) && (i <= k); i += 1, j += 1) {
                result *= (n - k + i);
                result /= i;
            }
        }

        // check final result
        if (isFinite(result)) { return result; }
        throw ErrorCodes.NUM;
    }

    /**
     * Returns whether the passed number is a leap year.
     *
     * @param {Number} year
     *  The year to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed number is a leap year.
     */
    function isLeapYear(year) {
        return (((year % 4) === 0) && ((year % 100) !== 0)) || ((year % 400) === 0);
    }

    /**
     * Tries to convert a string containing a binary value to a number.
     *
     * @param {String} value
     *  A binary number string to be converted to a number if it is not longer
     *  than 10 characters.
     *
     * @returns {Number}
     *  The resulting decimal number.
     *
     * @throws {ErrorCode}
     *  If the passed value is not a binary number or if it is too long, a
     *  #NUM! error will be thrown.
     */
    function convertBIN2Number(value) {

        // check input value, maximum length of the input string is 10 binary digits
        if (!/^[01]{0,10}$/.test(value)) {
            throw ErrorCodes.NUM;
        }

        var // the parsed result
            result = parseInt(value, 2);

        if ((value.length === 10) && (value.charAt(0) === '1')) {
            result -= 1024;
        }

        return result;
    }

    /**
     * @param {Number} value
     *
     * @param {Number} order
     *
     * @returns {Number}
     *
     * @throws {ErrorCode}
     *  If the passed order is negative a #NUM! error will be thrown.
     */
    function getBesselI(value, order) {

        var result = 0,
            epsilon = 1.0E-15,
            maxIteration = 2000,
            xHalf = value / 2.0,
            k = 0,
            term = 1.0;

        if (order < 0) {
            throw ErrorCodes.NUM;
        }

        for (k = 1; k <= order; ++k) {
           term = term / ( k ) * xHalf;
        }
        result = term;
        if (term !== 0) {
            k = 1;
            do {
                term = term * xHalf / k * xHalf / (k + order);
                result += term;
                k += 1;
            } while ((Math.abs(term) > Math.abs(result) * epsilon) && (k < maxIteration));
        }
        return result;
    }
    /**
     * @param {Date} date
     *
     * @param {Number} mode
     *  determines the start day of the week
     *  undefined, 1 or 17 : Sunday
     *  2, 11: Monday
     *  12...16 : Tuesday...Saturday
     *
     * @param {bool} iso
     * determines that iso mode is required, weeks start on Monday, first week is the one containing a Thursday
     *
     * @returns {Number}
     *  the resulting week number
     *
     * @throws {ErrorCode}
     *  if mode is not valid it throws a #NUM! error
     */
    function getWeeknum(date, mode, iso) {
        var dayNo,
        tmpDate = new Date( date.valueOf() ),
        firstThursday,
        startDay,
        targetDay;
        if (!mode) {
            mode = 1;
        }
        switch(mode) {
            case 1:
            case 17:
            case 2:
                targetDay = 1;
            break;
            case 11:
            case 22:
                targetDay = 1;
            break;
            case 12: targetDay = 2; break;
            case 13: targetDay = 3; break;
            case 14: targetDay = 4; break;
            case 15: targetDay = 5; break;
            case 16: targetDay = 6; break;
            default:    throw ErrorCodes.NUM;
        }
        if(iso) {
            startDay = 6;
        } else {
            startDay = targetDay;
        }
        dayNo = ( (date.getDay() + startDay) % 7 );
        tmpDate.setDate( date.getDate() - dayNo + 3 );
        firstThursday = tmpDate.valueOf();
        tmpDate.setMonth( 0, 1 );
        if  ( tmpDate.getDay() !==  targetDay ) {
            tmpDate.setMonth( 0, 1 + (( targetDay - tmpDate.getDay() ) + 7 ) % 7);
        }
        return 1 + Math.ceil(( firstThursday - tmpDate) / 604800000 );
    }

    /**
     * @param {Number} value
     * the value to be rounded
     * @param {Number} mult
     * multiple to round to
     * @returns {Number} the value rounded to a multiple of mult
     * @throws {ErrorCode}
     *  if signs of value and mult are different
     */
    function mround(value, mult) {
        if( value * mult < 0) {
            throw ErrorCodes.NUM;
        }
        var mod = Math.abs(value % mult),
            result = Math.abs(value),
            amult = Math.abs(mult);
        if ( mod ) {
            if ( mod > amult / 2) {
                result += amult - mod;
            } else {
                result -= mod;
            }
        }
        return value < 0 ? -result : result;
    }

    // operators ==============================================================

    var OPERATORS = {

        // arithmetic operators -----------------------------------------------

        // binary addition operator, or unary plus operator
        '+': {
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: ['val', 'val:num'], // unary plus operator passes all values (no conversion to number)
            resolve: function (value1, value2) {
                return _.isUndefined(value2) ? value1 : (this.convertToNumber(value1) + value2);
            }
        },

        // binary subtraction operator, or unary minus operator
        '-': {
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: ['val:num', 'val:num'], // unary minus operator always converts to numbers
            resolve: function (value1, value2) {
                return _.isUndefined(value2) ? -value1: (value1 - value2);
            }
        },

        // binary multiplication operator
        '*': {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: ['val:num', 'val:num'],
            resolve: function (value1, value2) { return value1 * value2; }
        },

        // binary division operator
        '/': {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: ['val:num', 'val:num'],
            resolve: divide
        },

        // binary power operator
        '^': {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: ['val:num', 'val:num'],
            resolve: power
        },

        // unary percent operator
        '%': {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:num'],
            resolve: function (value) { return value / 100; }
        },

        // string operators ---------------------------------------------------

        // binary string concatenation operator
        '&': {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: ['val:str', 'val:str'],
            resolve: function (value1, value2) { return value1 + value2; }
        },

        // comparison operators -----------------------------------------------

        // binary less-than operator
        '<': {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: ['val', 'val'],
            resolve: function (value1, value2) {
                return compareValues(value1, value2) < 0;
            }
        },

        // binary less-than-or-equal operator
        '<=': {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: ['val', 'val'],
            resolve: function (value1, value2) {
                return compareValues(value1, value2) <= 0;
            }
        },

        // binary greater-than operator
        '>': {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: ['val', 'val'],
            resolve: function (value1, value2) {
                return compareValues(value1, value2) > 0;
            }
        },

        // binary greater-than-or-equal operator
        '>=': {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: ['val', 'val'],
            resolve: function (value1, value2) {
                return compareValues(value1, value2) >= 0;
            }
        },

        // binary equals operator
        '=': {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: ['val', 'val'],
            resolve: function (value1, value2) {
                return compareValues(value1, value2) === 0;
            }
        },

        // binary equals-not operator
        '<>': {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: ['val', 'val'],
            resolve: function (value1, value2) {
                return compareValues(value1, value2) !== 0;
            }
        },

        // reference operators ------------------------------------------------

        // binary list reference operator
        ',': {
            minParams: 2,
            maxParams: 2,
            type: 'ref',
            signature: ['ref', 'ref'],
            resolve: function (ref1, ref2) {
                return new Reference(ref1.getRanges().concat(ref2.getRanges()));
            }
        },

        // binary intersection reference operator
        ' ': {
            minParams: 2,
            maxParams: 2,
            type: 'ref',
            signature: ['ref', 'ref'],
            resolve: function (ref1, ref2) {

                var // get all ranges in the passed operands
                    ranges1 = ref1.getRanges({ sameSheet: true }),
                    ranges2 = ref2.getRanges({ sameSheet: true }),
                    sheet = ranges1[0].sheet1,
                    intersectRanges = [];

                // ranges in both operands must refer to the same sheet
                if (sheet !== ranges2[0].sheet1) {
                    throw ErrorCodes.VALUE;
                }

                // calculate the intersection of each range pair from ranges1 and ranges2
                _.each(ranges1, function (range1) {
                    intersectRanges = intersectRanges.concat(SheetUtils.getIntersectionRanges(ranges2, range1));
                    // check resulting list size in loop, otherwise this loop may create a MAX^2 list
                    if (intersectRanges.length > TokenUtils.MAX_REF_LIST_SIZE) { throw TokenUtils.UNSUPPORTED_ERROR; }
                });

                // insert sheet index into all ranges
                _.each(intersectRanges, function (range) {
                    range.sheet1 = range.sheet2 = sheet;
                });

                return new Reference(intersectRanges);
            }
        },

        // binary range reference operator
        ':': {
            minParams: 2,
            maxParams: 2,
            type: 'ref',
            signature: ['ref', 'ref'],
            resolve: function (ref1, ref2) {

                var // get all ranges in the passed operands
                    ranges1 = ref1.getRanges({ sameSheet: true }),
                    ranges2 = ref2.getRanges({ sameSheet: true }),
                    boundRange = null;

                // ranges in both operands must refer to the same sheet
                if (ranges1[0].sheet1 !== ranges2[0].sheet1) {
                    throw ErrorCodes.VALUE;
                }

                // build the bounding range from all ranges
                boundRange = SheetUtils.getBoundingRange(ranges1, ranges2);
                boundRange.sheet1 = boundRange.sheet2 = ranges1[0].sheet1;
                return new Reference(boundRange);
            }
        }
    };

    // functions ==============================================================

    // ordered alphabetically to reduce risk of merge conflicts
    var FUNCTIONS = {

        ABS: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:num'],
            resolve: Math.abs
        },

        ACOS: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:num'],
            resolve: Math.acos
        },

        ACOSH: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:num'],
            resolve: function (value) {
                return Math.log(value + Math.sqrt(value * value - 1));
            }
        },

        ACOT: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:num'],
            resolve: function (value) {
                return Math.PI / 2 - Math.atan(value);
            }
        },

        ACOTH: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:num'],
            resolve: function (value) {
                return Math.log((value + 1) / (value - 1)) / 2;
            }
        },

        ADDRESS: {
            minParams: 2,
            maxParams: 5,
            type: 'val',
            signature: ['val:int', 'val:int', 'val:int', 'val:bool', 'val:str'],
            resolve: function (row, col, absMode, a1Style, sheetName) {

                var // document model
                    model = this.getModel(),
                    // absolute columns or row index
                    absCol = false, absRow = false,
                    // the resulting address string
                    result = '';

                // returns the string representation of the passed column/row index for A1 notation
                function createA1Index(index, abs, columns) {
                    var maxIndex = model.getMaxIndex(columns);
                    if ((1 <= index) && (index <= maxIndex + 1)) {
                        return (abs ? '$' : '') + SheetUtils.getIndexName(index - 1, columns);
                    }
                    throw ErrorCodes.VALUE;
                }

                // returns the string representation of the passed column/row index for RC notation
                function createRCIndex(index, abs, columns) {
                    var maxIndex = model.getMaxIndex(columns);
                    if (abs) {
                        // absolute mode: always insert one-based column/row index (e.g. R1C4)
                        if ((1 <= index) && (index <= maxIndex + 1)) { return '' + index; }
                    } else {
                        // relative mode: leave out zero values (e.g. RC[-1])
                        if (index === 0) { return ''; }
                        if ((-maxIndex <= index) && (index <= maxIndex)) { return '[' + index + ']'; }
                    }
                    throw ErrorCodes.VALUE;
                }

                // default for missing or empty absolute mode (always defaults to 1)
                if (this.isMissingOrEmptyOperand(2)) {
                    absMode = 1;
                } else if ((absMode < 1) || (absMode > 4)) {
                    throw ErrorCodes.VALUE;
                }
                absCol = (absMode % 2) > 0;
                absRow = absMode <= 2;

                // default for missing or empty A1 style (always defaults to TRUE)
                if (this.isMissingOrEmptyOperand(3)) {
                    a1Style = true;
                }

                if (a1Style) {
                    result = createA1Index(col, absCol, true) + createA1Index(row, absRow, false);
                } else {
                    // TODO: localized R/C names
                    result = 'R' + createRCIndex(row, absRow, false) + 'C' + createRCIndex(col, absCol, true);
                }

                // add encoded sheet name (enclosed in apostrophes if needed)
                // (not for missing or empty parameter, but for explicit empty string)
                if (!this.isMissingOrEmptyOperand(4)) {
                    result = ((sheetName.length > 0) ? this.getTokenizer().encodeSheetName(sheetName) : '') + '!' + result;
                }

                return result;
            }
        },

        AND: {
            minParams: 1,
            type: 'val',
            signature: ['any'],
            resolve: function () {
                var result = {}; // start with non-Boolean truthy value
                this.iterateOperands(0, function (operand) {
                    this.iterateBooleans(operand, function (value) {
                        result = result && value;
                    }, {
                        valMode: 'convert', // value operands: convert strings and numbers to Booleans
                        matMode: 'skip', // matrix operands: skip all strings
                        refMode: 'skip', // reference operands: skip all strings
                        skipEmpty: false, // empty parameters count as FALSE: AND(TRUE,) = FALSE
                        complexRef: true // accept multi-range and multi-sheet references
                    });
                });
                if (_.isBoolean(result)) { return result; }
                // default result (no Booleans found in parameters) is the #VALUE! error code
                throw ErrorCodes.VALUE;
            }
        },

        AREAS: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['ref'],
            resolve: function (ref) {
                return ref.getLength();
            }
        },

        ASIN: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:num'],
            resolve: Math.asin
        },

        ASINH: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:num'],
            resolve: function (value) {
                return Math.log(value + Math.sqrt(value * value + 1));
            }
        },

        ATAN: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:num'],
            resolve: Math.atan
        },

        ATAN2: {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: ['val:num', 'val:num'],
            resolve: function (value1, value2) {
                return Math.atan2(value2, value1);
            }
        },

        ATANH: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:num'],
            resolve: function (value) {
                return Math.log((1 + value) / (1 - value)) / 2;
            }
        },

        AVEDEV: {
            minParams: 1,
            type: 'val',
            signature: ['any'],
            resolve: function () {
                var values = [], sum = 0, count = 0, average = 0;
                this.iterateOperands(0, function (operand) {
                    this.iterateNumbers(operand, function (value) {
                        values.push(value);
                        sum += value;
                        count += 1;
                    }, {
                        valMode: 'convert', // value operands: convert strings and Booleans to numbers
                        matMode: 'skip', // matrix operands: skip strings and Booleans
                        refMode: 'skip', // reference operands: skip strings and Booleans
                        rejectBoolean: false, // Booleans count as numbers: AVEDEV(1,FALSE) = 0.5
                        skipEmpty: false, // empty parameters count as zero: AVEDEV(1,) = 0.5
                        complexRef: true // accept multi-range and multi-sheet references
                    });
                });
                // default result (no numbers found in parameters) is the #NUM! error code (not #DIV/0!)
                if (count === 0) { throw ErrorCodes.NUM; }
                average = sum / count;
                return Utils.getSum(values, function (value) { return Math.abs(value - average); }) / count;
            }
        },

        AVERAGE: {
            minParams: 1,
            type: 'val',
            signature: ['any'],
            resolve: function () {
                var result = 0, count = 0;
                this.iterateOperands(0, function (operand) {
                    this.iterateNumbers(operand, function (value) {
                        result += value;
                        count += 1;
                    }, {
                        valMode: 'convert', // value operands: convert strings and Booleans to numbers
                        matMode: 'skip', // matrix operands: skip strings and Booleans
                        refMode: 'skip', // reference operands: skip strings and Booleans
                        rejectBoolean: false, // Booleans count as numbers: AVERAGE(1,FALSE) = 0.5
                        skipEmpty: false, // empty parameters count as zero: AVERAGE(1,) = 0.5
                        complexRef: true // accept multi-range and multi-sheet references
                    });
                });
                // default result (no numbers found in parameters) is the #DIV/0! error code
                return divide(result, count);
            }
        },

        AVERAGEA: {
            minParams: 1,
            type: 'val',
            signature: ['any'],
            resolve: function () {
                var result = 0, count = 0;
                this.iterateOperands(0, function (operand) {
                    this.iterateNumbers(operand, function (value) {
                        result += value;
                        count += 1;
                    }, {
                        valMode: 'convert', // value operands: convert strings and Booleans to numbers
                        matMode: 'zero', // matrix operands: replace strings with zeros: AVERAGEA(1,{"a"}) = 0.5
                        refMode: 'zero', // reference operands: replace strings with zeros
                        rejectBoolean: false, // Booleans count as numbers: AVERAGEA(1,FALSE) = 0.5
                        skipEmpty: false, // empty parameters count as zero: AVERAGEA(1,) = 0.5
                        complexRef: true // accept multi-range and multi-sheet references
                    });
                });
                // default result (no numbers found in parameters) is the #DIV/0! error code
                return divide(result, count);
            }
        },

        BASE: {
            minParams: 2,
            maxParams: 3,
            type: 'val',
            signature: ['val:int', 'val:int', 'val:int'],
            resolve: function (number, radix, minLength) {
                var result;
                minLength = _.isUndefined(minLength) ? 0 : minLength;
                if (number < 0 || number > 0x1FFFFFFFFFFFFF || radix < 2 || radix > 36 || minLength < 0 || minLength > 255) {
                    throw ErrorCodes.NUM;
                }
                result = number.toString(radix);
                while (minLength > result.length) {
                    result = '0' + result;
                }
                return result.toUpperCase();
            }
        },

        BESSELI: {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: ['val:num', 'val:int'],
            resolve: getBesselI
        },

        BESSELJ: {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: ['val:num', 'val:num'],
            resolve: function (value, order) {

                var sign, maxIteration, estimateIteration, asymptoticPossible,
                    epsilon = 1.0e-15,
                    hasfound = false,
                    k= 0,
                    u,
                    mBar, gBar, gBarDeltaU,
                    g = 0,
                    deltaU = 0,
                    bar = -1.0,
                    PIDiv2 = Math.PI / 2.0,
                    PIDiv4 = Math.PI / 4.0,
                    twoDivPI = 2.0 / Math.PI;

                function fmod(v1, v2) {
                    return v1 - Math.floor(v1 / v2) * v2;
                }

                if( order < 0 ) {
                    throw ErrorCodes.NUM;
                }
                if (value === 0) {
                    return (order === 0) ? 1.0 : 0.0;
                }
                sign = ( value % 2 === 1 && value < 0) ? -1.0 : 1.0;
                value = Math.abs(value);

                maxIteration = 9000000.0; //experimental, for to return in < 3 seconds in C++!
                estimateIteration = value * 1.5 + order;
                asymptoticPossible = Math.pow(value, 0.4) > order;
                if ( estimateIteration > maxIteration) {
                    if (asymptoticPossible) {
                        return sign * Math.sqrt( twoDivPI / value ) * Math.cos( value - order * PIDiv2 - PIDiv4 );
                    }
                    throw ErrorCodes.NUM;
                }

                if (value === 0) {
                    u = 1;
                    gBarDeltaU = 0;
                    gBar = - 2.0 / value;
                    deltaU = gBarDeltaU / gBar;
                    u = u + deltaU ;
                    g = -1.0 / gBar;
                    bar = bar * g;
                    k = 2;
                } else  {
                    u=0;
                    for (k =1; k<= order - 1; k = k + 1) {
                        mBar = 2 * fmod( k - 1, 2 ) * bar;
                        gBarDeltaU = - g * deltaU - mBar * u; // alpha_k = 0.0
                        gBar = mBar - 2 * k / value + g;
                        deltaU = gBarDeltaU / gBar;
                        u = u + deltaU;
                        g = -1 / gBar;
                        bar = bar * g;
                    }
                    // Step alpha_N = 1.0
                    mBar=2.0 * fmod(k-1.0, 2) * bar;
                    gBarDeltaU = bar - g * deltaU - mBar * u; // alpha_k = 1.0
                    gBar = mBar - 2.0*k / value + g;
                    deltaU = gBarDeltaU / gBar;
                    u = u + deltaU;
                    g = -1 / gBar;
                    bar = bar * g;
                    k = k + 1;
                }
                // Loop until desired accuracy, always alpha_k = 0.0
                do {
                    mBar = 2.0 * fmod (k - 1,  2 ) * bar;
                    gBarDeltaU = - g * deltaU - mBar * u;
                    gBar = mBar - 2 * k / value + g;
                    deltaU = gBarDeltaU / gBar;
                    u = u + deltaU;
                    g = -1 / gBar;
                    bar = bar * g;
                    hasfound = (Math.abs(deltaU)<= Math.abs( u ) * epsilon);
                    k = k + 1;
                } while (!hasfound && k <= maxIteration);

                if (hasfound) {
                    return u * sign;
                }

                throw ErrorCodes.VALUE;
            }
        },

        BESSELK: {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: ['val:num', 'val:num'],
            resolve: function (value, order) {

                var result = 0,
                    tox, bkm, bkp, bk, n;

                function besselK0(value) {

                    var result, num2, y;

                     if( value <= 2.0 ) {
                         num2 = value * 0.5;
                         y = num2 * num2;

                         result = -Math.log( num2 ) * getBesselI( value, 0 ) +
                                ( -0.57721566 + y * ( 0.42278420 + y * ( 0.23069756 + y * ( 0.3488590e-1 +
                                    y * ( 0.262698e-2 + y * ( 0.10750e-3 + y * 0.74e-5 ) ) ) ) ) );
                     } else {
                         y = 2.0 / value;

                         result = Math.exp( -value ) / Math.sqrt( value ) * ( 1.25331414 + y * ( -0.7832358e-1 +
                                y * ( 0.2189568e-1 + y * ( -0.1062446e-1 + y * ( 0.587872e-2 +
                                y * ( -0.251540e-2 + y * 0.53208e-3 ) ) ) ) ) );
                     }
                    return result;
                }

                function besselK1(value) {

                    var result, num2, y;

                    if( value <= 2.0 ) {
                        num2 = value * 0.5;
                        y = num2 * num2;

                        result = Math.log( num2 ) * getBesselI( value, 1 ) +
                                ( 1.0 + y * ( 0.15443144 + y * ( -0.67278579 + y * ( -0.18156897 + y * ( -0.1919402e-1 +
                                    y * ( -0.110404e-2 + y * ( -0.4686e-4 ) ) ) ) ) ) ) / value;
                    } else {
                        y = 2.0 / value;

                        result = Math.exp( -value ) / Math.sqrt( value ) * ( 1.25331414 + y * ( 0.23498619 +
                                y * ( -0.3655620e-1 + y * ( 0.1504268e-1 + y * ( -0.780353e-2 +
                                y * ( 0.325614e-2 + y * ( -0.68245e-3 ) ) ) ) ) ) );
                    }

                    return result;
                }

                switch (order) {
                case 0:
                    result = besselK0( value );
                    break;
                case 1:
                    result = besselK1( value );
                    break;
                default:
                    tox = 2.0 / value;
                    bkm = besselK0( value );
                    bk = besselK1( value );

                    for( n = 1 ; n < order ; n++ ) {
                        bkp = bkm +  n * tox * bk;
                        bkm = bk;
                        bk = bkp;
                    }

                    result = bk;
                }
                return result;
            }
        },

        BESSELY: {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: ['val:num', 'val:num'],
            resolve: function (value, order) {
                var result,
                    maxIteration = 9000000,
                    epsilon = 1.0e-15,
                    eulerGamma = 0.57721566490153286060,
                    byp,
                    tox = 2 / value,
                    bym,
                    by,
                    n;

                function besselY0(value) {
                    var alpha = Math.log( value /2 ) + eulerGamma,
                        u,
                        k = 1,
                        mBar = 0,
                        gBarDeltaU = 0,
                        gBar = -2.0 / value,
                        deltaU = gBarDeltaU / gBar,
                        g = -1.0 / gBar,
                        fBar = -1 * g,
                        signAlpha = 1,
                        km1mod2,
                        hasFound = false,
                        twoDivPI = ( 2 / Math.PI );

                    if (value <=  0) {
                        throw ErrorCodes.VALUE;
                    }
                    if ( value > 5.0e+6 ) {
                        return Math.sqrt( 1/ Math.PI / value) * ( Math.sin( value ) - Math.cos(value));
                    }
                    u = alpha;
                    k = k + 1;
                    do {
                        km1mod2 = ( (k-1) % 2 );
                        mBar = (2 * km1mod2) * fBar;
                        if  ( km1mod2 === 0) {
                            alpha = 0;
                        } else {
                           alpha = signAlpha * (4 / k);
                           signAlpha = -signAlpha;
                        }
                        gBarDeltaU = fBar * alpha - g * deltaU - mBar * u;
                        gBar = mBar - (2 * k ) / value + g;
                        deltaU = gBarDeltaU / gBar;
                        u = u + deltaU;
                        g = -1.0 / gBar;
                        fBar = fBar*g;
                        hasFound = ( Math.abs( deltaU ) <= Math.abs( u ) * epsilon );
                        k = k + 1;
                    }
                    while (!hasFound && k< maxIteration);
                    if ( hasFound) {
                        return u * twoDivPI;
                    }
                    throw ErrorCodes.VALUE;
                }

                function besselY1(value) {

                    var alpha = 1.0 / value,
                        fBar = -1,
                        g = 0.0,
                        u = alpha,
                        k = 1,
                        mBar = 0.0,
                        gBarDeltaU,
                        gBar = -2 / value,
                        deltaU,
                        signAlpha = -1.0,
                        km1mod2,
                        q,
                        hasFound = false;

                    if (value <=  0) {
                        throw ErrorCodes.VALUE;
                    }
                    if ( value > 5.0e+6 ) {
                        return - Math.sqrt( 1 / Math.PI / value) *(Math.sin( value ) + Math.cos( value ));
                    }
                    alpha = 1.0 - eulerGamma - Math.log( value / 2 );
                    gBarDeltaU = -alpha;
                    deltaU = gBarDeltaU / gBar;
                    u = u + deltaU;
                    g = -1 / gBar;
                    fBar = fBar * g;

                    k = k + 1;
                    do
                    {
                        km1mod2 = ( (k - 1) % 2);
                        mBar = ( 2.0 * km1mod2 ) * fBar;
                        q = ( k - 1 ) / 2;
                        if ( km1mod2 === 0) {
                           alpha = signAlpha * (1.0/q + 1.0/(q+1.0));
                           signAlpha = -signAlpha;
                        } else {
                            alpha = 0;
                        }
                        gBarDeltaU = fBar * alpha - g * deltaU - mBar * u;
                        gBar = mBar - (2.0*k)/ value + g;
                        deltaU = gBarDeltaU / gBar;
                        u = u + deltaU;
                        g = -1 / gBar;
                        fBar = fBar*g;
                        hasFound = ( Math.abs( deltaU) <= Math.abs( u ) * epsilon );
                        k = k + 1;
                    }
                    while (!hasFound && k < maxIteration);
                    if (hasFound) {
                        return -u * 2 / Math.PI;
                    }
                    throw ErrorCodes.VALUE;
                }

                switch (order) {
                case 0:
                    result = besselY0( value );
                    break;
                case 1:
                    result = besselY1( value );
                    break;
                default:
                    bym = besselY0( value );
                    by = besselY1( value );

                    for( n = 1 ; n < order ; n++ ) {
                        byp = n * tox * by - bym;
                        bym = by;
                        by = byp;
                    }
                    result = by;
                }

                return result;
            }
        },

        BIN2DEC: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:str'],
            resolve: convertBIN2Number
        },

        BIN2HEX: {
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: ['val:str'],
            resolve: function (value, length) {
                var result = convertBIN2Number(value);
                if (result < 0) {
                    result = 0x10000000000 + result;
                }
                result = (result).toString( 16 );
                if(length && length > result.length){
                    if(length > 10){
                        throw ErrorCodes.NUM;
                    }
                    result = ('0000000000' + result).slice(-length);
                }                    
                return result.toUpperCase();
            }
        },

        BIN2OCT: {
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: ['val:str'],
            resolve: function (value, length) {
                var result = convertBIN2Number(value);
                if (result < 0) {
                    result = 0x40000000 + result;
                }
                result = (result).toString( 8 );
                if(length && length > result.length){
                    if(length > 10){
                        throw ErrorCodes.NUM;
                    }
                    result = ('0000000000' + result).slice(-length);
                }                    
                return result;
            }
        },

        CEILING: {
            minParams: 2,
            maxParams: 3,
            type: 'val',
            signature: ['val:num', 'val:num', 'val:num'],
            resolve: function (number, significance) {
                var result, modulo, minBorder = 1E-16;
                if (significance === 0) {
                    throw ErrorCodes.VALUE;
                } else if(significance < 0 && number > 0) {
                    throw ErrorCodes.NUM;
                }
                modulo = Math.abs( significance - (number % significance) );
                if (number === 0 || modulo < minBorder) {
                    result = number;
                } else {
                    result = (Math.floor(number / significance) + 1) * significance;
                }
                return result;
            }
        },

        CHAR: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:int'],
            resolve: function (value) {
                if ((value < 1) || (value > 255)) { throw ErrorCodes.VALUE; }
                // TODO: not exact, needs codepage conversion
                return String.fromCharCode(value);
            }
        },

        CHOOSE: {
            minParams: 2,
            type: 'any',
            signature: ['val:int', 'any'],
            resolve: function (index) {
                return this.getOperand(index - 1);
            }
        },

        CLEAN: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:str'],
            resolve: function (value) {
                // TODO: other Unicode characters?
                return value.replace(/[\x00-\x1f\x81\x8d\x8f\x90\x9d]/g, '');
            }
        },

        CODE: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:str'],
            resolve: function (value) {
                // TODO: not exact, needs codepage conversion
                return (value.length === 0) ? ErrorCodes.VALUE : value.charCodeAt(0);
            }
        },

        COLUMN: {
            minParams: 0,
            maxParams: 1,
            type: 'val',
            signature: ['ref'],
            resolve: function (ref) {
                // use reference address, if parameter is missing
                var address = _.isUndefined(ref) ? this.getRefAddress() : ref.getSingleRange().start;
                // function returns one-based column index
                return address[0] + 1;
            }
        },

        COLUMNS: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['any'],
            resolve: function (operand) {
                switch (operand.getType()) {
                case 'val':
                    var value = operand.getValue();
                    return SheetUtils.isErrorCode(value) ? value : 1;
                case 'mat':
                    return operand.getMatrix().getColCount();
                case 'ref':
                    return SheetUtils.getColCount(operand.getReference().getSingleRange());
                default:
                    throw 'fatal';
                }
            }
        },

        COMBIN: {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: ['val:int', 'val:int'],
            resolve: getBinomialCoefficient
        },

        COMBINA: {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: ['val:int', 'val:int'],
            resolve: function (value1, value2) {
                return getBinomialCoefficient(value1 + value2 - 1, value2);
            }
        },

        COMPLEX: {
            minParams: 2,
            maxParams: 3,
            type: 'val',
            signature: ['val:num', 'val:num', 'val:str'],
            resolve: function (value1, value2, suffix) {
                // use 'i', if suffix is missing (undefined), empty parameter (null), or empty string
                if (!suffix) { suffix = 'i'; }
                // suffix must be lower-case 'i' or 'j'
                if (!/^[ij]$/.test(suffix)) { throw ErrorCodes.VALUE; }
                // leave out imaginary part if zero
                if (value2 === 0) { return '' + value1; }
                // leave out real part if missing; do not add single '1' for imaginary part
                return ((value1 === 0) ? '' : value1) + ((value2 < 0) ? '-' : (value1 === 0) ? '' : '+') + ((Math.abs(value2) === 1) ? '' : Math.abs(value2)) + suffix;
            }
        },

        CONCATENATE: {
            minParams: 1,
            type: 'val',
            // each parameter is a single value (function does not concatenate cell ranges or matrixes!)
            signature: ['val:str'],
            resolve: function () {
                return _.reduce(arguments, function (result, value) { return result + value; }, '');
            }
        },

        COS: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:num'],
            resolve: Math.cos
        },

        COSH: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:num'],
            resolve: function(value) {
                return (Math.exp(value) + Math.exp(-value)) / 2;
            }
        },

        COT: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:num'],
            resolve: function (value) { return divide(1, Math.tan(value)); }
        },

        COTH: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:num'],
            resolve: function (value) {
                return divide(Math.exp(value) + Math.exp(-value), Math.exp(value) - Math.exp(-value));
            }
        },

        COUNT: {
            minParams: 1,
            type: 'val',
            signature: ['any'],
            resolve: function () {
                var count = 0;
                this.iterateOperands(0, function (operand) {
                    if (operand.isValue()) {
                        // count Booleans and strings that are convertible to numbers
                        // (simple values only, not counted in matrixes and references)
                        try {
                            this.convertToNumber(operand.getValue());
                            count += 1;
                        } catch (ex) {}
                    } else {
                        // matrixes and references: count real numbers only (no strings, no Booleans)
                        this.iterateValues(operand, function (value) {
                            if (_.isNumber(value)) { count += 1; }
                        }, {
                            acceptErrors: true, // do not fail on error codes: =COUNT({1,#VALUE!}) = 1
                            complexRef: true // accept multi-range and multi-sheet references
                        });
                    }
                });
                return count;
            }
        },

        COUNTA: {
            minParams: 1,
            type: 'val',
            signature: ['any'],
            resolve: function () {
                var count = 0;
                this.iterateOperands(0, function (operand) {
                    switch (operand.getType()) {
                    case 'val':
                        // all values are counted (also zeros, empty strings, FALSE,
                        // error codes, empty parameters): COUNTA(1,,#VALUE!) = 3
                        count += 1;
                        break;
                    case 'mat':
                        // all matrix elements are counted (also zeros, empty strings, FALSE, error codes)
                        count += operand.getMatrix().getElementCount();
                        break;
                    case 'ref':
                        // all filled cells are counted (also error codes)
                        this.iterateValues(operand, function () { count += 1; }, {
                            acceptErrors: true, // count all error codes
                            complexRef: true // accept multi-range and multi-sheet references
                        });
                    }
                });
                return count;
            }
        },

        COUNTBLANK: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['ref'],
            resolve: function (ref) {

                var // number of non-blank cells
                    count = 0,
                    // whether empty strings (formula results) are blank (Excel only)
                    emptyStr = !this.getApp().isODF();

                // count all non-blank cells in the reference
                this.iterateValues(ref, function (value) {
                    if (!emptyStr || (value !== '')) { count += 1; }
                }, {
                    acceptErrors: true, // count all error codes
                    complexRef: false // do not accept multi-range and multi-sheet references
                });

                // return number of remaining blank cells
                return ref.getCellCount() - count;
            }
        },

        CSC: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:num'],
            resolve: function (value) { return divide(1, Math.sin(value)); }
        },

        CSCH: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:num'],
            resolve: function (value) {
                return divide(2 * Math.exp(-value), 1 - Math.exp(-2 * value));
            }
        },

        DATE: {
            minParams: 3,
            maxParams: 3,
            type: 'val',
            signature: ['val:int'],
            resolve: function (year, month, day) {
                // Excel adds 1900 to all years less than 1900; e.g. 1899 becomes 3799 (TODO: different behavior for ODF?)
                if (year < 1900) { year += 1900; }
                // create a UTC date object (Date object expects zero-based month)
                return new Date(Date.UTC(year, month - 1, day));
            }
        },

        DAY: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:date'],
            resolve: function (date) {
                return date.getUTCDate();
            }
        },

        DEGREES: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:num'],
            resolve: function (value) {
                return value / Math.PI * 180;
            }
        },

        EVEN: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:num'],
            resolve: function (value) {
                var result = Math.ceil(Math.abs(value));
                if((result % 2) !== 0) {
                    ++result;
                }
                return value < 0 ? -result : result;
            }
        },

        EXACT: {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: ['val:str', 'val:str'],
            resolve: function (value1, value2) {
                return value1 === value2; // exact case-sensitive comparison
            }
        },

        EXP: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:num'],
            resolve: Math.exp
        },

        FALSE: {
            minParams: 0,
            maxParams: 0,
            type: 'val',
            resolve: _.constant(false)
        },

        FLOOR: {
            minParams: 2,
            maxParams: 3,
            type: 'val',
            signature: ['val:num', 'val:num', 'val:num'],
            resolve: function (number, significance, mode) { //mode in odf, only!
                var result, modulo, minBorder = 1E-16;
                if (mode === undefined) {
                    mode = this.getApp().isODF() ? 0 : 1;
                }
                if (significance === 0 || (number > 0 && significance < 0) ) {
                    throw ErrorCodes.VALUE;
                }
                modulo = Math.abs( significance - (number % significance) );
                if (number === 0 || modulo < minBorder) {
                    result = number;
                } else {
                    if ( mode === 0 && number < 0 ) {
                        result = (Math.floor(number / significance) + 1) * significance;
                    } else {
                        result = (Math.floor(number / significance)) * significance;
                    }
                }
                return result;
            }
        },

        GCD: {
            minParams: 1,
            type: 'val',
            signature: ['any'],
            resolve: function () {
                var result = null, hasZero = false;
                this.iterateOperands(0, function (operand) {
                    this.iterateNumbers(operand, function (value) {
                        // round down to integers
                        value = Math.floor(value);
                        // fail on negative numbers
                        if (value < 0) { throw ErrorCodes.NUM; }
                        // always skip zeros (in difference to LCM)
                        if (value > 0) {
                            result = _.isNumber(result) ? getGCD(result, value) : value;
                        } else {
                            hasZero = true;
                        }
                    }, {
                        valMode: 'convert', // value operands: convert strings and Booleans to numbers
                        matMode: 'convert', // matrix operands: convert strings and Booleans to numbers
                        refMode: 'convert', // reference operands: convert strings and Booleans to numbers
                        rejectBoolean: true, // Booleans lead to error: GCD(6,TRUE) = #VALUE!
                        skipEmpty: true, // empty parameters are ignored: GCD(6,) = 6
                        complexRef: false // do not accept multi-range and multi-sheet references
                    });
                });
                if (_.isNumber(result)) { return result; }
                // only zeros as arguments: return zero instead of error code
                if (hasZero) { return 0; }
                // default result (no numbers found in parameters) is the #VALUE! error code
                throw ErrorCodes.VALUE;
            }
        },

        HOUR: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:date'],
            resolve: function (date) {
                return date.getUTCHours();
            }
        },

        IF: {
            minParams: 2,
            maxParams: 3,
            type: 'any',
            signature: ['val:bool', 'any', 'any'],
            resolve: function (cond, operand1, operand2) {
                return cond ? operand1 : _.isUndefined(operand2) ? false : operand2;
            }
        },

        IFERROR: {
            minParams: 2,
            maxParams: 2,
            type: 'any',
            signature: ['val:any', 'val:any'], // always return values (in difference to IF)
            resolve: function (value1, value2) {
                return SheetUtils.isErrorCode(value1) ? value2 : value1;
            }
        },

        IFNA: {
            minParams: 2,
            maxParams: 2,
            type: 'any',
            signature: ['val:any', 'val:any'], // always return values (in difference to IF)
            resolve: function (value1, value2) {
                return ErrorCodes.NA.equals(value1) ? value2 : value1;
            }
        },

        INT: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:int'],
            resolve: _.identity // always rounds down, e.g. INT(-1.5) = -2
        },

        ISBLANK: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:any'],
            resolve: _.isNull // error codes result in FALSE
        },

        ISERR: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:any'],
            resolve: function (value) {
                return SheetUtils.isErrorCode(value) && !ErrorCodes.NA.equals(value);
            }
        },

        ISERROR: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:any'],
            resolve: SheetUtils.isErrorCode
        },

        ISEVEN: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:int'],
            resolve: function (value) {
                return (value % 2) === 0;
            }
        },

        ISFORMULA: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['ref'],
            resolve: function (ref) {

                var // parameter must be a single range
                    range = ref.getSingleRange(),
                    // use the first cell in the range (TODO: matrix context)
                    address = range.start;

                // accept reference to own cell (no circular reference error)
                if ((range.sheet1 === this.getRefSheet()) && _.isEqual(address, this.getRefAddress())) {
                    return true; // do not check the cell, a new formula has not been inserted yet
                }

                // getCellFormula() returns null for value cells
                return _.isString(this.getCellFormula(range.sheet1, address));
            }
        },

        ISLEAPYEAR: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:date'],
            resolve: function (value) {
                return isLeapYear(value.getUTCFullYear());
            }
        },

        ISLOGICAL: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:any'],
            resolve: _.isBoolean // error codes and empty cells result in FALSE
        },

        ISNA: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:any'],
            resolve: ErrorCodes.NA.equals
        },

        ISNONTEXT: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:any'],
            resolve: function (value) {
                return !_.isString(value); // error codes and empty cells result in TRUE
            }
        },

        ISNUMBER: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:any'],
            resolve: _.isNumber // error codes and empty cells result in FALSE
        },

        ISODD: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:int'],
            resolve: function (value) {
                return (value % 2) === 1;
            }
        },

        ISOWEEKNUM: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:date'],
            resolve: function (date) {
                return getWeeknum( date, 14, true );
            }
        },

        ISREF: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['any'],
            resolve: function (operand) {
                return operand.isReference(); // error codes result in FALSE
            }
        },

        ISTEXT: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:any'],
            resolve: _.isString // error codes and empty cells result in FALSE
        },

        LCM: {
            minParams: 1,
            type: 'val',
            signature: ['any'],
            resolve: function () {
                var result = null;
                this.iterateOperands(0, function (operand) {
                    this.iterateNumbers(operand, function (value) {
                        // round down to integers
                        value = Math.floor(value);
                        // fail on negative numbers
                        if (value < 0) { throw ErrorCodes.NUM; }
                        // do not ignore zeros (in difference to GCD)
                        result = _.isNumber(result) ? (result * value / getGCD(value, result)) : value;
                    }, {
                        valMode: 'convert', // value operands: convert strings and Booleans to numbers
                        matMode: 'convert', // matrix operands: convert strings and Booleans to numbers
                        refMode: 'convert', // reference operands: convert strings and Booleans to numbers
                        rejectBoolean: true, // Booleans lead to error: LCM(6,TRUE) = #VALUE!
                        skipEmpty: true, // empty parameters are ignored: LCM(6,) = 6 while LCM(6,0) = 0
                        complexRef: false // do not accept multi-range and multi-sheet references
                    });
                });
                if (_.isNumber(result)) { return result; }
                // default result (no numbers found in parameters) is the #VALUE! error code
                throw ErrorCodes.VALUE;
            }
        },

        LEFT: {
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: ['val:str', 'val:int'],
            resolve: function (text, count) {
                if (_.isUndefined(count)) { count = 1; }
                this.checkStringLength(count);
                return (count === 0) ? text : text.substr(0, count);
            }
        },

        LEN: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:str'],
            resolve: function (text) {
                return text.length;
            }
        },

        LN: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:num'],
            resolve: Math.log
        },

        LOG: {
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: ['val:num', 'val:num'],
            resolve: function (value, base) {
                if (_.isUndefined(base)) { base = 10; }
                return Math.log(value) / Math.log(base);
            }
        },

        LOG10: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:num'],
            resolve: function (value) {
                return Math.log(value) / Math.LN10;
            }
        },

        LOWER: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:str'],
            resolve: function (text) {
                return text.toLowerCase();
            }
        },

        MAX: {
            minParams: 1,
            type: 'val',
            signature: ['any'],
            resolve: function () {
                var result = Number.NEGATIVE_INFINITY;
                this.iterateOperands(0, function (operand) {
                    this.iterateNumbers(operand, function (value) {
                        result = Math.max(result, value);
                    }, {
                        valMode: 'convert', // value operands: convert strings and Booleans to numbers
                        matMode: 'skip', // matrix operands: skip strings and Booleans
                        refMode: 'skip', // reference operands: skip strings and Booleans
                        rejectBoolean: false, // Booleans count as numbers: MAX(0,TRUE) = 1
                        skipEmpty: false, // empty parameters count as zero: MAX(-1,) = 0
                        complexRef: true // accept multi-range and multi-sheet references
                    });
                });
                // default result (no numbers found in parameters) is zero (no error code)
                return isFinite(result) ? result : 0;
            }
        },

        MAXA: {
            minParams: 1,
            type: 'val',
            signature: ['any'],
            resolve: function () {
                var result = Number.NEGATIVE_INFINITY;
                this.iterateOperands(0, function (operand) {
                    this.iterateNumbers(operand, function (value) {
                        result = Math.max(result, value);
                    }, {
                        valMode: 'convert', // value operands: convert strings and Booleans to numbers
                        matMode: 'skip', // matrix operands: skip strings and Booleans
                        refMode: 'zero', // reference operands: replace all strings with zeros
                        rejectBoolean: false, // Booleans count as numbers: MAXA(0,TRUE) = 1
                        skipEmpty: false, // empty parameters count as zero: MAXA(-1,) = 0
                        complexRef: true // accept multi-range and multi-sheet references
                    });
                });
                // default result (no numbers found in parameters) is zero (no error code)
                return isFinite(result) ? result : 0;
            }
        },

        MID: {
            minParams: 3,
            maxParams: 3,
            type: 'val',
            signature: ['val:str', 'val:int', 'val:int'],
            resolve: function (text, start, count) {
                this.checkStringIndex(start);
                this.checkStringLength(count);
                return text.substr(start - 1, count);
            }
        },

        MIN: {
            minParams: 1,
            type: 'val',
            signature: ['any'],
            resolve: function () {
                var result = Number.POSITIVE_INFINITY;
                this.iterateOperands(0, function (operand) {
                    this.iterateNumbers(operand, function (value) {
                        result = Math.min(result, value);
                    }, {
                        valMode: 'convert', // value operands: convert strings and Booleans to numbers
                        matMode: 'skip', // matrix operands: skip strings and Booleans
                        refMode: 'skip', // reference operands: skip strings and Booleans
                        rejectBoolean: false, // Booleans count as numbers: MIN(2,TRUE) = 1
                        skipEmpty: false, // empty parameters count as zero: MIN(1,) = 0
                        complexRef: true // accept multi-range and multi-sheet references
                    });
                });
                // default result (no numbers found in parameters) is zero (no error code)
                return isFinite(result) ? result : 0;
            }
        },

        MINA: {
            minParams: 1,
            type: 'val',
            signature: ['any'],
            resolve: function () {
                var result = Number.POSITIVE_INFINITY;
                this.iterateOperands(0, function (operand) {
                    this.iterateNumbers(operand, function (value) {
                        result = Math.min(result, value);
                    }, {
                        valMode: 'convert', // value operands: convert strings and Booleans to numbers
                        matMode: 'skip', // matrix operands: skip strings and Booleans
                        refMode: 'zero', // reference operands: replace all strings with zeros
                        rejectBoolean: false, // Booleans count as numbers: MINA(2,TRUE) = 1
                        skipEmpty: false, // empty parameters count as zero: MINA(1,) = 0
                        complexRef: true // accept multi-range and multi-sheet references
                    });
                });
                // default result (no numbers found in parameters) is zero (no error code)
                return isFinite(result) ? result : 0;
            }
        },

        MINUTE: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:date'],
            resolve: function (date) {
                return date.getUTCMinutes();
            }
        },

        MMULT: {
            minParams: 2,
            maxParams: 2,
            type: 'mat',
            signature: ['mat:num'],
            resolve: function (matrix1, matrix2) {

                // width of first matrix must match height of second matrix
                if (matrix1.getColCount() !== matrix2.getRowCount()) { throw ErrorCodes.VALUE; }

                // calculate all elements of the result matrix
                return new Matrix(matrix1.getRowCount(), matrix2.getColCount(), function (row, col) {
                    var elem = 0;
                    for (var i = 0, l = matrix1.getColCount(); i < l; i += 1) {
                        elem += (matrix1.getElement(row, i) * matrix2.getElement(i, col));
                    }
                    return elem;
                });
            }
        },

        MOD: {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: ['val:num', 'val:num'],
            resolve: function (value, divisor) {
                var result = value % divisor;
                if( result !== 0 && value * divisor < 0) {
                    result = divisor + result;
                }
                return result;
            }
        },

        MONTH: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:date'],
            resolve: function (date) {
                return date.getUTCMonth() + 1; // Date class returns zero-based month
            }
        },

        MROUND: {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: ['val:num', 'val:num'],
            resolve: mround
        },

        N: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val'],
            resolve: function (value) {
                return _.isNumber(value) ? value : (value === true) ? 1 : 0;
            }
        },

        NA: {
            minParams: 0,
            maxParams: 0,
            type: 'val',
            resolve: _.constant(ErrorCodes.NA)
        },

        NOT: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:bool'],
            resolve: function (value) { return !value; }
        },

        NOW: {
            minParams: 0,
            maxParams: 0,
            type: 'val',
            resolve: function() {
                var now = new Date();
                // create a UTC date object representing the current local date/time
                return new Date(Date.UTC(now.getFullYear(), now.getMonth(), now.getDate(), now.getHours(), now.getMinutes(), now.getSeconds(), now.getMilliseconds()));
            }
        },

        ODD: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:num'],
            resolve: function (value) {
                var result = Math.ceil(Math.abs(value));
                if((result % 2) === 0) {
                    ++result;
                }
                return value < 0 ? -result : result;
            }
        },

        OR: {
            minParams: 1,
            type: 'val',
            signature: ['any'],
            resolve: function () {
                var result = null; // start with non-Boolean falsy value
                this.iterateOperands(0, function (operand) {
                    this.iterateBooleans(operand, function (value) {
                        result = result || value;
                    }, {
                        valMode: 'convert', // value operands: convert strings and numbers to Booleans
                        matMode: 'skip', // matrix operands: skip all strings
                        refMode: 'skip', // reference operands: skip all strings
                        skipEmpty: true, // empty parameters count as FALSE and can be skipped
                        complexRef: true // accept multi-range and multi-sheet references
                    });
                });
                if (_.isBoolean(result)) { return result; }
                // default result (no Booleans found in parameters) is the #VALUE! error code
                throw ErrorCodes.VALUE;
            }
        },

        PI: {
            minParams: 0,
            maxParams: 0,
            type: 'val',
            resolve: _.constant(Math.PI)
        },

        POWER: {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: ['val:num', 'val:num'],
            resolve: power
        },

        PRODUCT: {
            minParams: 1,
            type: 'val',
            signature: ['any'],
            resolve: function () {
                var result = null;
                this.iterateOperands(0, function (operand) {
                    this.iterateNumbers(operand, function (value) {
                        result = _.isNumber(result) ? (result * value) : value;
                    }, {
                        valMode: 'convert', // value operands: convert strings and Booleans to numbers
                        matMode: 'skip', // matrix operands: skip strings and Booleans
                        refMode: 'skip', // reference operands: skip strings and Booleans
                        rejectBoolean: false, // Booleans count as numbers: PRODUCT(1,FALSE) = 0
                        skipEmpty: true, // empty parameters are ignored: PRODUCT(1,) = 1
                        complexRef: true // accept multi-range and multi-sheet references
                    });
                });
                // default result (no numbers found in parameters) is zero (no error code)
                return _.isNumber(result) ? result : 0;
            }
        },

        PROPER: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:str'],
            resolve: function (text) {
                // TODO: the reg-exp does not include all characters that count as letters
                return text.replace(/[A-Za-z\xAA\xB5\xBA\xC0-\xD6\xD8-\xF6\xF8-\u02B8\u02BB-\u02C1\u02D0-\u02D1\u02E0-\u02E4\u02EE\u0370-\u0373\u0376-\u0377\u037A-\u037D\u0386\u0388-\u038A\u038C\u038E-\u03A1\u03A3-\u03F5\u03F7-\u0481\u048A-\u0523]+/g, Utils.capitalize);
            }
        },

        QUOTIENT: {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: ['val:num', 'val:num'],
            resolve: function (value1, value2) {
                return trunc(divide(value1, value2));
            }
        },

        RADIANS: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:num'],
            resolve: function (value) {
                return value * Math.PI / 180;
            }
        },

        RIGHT: {
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: ['val:str', 'val:int'],
            resolve: function (text, count) {
                if (_.isUndefined(count)) { count = 1; }
                this.checkStringLength(count);
                return (count === 0) ? text : text.substr(-count);
            }
        },

        REPLACE: {
            minParams: 4,
            maxParams: 4,
            type: 'val',
            signature: ['val:str', 'val:int', 'val:int', 'val:str'],
            resolve: function (oldText, start, count, newText) {
                this.checkStringIndex(start);
                this.checkStringLength(count);
                start -= 1;
                return Utils.replaceSubString(oldText, start, start + count, newText);
            }
        },

        REPT: {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: ['val:str', 'val:int'],
            resolve: function (text, repeat) {
                this.checkStringLength(text.length * repeat);
                return (text.length === 0) ? '' : Utils.repeatString(text, repeat);
            }
        },

        ROUND: {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: ['val:num', 'val:int'],
            resolve: function (value, digits) {
                var mult = Math.pow(10, -digits);
                return mround( value, value < 0 ? -mult : mult );
            }
        },

        ROUNDDOWN: {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: ['val:num', 'val:int'],
            resolve: function (value, digits) {
                var mult = Math.pow(10, -digits),
                    smult = value < 0 ? -mult : mult,
                    result = mround( value, smult );
                return result !== value && Math.abs(result) > Math.abs(value) ? result - smult : result;
            }
        },

        ROUNDUP: {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: ['val:num', 'val:int'],
            resolve: function (value, digits) {
                var mult = Math.pow(10, -digits),
                    smult = value < 0 ? -mult : mult,
                    result = mround( value, smult );
                return result !== value && Math.abs(result) < Math.abs(value) ? result + smult : result;
            }
        },

        ROW: {
            minParams: 0,
            maxParams: 1,
            type: 'val',
            signature: ['ref'],
            resolve: function (ref) {
                // use reference address, if parameter is missing
                var address = _.isUndefined(ref) ? this.getRefAddress() : ref.getSingleRange().start;
                // function returns one-based row index
                return address[1] + 1;
            }
        },

        ROWS: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['any'],
            resolve: function (operand) {
                switch (operand.getType()) {
                case 'val':
                    var value = operand.getValue();
                    return SheetUtils.isErrorCode(value) ? value : 1;
                case 'mat':
                    return operand.getMatrix().getRowCount();
                case 'ref':
                    return SheetUtils.getRowCount(operand.getReference().getSingleRange());
                default:
                    throw 'fatal';
                }
            }
        },

        SECOND: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:date'],
            resolve: function (date) {
                return date.getUTCSeconds();
            }
        },

        SEC: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:num'],
            resolve: function (value) {
                return divide(1, Math.cos(value));
            }
        },

        SECH: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:num'],
            resolve: function (value) {
                return divide(2 * Math.exp(-value), 1 + Math.exp(-2 * value));
            }
        },

        SHEET: {
            minParams: 0,
            maxParams: 1,
            type: 'val',
            signature: ['any'], // string (sheet name) or reference
            resolve: function (operand) {
                // use reference sheet, if parameter is missing
                if (_.isUndefined(operand)) { return this.getRefSheet() + 1; }
                // use sheet index of a reference
                if (operand.isReference()) { return operand.getReference().getSingleRange().sheet1 + 1; }
                // convert values to strings (sheet name may look like a number or Boolean)
                if (operand.isValue()) {
                    var sheet = this.getModel().getSheetIndex(this.convertToString(operand.getValue()));
                    return (sheet < 0) ? ErrorCodes.NA : (sheet + 1);
                }
                // always fail for matrix operands, regardless of context
                return ErrorCodes.NA;
            }
        },

        SHEETS: {
            minParams: 0,
            maxParams: 1,
            type: 'val',
            signature: ['ref'],
            resolve: function (ref) {
                // return total number of sheets in document, if parameter is missing
                if (_.isUndefined(ref)) { return this.getModel().getSheetCount(); }
                // parameter must be a single range, but may point to multiple sheets
                var range = ref.getSingleRange({ multiSheet: true });
                return range.sheet2 - range.sheet1 + 1;
            }
        },

        SIGN: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:num'],
            resolve: function (value) {
                return (value < 0) ? -1 : (value > 0) ? 1 : 0;
            }
        },

        SIN: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:num'],
            resolve: Math.sin
        },

        SINH: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:num'],
            resolve: function (value) {
                return (Math.exp(value) - Math.exp(-value)) / 2;
            }
        },

        SQRT: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:num'],
            resolve: Math.sqrt
        },

        SQRTPI: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:num'],
            resolve: function (value) {
                return Math.sqrt(Math.PI * value);
            }
        },

        SUM: {
            minParams: 1,
            type: 'val',
            signature: ['any'],
            resolve: function () {
                var result = 0;
                this.iterateOperands(0, function (operand) {
                    this.iterateNumbers(operand, function (value) {
                        result += value;
                    }, {
                        valMode: 'convert', // value operands: convert strings and Booleans to numbers
                        matMode: 'skip', // matrix operands: skip strings and Booleans
                        refMode: 'skip', // reference operands: skip strings and Booleans
                        rejectBoolean: false, // Booleans count as numbers: SUM(1,TRUE) = 2
                        skipEmpty: true, // empty parameters are ignored
                        complexRef: true // accept multi-range and multi-sheet references
                    });
                });
                // default result (no numbers found in parameters) is zero (no error code)
                return result;
            }
        },

        SUMSQ: {
            minParams: 1,
            type: 'val',
            signature: ['any'],
            resolve: function () {
                var result = 0;
                this.iterateOperands(0, function (operand) {
                    this.iterateNumbers(operand, function (value) {
                        result += (value * value);
                    }, {
                        valMode: 'convert', // value operands: convert strings and Booleans to numbers
                        matMode: 'skip', // matrix operands: skip strings and Booleans
                        refMode: 'skip', // reference operands: skip strings and Booleans
                        rejectBoolean: false, // Booleans count as numbers: SUMSQ(2,TRUE) = 5
                        skipEmpty: true, // empty parameters are ignored
                        complexRef: true // accept multi-range and multi-sheet references
                    });
                });
                // default result (no numbers found in parameters) is zero (no error code)
                return result;
            }
        },

        T: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val'],
            resolve: function (value) {
                return _.isString(value) ? value : '';
            }
        },

        TAN: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:num'],
            resolve: Math.tan
        },

        TANH: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:num'],
            resolve: function (value) {
                return (Math.exp(value) - Math.exp(-value)) / (Math.exp(value) + Math.exp(-value));
            }
        },

        TIME: {
            minParams: 3,
            maxParams: 3,
            type: 'val',
            signature: ['val:int'],
            resolve: function (hour, minute, second) {

                var // the time value, as fraction of a day
                    time = (hour / 24 + minute / 1440 + second / 86400) % 1;

                // must not result in a negative value
                if (time < 0) { throw ErrorCodes.NUM; }

                // convert to a date object, using the document's current null date
                return this.convertToDate(time);
            }
        },

        TODAY: {
            minParams: 0,
            maxParams: 0,
            type: 'val',
            resolve: function() {
                var now = new Date();
                // create a UTC date object representing the current local date (but not the time)
                return new Date(Date.UTC(now.getFullYear(), now.getMonth(), now.getDate()));
            }
        },

        TRANSPOSE: {
            minParams: 1,
            maxParams: 1,
            type: 'mat',
            signature: ['mat:any'],
            resolve: function (matrix) {
                return new Matrix(matrix.getColCount(), matrix.getRowCount(), function (row, col) {
                    return matrix.getElement(col, row);
                });
            }
        },

        TRIM: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:str'],
            resolve: function (text) {
                // TRIM removes space characters (U+0020) only, no other white-space or control characters
                return text.replace(/^ +| +$/, '').replace(/  +/g, ' ');
            }
        },

        TRUE: {
            minParams: 0,
            maxParams: 0,
            type: 'val',
            resolve: _.constant(true)
        },

        TRUNC: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:num'],
            resolve: trunc
        },

        UNICHAR: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:int'],
            resolve: function (value) {
                if ((value < 1) || (value > 65535)) { throw ErrorCodes.VALUE; }
                var char = String.fromCharCode(value);
                if (/[\ud800-\udfff\ufdd0-\ufdef\ufffe\uffff]/.test(char)) { throw ErrorCodes.NA; }
                return char;
            }
        },

        UNICODE: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:str'],
            resolve: function (value) {
                return (value.length === 0) ? ErrorCodes.VALUE : value.charCodeAt(0);
            }
        },

        UPPER: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:str'],
            resolve: function (text) {
                return text.toUpperCase();
            }
        },

        WEEKDAY: {
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: ['val:date', 'val:int'],
            resolve: function (date, mode) {

                var // weekday index, from 0 to 6, starting at Sunday
                    day = date.getUTCDay();

                // missing parameter defaults to 1 (but: empty parameter defaults to 0)
                switch (_.isUndefined(mode) ? 1 : mode) {
                case 1:
                    return day + 1; // 1 to 7, starting at Sunday
                case 2:
                    return (day === 0) ? 7 : day; // 1 to 7, starting at Monday
                case 3:
                    return (day + 6) % 7; // 0 to 6, starting at Monday
                }

                // other modes result in #NUM! error
                throw ErrorCodes.NUM;
            }
        },

        WEEKNUM: {
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: ['val:date', 'val:int'],
            resolve: function (date, mode) {
                return getWeeknum( date, mode);
            }
        },

        XOR: {
            minParams: 1,
            type: 'val',
            signature: ['any'],
            resolve: function () {
                var result = null; // start with non-Boolean falsy value
                this.iterateOperands(0, function (operand) {
                    this.iterateBooleans(operand, function (value) {
                        result = result ? !value : value; // including null to Boolean
                    }, {
                        valMode: 'convert', // value operands: convert strings and numbers to Booleans
                        matMode: 'skip', // matrix operands: skip all strings
                        refMode: 'skip', // reference operands: skip all strings
                        skipEmpty: true, // empty parameters count as FALSE and can be skipped
                        complexRef: true // accept multi-range and multi-sheet references
                    });
                });
                if (_.isBoolean(result)) { return result; }
                // default result (no Booleans found in parameters) is the #VALUE! error code
                throw ErrorCodes.VALUE;
            }
        },

        YEAR: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: ['val:date'],
            resolve: function (date) {
                return date.getUTCFullYear();
            }
        }

    };

    // exports ================================================================

    return { OPERATORS: OPERATORS, FUNCTIONS: FUNCTIONS };

});
