/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/drawing/axismodel',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/config',
     'io.ox/office/drawinglayer/view/chartstyleutil',
     'io.ox/office/editframework/model/attributedmodel',
     'io.ox/office/spreadsheet/model/drawing/gridlinemodel',
     'io.ox/office/spreadsheet/model/drawing/axistitlemodel'
    ], function (Utils, Config, ChartStyleUtil, AttributedModel, GridlineModel, AxisTitleModel) {

    'use strict';

    // class AxisModel ========================================================

    var PERCENTFORMAT = {
        exp: -2,
        format: '0%'
    };

    ///////////////////////////////////////////////////////////////////////////

    /**
     * visualization model for chart axes,
     * it defines colors and textual colors
     */
    function AxisModel(chart, type, otherType) {
        var self = this;
        var title = null;
        var zeroLine = null;
        var grid = null;
        var format = null;

        var dataAxis = chart.getModelData()['axis' + type.toUpperCase()];
        if(dataAxis){
            grid = new GridlineModel(chart, {}, dataAxis);
            title = new AxisTitleModel(chart);

            var otherAxis = chart.getModelData()['axis' + otherType.toUpperCase()];
            if(otherAxis.stripLines){
                zeroLine = otherAxis.stripLines[0];
                if(zeroLine && !zeroLine.zero){
                    zeroLine = null;
                }
            }
        }


        // base constructor ---------------------------------------------------

        AttributedModel.call(this, chart.getApp(), {}, { additionalFamilies: ['axis', 'line', 'character'] });

        // methods ------------------------------------------------------------

        this.refreshInfo = function () {

            refreshMinMax();

            var app = chart.getApp();
            if (!dataAxis) {
                return;
            }
            var view = app.getView();
            if (!view) {
                return;
            }
            var attrs = this.getMergedAttributes();

            ChartStyleUtil.handleLineProps(chart, attrs.line, dataAxis, 'line');
            ChartStyleUtil.handleLineProps(chart, attrs.line, dataAxis, 'tick');

            if (attrs.axis.label) {
                ChartStyleUtil.handleCharacterProps(chart, attrs.character, dataAxis, 'label');
            } else {
                dataAxis.labelFontColor = 'transparent';
                dataAxis.labelFontSize = 1;
            }

            if (zeroLine) {
                zeroLine.color = dataAxis.lineColor;
                zeroLine.thickness = dataAxis.lineThickness;
            }

            var overwriteLabels = false;
            var stripLines = dataAxis.stripLines;
            _.each(stripLines, function(stripLine) {
                if(!stripLine.zero){
                    if (attrs.axis.label) {
                        ChartStyleUtil.handleCharacterProps(chart, attrs.character, stripLine, 'label');
                    }else{
                        stripLine.labelFontColor = 'transparent';
                        stripLine.labelFontSize = 1;
                    }
                    overwriteLabels = true;
                }
            });
            if(overwriteLabels){
                dataAxis.labelFontColor = 'transparent';
            }

            grid.refreshInfo();

            var titleAtt = title.getMergedAttributes();
            dataAxis.title = titleAtt.text.link[0];
            if (dataAxis.title) {
                ChartStyleUtil.handleCharacterProps(chart, titleAtt.character, dataAxis, 'title');
            }

        };

        this.setFormat = function(newFormat) {
            format = newFormat;
        };

        this.getGrid = function () {
            return grid;
        };

        this.getTitle = function () {
            return title;
        };

        this.registerDestructor(function () {
            if (grid) {
                grid.destroy();
            }
            if (title) {
                title.destroy();
            }
            grid = title = null;
        });


        function refreshMinMax() {
            var data = chart.getModelData();
            if (!data.data.length) {
                return;
            }
            if(type === 'x'){
                refreshAxisX(data);
            }else{
                refreshAxisY(data);
            }
        }

        function makeMin(min, max) {
            if ((!format || !format.date) && (Math.abs(max - min) < 0.1 || (min / max < 5 / 6))) {
                return 0;
            } else {
                return min - ((max - min) / 2);
            }
        }

        function makeMax(min, max) {
            var res = max + 0.2 * (max - min);
            if (res < max) {
                res = max;
            }
            return res;
        }

        function setMinMax(min, max, interval, interlaced) {
            dataAxis.minimum = min;
            dataAxis.maximum = max;
            dataAxis.interval = interval;

            var stripLines = dataAxis.stripLines;

            if(!interval){
                if(stripLines.length>1){
                    for (var i = stripLines.length-1; i >= 0 ; i--) {
                        var stripLine = stripLines[i];
                        if(!stripLine.zero){
                            stripLines.splice(i, 1);
                        }
                    }
                }
                return;
            }

            var index = 0;
            var formatter = chart.getApp().getModel().getNumberFormatter();
            var steps = (dataAxis.maximum - dataAxis.minimum) / dataAxis.interval;
            var lastLine = null;

            for (var i = 0; i <= steps; i ++) {
                var value = dataAxis.minimum + i * dataAxis.interval;

                var line = stripLines[index];
                if(line && line.zero){
                    index++;
                    line = stripLines[index];
                }
                if(!line){
                    line = {
                        value: 0,
                        color: 'transparent',
                        label : '',
                        labelBackgroundColor: 'transparent'
                    };
                    stripLines[index] = line;
                }
                line.labelFontColor = 'transparent';
                line.label = '';
                line.value = value;

                if(!interlaced || index%2){
                    var realValue = value * Math.pow(10, format.exp);
                    if(format && format.format){
                        line.label = formatter.formatValue(realValue, format.format);
                    }else{
                        line.label = formatter.formatStandardNumber(realValue, 8).text;
                    }
                    if(lastLine && lastLine.label === line.label){
                        line.label = '';
                    }else{
                        lastLine = line;
                    }
                }
                index++;
            }
            for (; index < stripLines.length; index++) {
                var line = stripLines[index];
                line.labelFontColor = 'transparent';
                line.label = '';
            }
        }

        function refreshAxisX(data) {
            var area = chart.getMergedAttributes().chart.type.indexOf('area') === 0;
            if (area && (!format || !format.format)) {
                area = false;
            }
            if (!data.linearX && (chart.isXYType() || area)) {
                var minValue = Number.POSITIVE_INFINITY;
                var maxValue = Number.NEGATIVE_INFINITY;
                var intervalCount = Math.max(data.data.length, 8);

                _.each(data.data, function (dd) {
                    _.each(dd.dataPoints, function (dataPoint) {
                        var value = dataPoint[type];
                        minValue = Math.min(minValue, value);
                        maxValue = Math.max(maxValue, value);
                    });
                });

//                if (chart.isXYType()) {
//                    var diff = maxValue - minValue;
//                    maxValue += 1 + diff / intervalCount;
//                    minValue -= 1 + diff / intervalCount;
//                } else {
//                    minValue -= 0.25;
//                    maxValue += 0.25;
//                }

                if(!isFinite(minValue) || !isFinite(maxValue)){
                    setMinMax();
                    return;
                }
                if (minValue === maxValue) {
                    maxValue = minValue + 0.5;
                    minValue -= 0.5;
                }

                var interval = (maxValue - minValue) / intervalCount;
                setMinMax(minValue, maxValue, interval, true);
            } else {
                setMinMax(null, null, null);
            }
        }

        function round(value, precision) {
            if(precision>0&&precision<20){
                return parseFloat(value.toPrecision(precision));
            }else{
                return value;
            }
        }

        function refreshAxisY(data) {
            var chartAttrs = chart.getMergedAttributes().chart;
            var stacking = chartAttrs.stacking;

            var givenMin = null;
            var givenMax = null;

            var axisInfo = self.getMergedAttributes();
            if (axisInfo) {
                if (axisInfo.min !== 'auto') {
                    givenMin = axisInfo.min;
                }
                if (axisInfo.max !== 'auto') {
                    givenMax = axisInfo.max;
                }
            }

            var minValue = Number.POSITIVE_INFINITY;
            var maxValue = Number.NEGATIVE_INFINITY;

            if (stacking === 'percentStacked' || (stacking === 'stacked' && data.data.length > 1)) {
                var first = data.data[0];
                _.times(first.dataPoints.length, function (j) {
                    var value = 0;
                    _.each(data.data, function (dd) {
                        var useValue = dd.dataPoints[j][type];
                        value += useValue;
                    });
                    minValue = Math.min(minValue, value);
                    maxValue = Math.max(maxValue, value);
                });
            } else {
                _.each(data.data, function (dd) {
                    _.each(dd.dataPoints, function (dataPoint) {
                        var value = dataPoint[type];
                        minValue = Math.min(minValue, value);
                        maxValue = Math.max(maxValue, value);
                    });
                });
            }

            var intervalCount = 8;

            if (chart.isXYType()) {
                var diff = maxValue - minValue;
                maxValue += 0.005 + diff / intervalCount;
                minValue -= 0.005 + diff / intervalCount;
            } else if (stacking === 'percentStacked') {
                format = PERCENTFORMAT;
                //Trial & error...
                if (maxValue >= 0 && minValue >= 0) {
                    //Szenario 1
                    minValue = givenMin || 0;
                    maxValue = givenMax || 110;
                } else if (maxValue <= 0 && minValue <= 0) {
                    //Szenario 2
                    maxValue = givenMax || 0;
                    minValue = givenMin || -110;
                } else {
                    //Szenario 3
                    maxValue =  givenMax || 110;
                    minValue =  givenMin || -110;
                }
            } else {
                if (maxValue >= 0 && minValue >= 0) {
                    //Szenario 1
                    minValue = givenMin || makeMin(minValue, maxValue);
                    maxValue = givenMax || makeMax(minValue, maxValue);
                } else if (maxValue <= 0 && minValue <= 0) {
                    //Szenario 2
                    maxValue = givenMax || -makeMin(-maxValue, -minValue);
                    minValue = givenMin || -makeMax(-maxValue, -minValue);
                } else {
                    //Szenario 3
                    var ma = maxValue;
                    var mi = minValue;
                    maxValue =  givenMax || makeMax(mi, ma);
                    minValue =  givenMin || -makeMax(-ma, -mi);
                }
            }

            if(!isFinite(minValue) || !isFinite(maxValue)){
                setMinMax();
                return;
            }

            if (minValue === maxValue) {
                maxValue = minValue + 0.5;
                minValue -= 0.5;
            }


            var precision = 2;
            if(format.date){
                precision = 0;
            }

            minValue = round(minValue, precision);
            maxValue = round(maxValue, precision);


            if (minValue < maxValue) {
                var interval = (maxValue - minValue) / intervalCount;
				interval = round(interval, precision - 1);

                if (interval === 0) {
                    interval = null;
                    dataAxis.minimum = minValue;
                    dataAxis.maximum = maxValue;
                    dataAxis.interval = null;
                } else {
                    var intCount = parseFloat((maxValue - minValue) / interval);

                    if (Math.abs(maxValue) < Math.abs(minValue)) {
                        maxValue = Utils.roundUp(maxValue, interval);
                        minValue = maxValue - (interval * intCount);
                    } else {
                        minValue = Utils.roundDown(minValue, interval);
                        maxValue = minValue + (interval * intCount);
                    }
                }
                setMinMax(minValue, maxValue, interval);
            }
        }

    } // class AxisModel

    // exports ================================================================
    return AttributedModel.extend({ constructor: AxisModel });

});
