/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/app/application',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/io',
     'io.ox/office/tk/errorcode',
     'io.ox/office/editframework/app/editapplication',
     'io.ox/office/spreadsheet/utils/config',
     'io.ox/office/spreadsheet/utils/sheetutils',
     'io.ox/office/spreadsheet/model/model',
     'io.ox/office/spreadsheet/view/view',
     'io.ox/office/spreadsheet/app/controller',
     'gettext!io.ox/office/spreadsheet'
    ], function (Utils, IO, ErrorCode, EditApplication, Config, SheetUtils, SpreadsheetModel, SpreadsheetView, SpreadsheetController, gt) {

    'use strict';

    var // default settings for locale data
        DEFAULT_LOCALE_DATA = {
            functions: {},
            errorCodes: _.chain(SheetUtils.ErrorCodes).map(function (code) { return [code.code, code.code]; }).object().value(),
            decSeparator: '.',
            groupSeparator: ',',
            listSeparator: ',',
            trueLiteral: 'TRUE',
            falseLiteral: 'FALSE'
        },

        // RE pattern for a valid error code literal
        RE_ERROR_CODE = /^#[^ !?]+[!?]?$/,

        // global cache for locale data, mapped by locale identifier
        localeDataCache = {};

    // private static functions ===============================================

    /**
     * Inserts additional information into the passed locale data object.
     */
    function postProcessLocaleData(localeData) {

        // build inverted function name map (translated to native), and arrays of function names
        localeData.invFunctions = _.invert(localeData.functions);
        localeData.nativeFunctionNames = _.chain(localeData.functions).keys().sort().value();
        localeData.localizedFunctionNames = _.chain(localeData.invFunctions).keys().sort().value();

        // build inverted error code map (translated to native), and arrays of error codes
        localeData.invErrorCodes = _.invert(localeData.errorCodes);
        localeData.nativeErrorCodes = _.chain(localeData.errorCodes).keys().sort().value();
        localeData.localizedErrorCodes = _.chain(localeData.invErrorCodes).keys().sort().value();

        // print error messages to browser console
        if (Config.DEBUG) {

            // check for duplicates in translated function names
            if (localeData.nativeFunctionNames.length !== localeData.localizedFunctionNames.length) {
                Utils.error('SpreadsheetApplication.postProcessLocaleData(): list of translated function names contains duplicates');
            }

            // check for duplicates in translated error code names
            if (localeData.nativeErrorCodes.length !== localeData.localizedErrorCodes.length) {
                Utils.error('SpreadsheetApplication.postProcessLocaleData(): list of translated error codes contains duplicates');
            }

            // check native error codes
            _.each(localeData.nativeErrorCodes, function (errorCode) {
                if (!RE_ERROR_CODE.test(errorCode)) {
                    Utils.error('SpreadsheetApplication.postProcessLocaleData(): invalid native error code: "' + errorCode + '"');
                }
            });

            // check translated error codes
            _.each(localeData.localizedErrorCodes, function (errorCode) {
                if (!RE_ERROR_CODE.test(errorCode)) {
                    Utils.error('SpreadsheetApplication.postProcessLocaleData(): invalid translated error code: "' + errorCode + '"');
                }
            });
        }
    }

    // class SpreadsheetApplication ===========================================

    /**
     * The OX Spreadsheet application.
     *
     * Triggers the following additional events:
     * - 'docs:update:cells'
     *      After the application has received a 'docs:update' event which
     *      contains the contents and/or addresses of changed cells in the
     *      document. Event handlers receive the following parameter:
     *      (1) {Object} changedData
     *          A map of change descriptors for each changed sheet, mapped by
     *          the zero-based sheet index. Each change descriptor object
     *          contains the following properties:
     *          - {Number} sheet
     *              The zero-based sheet index. Equal to the map key of this
     *              object, but as number for convenience.
     *          - {Array} [rangeContents]
     *              An array with cell range descriptors. Each descriptor
     *              object contains the same properties as also sent in the
     *              response data of view update requests. If omitted, no cell
     *              content data is available for the sheet. If omitted, no
     *              cells have been changed in the sheet, or no explicit cell
     *              content data is available yet (see property 'dirtyRanges').
     *          - {Array} [dirtyRanges]
     *              An array with range addresses of additional cells that are
     *              not included in the 'rangeContents' property, but whose
     *              contents have been changed in the sheet. These ranges need
     *              to be updated with further server requests. If omitted, the
     *              sheet does not contain additional dirty cell ranges.
     *          - {Array} [changedRanges]
     *              An array containing the unified range addresses from the
     *              properties 'rangeContents' and 'dirtyRanges'. Will exist,
     *              if either of these properties exists too.
     *          - {Number} [usedCols]
     *              The number of used columns in the sheet. If omitted, the
     *              number of used columns has not changed.
     *          - {Number} [usedRows]
     *              The number of used rows in the sheet. If omitted, the
     *              number of used rows has not changed.
     *      (2) {Array} allChangedRanges
     *          An array containing the range addresses of all changed ranges
     *          in all sheets. This is the concatenation of all cell ranges
     *          contained in the 'changedRanges' properties for all sheets in
     *          the 'changedData' event parameter. Each range object in this
     *          array contains the additional property 'sheet' with the
     *          zero-based sheet index of the range.
     *      (3) {String} [errorCode]
     *          A specific error code associated to the changed cells. The
     *          following error codes are supported:
     *          - 'circular': The changed cells contain at least one formula
     *              cell with a circular reference.
     *
     * @constructor
     *
     * @extends EditApplication
     *
     * @param {Object} launchOptions
     *  All options passed to the core launcher (the ox.launch() method).
     *
     * @param {Object} [appOptions]
     *  Static application options that have been passed to the static method
     *  BaseApplication.createLauncher().
     */
    var SpreadsheetApplication = EditApplication.extend({ constructor: function (launchOptions, appOptions) {

        var // self reference
            self = this,

            // the spreadsheet document model and view
            model = null,
            view = null,

            // the locale data (translated function names etc.)
            localeData = _.copy(DEFAULT_LOCALE_DATA, true),

            // the Deferred object that will be resolved when locale data is available
            localeDef = $.Deferred(),

            // the function help resource
            functionHelp = {};

        // base constructor ---------------------------------------------------

        EditApplication.call(this, SpreadsheetModel, SpreadsheetView, SpreadsheetController, launchOptions, appOptions, {
            newDocumentParams: { initial_sheetname: SheetUtils.generateSheetName(0) },
            preProcessHandler: preProcessDocument,
            postProcessHandler: postProcessDocument,
            previewHandler: previewHandler,
            importFailedHandler: importFailedHandler,
            prepareFlushHandler: prepareFlushDocument,
            prepareLoseEditRightsHandler: prepareLoseEditRights,
            realTimeDelay: 10
        });

        // private methods ----------------------------------------------------

        /**
         * Preparation of the document, before its operations will be applied.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  document has been prepared successfully; or rejected on any error.
         */
        function preProcessDocument() {

            var // load function help data asynchronously
                functionHelpPromise = IO.loadResource('io.ox/office/spreadsheet/resource/functions'),
                // the server request for the locale data
                localeRequest = null;

            // store function help data, but ignore the success/failure result
            // (application may run even without function help)
            functionHelpPromise.done(function (data) { functionHelp = data; });

            // try to receive cached locale data already downloaded before
            if (Config.LOCALE in localeDataCache) {
                localeData = localeDataCache[Config.LOCALE];
                localeDef.resolve();
                return $.when();
            }

            // request locale data from server
            localeRequest = self.sendFilterRequest({
                method: 'POST',
                params: {
                    action: 'query',
                    requestdata: JSON.stringify({
                        type: 'localeData',
                        locale: Config.LOCALE
                    })
                }
            });

            // process the response data
            localeRequest = localeRequest.then(function (result) {

                Utils.log('SpreadsheetApplication.preProcessDocument(): locale data response:', result);
                if (!_.isObject(result)) {
                    Utils.error('SpreadsheetApplication.preProcessDocument(): missing locale data');
                    return $.Deferred().reject();
                }

                // print console warnings for missing properties
                if (Config.DEBUG) {
                    if (!result.decSeparator) { Utils.error('SpreadsheetApplication.preProcessDocument(): missing decimal separator'); }
                    if (!result.groupSeparator) { Utils.error('SpreadsheetApplication.preProcessDocument(): missing group separator'); }
                    if (!result.listSeparator) { Utils.error('SpreadsheetApplication.preProcessDocument(): missing list separator'); }
                    if (!result.trueLiteral) { Utils.error('SpreadsheetApplication.preProcessDocument(): missing literal for TRUE'); }
                    if (!result.falseLiteral) { Utils.error('SpreadsheetApplication.preProcessDocument(): missing literal for FALSE'); }
                }

                // the map of all function names
                localeData = { functions: Utils.getObjectOption(result, 'functions', {}) };

                // error code literals
                localeData.errorCodes = _.extend({}, DEFAULT_LOCALE_DATA.errorCodes, Utils.getObjectOption(result, 'errorCodes'));

                // all separator characters
                _.each(['decSeparator', 'groupSeparator', 'listSeparator'], function (propertyName) {
                    localeData[propertyName] = Utils.getStringOption(result, propertyName, DEFAULT_LOCALE_DATA[propertyName], true)[0];
                });

                // Boolean literals
                localeData.trueLiteral = Utils.getStringOption(result, 'trueLiteral', DEFAULT_LOCALE_DATA.trueLiteral, true).toUpperCase();
                localeData.falseLiteral = Utils.getStringOption(result, 'falseLiteral', DEFAULT_LOCALE_DATA.falseLiteral, true).toUpperCase();

                localeData.engineVersion = Utils.getStringOption(result, 'engineVersion', '');
                postProcessLocaleData(localeData);
                Utils.info('SpreadsheetApplication.preProcessDocument(): new locale data:', localeData);

                // check that the translated function names exist in the function help
                functionHelpPromise.done(function (data) {
                    var supportedFuncs = _.values(localeData.functions),
                        annotatedFuncs = _.keys(data),
                        missingFuncs = _.difference(supportedFuncs, annotatedFuncs),
                        unknownFuncs = _.difference(annotatedFuncs, supportedFuncs);
                    if (missingFuncs.length > 0) {
                        missingFuncs.sort();
                        Utils.warn('SpreadsheetApplication.preProcessDocument(): missing help texts for functions: ' + missingFuncs.join(', '));
                    }
                    if (unknownFuncs.length > 0) {
                        unknownFuncs.sort();
                        Utils.info('SpreadsheetApplication.preProcessDocument(): help available for unknown functions: ' + unknownFuncs.join(', '));
                    }
                });

                // cache complete locale data object in the global cache
                localeDataCache[Config.LOCALE] = localeData;

            }, function () {
                Utils.error('SpreadsheetApplication.preProcessDocument(): could not download locale data for "' + Config.LOCALE + '"');
                return { headline: gt('Server Error'), message: gt('The engine used for calculating formulas is not available.') };
            });

            // resolve the internal Deferred object used for the onInitLocale() method
            return localeRequest.always(function () { localeDef.resolve(); });
        }

        /**
         * Post-processing of the document, after all its import operations
         * have been applied successfully.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  document has been post-processed successfully; or rejected when the
         *  document is invalid, or an error has occurred.
         */
        function postProcessDocument() {
            // document must contain at least one sheet (returning a rejected Deferred
            // object will cause invocation of the prepareInvalidDocument() method)
            return (model.getSheetCount() === 0) ? $.Deferred().reject() : model.postProcessDocument();
        }

        /**
         * Shows an early preview of the document. Activates the sheet that is
         * specified in the passed preview data, and starts querying cell
         * contents for the active sheet.
         *
         * @param {Object} previewData
         *  The preview data containing the index of the active sheet.
         *
         * @returns {Boolean}
         *  Whether a sheet has been activated successfully.
         */
        function previewHandler(previewData) {
            var activeSheet = Utils.getIntegerOption(previewData, 'activeSheet', 0);
            return self.getView().activatePreviewSheet(activeSheet);
        }

        /**
         * Handler will be called by base class if importing the document
         * failed. The functions handles Spreadsheet specific errors.
         */
        function importFailedHandler(response) {

            var // specific error code sent by the server
                error = new ErrorCode(response);

            switch (error.getCodeAsConstant()) {
            case 'LOADDOCUMENT_COMPLEXITY_TOO_HIGH_ERROR':
                response.message = gt('This document exceeds the spreadsheet size and complexity limits.');
                break;
            case 'LOADDOCUMENT_CALCENGINE_NOT_WORKING_ERROR':
                response.headline = gt('Server Error');
                response.message = gt('The engine used for calculating formulas is not available.');
                break;
            }

            // document must contain at least one sheet, to prevent if-else
            // statements for empty documents at thousand places in the view code
            if (!self.isInQuit()) {
                model.prepareInvalidDocument();
            }
        }

        /**
         * Preprocessing before the document will be flushed for downloading,
         * printing, sending as mail, or closing.
         */
        function prepareFlushDocument() {
            // commit current text of the cell in-place edit mode (without validation)
            view.leaveCellEditMode('auto');
            // send all changed view settings before flushing editable document
            view.sendChangedViewAttributes();
        }

        /**
         * Preparations before the edit mode will be switched off.
         */
        function prepareLoseEditRights() {
            // commit current text of the cell in-place edit mode (without validation)
            view.leaveCellEditMode('auto');
        }

        /**
         * Handles 'docs:update' notifications. If the message data contains
         * information about changed cells in the document, this method
         * triggers a 'docs:update:cells' event with preprocessed user data.
         *
         * @param {Object} data
         *  The message data of the 'docs:update' application event.
         */
        function updateNotificationHandler(data) {

            var // complete changed data
                changedData = Utils.getObjectOption(data, 'changed'),
                // the locale used to format the cell contents
                locale = Utils.getStringOption(data, 'locale'),
                // a special error code associated to the changed cells
                errorCode = Utils.getStringOption(changedData, 'error'),
                // all changed cell information for the own 'docs:update:cells' event
                eventData = null,
                // all changed range addresses, with sheet index
                allChangedRanges = null;

            // update message may not contain any data about changed cells
            if (!_.isObject(changedData)) { return; }
            Utils.info('SpreadsheetApplication.updateNotificationHandler(): notification received:', changedData);

            // extract the map of changed data per sheet
            changedData = Utils.getObjectOption(changedData, 'sheets');
            if (!changedData) {
                Utils.warn('SpreadsheetApplication.updateNotificationHandler(): no sheets defined');
                return;
            }

            // process all changed sheets
            eventData = {};
            allChangedRanges = [];
            _.each(changedData, function (changedSheetData, sheetKey) {

                var sheet = parseInt(sheetKey, 10),
                    rangeContents = Utils.getArrayOption(changedSheetData, 'contents', []),
                    dirtyRanges = Utils.getArrayOption(changedSheetData, 'cells', []),
                    colIntervals = Utils.getArrayOption(changedSheetData, 'columns'),
                    rowIntervals = Utils.getArrayOption(changedSheetData, 'rows'),
                    usedCols = Utils.getIntegerOption(changedSheetData, 'usedCols'),
                    usedRows = Utils.getIntegerOption(changedSheetData, 'usedRows'),
                    eventSheetData = {},
                    changedRanges = null;

                if (!isFinite(sheet) || (sheet < 0) || (sheet >= model.getSheetCount())) {
                    Utils.warn('SpreadsheetApplication.updateNotificationHandler(): invalid sheet index: "' + sheetKey + '"');
                    return;
                }

                // insert cell contents
                if (_.isArray(rangeContents) && (rangeContents.length > 0)) {
                    _.each(rangeContents, function (rangeData) {
                        if (!rangeData.end) { rangeData.end = _.clone(rangeData.start); }
                    });
                    // bug 32624: use content data only if formatted with the own locale; otherwise store as 'dirty ranges'
                    if (locale === Config.LOCALE) {
                        eventSheetData.rangeContents = rangeContents;
                    } else {
                        dirtyRanges = dirtyRanges.concat(rangeContents);
                    }
                }

                // insert addresses of dirty ranges
                _.each(dirtyRanges, function (range) {
                    if (!range.end) { range.end = _.clone(range.start); }
                });
                _.each(colIntervals, function (interval) {
                    dirtyRanges.push(model.makeColRange(('last' in interval) ? interval : interval.first));
                });
                _.each(rowIntervals, function (interval) {
                    dirtyRanges.push(model.makeRowRange(('last' in interval) ? interval : interval.first));
                });
                if (dirtyRanges.length > 0) {
                    eventSheetData.dirtyRanges = SheetUtils.getUnifiedRanges(dirtyRanges);
                }

                // insert number of used columns/rows
                if (_.isNumber(usedCols)) { eventSheetData.usedCols = usedCols; }
                if (_.isNumber(usedRows)) { eventSheetData.usedRows = usedRows; }

                // insert sheet entry into the own event data
                if (!_.isEmpty(eventSheetData)) {
                    eventSheetData.sheet = sheet;
                    eventData[sheetKey] = eventSheetData;
                }

                // extend array of all changed ranges
                if (eventSheetData.rangeContents && eventSheetData.dirtyRanges) {
                    changedRanges = SheetUtils.getUnifiedRanges(eventSheetData.rangeContents.concat(eventSheetData.dirtyRanges));
                } else if (eventSheetData.rangeContents) {
                    changedRanges = SheetUtils.getUnifiedRanges(eventSheetData.rangeContents);
                } else if (eventSheetData.dirtyRanges) {
                    changedRanges = _.copy(dirtyRanges, true);
                }
                _.each(changedRanges, function (range) { range.sheet = sheet; });
                eventSheetData.changedRanges = changedRanges;
                allChangedRanges = allChangedRanges.concat(changedRanges);
            });

            // trigger the event, if the event data object contains any useful information
            if (!_.isEmpty(eventData)) {
                self.trigger('docs:update:cells', eventData, allChangedRanges, errorCode);
            }
        }

        // methods ------------------------------------------------------------

        /**
         * Registers a callback function that will be invoked when the locale
         * settings (number format settings, translated function names, etc.)
         * for this document have been initialized. In the early initialization
         * phase of the document, the callback function will be invoked
         * deferred, when the server request for the locale data has returned.
         * Later, the callback function will be invoked immediately when
         * calling this method.
         *
         * @param {Function} callback
         *  The callback function invoked after the locale data has been
         *  initialized.
         *
         * @param {Object} [context]
         *  The context bound to the invoked callback function.
         *
         * @returns {SpreadsheetApplication}
         *  A reference to this instance.
         */
        this.onInitLocale = function (callback, context) {
            localeDef.done(_.bind(callback, context));
            return this;
        };

        /**
         * Returns the sorted list with the names of all supported functions
         * used in formulas, translated for the current UI language.
         *
         * @returns {Array}
         *  The sorted list with the names of all supported functions.
         */
        this.getLocalizedFunctionNames = function () {
            return localeData.localizedFunctionNames;
        };

        /**
         * Returns the name of the specified function, translated for the
         * current UI language.
         *
         * @param {String} nativeFuncName
         *  The English name of the function.
         *
         * @returns {String}
         *  The translated name of the specified function if available,
         *  otherwise the passed function name.
         */
        this.getLocalizedFunctionName = function (nativeFuncName) {
            nativeFuncName = nativeFuncName.toUpperCase();
            return (nativeFuncName in localeData.functions) ? localeData.functions[nativeFuncName] : nativeFuncName;
        };

        /**
         * Returns the sorted list with the native (English) names of all
         * supported functions used in formulas.
         *
         * @returns {Array}
         *  The sorted list with the native names of all supported functions.
         */
        this.getNativeFunctionNames = function () {
            return localeData.nativeFunctionNames;
        };

        /**
         * Returns the native (English) name of the specified function name in
         * the current UI language.
         *
         * @param {String} localFuncName
         *  The translated name of the function.
         *
         * @returns {String}
         *  The native name of the specified function if available, otherwise
         *  the passed function name.
         */
        this.getNativeFunctionName = function (localFuncName) {
            localFuncName = localFuncName.toUpperCase();
            return (localFuncName in localeData.invFunctions) ? localeData.invFunctions[localFuncName] : localFuncName;
        };

        /**
         * Returns a help text descriptor for the specified translated function
         * name.
         *
         * @param {String} localFuncName
         *  The translated name of the function.
         *
         * @returns {Object|Null}
         *  A help text descriptor if available, with the following properties:
         *  - {String} description
         *      A text describing the entire formula.
         *  - {String[]} params
         *      An array with the names of all function parameters. Functions
         *      with trailing repeating parameters will be mentioned only once
         *      in this array.
         *  - {String[]} paramshelp
         *      An array describing each parameter more detailed. This array
         *      MUST have the same length as the 'params' array.
         *  - {Number} [varargs]
         *      If contained, the zero-based index of the first parameter that
         *      can be used repeatedly. MUST be less than the length of the
         *      'params' array. If this index does not point to the last array
         *      element, the function expects pairs or even larger tuples of
         *      repeating parameters.
         */
        this.getFunctionHelp = function (localFuncName) {
            localFuncName = localFuncName.toUpperCase();
            // bug 33438: do not show help for unsupported functions (e.g. due to current file format)
            return ((localFuncName in functionHelp) && (localFuncName in localeData.invFunctions)) ? functionHelp[localFuncName] : null;
        };

        /**
         * Returns the sorted list with the names of all supported error codes
         * used in formulas, translated for the current UI language.
         *
         * @returns {Array}
         *  The sorted list with the names of all supported error codes.
         */
        this.getLocalizedErrorCodes = function () {
            return localeData.localizedErrorCodes;
        };

        /**
         * Returns whether the passed string is a known error code literal in
         * the current UI language.
         *
         * @param {String} errorCode
         *  The string to be checked.
         *
         * @returns {Boolean}
         *  Whether the passed string is a known error code literal in the
         *  current UI language.
         */
        this.isLocalizedErrorCode = function (errorCode) {
            return errorCode.toUpperCase() in localeData.invErrorCodes;
        };

        /**
         * Returns the sorted list with the internal values of all supported
         * error codes used in formulas.
         *
         * @returns {Array}
         *  The sorted list with the internal values of all supported error
         *  codes.
         */
        this.getNativeErrorCodes = function () {
            return localeData.nativeErrorCodes;
        };

        /**
         * Returns whether the passed string is a known internal error code
         * literal.
         *
         * @param {String} errorCode
         *  The string to be checked.
         *
         * @returns {Boolean}
         *  Whether the passed string is a known internal error code literal.
         */
        this.isNativeErrorCode = function (errorCode) {
            return errorCode.toUpperCase() in localeData.errorCodes;
        };

        /**
         * Returns the decimal separator for the current UI language.
         *
         * @returns {String}
         *  The decimal separator for the current UI language.
         */
        this.getDecimalSeparator = function () {
            return localeData.decSeparator;
        };

        /**
         * Returns the group separator (usually the character that separates
         * groups of 3 digits in the integral part of a number, but different
         * rules may apply in specific locales) for the current UI language.
         *
         * @returns {String}
         *  The group separator for the current UI language.
         */
        this.getGroupSeparator = function () {
            return localeData.groupSeparator;
        };

        /**
         * Returns the list separator for the current UI language. The list
         * separator is used to separate parameters of functions, and for the
         * range list operator.
         *
         * @returns {String}
         *  The list separator for the current UI language.
         */
        this.getListSeparator = function () {
            return localeData.listSeparator;
        };

        /**
         * Returns the string literal of the specified Boolean value for the
         * current UI language.
         *
         * @param {Boolean} value
         *  The Boolean value.
         *
         * @returns {String}
         *  The string literal of the specified Boolean value.
         */
        this.getBooleanLiteral = function (value) {
            return localeData[value ? 'trueLiteral' : 'falseLiteral'];
        };

        /**
         * Converts the passed number to its string representation.
         *
         * @param {Number} number
         *  The floating-point number to be converted to a string.
         *
         * @param {Number} maxLength
         *  The maximum number of characters allowed for the absolute part of
         *  the passed number, including the decimal separator and the complete
         *  exponent in scientific notation.
         *
         * @param {String} [sep]
         *  The decimal separator. If omitted, uses the decimal separator of
         *  the current GUI language.
         *
         * @returns {String}
         *  The string representation of the passed floating-point number.
         */
        this.convertNumberToString = function (number, maxLength, sep) {
            var result = model.getNumberFormatter().formatStandardNumber(number, maxLength).text;
            if (_.isString(sep)) { result = result.replace(localeData.decSeparator, sep); }
            return result;
        };

        /**
         * Converts the passed text to a floating-point number.
         *
         * @param {String} text
         *  The string to be converted to a floating-point number.
         *
         * @param {String} [sep]
         *  The decimal separator expected in the string. If omitted, uses the
         *  decimal separator of the current GUI language.
         *
         * @returns {Number}
         *  The floating-point number represented by the passed string; or NaN,
         *  if the string cannot be parsed to a number.
         */
        this.convertStringToNumber = (function () {

            var getNumberRegExp = _.memoize(function (sep) {
                sep = _.escapeRegExp(sep);
                return new RegExp('^[-+]?(\\d+' + sep + '?|\\d*' + sep + '\\d+)(E[-+]?\\d+)?$', 'i');
            });

            return function (text, sep) {
                sep = sep || localeData.decSeparator;
                return getNumberRegExp(sep).test(text) ? parseFloat(text.replace(sep, '.')) : Number.NaN;
            };
        }());

        /**
         * Returns the name of the specified error code, translated for the
         * current UI language.
         *
         * @param {ErrorCode} errorCode
         *  The error code object.
         *
         * @returns {String}
         *  The translated name of the specified error code if available,
         *  otherwise the internal error code (the 'code' property) of the
         *  passed error code object.
         */
        this.convertErrorCodeToString = function (errorCode) {
            return (errorCode.code in localeData.errorCodes) ? localeData.errorCodes[errorCode.code] : errorCode.code;
        };

        /**
         * Returns the error code object representing the the passed translated
         * error code literal.
         *
         * @param {String} errorCode
         *  The translated name of the error code.
         *
         * @returns {ErrorCode}
         *  The error code object representing the specified translated error
         *  code.
         */
        this.convertStringToErrorCode = function (errorCode) {
            errorCode = errorCode.toUpperCase();
            return SheetUtils.makeErrorCode((errorCode in localeData.invErrorCodes) ? localeData.invErrorCodes[errorCode] : errorCode);
        };

        /**
         * Returns the version of the server-side Calc Engine.
         *
         * @returns {String}
         *  The version of the Calc Engine.
         */
        this.getEngineVersion = function () {
            return localeData.engineVersion;
        };

        // initialization -----------------------------------------------------

        // initialization of class members
        this.onInit(function () {
            model = self.getModel();
            view = self.getView();
            // initialize default locale data
            postProcessLocaleData(localeData);
        });

        // application notifies changed contents/results of cells create an own 'docs:update:cells' event
        this.on('docs:update', updateNotificationHandler);

        // destroy all class members
        this.on('docs:destroy', function () {
            launchOptions = appOptions = null;
            model = view = localeData = functionHelp = null;
        });

    }}); // class SpreadsheetApplication

    // static methods ---------------------------------------------------------

    /**
     * Replacement for the generic method EditApplication.createLauncher()
     * without parameters, to launch spreadsheet applications.
     */
    SpreadsheetApplication.createLauncher = function () {
        return EditApplication.createLauncher('io.ox/office/spreadsheet', SpreadsheetApplication, { icon: 'fa-table' });
    };

    // exports ================================================================

    return SpreadsheetApplication;

});
