/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/preview/view/toppane',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/forms',
     'io.ox/office/baseframework/view/toolpane',
     'io.ox/office/baseframework/view/toolbar',
     'io.ox/office/preview/view/controls',
     'gettext!io.ox/office/preview'
    ], function (Utils, Forms, ToolPane, ToolBar, Controls, gt) {

    'use strict';

    var // class name shortcuts
        Group = Controls.Group,
        Label = Controls.Label,
        Button = Controls.Button,
        CheckBox = Controls.CheckBox,
        TextField = Controls.TextField;

    // class TopPane ==========================================================

    /**
     * Represents the top-level pane, containing global
     * controls such as the Close, Edit button.
     *
     * The TopPane controls are constructed as if there is only one page available,
     * each other control will be enabled after updating the page count via
     * updatePageCount function.
     *
     * @constructor
     *
     * @extends ToolPane
     *
     * @param {EditApplication} app
     *  The application containing this tool bar.
     */
    function TopPane(app) {

        var pageCount = null,

            leftToolBar = new ToolBar(app),
            centerToolBar = new ToolBar(app),
            rightToolBar = new ToolBar(app),

            // sizeableControls is an array of following object
            // {control: singlePageHidden: on{from:, to: }, off{from:, to:}}
            sizeableControls = [],

            prio1long = { singlePageDisabled: true, on: {from: 1, to: 1},  off: {from: 2, to: 10}},
            prio1short= { singlePageHidden: true,   on: {from: 2, to: 10}, off: {from: 1, to: 1}},
            prio2long = {                           on: {from: 1, to: 2},  off: {from: 3, to: 10}},
            prio2short= {                           on: {from: 3, to: 10}, off: {from: 1, to: 2}},
            prio3long = { singlePageDisabled: true, on: {from: 1, to: 3},  off: {from: 4, to: 10}},
            prio3short= { singlePageHidden: true,   on: {from: 4, to: 10}, off: {from: 1, to: 3}},

            labelMenuShort1 = null,
            labelMenuShort2 = null,
            textFieldShort = null,
            labelMenuLong2 = null,
            textFieldLong = null,

            actionsMenuGroupShort = null,

            viewMenuGroup = null;

        // base constructor ---------------------------------------------------

        ToolPane.call(this, app, { position: 'top', classes: 'top-pane' });

        // methods ------------------------------------------------------------

        function paneLayoutHandler() {
            if(!pageCount) {
                return;
            }
            for(var prio=1; prio<5; prio++) {
                for(var i = 0; i < sizeableControls.length; i++) {
                    var options = sizeableControls[i];
                    if((pageCount===1)&&options.singlePageHidden) {
                        sizeableControls[i].control.hide();
                    }
                    else if((pageCount===1)&&typeof options.singlePageDisabled==='boolean') {
                        sizeableControls[i].control.enable(!options.singlePageDisabled);
                    }
                    else {
                        if(options.on) {
                            var from = options.on.from,
                                to = options.on.to;
                            if(from&&prio>=from) {
                                if(!to||to>=prio) {
                                    options.control.show();
                                }
                            }
                        }
                        if(options.off) {
                            var from = options.off.from,
                                to = options.off.to;
                            if(from&&prio>=from) {
                                if(!to||to>=prio) {
                                    options.control.hide();
                                }
                            }
                        }
                    }
                }
                // check if the size now is sufficient
                var paneWidth = this.getNode().width(),
                    left = leftToolBar.getNode().outerWidth(),
                    center = centerToolBar.getNode().outerWidth(),
                    right = rightToolBar.getNode().outerWidth();
                if(paneWidth>(left+right+center)) {
                    return;
                }
            }
        }

        function createSidePaneControls(group) {
            if(!Modernizr.touch) {
                addGroup(group, null, createSeparatorGroup(), { inline: false }, { singlePageHidden: true });
                addGroup(group, null, createHeaderLabelGroup(gt.pgettext('menu-title', 'Options')), { inline: false }, { singlePageHidden: true });
                addGroup(group, 'view/sidepane', new CheckBox({ label: _.noI18n('Show side panel')}), null, { singlePageHidden: true });
            }
            return group;
        }

        /*
         *  @param {object} destination
         *  the destination object where the control is added
         *
         *  @param {string} controller
         *  the controller that is used for the control
         *
         *  @param {object} control
         *  the source control that has to be added at the destination
         *
         *  @param {object} control options
         *  specifies the options that are to be used when inserting the control into the destination
         *
         *  @param {object} sizingOption
         *  an object containing the sizing options, this object has following structure:
         *  {singlePageHidden:... on{from:..., to:... }, off{from:..., to:...}}
         *      ('singlePageHidden', 'on' and 'off' are optional)
         *
         */
        function addGroup(destination, controller, control, controlOptions, sizingOptions) {
            if(sizingOptions&&sizingOptions.singlePageHidden) {
                control.hide();
            }
            destination.addGroup(controller, control, controlOptions);
            if(sizingOptions) {
                var v = {control: control};
                for(var k in sizingOptions) {
                    v[k]=sizingOptions[k];
                }
                sizeableControls.push(v);
            }
        }

        // creating a header label by exchanging class 'label' to 'header-label'
        function createHeaderLabelGroup(label) {
            var labelGroup = new Label({ label: label, classes: 'header-label'});
            labelGroup.getLabelNode().attr('class', 'header-label');
            return labelGroup;
        }

        function createSeparatorGroup() {
            return new Group({classes: 'separator-line'});
        }

        this.updatePageCount = function(_pageCount) {

            pageCount = _pageCount;
            // the label for the 'Go to page' menu entry
            // the leading and trailing part of the 'Go to page' label
            var goToLabelParts = _.noI18n.fix(
                    //#. an interactive menu entry for the current page number in a document
                    //#. "##" is a placeholder for the current page number (can be modified by the user) - do not change
                    //#. %1$d is the total count of pages in the document
                    //#, c-format
                    gt('Go to page ## of %1$d', _.noI18n(pageCount)).split('##').map(Utils.trimString));

            if(goToLabelParts) {
                if(goToLabelParts[0]) {
                    labelMenuShort1.setLabel(goToLabelParts[0]);
                }
                if(goToLabelParts[1]) {
                    labelMenuShort2.setLabel(goToLabelParts[1]);
                    labelMenuLong2.setLabel(goToLabelParts[1]);
                }
            }
            var validator = new TextField.NumberValidator({ min: 1, max: pageCount });
            textFieldShort.setValidator(validator);
            textFieldLong.setValidator(validator);
            if(pageCount>1) {
                for(var i = 0; i < sizeableControls.length; i++) {
                    if(sizeableControls[i].singlePageHidden) {
                        sizeableControls[i].control.show();
                    }
                }
            }
        };

        // initialization -----------------------------------------------------
        //
        // Left ToolBar
        //
        // create the 'actions' drop-down menu
        actionsMenuGroupShort = new Controls.ComponentMenuButton(app, { label: Controls.ACTIONS_LABEL, anchorPadding: -3, classes: 'link-style' })
            .addGroup('document/download', new Button({ icon: 'fa-cloud-download', label: gt('Download')}))
            .addGroup('document/print',    new Button({ icon: 'fa-print', label: gt('Download as PDF')}))
            .addGroup('document/sendmail', new Button({ icon: 'fa-envelope-o', label: gt('Send as mail')}))
            .addGroup('document/edit', new Controls.EditDocumentButton({label: gt('Edit document')}));

        addGroup(leftToolBar, null, actionsMenuGroupShort, null, prio2short);
        addGroup(leftToolBar, 'document/download', new Button({ icon: 'fa-cloud-download', tooltip: gt('Download'), classes: 'link-style' }), null, prio2long);
        addGroup(leftToolBar, 'document/print', new Button({ icon: 'fa-print', tooltip: gt('Download as PDF'), classes: 'link-style' }), null, prio2long);
        addGroup(leftToolBar, 'document/sendmail', new Button({ icon: 'fa-envelope-o', tooltip: gt('Send as mail'), classes: 'link-style' }), null, prio2long);
        addGroup(leftToolBar, 'document/edit', new Controls.EditDocumentButton({ classes: 'link-style' }), null, prio2long);

        this.addViewComponent(leftToolBar, { targetArea: 'leading' });

        //
        // center ToolBar
        //
        addGroup(centerToolBar, 'pages/first', new Button({ icon: 'fa-fast-backward', tooltip: gt('Show first page'), classes: 'link-style' }), null, prio3long);
        addGroup(centerToolBar, 'pages/previous', new Button({ icon: 'fa-step-backward', tooltip: gt('Show previous page'), classes: 'link-style' }), null, prio3long);
        addGroup(centerToolBar, 'pages/current', textFieldLong = new TextField({ tooltip: gt('Page number'), width: 45, style: 'text-align:right;', keyboards: 'number' }), null, prio1long);
        addGroup(centerToolBar, 'pages/current', labelMenuLong2 = new Label({ classes: 'link-style' }), null, prio1long);
        addGroup(centerToolBar, 'pages/next', new Button({ icon: 'fa-step-forward', tooltip: gt('Show next page'), classes: 'link-style' }), null, prio3long);
        addGroup(centerToolBar, 'pages/last', new Button({ icon: 'fa-fast-forward', tooltip: gt('Show last page'), classes: 'link-style' }), null, prio3long);
        this.addViewComponent(centerToolBar, { targetArea: 'center', classes: 'centerToolPane' });

        //
        // right ToolBar
        //
        // create the 'View' drop-down menu

        viewMenuGroup = new Controls.ComponentMenuButton(app, { label: Controls.VIEW_LABEL, anchorPadding: -3, classes: 'link-style' })
            .addGroup(null, createHeaderLabelGroup(gt.pgettext('menu-title', 'Zoom')), { inline: false })
            .addGroup('zoom/dec', new Button(Controls.ZOOMOUT_BUTTON_OPTIONS), { inline: false, sticky: true })
            .addGroup('zoom/inc', new Button(Controls.ZOOMIN_BUTTON_OPTIONS), { inline: true, sticky: true })
            .addGroup('zoom/percentage', new Controls.PercentageLabel(), { inline: true })
            .addGroup(null, createSeparatorGroup())
            .addGroup('zoom/width', new CheckBox({ label: gt('Fit to screen width')}))
            .addGroup('zoom/page', new CheckBox({ label: gt('Fit to screen size')}));
        createSidePaneControls(viewMenuGroup);
        addGroup(viewMenuGroup, null, createSeparatorGroup(), { inline: false }, prio3short);
        addGroup(viewMenuGroup, null, createHeaderLabelGroup(gt.pgettext('menu-title', 'Pages')), { inline: false }, prio3short);
        addGroup(viewMenuGroup, 'pages/first', new Button({ icon: 'fa-fast-backward', tooltip: gt('Show first page')}), { inline: false }, prio3short);
        addGroup(viewMenuGroup, 'pages/previous', new Button({ icon: 'fa-step-backward', tooltip: gt('Show previous page')}), { inline: true, sticky: true }, prio3short);
        addGroup(viewMenuGroup, 'pages/next', new Button({ icon: 'fa-step-forward', tooltip: gt('Show next page')}), { inline: true, sticky: true }, prio3short);
        addGroup(viewMenuGroup, 'pages/last', new Button({ icon: 'fa-fast-forward', tooltip: gt('Show last page')}), { inline: true }, prio3short);
        addGroup(viewMenuGroup, null, createSeparatorGroup(), { inline: false }, prio1short);
        addGroup(viewMenuGroup, 'pages/current', labelMenuShort1 = new Label(), { inline: true }, prio1short);
        addGroup(viewMenuGroup, 'pages/current', textFieldShort = new TextField({ tooltip: gt('Page number'), width: 45, style: 'text-align:right;', keyboard: 'number' }), { inline: true }, prio1short);
        addGroup(viewMenuGroup, 'pages/current', labelMenuShort2 = new Label(), { inline: true }, prio1short);

        rightToolBar
            .addGroup(null, viewMenuGroup)
            .addGap()
            .addGroup('app/quit', new Button(Utils.extendOptions(Controls.QUIT_BUTTON_OPTIONS, { classes: 'link-style' })));
        this.addViewComponent(rightToolBar, { targetArea: 'trailing' });

        // update view pane after it has been resized, or contents have been changed
        this.on('pane:layout', paneLayoutHandler);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            app = actionsMenuGroupShort = null;
        });
    } // class TopPane

    // exports ================================================================

    // derive this class from class ToolPane
    return ToolPane.extend({ constructor: TopPane });
});
