/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/view/operationspane',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/keycodes',
     'io.ox/office/baseframework/view/pane',
     'io.ox/office/editframework/app/rtconnection'
    ], function (Utils, KeyCodes, Pane, RTConnection) {

    'use strict';

    // class OperationsPane ===================================================

    /**
     * A view pane attached to the bottom border of the application view that
     * contains the operations log, and other custom debug information defined
     * by the application.
     *
     * @constructor
     *
     * @extends Pane
     */
    function OperationsPane(app) {

        var // self reference
            self = this,

            // initialize the operations log
            operationsNode = null,

            // the debug info table
            infoNode = $('<table style="table-layout:fixed;"><colgroup><col width="40"></col><col></col></colgroup></table>'),

            // number of operations already logged
            operationsCount = 0,

            // operations not yet logged
            pendingOperations = [],

            // background loop logging all pending operations
            logOperationsTimer = null;

        // base constructor ---------------------------------------------------

        Pane.call(this, app, { position: 'bottom', classes: 'operations-pane', enableContextMenu: true });

        // private methods ----------------------------------------------------

        /**
         * Scrolls the operations output down to its end.
         */
        function scrollDownOperationsNode() {
            var scrollNode = operationsNode.parent();
            scrollNode.scrollTop(scrollNode[0].scrollHeight);
        }

        /**
         * Direct callback for logOperations(): called every time when
         * logOperations() is called.
         */
        function registerOperations(event, operations, external) {
            // append the new operations to the pending operations queue
            pendingOperations = pendingOperations.concat(_.map(operations, function (operation) {
                if (app.isPrepareLosingEditRights()) {
                    RTConnection.log('registerOperation: ' + JSON.stringify(operation));
                }
                return { operation: operation, external: external };
            }));
        }

        /**
         * Deferred callback for logOperations(): called once after current
         * script execution ends.
         */
        function printSomeOperations() {

            // check if the background loop is already running
            if (logOperationsTimer) { return; }

            // create a new background loop that processes all pending operations
            logOperationsTimer = app.repeatDelayed(function () {

                // abort the background loop, if all operations have been logged
                if (pendingOperations.length === 0) { return Utils.BREAK; }

                var // the operations to insert into the table
                    operations = pendingOperations.splice(0, 20),
                    // the number of operations applied successfully by the document model
                    successCount = app.getModel().getOperationsCount(),
                    // checking the operation length, so that operations removed from the
                    // 'optimizeOperationHandler' will not be displayed in the debug pane.
                    oplCounter = 1,
                    // the entire HTML mark-up to be appended to the operations log
                    markup = '';

                // log the next operations chunk
                _.each(operations, function (entry) {

                    var operation = _.clone(entry.operation),
                        name = operation.name || '',
                        osn = ('osn' in operation) ? operation.osn : '',
                        opl = ('opl' in operation) ? operation.opl : '',
                        // skipping operations, that were 'optimized' in 'sendActions' in the
                        // optimizeOperationHandler. Otherwise the debug pane will be irritating, because
                        // 'merged operations' are followed by the operations, that will not be sent to
                        // the server, because they were removed by the 'optimizeOperationHandler'.
                        skipOperation = false;

                    // checking if this operation was merged by the 'optimizeOperationHandler'. This can
                    // only happen for internal operations (task 29601)
                    if (! entry.external) {
                        if (_.isNumber(opl) && (opl > 1)) {
                            oplCounter = opl;
                        } else if (oplCounter > 1) {
                            oplCounter--;
                            skipOperation = true;
                        }
                    }

                    if (skipOperation) { return; }

                    delete operation.name;
                    delete operation.osn;
                    delete operation.opl;
                    operation = JSON.stringify(operation).replace(/^\{(.*)\}$/, '$1').replace(/"(\w+)":/g, '$1:').replace(/ /g, '\xb7');
                    operationsCount += 1;

                    markup += '<tr' + ((successCount < operationsCount) ? ' class="error"' : '') + '>';
                    markup += '<td>' + operationsCount + '</td>';
                    markup += '<td title="' + (entry.external ? 'Ex' : 'In') + 'ternal operation">' + (entry.external ? 'E' : 'I') + '</td>';
                    markup += '<td style="text-align:right;">' + osn + '</td>';
                    markup += '<td style="text-align:right;">' + opl + '</td>';
                    markup += '<td>' + Utils.escapeHTML(name) + '</td>';
                    markup += '<td>' + Utils.escapeHTML(operation) + '</td>';
                    markup += '</tr>';
                });

                // append the mark-up, and scroll down to the last log entry
                operationsNode.append(markup);
                scrollDownOperationsNode();

            }, { delay: 50 });

            // forget reference to the timer, when all operations have been logged
            logOperationsTimer.always(function () { logOperationsTimer = null; });
        }

        // methods ------------------------------------------------------------

        this.addDebugInfoHeader = function (header, options) {
            var tooltip = Utils.escapeHTML(Utils.getStringOption(options, 'tooltip', ''));
            infoNode.append('<tr data-header="' + header + '" title="' + tooltip + '"><th colspan="2">' + Utils.capitalizeWords(header) + '</th></tr>');
            return this;
        };

        this.addDebugInfoNode = function (label, options) {
            var header = Utils.getStringOption(options, 'header', infoNode.find('tr').last().attr('data-header')),
                tooltip = Utils.escapeHTML(Utils.getStringOption(options, 'tooltip', '')),
                node = $('<tr data-header="' + header + '" title="' + tooltip + '"><td>' + label + '</td><td data-id="' + label + '"></td></tr>');
            node.insertAfter(infoNode.find('tr[data-header="' + header + '"]').last());
            return this;
        };

        this.setDebugInfoText = function (header, label, text) {
            infoNode.find('tr[data-header="' + header + '"]>td[data-id="' + label + '"]').text(text);
            return this;
        };

        // initialization -----------------------------------------------------

        // prepare the table element for the operations log
        (function () {
            var markup = '<table><tbody><tr>';
            markup += '<th></th>';
            markup += '<th title="Operation type">T</th>';
            markup += '<th title="Operation State Number" style="font-size:70%;">OSN</th>';
            markup += '<th title="Operation Length" style="font-size:70%;">OPL</th>';
            markup += '<th>Name</th>';
            markup += '<th>Parameters</th>';
            markup += '</tr></tbody></table>';
            operationsNode = $(markup);
        }());

        // add the operations table and the info table to the debug pane
        (function () {
            var tableNode = $('<table><colgroup><col width="70%"></col></colgroup><tbody><tr></tr></tbody></table>');
            _.each([operationsNode, infoNode], function (node) {
                var cellNode = $('<td><div tabindex="1"></div></td>');
                cellNode.children().append(node);
                tableNode.find('>tbody>tr').append(cellNode);
            });
            self.getNode().append(tableNode);
        }());

        // create the output nodes in the debug pane
        this.addDebugInfoHeader('application', { tooltip: 'Global application state' })
            .addDebugInfoNode('state', { tooltip: 'Application state' })
            .addDebugInfoNode('osn', { tooltip: 'Operation state number' });

        // log the application state
        this.listenTo(app, 'docs:state', function (state) {
            self.setDebugInfoText('application', 'state', state);
        });

        // log the operation state number of the document
        this.listenTo(app.getModel(), 'operations:after change:osn', function () {
            self.setDebugInfoText('application', 'osn', app.getModel().getOperationStateNumber());
        });

        // simply collect all operations while importing the document
        this.listenTo(app.getModel(), 'operations:after', registerOperations);

        // log operations after importing the document
        app.onImport(function () {

            var // create the debounced logOperations() method
                logOperations = app.createDebouncedMethod(registerOperations, printSomeOperations);

            // reconnect the 'operations:after' event to the debounced method
            app.getModel().off('operations:after', registerOperations);
            self.listenTo(app.getModel(), 'operations:after', logOperations);

            // trigger deferred logging by calling 'logOperations' once without new operations
            logOperations({}, []);
        });

        // scroll to last operation when showing the debug pane
        this.on('pane:show', scrollDownOperationsNode);

        // keyboard shortcuts
        this.listenTo(operationsNode.add(infoNode).parent(), 'keydown', function (event) {
            if (KeyCodes.matchKeyCode(event, 'A', { ctrlOrMeta: true })) {
                var docRange = window.document.createRange();
                docRange.setStart(event.delegateTarget, 0);
                docRange.setEnd(event.delegateTarget, 1);
                window.getSelection().removeAllRanges();
                window.getSelection().addRange(docRange);
                return false;
            }
            if (event.keyCode === KeyCodes.ESCAPE) {
                app.getView().grabFocus();
                return false;
            }
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            operationsNode.remove();
            infoNode.remove();
            self = operationsNode = infoNode = pendingOperations = logOperationsTimer = null;
        });

    } // class OperationsPane

    // exports ================================================================

    // derive this class from class Pane
    return Pane.extend({ constructor: OperationsPane });

});
