/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/drawinglayer/view/drawingframe',
    ['io.ox/core/date',
     'io.ox/office/tk/utils',
     'io.ox/office/tk/forms',
     'io.ox/office/drawinglayer/utils/drawingutils',
     'io.ox/office/drawinglayer/lib/canvasjs.min'
    ], function (CoreDate, Utils, Forms, DrawingUtils, CanvasJS) {

    'use strict';

    var // the CSS class name of the drawing content node
        CONTENT_CLASS = 'content',

        // the CSS class name of the selection root node
        SELECTION_CLASS = 'selection',

        // the CSS class name for active tracking mode
        TRACKING_CLASS = 'tracking-active',

        CULTURE_INFO = null;

    // private global functions ===============================================

    /**
     * Returns the selection root node of the specified drawing frame.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {jQuery}
     *  The selection root node of the specified drawing frame. Will be an
     *  empty collection, if the drawing frame is not selected currently.
     */
    function getSelectionNode(drawingFrame) {
        return $(drawingFrame).first().children('.' + SELECTION_CLASS);
    }

    /**
     * Calculates the offset and size of the bitmap in an image frame for one
     * dimension (either horizontally or vertically), according to the passed
     * cropping settings.
     *
     * @param {Number} frameSize
     *  Visible size (with or height) of the drawing frame containing the image
     *  bitmap, in 1/100 of millimeters.
     *
     * @param {Number} leadingCrop
     *  The leading cropping value (left/top), in percent.
     *
     * @param {Number} trailingCrop
     *  The trailing cropping value (right/bottom), in percent.
     *
     * @returns {Object}
     *  An object containing 'offset' and 'size' CSS attributes specifying how
     *  to format the bitmap node (in pixels with 'px' unit name).
     */
    function calculateBitmapSettings(frameSize, leadingCrop, trailingCrop) {

        var // sum of leading and trailing cropping (must not exceed a specific amount)
            totalCrop = leadingCrop + trailingCrop,
            // resulting settings for the bitmap
            size = 0, offset = 0;

        // do not crop away more than 90% of the bitmap
        if (totalCrop > 90) {
            leadingCrop *= (90 / totalCrop);
            trailingCrop *= (90 / totalCrop);
        }

        // bitmap size and offset, according to object size and cropping
        size = frameSize * 100 / (100 - leadingCrop - trailingCrop);
        offset = (size * leadingCrop) / 100;

        // convert to CSS pixels
        return {
            offset: Utils.convertHmmToCssLength(-offset, 'px', 1),
            size: Utils.convertHmmToCssLength(size, 'px', 1)
        };
    }

    // static class DrawingFrame ==============================================

    /**
     * Contains common static helper functions to display and handle drawing
     * frames in any document.
     */
    var DrawingFrame = {};

    // constants --------------------------------------------------------------

    /**
     * The CSS class used to mark drawing frame nodes.
     *
     * @constant
     */
    DrawingFrame.NODE_CLASS = 'drawing';

    /**
     * A jQuery selector that matches nodes representing a drawing frame.
     *
     * @constant
     */
    DrawingFrame.NODE_SELECTOR = '.' + DrawingFrame.NODE_CLASS;

    /**
     * The name of the jQuery data attribute that stores the unique node
     * identifier of a drawing frame node.
     *
     * @constant
     */
    DrawingFrame.DATA_UID = 'uid';

    /**
     * The name of the jQuery data attribute that stores the object type of a
     * drawing frame node.
     *
     * @constant
     */
    DrawingFrame.DATA_TYPE = 'type';

    /**
     * The name of the jQuery data attribute that stores the drawing model
     * instance for a drawing frame node.
     *
     * @constant
     */
    DrawingFrame.DATA_MODEL = 'model';

    // methods ----------------------------------------------------------------

    /**
     * Returns a new drawing frame node.
     *
     * @param {String} type
     *  The type of the drawing.
     *
     * @param {DrawingModel|String} model
     *  The drawing model instance this drawing frame is based on.
     *  Alternatively (for applications not supporting drawing models), this
     *  value can be a string with the type of the drawing object.
     *
     * @returns {jQuery}
     *  A new drawing frame with empty content node, as jQuery object.
     */
    DrawingFrame.createDrawingFrame = function (model) {
        return $('<div>', { contenteditable: false })
            .addClass(DrawingFrame.NODE_CLASS)
            .data(DrawingFrame.DATA_UID, 'frame' + _.uniqueId())
            .data(DrawingFrame.DATA_TYPE, _.isObject(model) ? model.getType() : model)
            .data(DrawingFrame.DATA_MODEL, _.isObject(model) ? model : null)
            .on('dragstart', false)
            .append($('<div>').addClass(CONTENT_CLASS));
    };

    /**
     * Returns whether the passed node is the root node of a drawing frame.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is the root node of a drawing frame.
     */
    DrawingFrame.isDrawingFrame = function (node) {
        return $(node).is(DrawingFrame.NODE_SELECTOR);
    };

    /**
     * Returns the type of the specified drawing frame.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {String}
     *  The type of the specified drawing frame, as specified while the frame
     *  has been created.
     */
    DrawingFrame.getDrawingType = function (drawingFrame) {
        return $(drawingFrame).first().data(DrawingFrame.DATA_TYPE);
    };

    /**
     * Returns the model of the specified drawing frame.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {Object|Null}
     *  The model object of the specified drawing frame, as specified while the
     *  frame has been created.
     */
    DrawingFrame.getModel = function (drawingFrame) {
        return $(drawingFrame).first().data(DrawingFrame.DATA_MODEL);
    };

    /**
     * Returns the unique identifier of the specified drawing frame.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {String}
     *  The unique identifier of the specified drawing frame, as specified
     *  while the frame has been created.
     */
    DrawingFrame.getUid = function (drawingFrame) {
        return $(drawingFrame).first().data(DrawingFrame.DATA_UID);
    };

    /**
     * Returns the root content node of the specified drawing frame containing
     * all type specific contents of the drawing.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {jQuery}
     *  The root content node of the specified drawing frame.
     */
    DrawingFrame.getContentNode = function (drawingFrame) {
        return $(drawingFrame).first().children('.' + CONTENT_CLASS);
    };

    /**
     * Clears and returns the root content node of the specified drawing frame.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {jQuery}
     *  The empty root content node of the specified drawing frame.
     */
    DrawingFrame.getAndClearContentNode = function (drawingFrame) {
        return DrawingFrame.getContentNode(drawingFrame).empty().removeClass('placeholder').attr('style', '');
    };

    /**
     * Inserts replacement layout nodes for unsupported drawing types. Inserts
     * the passed name and description of the drawing as text.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @param {String} [name]
     *  The name of the drawing frame.
     *
     * @param {String} [description]
     *  The description text for the drawing frame.
     */
    DrawingFrame.insertReplacementNodes = function (drawingFrame, name, description) {

        var // the type of the drawing frame
            type = DrawingFrame.getDrawingType(drawingFrame),
            // the empty content node
            contentNode = DrawingFrame.getAndClearContentNode(drawingFrame),
            // the inner width and height available in the content node
            innerWidth = Math.max(0, drawingFrame.outerWidth() - 2),
            innerHeight = Math.max(0, drawingFrame.outerHeight() - 2),
            // the vertical padding in the content node
            verticalPadding = Utils.minMax(Math.min(innerWidth, innerHeight) / 24, 1, 6),
            // the font size of the picture icon
            pictureIconSize = Utils.minMax(Math.min(innerWidth - 16, innerHeight - 28), 8, 72),
            // the base font size of the text
            fontSize = Utils.minMax(Math.min(innerWidth, innerHeight) / 4, 9, 13);

        // set border width at the content node, insert the picture icon
        contentNode
            .addClass('placeholder')
            .css({
                padding: Math.floor(verticalPadding) + 'px ' + Math.floor(verticalPadding * 2) + 'px',
                fontSize: fontSize + 'px'
            })
            .append(
                $('<div>')
                    .addClass('abs background-icon')
                    .css('line-height', innerHeight + 'px')
                    .append(Forms.createIconMarkup(DrawingUtils.getDrawingTypeIcon(type), { style: 'font-size:' + pictureIconSize + 'px' })),
                $('<p>').text(_.noI18n(name) || DrawingUtils.getDrawingTypeLabel(type))
            );

        // insert description if there is a reasonable amount of space available
        if ((innerWidth >= 20) && (innerHeight >= 20)) {
            contentNode.append($('<p>').text(_.noI18n(description)));
        }
    };

    // formatting -------------------------------------------------------------

    /**
     * Updates the CSS formatting of the passed drawing frame, according to the
     * passed generic formatting attributes.
     *
     * @param {EditApplication} app
     *  The application instance containing the drawing frame.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame to be updated. If this object is a
     *  jQuery collection, uses the first DOM node it contains.
     *
     * @param {Object} mergedAttributes
     *  The drawing attribute set (a map of attribute maps as name/value pairs,
     *  keyed by the 'drawing' attribute family), containing the effective
     *  attribute values merged from style sheets and explicit attributes.
     */
    DrawingFrame.updateFormatting = function (app, drawingFrame, mergedAttributes) {

        var // the drawing attributes of the passed attribute set
            drawingAttributes = mergedAttributes.drawing,
            // value of the 'transform' CSS attribute, e.g. for flipped drawings
            transformations = '',
            // type of the drawing object: 'image', ...
            type = DrawingFrame.getDrawingType(drawingFrame),
            // the content node inside the drawing frame
            contentNode = DrawingFrame.getContentNode(drawingFrame),
            // the replacement data for unsupported drawings
            replacementData = Utils.getStringOption(drawingAttributes, 'replacementData', ''),
            // rendering result (false to fall-back to placeholder graphic)
            rendered = false;

        // creates an image node in the content node with the specified source URL
        function createImageNodeFromUrl(srcUrl) {
            return app.createImageNode(srcUrl, { timeout: 15000 }).done(function (imgNode) {
                DrawingFrame.getAndClearContentNode(drawingFrame).append(imgNode);
            });
        }

        // creates an image node or an SVG image according to the passed image data
        function createImageNodeFromData(imageData) {
            if (/^data:/.test(imageData)) {
                return createImageNodeFromUrl(imageData);
            }
            if (/^<svg/.test(imageData)) {
                DrawingFrame.getAndClearContentNode(drawingFrame)[0].innerHTML = imageData;
                return $.when();
            }
            Utils.warn('DrawingFrame.createImageNodeFromData() - unsupported source data for the image object');
            return $.Deferred().reject();
        }

        // updates all formatting for unsupported objects
        function updateUnsupportedFormatting() {
            if (replacementData.length === 0) {
                // insert default replacement nodes, if no replacement data is available
                DrawingFrame.insertReplacementNodes(drawingFrame, drawingAttributes.name, drawingAttributes.description);
            } else if (!drawingFrame.data('has-replacement')) {
                // replacement data MUST NOT change at runtime
                createImageNodeFromData(replacementData);
                drawingFrame.data('has-replacement', true);
            }
        }

        // updates all formatting for image objects
        function updateImageFormatting() {

            var // current width of the drawing frame, in 1/100 mm
                drawingWidth = Utils.convertLengthToHmm(drawingFrame.width(), 'px'),
                // current height of the drawing frame, in 1/100 mm
                drawingHeight = Utils.convertLengthToHmm(drawingFrame.height(), 'px'),
                // horizontal offset/size of cropped bitmaps, as CSS attributes
                horizontalSettings = null, verticalSettings = null,
                // IE requires to swap cropping settings if image is flipped
                swapHorCrop = _.browser.IE && drawingAttributes.fliph,
                swapVertCrop = _.browser.IE && drawingAttributes.flipv,
                // effective cropping values
                cropLeft = swapHorCrop ? drawingAttributes.cropRight : drawingAttributes.cropLeft,
                cropRight = swapHorCrop ? drawingAttributes.cropLeft : drawingAttributes.cropRight,
                cropTop = swapVertCrop ? drawingAttributes.cropBottom : drawingAttributes.cropTop,
                cropBottom = swapVertCrop ? drawingAttributes.cropTop : drawingAttributes.cropBottom;

            // updates attributes of the image node after loading
            function updateImageAttributes() {
                if ((drawingWidth > 0) && (drawingHeight > 0)) {
                    horizontalSettings = calculateBitmapSettings(drawingWidth, cropLeft, cropRight);
                    verticalSettings = calculateBitmapSettings(drawingHeight, cropTop, cropBottom);
                    contentNode.find('>img').css({
                        left: horizontalSettings.offset,
                        top: verticalSettings.offset,
                        width: horizontalSettings.size,
                        height: verticalSettings.size
                    });
                }
            }

            // create the image or SVG node (considered to be constant at runtime)
            if (contentNode.children().length === 0) {
                // start with replacement graphics (image data will be loaded asynchronously)
                updateUnsupportedFormatting();
                if (drawingAttributes.imageData.length > 0) {
                    createImageNodeFromData(drawingAttributes.imageData).done(updateImageAttributes);
                } else if (drawingAttributes.imageUrl.length > 0) {
                    // convert relative URL to an application-specific absolute URL
                    createImageNodeFromUrl(/:\/\//.test(drawingAttributes.imageUrl) ? drawingAttributes.imageUrl : app.getFilterModuleUrl({ action: 'getfile', get_filename: drawingAttributes.imageUrl })).done(updateImageAttributes);
                } else {
                    Utils.warn('DrawingFrame.updateFormatting() - missing source data for the image object');
                }
            } else {
                // update attributes of an existing image element
                updateImageAttributes();
            }

            // image successfully rendered
            return true;
        }

        // updates chart formatting
        function updateChartFormatting() {

            var // the drawing object model
                model = DrawingFrame.getModel(drawingFrame),
                // the chart rendering engine (lazy initialization, see below)
                renderer = drawingFrame.data('chart-renderer'),
                // unique DOM element identifier for the renderer
                chartId = null,
                drawingNode = null;

            // drawing object model must exist for chart objects
            if (!model) { return false; }

            // lazy creation of the renderer
            if (!renderer) {
                chartId = 'io-ox-documents-chart-' + DrawingFrame.getUid(drawingFrame);
                contentNode.addClass('chartholder');

                // localization for CanvasJS
                if (!CULTURE_INFO) {
                    CULTURE_INFO = {
                        decimalSeparator: app.getDecimalSeparator(),
                        digitGroupSeparator: app.getGroupSeparator(),
                        shortDays: CoreDate.locale.daysShort,
                        days: CoreDate.locale.days,
                        months: CoreDate.locale.months,
                        shortMonths: CoreDate.locale.monthsShort
                    };
                    CanvasJS.addCultureInfo('en', CULTURE_INFO);
                }

                drawingNode = $('<div>');
                drawingNode.addClass('chartnode');
                drawingNode.attr('id', chartId);
                contentNode.append(drawingNode);

                //realy important call, because of a bug in canvasJS
                model.resetData();

                renderer = new CanvasJS.Chart(chartId, model.getModelData());
                drawingFrame.data('chart-renderer', renderer);
                model.firstInit();
            }

            // render the chart object
            model.refresh();
            try {
                renderer.render();
            } catch (ex) {
                Utils.exception(ex, 'chart rendering error');
            }

            // background color of model data is already prepared for css
            contentNode.css('backgroundColor', model.getModelData().cssBackgroundColor);

            // chart successfully rendered
            return true;
        }

        // convert to single jQuery object
        drawingFrame = $(drawingFrame).first();

        // set text attribute to support tooltip for the drawing
        Forms.setToolTip(drawingFrame, drawingAttributes.name);

        // vertically or horizontally flipped drawings
        if (drawingAttributes.flipH && drawingAttributes.flipV) {
            transformations = 'scale(-1)';
        } else if (drawingAttributes.flipH) {
            transformations = 'scaleX(-1)';
        } else if (drawingAttributes.flipV) {
            transformations = 'scaleY(-1)';
        }

        // always set the CSS attribute, to be able to remove existing transformations
        Utils.setCssAttributeWithPrefixes(contentNode, 'transform', transformations);

        // update drawing frame specific to the drawing type
        switch (type) {
        case 'image':
        case 'ole':
            rendered = updateImageFormatting();
            break;
        case 'chart':
            rendered = updateChartFormatting();
            break;
        }

        // add replacement in case the drawing type is not supported, or if rendering was not successful
        if (!rendered) {
            updateUnsupportedFormatting();
        }
    };

    // selection --------------------------------------------------------------

    /**
     * Returns whether the specified drawing frame is currently selected.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {Boolean}
     *  Whether the specified drawing frame is selected.
     */
    DrawingFrame.isSelected = function (drawingFrame) {
        return Forms.isSelectedNode(drawingFrame);
    };

    /**
     * Creates or updates additional nodes displayed while a drawing frame is
     * selected.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @param {Object} [options]
     *  A map with options to control the appearance of the selection. The
     *  following options are supported:
     *  @param {Boolean} [options.movable=false]
     *      If set to true, mouse pointer will be changed to a movable pointer
     *      while the mouse hovers the drawing.
     *  @param {Boolean} [options.resizable=false]
     *      If set to true, mouse pointer will be changed to an appropriate
     *      resize pointer while the mouse hovers a resize handle.
     *
     * @returns {jQuery}
     *  The selection root node contained in the drawing frame, as jQuery
     *  object.
     */
    DrawingFrame.drawSelection = function (drawingFrame, options) {

        var // HTML mark-up for the selection
            markup = null;

        // create new selection nodes if missing
        if (!DrawingFrame.isSelected(drawingFrame)) {

            // create a new selection node
            markup = '<div class="' + SELECTION_CLASS + '">';

            // create the border nodes
            markup += '<div class="borders">';
            _.each('tblr', function (pos) {
                markup += '<div data-pos="' + pos + '"></div>';
            });
            markup += '</div>';

            // create resizer handles
            markup += '<div class="resizers">';
            _.each(['t', 'b', 'l', 'r', 'tl', 'tr', 'bl', 'br'], function (pos) {
                markup += '<div data-pos="' + pos + '"></div>';
            });
            markup += '</div>';

            // create the tracker node
            markup += '<div class="tracker"></div>';

            // close the selection node
            markup += '</div>';

            $(drawingFrame).first().append(markup);
        }

        // update state classes at the drawing frame
        $(drawingFrame).first()
            .addClass(Forms.SELECTED_CLASS)
            .toggleClass('movable', Utils.getBooleanOption(options, 'movable', false))
            .toggleClass('resizable', Utils.getBooleanOption(options, 'resizable', false));

        return getSelectionNode(drawingFrame);
    };

    /**
     * Removes the selection node from the specified drawing frame.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @param {Object} [options]
     *  A map with options to control the appearance of the selection. The
     *  following options are supported:
     *  @param {Boolean} [options.movable=false]
     *      If set to true, mouse pointer will be changed to a movable pointer
     *      while the mouse hovers the drawing.
     *  @param {Boolean} [options.resizable=false]
     *      If set to true, mouse pointer will be changed to an appropriate
     *      resize pointer while the mouse hovers a resize handle.
     *
     * @returns {jQuery}
     *  The selection root node contained in the drawing frame, as jQuery
     *  object.
     */
    DrawingFrame.clearSelection = function (drawingFrame) {
        $(drawingFrame).first().removeClass(Forms.SELECTED_CLASS + ' movable resizable');
        getSelectionNode(drawingFrame).remove();
    };

    /**
     * Sets or resets the CSS class representing the active tracking mode for
     * the specified drawing frames.
     *
     * @param {HTMLElement|jQuery} drawingFrames
     *  The root node of a single drawing frame, or multiple drawing frames in
     *  a jQuery collection. This method changes the tracking mode of all
     *  drawing frames at once.
     *
     * @param {Boolean} active
     *  Whether tracking mode is active.
     */
    DrawingFrame.toggleTracking = function (drawingFrames, active) {
        $(drawingFrames).find('>.' + SELECTION_CLASS).toggleClass(TRACKING_CLASS, active === true);
    };

    /**
     * Returns the tracking direction stored in a resizer handle node of a
     * drawing frame selection box.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {String|Null}
     *  If the passed node is a resizer handle of a drawing frame selection
     *  box, returns its direction identifier (one of 'l', 'r', 't', 'b', 'tl',
     *  'tr', 'bl', or 'br'). Otherwise, returns null.
     */
    DrawingFrame.getResizerHandleType = function (node) {
        return $(node).is(DrawingFrame.NODE_SELECTOR + '>.' + SELECTION_CLASS + '>.resizers>[data-pos]') ? $(node).attr('data-pos') : null;
    };

    // exports ================================================================

    return DrawingFrame;

});
