/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/drawinglayer/utils/drawingutils',
    ['io.ox/office/tk/utils',
     'gettext!io.ox/office/drawing'
    ], function (Utils, gt) {

    'use strict';

    var // maps drawing object types to GUI names and icons
        DRAWING_TYPE_INFOS = {
            chart:    { icon: 'fa-bar-chart-o', label: /*#. bar charts, line charts, pie charts, etc. */ gt.pgettext('drawing', 'Chart') },
            image:    { icon: 'fa-picture-o',   label: /*#. bitmaps, vector graphics, etc. */ gt.pgettext('drawing', 'Image') },
            shape:    { icon: 'fa-picture-o',   label: /*#. rectangles, circles, lines, stars, etc. */ gt.pgettext('drawing', 'Shape') },
            diagram:  { icon: 'fa-sitemap',     label: /*#. complex diagrams, organigrams, etc. */ gt.pgettext('drawing', 'Diagram') }
        },

        DRAWING_TYPE_DEFAULT_INFO = { icon: 'fa-picture-o', label: gt.pgettext('drawing', 'Drawing') };

    // static class DrawingUtils ==============================================

    /**
     * Contains common static helper functions to handle drawing objects.
     */
    var DrawingUtils = {};

    // methods ----------------------------------------------------------------

    /**
     * Returns whether the passed rectangle contains the position.
     *
     * @param {Object} rectangle
     *  The position of the rectangle, in the numeric properties 'left', 'top',
     *  'width', and 'height'.
     *
     * @param {Object} rectangle
     *  The position to test, in the numeric properties 'left', and 'top'.
     *
     * @returns {Boolean}
     *  Whether the position is inside the rectangle.
     */
    DrawingUtils.rectangleContainsPosition = function (rectangle, position) {
        return (rectangle.left <= position.left) && (position.left < rectangle.left + rectangle.width) &&
            (rectangle.top <= position.top) && (position.top < rectangle.top + rectangle.height);
    };

    /**
     * Returns the intersecting rectangle of the passed rectangles.
     *
     * @param {Object} rectangle1
     *  The position of the first rectangle, in the numeric properties 'left',
     *  'top', 'width', and 'height'.
     *
     * @param {Object} rectangle2
     *  The position of the second rectangle, in the numeric properties 'left',
     *  'top', 'width', and 'height'.
     *
     * @returns {Object|Null}
     *  The position of the rectangle covered by both passed rectangles, if
     *  existing; otherwise null. If one of the rectangles has zero width or
     *  height and is covered by the other rectangle, the resulting rectangle
     *  will have zero width or height too. Otherwise, the rectangles must
     *  overlap in order to return an existing intersection rectangle.
     */
    DrawingUtils.getIntersectionRectangle = function (rectangle1, rectangle2) {

        var // the left position of the intersection rectangle
            left = Math.max(rectangle1.left, rectangle2.left),
            // the top position of the intersection rectangle
            top = Math.max(rectangle1.top, rectangle2.top),
            // the right position of the intersection rectangle
            right = Math.min(rectangle1.left + rectangle1.width, rectangle2.left + rectangle2.width),
            // the bottom position of the intersection rectangle
            bottom = Math.min(rectangle1.top + rectangle1.height, rectangle2.top + rectangle2.height);

        if (((left < right) || ((left === right) && ((rectangle1.width === 0) || (rectangle2.width === 0)))) &&
            ((top < bottom) || ((top === bottom) && ((rectangle1.height === 0) || (rectangle2.height === 0))))
        ) {
            return { left: left, top: top, width: right - left, height: bottom - top };
        }

        return null;
    };

    /**
     * Returns the bounding rectangle of the passed rectangles (the smallest
     * rectangle that contains all passed rectangles).
     *
     * @param {Object|Array} [...]
     *  A single rectangle, or an array with rectangles. The number of
     *  parameters that can be passed to this method is not limited.
     *
     * @returns {Object|Null}
     *  The bounding rectangle containing all passed rectangles; or null, if no
     *  rectangle has been passed.
     */
    DrawingUtils.getBoundingRectangle = function () {

        var // the resulting bounding rectangle
            boundRectangle = null;

        _.each(arguments, function (rectangles) {
            _.chain(rectangles).getArray().each(function (rectangle) {
                var right = 0, bottom = 0;
                if (!boundRectangle) {
                    boundRectangle = _.copy(rectangle, true);
                } else {
                    right = Math.max(boundRectangle.left + boundRectangle.width, rectangle.left + rectangle.width);
                    bottom = Math.max(boundRectangle.top + boundRectangle.height, rectangle.top + rectangle.height);
                    boundRectangle.left = Math.min(boundRectangle.left, rectangle.left);
                    boundRectangle.top = Math.min(boundRectangle.top, rectangle.top);
                    boundRectangle.width = right - boundRectangle.left;
                    boundRectangle.height = bottom - boundRectangle.top;
                }
            });
        });

        return boundRectangle;
    };

    // logical positions ------------------------------------------------------

    /**
     * Sorts the passed array of logical drawing positions in-place. The
     * positions of embedded drawings will be kept behind the positions of
     * their parent drawings.
     *
     * @param {Array} positions
     *  (in/out) An array with logical drawing positions. Each drawing position
     *  (each array element) MUST be an array with integers. The passed array
     *  will be sorted in-place.
     *
     * @returns {Array}
     *  A reference to the passed original array that has been sorted in-place.
     */
    DrawingUtils.sortDrawingPositions = function (positions) {
        positions.sort(Utils.compareNumberArrays);
        return positions;
    };

    /**
     * Sorts the passed array of logical drawing positions in-place, and
     * removes the positions of all embedded drawings whose parent drawing
     * positions are contained in the array too.
     *
     * @param {Array} positions
     *  (in/out) An array with logical drawing positions. Each drawing position
     *  (each array element) MUST be an array with integers. The passed array
     *  will be sorted and modified in-place.
     *
     * @returns {Array}
     *  A reference to the passed original array that has been sorted and
     *  modified in-place.
     */
    DrawingUtils.optimizeDrawingPositions = function (positions) {

        var // array index for deleting positions from the array
            index = 0,
            // array elements while deleting positions
            pos1 = null, pos2 = null;

        // first, sort the positions
        DrawingUtils.sortDrawingPositions(positions);

        // remove positions of embedded drawings whose parent position is contained too
        while (index + 1 < positions.length) {
            pos1 = positions[index];
            pos2 = positions[index + 1];
            if ((pos1.length < pos2.length) && _.isEqual(pos1, pos2.slice(0, pos1.length))) {
                positions.splice(index + 1, 1);
            } else {
                index += 1;
            }
        }

        return positions;
    };

    // GUI labels and icons ---------------------------------------------------

    /**
     * Returns an appropriate CSS icon class for the passed drawing type.
     *
     * @param {String} type
     *  The type identifier of a drawing object.
     *
     * @returns {String}
     *  A CSS icon class for the passed drawing type.
     */
    DrawingUtils.getDrawingTypeIcon = function (type) {
        return (DRAWING_TYPE_INFOS[type] || DRAWING_TYPE_DEFAULT_INFO).icon;
    };

    /**
     * Returns an appropriate text label for the passed drawing type.
     *
     * @param {String} type
     *  The type identifier of a drawing object.
     *
     * @returns {String}
     *  A text label for the passed drawing type.
     */
    DrawingUtils.getDrawingTypeLabel = function (type) {
        return (DRAWING_TYPE_INFOS[type] || DRAWING_TYPE_DEFAULT_INFO).label;
    };

    // exports ================================================================

    return DrawingUtils;

});
