/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/baseframework/view/toolbar',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/forms',
     'io.ox/office/baseframework/view/component'
    ], function (Utils, Forms, Component) {

    'use strict';

    // class ToolBar ==========================================================

    /**
     * Represents a view component intended to be inserted into a horizontal
     * tool pane.
     *
     * @constructor
     *
     * @extends Component
     *
     * @param {EditApplication} app
     *  The application containing this tool bar instance.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options of the Component base class.
     */
    function ToolBar(app, initOptions) {

        var // self reference
            self = this,

            // the current container node as target for new groups
            containerNode = null,

            // whether to add a separator line into the next gap
            separator = false,
            // special classes for the separator
            separatorClasses = null;

        // base constructor ---------------------------------------------------

        Component.call(this, app, initOptions);

        // private methods ----------------------------------------------------

        /**
         * Prepares to add a gap before the next inserted control group.
         */
        function addGap(sep) {
            containerNode = null;
            separator = sep;
        }

        /**
         * Inserts the root DOM node of the passed group into this tool bar.
         */
        function groupInserter(group) {

            // create missing group container node
            if (!containerNode) {
                containerNode = $('<div class="group-container' + (separator ? ' separator '+separatorClasses : '') + '">');
                self.getNode().append(containerNode);
            }

            // insert the group into the container
            containerNode.append(group.getNode());

            // immediately update visibility of group container nodes according to group visibility
            (function (contNode) {

                // show the container node before showing the group, for correct group:resize events
                self.listenTo(group, 'group:beforeshow', function () {
                    Forms.showNodes(contNode, true);
                });

                // hide the container node if no visible groups remain
                self.listenTo(group, 'group:hide', function () {
                    var visibleGroups = Forms.filterVisibleNodes(contNode.children());
                    Forms.showNodes(contNode, visibleGroups.length > 0);
                });
            }(containerNode)); // pass local copy of the jQuery object
        }

        // methods ------------------------------------------------------------

        /**
         * Inserts some white-space after the last group into the tool bar.
         *
         * @returns {ToolBar}
         *  A reference to this instance.
         */
        this.addGap = function () {
            addGap(false);
            return this;
        };

        /**
         * Inserts a vertical separator line after the last group into the tool
         * bar.
         *
         * @returns {ToolBar}
         *  A reference to this instance.
         */
        this.addSeparator = function (options) {
            separatorClasses = Utils.getStringOption(options, 'classes', '');

            addGap(true);
            return this;
        };

        // initialization -----------------------------------------------------

        this.getNode().addClass('toolbar').attr({ role: 'navigation' });

        // insert new groups into the current container node
        this.registerGroupInserter(groupInserter);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            app = initOptions = self = containerNode = null;
        });

    } // class ToolBar

    // exports ================================================================

    // derive this class from class Component
    return Component.extend({ constructor: ToolBar });

});
