/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2014 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

/**
 * GetText equivalent.  Loads PO files and parses Templates
 * 
 */

package com.openexchange.guard.translate;

import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


public class GetText {

	private static Logger logger = LoggerFactory.getLogger(GetText.class);
	
    private static String def_lang = "en_US";

    private static Map<String, Map<String, Translation>> langmaps = new HashMap<String, Map<String, Translation>>();

    public static void setDefault(String def) {
        def_lang = def;
    }

    public static void loadFiles(String directory, String domain) {
        if (domain == null)
            domain = "";
        java.io.File i18folder = new java.io.File(directory);
        java.io.File[] list = i18folder.listFiles();
        for (int i = 0; i < list.length; i++) {
            if (list[i].getName().startsWith(domain) || (domain == "")) {// make sure for the domain specified
                String name = list[i].getName();
                if (name.endsWith(".po")) {// only load the po files
                    String lang = name;
                    if (name.contains("-")) {
                        lang = name.substring(name.indexOf("-") + 1);
                    }
                    if (lang.contains(".")) {
                        lang = lang.substring(0, lang.indexOf("."));
                    }
                    loadPOfile(list[i].getAbsolutePath(), lang);
                }
            }
        }
    }

    public static void loadPOfile(String filename, String language) {
        Map<String, Translation> mapping = new HashMap<String, Translation>();
    	if (langmaps.containsKey(language)) mapping = langmaps.get(language);
        Path path = Paths.get(filename);
        java.io.File f = new java.io.File(filename);
        if (!f.exists()) {
            logger.error("Unable to load file " + filename);
            return;
        }
        try {
            List<String> lines = Files.readAllLines(path, StandardCharsets.UTF_8);
            for (int i = 0; i < lines.size(); i++) {
                if (lines.get(i).contains("msgid")) { // First message ID line
                    String id = getBetweenQuotes(lines.get(i));
                    i++;
                    String line2 = lines.get(i);
                    while (line2.trim().startsWith("\"")) {
                    	id = id + getBetweenQuotes(line2);
                    	i++;
                    	line2 = lines.get(i);
                    }
                    ArrayList<String> plurals = new ArrayList<String>();
                    if (line2.contains("msgid_plural")) {// Pleural based field
                        String plural = getBetweenQuotes(line2);
                        i++;
                        while (lines.get(i).contains("msgstr")) {
                            plurals.add(getBetweenQuotes(lines.get(i)));
                            i++;
                        }
                        Translation trans = new Translation(id, plural, plurals);
                        mapping.put(id, trans);
                        mapping.put(plural, trans);
                    } else {
                        // Singular based field
                        String data = lines.get(i);
                        if (data.contains("msgstr")) {
                        	String transl = getBetweenQuotes(data);
                        	while (((i + 1) < lines.size()) && lines.get(i+1).trim().startsWith("\"")) {
                        		i++;
                        		transl = transl + getBetweenQuotes(lines.get(i));
                        	}
                            Translation trans = new Translation(id, transl);
                            mapping.put(id, trans);
                        }
                    }
                }
            }
            langmaps.put(language, mapping);
        } catch (Exception ex) {
            logger.error("Error loading .po file " + filename + " for language " + language, ex);
        }
    }

    public static String gettext(String msgid, String lang) {
        try {
            Translation trans = langmaps.get(checklang(lang)).get(msgid);
            if (trans == null)
                return (msgid);
            return (trans.translation);
        } catch (Exception ex) {
            logger.error("Error translating '" + msgid + "' into " + lang, ex);
            return (msgid);
        }
    }
    
    public static String gettext(String msgid, String lang, String variable) {
    	String translated = gettext(msgid, lang);
    	return (translated.replace("%s", variable));
    }

    /**
     * Check if we have that language trasnlation. If not, return default
     * 
     * @param lang
     * @return
     */
    private static String checklang(String lang) {
        if ((lang == null) || lang.length() < 2)
            return (def_lang);
        if (!langmaps.containsKey(lang)) {
            if (langmaps.containsKey(lang.substring(0, 2))) {// If we don't have the 4 character translation, see if we have 2
                lang = lang.substring(0, 2);
            } else
                lang = def_lang; // If we don't have that language, go to default
        }
        return (lang);
    }

    /**
     * Parse a template. Gettext to be deliniated as <$gt(sometext)>
     * 
     * @param filename
     * @param lang
     * @return
     */
    public static String processTemplate(String filename, String lang) {
        try {
            lang = checklang(lang);
            java.io.File f = new java.io.File(filename);
            if (!f.exists()) {
                logger.error("Unable to load file " + filename);
                return (null);
            }
            String templ = readFile(filename);
            int counter = 0;
            while (templ.contains("<$gettext") && (counter < 1000)) {
                int i = templ.indexOf("<$gettext(");
                int j = templ.substring(i).indexOf(")>");
                if (j < 0) {
                    logger.error("Poorly formed template");
                    return (null);
                }
                String toreplace = templ.substring(i, i + j + 2);
                String totranslat = removeQuotes(toreplace.replace("<$gettext(", "").replace(")>", ""));
                templ = templ.replace(toreplace, gettext(totranslat, lang));
            }
            return (templ);
        } catch (Exception ex) {
            logger.error("Error processing template " + filename + " for language " + lang, ex);
            return (null);
        }

    }

    private static String removeQuotes(String data) {
        if (data.startsWith("\"")) {
            data = data.substring(1);
            if (data.endsWith("\"")) {
                data = data.substring(0, data.length() - 1);
            }
        }
        if (data.startsWith("'")) {
            data = data.substring(1);
            if (data.endsWith("'")) {
                data = data.substring(0, data.length() - 1);
            }
        }
        return (data);
    }

    private static String getBetweenQuotes(String id) {
        if (id.contains("\"")) {
            id = id.substring(id.indexOf("\"") + 1);
            if (id.contains("\"")) {
                id = id.substring(0, id.indexOf("\""));
            }
        }
        return (id);
    }

    private static String readFile(String fileName) throws IOException {
        BufferedReader br = new BufferedReader(new InputStreamReader(new FileInputStream(fileName), "UTF8"));
        try {
            StringBuilder sb = new StringBuilder();
            String line = br.readLine();

            while (line != null) {
                sb.append(line);
                sb.append("\n");
                line = br.readLine();
            }
            return sb.toString();
        } finally {
            br.close();
        }
    }

}
