/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2014 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.mailcreator;

import java.io.StringReader;
import java.util.ArrayList;
import java.util.Date;

import net.htmlparser.jericho.Source;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.gson.Gson;
import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.google.gson.JsonPrimitive;
import com.openexchange.guard.database.RecipKey;
import com.openexchange.guard.ox.Languages;

/**
 * Class for managing Open-Xchange mail json object
 * 
 * @author greg
 */
public class Mail {
	private static Logger logger = LoggerFactory.getLogger(Mail.class);
	
    public static String outsideTemplate = "";

    public static String passwordEmailTemplate = "";

    // Default blank email
    public JsonObject createBlankMail() {
        String blank = " {\"data\":{\"from\":\"\",\"to\":\"\",\"cc\":\"\",\"bcc\":\"\",\"reply_to\":\"\",\"subject\":\"\",\"priority\":\"3\",\"vcard\":\"0\",\"attachments\":[{\"content\":\"\",\"content_type\":\"ALTERNATIVE\"}],\"nested_msgs\":[],\"sendtype\":\"0\"},\"mode\":\"compose\",\"format\":\"html\",\"signature\":\"\",\"files\":[]} ";
        Gson g = new Gson();
        JsonObject json = g.fromJson(blank, JsonObject.class);
        return (json);
    }

    public JsonObject createFailureMail() {
        String blank = " {\"data\":{\"from\":\"\",\"to\":\"\",\"cc\":\"\",\"bcc\":\"\",\"reply_to\":\"\",\"subject\":\"\",\"priority\":\"3\",\"vcard\":\"0\",\"attachments\":[{\"content\":\"\",\"content_type\":\"ALTERNATIVE\"}],\"nested_msgs\":[],\"sendtype\":\"0\"},\"mode\":\"compose\",\"format\":\"html\",\"signature\":\"\",\"files\":[]} ";
        Gson g = new Gson();
        JsonObject json = g.fromJson(blank, JsonObject.class);
        return (json);
    }

    public JsonObject createBlankOutMail(String lang, int templId) {
        JsonObject email = createBlankMail();
        String content = Languages.getTranslation("oxtempl.html", lang, templId);
        if (content == null)
            return (null);
        return (setContent(email, content));
    }

    public JsonObject createBlankGuestMail(String lang, int templId) {
        JsonObject email = createBlankMail();
        String content = Languages.getTranslation("guesttempl.html", lang, templId);
        if (content == null)
            return (null);
        return (setContent(email, content));
    }

    public JsonObject addPlainText(JsonObject email, String data) {
        Gson gson = new Gson();
        String emailText = gson.toJson(email);
        if (data == null)
            data = "";
        emailText = emailText.replace("$plaintext", data);
        Gson g = new Gson();
        return (g.fromJson(emailText, JsonObject.class));
    }

    public JsonObject addURL(JsonObject email, String data) {
        Gson gson = new Gson();
        String emailText = gson.toJson(email);
        emailText = emailText.replace("$url", data);
        Gson g = new Gson();
        return (g.fromJson(emailText, JsonObject.class));
    }

    public JsonObject addTo(JsonObject mail, String name, String email, RecipKey.reciptype type) {
        switch (type) {
        case bcc:
            return (addBCC(mail, name, email));
        case cc:
            return (addCC(mail, name, email));
        case to:
            return (addTo(mail, name, email));

        }
        return (null);
    }

    // Add flattened to address to mail
    public JsonObject addTo(JsonObject mail, String name, String email) {
        String to = mail.getAsJsonObject("data").get("to").getAsString();
        if (to.length() > 0)
            to = to + ", ";
        if (name.length() > 0) {
        	if (name.startsWith("\"")) {
        		to = to + name + "<" + email + ">";
        	} else {
        		to = to + "\"" + name  + "\"<" + email + ">";
        	}
        } else {
            to = to + email;
        }
        mail.getAsJsonObject("data").add("to", new JsonPrimitive(to));
        return (mail);
    }

    // Add flattened to address to mail
    public JsonObject addCC(JsonObject mail, String name, String email) {
        String to = mail.getAsJsonObject("data").get("cc").getAsString();
        if (to.length() > 0)
            to = to + ", ";
        if (name.length() > 0) {
            to = to + name + "<" + email + ">";
        } else {
            to = to + email;
        }
        mail.getAsJsonObject("data").add("cc", new JsonPrimitive(to));
        return (mail);
    }

    // Add flattened to address to mail
    public JsonObject addBCC(JsonObject mail, String name, String email) {
        String to = mail.getAsJsonObject("data").get("bcc").getAsString();
        if (to.length() > 0)
            to = to + ", ";
        if (name.length() > 0) {
            to = to + name + "<" + email + ">";
        } else {
            to = to + email;
        }
        mail.getAsJsonObject("data").add("bcc", new JsonPrimitive(to));
        return (mail);
    }

    // Add flattened from address to mail
    public JsonObject addFrom(JsonObject mail, String name, String email) {
        String from = email;
        if (name.length() > 0) {
            from = "\"" + name + "\"<" + email + ">";
        }
        mail.getAsJsonObject("data").add("from", new JsonPrimitive(from));
        return (mail);
    }

    public JsonObject addSubject(JsonObject mail, String subject) {
        mail.getAsJsonObject("data").add("subject", new JsonPrimitive(subject));
        return (mail);
    }

    public JsonObject setContent(JsonObject mail, String content) {
        mail.getAsJsonObject("data").getAsJsonArray("attachments").get(0).getAsJsonObject().add("content", new JsonPrimitive(content));
        return (mail);
    }

    private JsonArray address(String name, String email) {
        JsonPrimitive element = new JsonPrimitive(name);
        JsonArray arr = new JsonArray();
        arr.add(element);
        element = new JsonPrimitive(email);
        arr.add(element);
        return (arr);
    }

    public static String PrepareOXEmail(JsonObject Email, ArrayList<Attachment> attachments, long expiration, String mailId) {

        Email.get("attachments").getAsJsonArray().get(0).getAsJsonObject().addProperty("disp", "inline");
        int size = Email.get("attachments").getAsJsonArray().get(0).getAsJsonObject().get("content").getAsString().length();
        Email.get("attachments").getAsJsonArray().get(0).getAsJsonObject().addProperty("size", size);
        Email.get("attachments").getAsJsonArray().get(0).getAsJsonObject().addProperty("id", "0");
        Email.get("headers").getAsJsonObject().addProperty("Message-ID", mailId);
        Email.addProperty("sent", (new Date()).toString());
        Gson g = new Gson();
        int id = 2;
        for (Attachment attach : attachments) {
            String contentType = attach.type + "; name=\\\"" + attach.filename + "\\\"";
            JsonObject attachjson = g.fromJson(
                "{\"content\" : null, \"content_type\" : \"" + contentType + "\", \"disp\" : \"attachment\", \"filename\" : \"" + attach.filename + "\", \"size\" : " + attach.content.length + ", \"id\" : \"" + id++ + "\", \"title\" : \"" + attach.filename + "\"}",
                JsonObject.class);
            Email.get("attachments").getAsJsonArray().add(attachjson);
            Email.addProperty("attachment", true);
        }
        Email.add("exp", new JsonPrimitive(expiration));
        return (Email.toString());
    }

    public JsonObject noSave(JsonObject email) {
        email.getAsJsonObject("data").add("copy2Sent", new JsonPrimitive("false"));
        return (email);
    }
    
    public JsonObject getOxPasswordEmail (String to, String sendername, String from, String from2, String password, String lang, int templId) {
    	return getPasswordEmail (to, sendername, from, from2, password, lang, templId, "", "oxpasswordtempl.html");
    }

    public JsonObject getPasswordEmail(String to, String sendername, String from, String from2, String password, String lang, int templId, String guestmessage) {
        return getPasswordEmail (to, sendername, from, from2, password, lang, templId, guestmessage, "passwordtempl.html");
    }
    
    private JsonObject getPasswordEmail (String to, String sendername, String from, String from2, String password, String lang, int templId, String guestmessage, String templateName) {
    	JsonObject email = createBlankMail();
        email = addTo(email, to, to);
        email = addFrom(email, from, from2);
        email.getAsJsonObject("data").add("copy2Sent", new JsonPrimitive("false"));// Do not copy this to sent folder
        String templ = Languages.getTranslation(templateName, lang, templId).replace("$from", sendername).replace("$password", password).replace("$plaintext", guestmessage);
        if (templ.contains("Subject:")) {
            String subject = templ.substring(templ.indexOf("Subject:"), templ.indexOf("\n")).replace("Subject:", "");
            email = addSubject(email, subject);
        } else {
            logger.error("Missing subject in password template! Make sure starts with 'Subject:'");
        }
        if (templ.contains("Body:")) {
            templ = templ.substring(templ.indexOf("Body:") + 5);
        } else {
            logger.error("Missing body in password template! Make sure starts with 'Body:'");
        }
        email = setContent(email, templ);
        return (email);
    }

    public JsonObject getResetEmail(String to, String from, String password, String lang, int templId) {
        JsonObject email = createBlankMail();
        email = addTo(email, to, to);
        email = addFrom(email, from, from);
        email.getAsJsonObject("data").add("copy2Sent", new JsonPrimitive("false"));// Do not copy this to sent folder
        String templ = Languages.getTranslation("resettempl.html", lang, templId).replace("$from", from).replace("$password", password);
        if (templ.contains("Subject:")) {
            String subject = templ.substring(templ.indexOf("Subject:"), templ.indexOf("\n", templ.indexOf("Subject:"))).replace("Subject:", "");
            email = addSubject(email, subject);
        } else {
            logger.error("Missing subject in password template! Make sure starts with 'Subject:'");
        }
        if (templ.contains("Body:")) {
            templ = templ.substring(templ.indexOf("Body:") + 5);
        } else {
            logger.error("Missing body in password template! Make sure starts with 'Body:'");
        }
        email = setContent(email, templ);
        return (email);
    }
    

}
