/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.rest.services.osgiservice;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import org.json.JSONException;
import org.json.JSONValue;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.ajax.requesthandler.AJAXRequestData;
import com.openexchange.ajax.tools.JSONCoercion;
import com.openexchange.exception.OXException;
import com.openexchange.rest.services.OXRESTMatch;
import com.openexchange.rest.services.OXRESTService;
import com.openexchange.rest.services.OXRESTService.HALT;
import com.openexchange.rest.services.Response;
import com.openexchange.rest.services.internal.OXRESTServiceWrapper;


/**
 * The {@link ReflectiveServiceWrapper} wraps an {@link OXRESTService} and knows how to execute a service method
 *
 * @author <a href="mailto:francisco.laguna@open-xchange.com">Francisco Laguna</a>
 */
public class ReflectiveServiceWrapper implements OXRESTServiceWrapper {

    private static final Logger LOGGER = LoggerFactory.getLogger(ReflectiveServiceWrapper.class);

    private final Method method;
    private final OXRESTService<?> delegate;
    private final OXRESTMatch match;
    private AJAXRequestData request;

    /**
     * Initializes a new {@link ReflectiveServiceWrapper}.
     *
     * @param method The associated reflection method
     * @param newInstance The new service instance for this wrapper
     * @param match The match providing parameters
     */
    public ReflectiveServiceWrapper(Method method, OXRESTService<?> newInstance, OXRESTMatch match) {
        super();
        this.method = method;
        this.delegate = newInstance;
        this.match = match;
        delegate.setMatch(match);
    }

    @Override
    public void setRequest(AJAXRequestData request) {
        this.request = request;
        delegate.setRequest(request);
    }

    @Override
    public OXRESTMatch getMatch() {
        return match;
    }

    @Override
    public Response execute() throws OXException {
        try {
            delegate.before();
            Class<?>[] parameterTypes = method.getParameterTypes();
            Object result;
            if (parameterTypes.length == 0) {
                // Just invoke the method
                result = method.invoke(delegate);
            } else {
                Object[] args = new Object[parameterTypes.length];
                int i = 0;
                for (Class<?> paramType : parameterTypes) {
                    if (paramType != String.class) {
                        args[i] = request.getParameter(match.getParameterName(i), paramType);
                    } else {
                        args[i] = request.getParameter(match.getParameterName(i));
                    }
                    i++;
                }
                result = method.invoke(delegate, args);
            }

            if (result != null) {
                if (result instanceof Iterable) {
                    final Iterable<String> body = (Iterable<String>) result;
                    delegate.body(body);
                } else if (result instanceof String){
                    delegate.body((String) result);
                } else {
                    try {
                        if (false && (result instanceof JSONValue)) {
                            JSONValue jsonValue = (JSONValue) result;
                            if (jsonValue.isObject()) {
                                delegate.body(jsonValue.toObject().toString(true));
                            } else {
                                delegate.body(jsonValue.toArray().toString(true));
                            }
                        } else {
                            delegate.header("Content-Type", "application/json; charset=UTF-8");
                            delegate.body(JSONCoercion.coerceToJSON(result).toString());
                        }
                    } catch (JSONException x) {
                        delegate.body(result.toString());
                    }
                }
            }
        } catch (final InvocationTargetException x) {
            final Throwable cause = x.getCause();
            if (cause instanceof OXException) {
                throw (OXException) cause;
            }
            if (cause instanceof HALT) {
                // processing has finished
            } else {
                LOGGER.error("Error invoking method {} of REST service {}", method.getName(), delegate.getClass().getSimpleName(), x);
            }
        } catch (final IllegalArgumentException e) {
            LOGGER.error("Passed wrong arguments when invoking method {} of REST service {}", method.getName(), delegate.getClass().getSimpleName(), e);
        } catch (final IllegalAccessException e) {
            LOGGER.error("Illegal access to method {} of REST service {}", method.getName(), delegate.getClass().getSimpleName(), e);
        } catch (final RuntimeException e) {
            LOGGER.error("Runtime error when invoking method {} of REST service {}", method.getName(), delegate.getClass().getSimpleName(), e);
        } finally {
            delegate.after();
        }
        return delegate.getResponse();
    }

    /**
     * Gets the (newly created) service instance associated with this wrapper.
     *
     * @return The service instance
     */
    public <T> OXRESTService<T> getDelegate() {
        return (OXRESTService<T>) delegate;
    }

}
