/*!
 * delims <https://github.com/jonschlinkert/delims>
 *
 * Copyright (c) 2014 Jon Schlinkert, contributors
 * Licensed under the MIT license.
 */

const util = require('util');
const expect = require('chai').expect;
const delims = require('../');


describe('create delims:', function () {

  it('should use the default delimiters for YAML front matter.', function () {
    var d = delims();

    var actual = util.inspect(d.evaluate);
    var expected = '/^---([\\s\\S]+?)---([\\s\\S]+|\\s?)$/';
    expect(actual).to.eql(expected);
  });

  it('should use and escape the default delimiters for YAML front matter.', function () {
    var d = delims({escape: true});

    var actual = util.inspect(d.evaluate);
    var expected = '/^\\-\\-\\-([\\s\\S]+?)\\-\\-\\-([\\s\\S]+|\\s?)$/';
    expect(actual).to.eql(expected);
  });

  it('should create escaped delimiters for YAML front matter.', function () {
    var d = delims(['~~~', '~~~'], {escape: true});

    var actual = util.inspect(d.evaluate);
    var expected = '/^\\~\\~\\~([\\s\\S]+?)\\~\\~\\~([\\s\\S]+|\\s?)$/';
    expect(actual).to.eql(expected);
  });

  it('should create non-escaped delimiters for YAML front matter.', function () {
    var d = delims(['~~~', '~~~']);

    var actual = util.inspect(d.evaluate);
    var expected = '/^~~~([\\s\\S]+?)~~~([\\s\\S]+|\\s?)$/';
    expect(actual).to.eql(expected);
  });

  it('should create non-escaped custom delimiters for YAML front matter.', function () {
    var d = delims(['-{3}', '-{3}']);

    var actual = util.inspect(d.evaluate);
    var expected = '/^-{3}([\\s\\S]+?)-{3}([\\s\\S]+|\\s?)$/';
    expect(actual).to.eql(expected);
  });

  it('should create non-escaped custom delimiters for YAML front matter.', function () {
    var d = delims(['-{3}', '~~~']);

    var actual = util.inspect(d.evaluate);
    var expected = '/^-{3}([\\s\\S]+?)~~~([\\s\\S]+|\\s?)$/';
    expect(actual).to.eql(expected);
  });

  it('should create non-escaped delimiters for YAML front matter with custom "body" regex.', function () {
    var d = delims(['---', '---', '([\\w\\W]+?)']);

    var actual = util.inspect(d.evaluate);
    var expected = '/^---([\\s\\S]+?)---([\\w\\W]+?)$/';
    expect(actual).to.eql(expected);
  });

  it('should create multiple non-escaped delimiters for YAML front matter.', function () {
    var d = delims([['---', '~~~', '= yaml ='], ['---', '~~~', '= yaml =']]);

    var actual = util.inspect(d.evaluate);
    var expected = '/^(?:---|~~~|= yaml =)([\\s\\S]+?)(?:---|~~~|= yaml =)([\\s\\S]+|\\s?)$/';
    expect(actual).to.eql(expected);
  });

  it('should create multiple non-escaped delimiters for YAML front matter with custom "body" regex.', function () {
    var d = delims([['---', '~~~', '= yaml ='], ['---', '~~~', '= yaml ='], '([\\w\\W]+?)']);

    var actual = util.inspect(d.evaluate);
    var expected = '/^(?:---|~~~|= yaml =)([\\s\\S]+?)(?:---|~~~|= yaml =)([\\w\\W]+?)$/';
    expect(actual).to.eql(expected);
  });

  it('should create escaped delimiters for Lo-Dash templates.', function () {
    var opts = {body: '', beginning: '', end: '', flags: 'g', noncapture: false, escape: true};
    var actual = util.inspect(delims(['{%', '%}'], opts));

    var expected = util.inspect({
      evaluate: new RegExp('\\{\\%([\\s\\S]+?)\\%\\}', 'g'),
      interpolate: new RegExp('\\{\\%=([\\s\\S]+?)\\%\\}', 'g'),
      escape: new RegExp('\\{\\%-([\\s\\S]+?)\\%\\}', 'g')
    });
    expect(actual).to.eql(expected);
  });


  it('should create escaped delimiters for Lo-Dash templates. empty value in array[2] should not throw an error.', function () {
    var opts = {body: '', beginning: '', end: '', flags: 'g', noncapture: false, escape: true};
    var actual = util.inspect(delims(['<%', '%>', ''], opts));

    var expected = util.inspect({
      evaluate: new RegExp('\\<\\%([\\s\\S]+?)\\%\\>', 'g'),
      interpolate: new RegExp('\\<\\%=([\\s\\S]+?)\\%\\>', 'g'),
      escape: new RegExp('\\<\\%-([\\s\\S]+?)\\%\\>', 'g')
    });
    expect(actual).to.eql(expected);
  });

  it('should create non-escaped delimiters for Lo-Dash templates.', function () {
    var opts = {body: '', beginning: '', end: '', flags: 'g', noncapture: false};
    var actual = util.inspect(delims(['{%', '%}'], opts));

    var expected = util.inspect({
      evaluate: new RegExp('{%([\\s\\S]+?)%}', 'g'),
      interpolate: new RegExp('{%=([\\s\\S]+?)%}', 'g'),
      escape: new RegExp('{%-([\\s\\S]+?)%}', 'g')
    });
    expect(actual).to.eql(expected);
  });
});