/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.tools;

import java.io.BufferedInputStream;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.logging.Log;
import com.openexchange.ajax.requesthandler.AJAXRequestData;
import com.openexchange.exception.OXException;
import com.openexchange.file.storage.DefaultFile;
import com.openexchange.file.storage.File;
import com.openexchange.file.storage.FileStorageFileAccess;
import com.openexchange.file.storage.FileStorageFolder;
import com.openexchange.file.storage.FileStoragePermission;
import com.openexchange.file.storage.composition.FolderAware;
import com.openexchange.file.storage.composition.IDBasedFileAccess;
import com.openexchange.file.storage.composition.IDBasedFileAccessFactory;
import com.openexchange.folderstorage.ContentType;
import com.openexchange.folderstorage.FolderService;
import com.openexchange.folderstorage.UserizedFolder;
import com.openexchange.folderstorage.database.contentType.InfostoreContentType;
import com.openexchange.groupware.infostore.InfostoreExceptionCodes;
import com.openexchange.groupware.ldap.User;
import com.openexchange.java.Strings;
import com.openexchange.server.ServiceLookup;
import com.openexchange.session.Session;
import com.openexchange.tools.file.external.QuotaFileStorageExceptionCodes;
import com.openexchange.tools.session.ServerSession;
import com.openexchange.tx.TransactionAwares;
import com.openexchange.user.UserService;

/**
 * {@link FileHelper}
 *
 * The FileHelper class contains many utility methods to retrieve a document
 * stream from a file or write a stream to a file. To retrieve meta data from
 * a file, e.g. permissions, locking and determine mine type, extension, document
 * type/format.
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @author <a href="mailto:carsten.driesner@open-xchange.com">Carsten Driesner</a>
 */
public class FileHelper {
    private static final Log LOG = com.openexchange.log.Log.loggerFor(FileHelper.class);

    static public final String OX_RESCUEDOCUMENT_DOCUMENTSTREAM_NAME = "oxoffice/documentStream";

    static public final String OX_RESCUEDOCUMENT_OPERATIONSSTREAM_NAME = "oxoffice/operationUpdates";

    static public final String OX_RESCUEDOCUMENT_RESOURCESTREAM_NAME_PREFIX = "oxoffice/resource/";

    static public final String OX_DOCUMENT_EXTENDEDDATA = "oxoffice/extendedData.json";

    // ------------------------------------------------------

    static public final String IMAGE_JPEG = "image/jpeg";

    static public final String EXTENSION_JPG_1 = "jpg";

    static public final String EXTENSION_JPG_2 = "jpeg";

    static public final String IMAGE_EMF = "image/x-emf";

    static public final String IMAGE_EMF2 = "image/emf";

    static public final String EXTENSION_EMF = "emf";

    static public final String IMAGE_WMF = "image/x-wmf";

    static public final String EXTENSION_WMF = "wmf";

    static public final String IMAGE_GIF = "image/gif";

    static public final String EXTENSION_GIF = "gif";

    static public final String IMAGE_PICT = "image/pict";

    static public final String EXTENSION_PICT = "pct";

    static public final String IMAGE_PNG = "image/png";

    static public final String EXTENSION_PNG = "png";

    public static final String IMAGE_TIFF = "image/tiff";

    public static final String EXTENSION_TIFF = "tiff";

    public static final String EXTENSION_TIFF2 = "tif";

    public static final String IMAGE_EPS = "application/postscript"; // as reported by XmlGraphics

    public static final String EXTENSION_EPS = "eps";

    public static final String IMAGE_BMP = "image/bmp";

    public static final String EXTENSION_BMP = "bmp";

    public static final String EXTENSION_ODFTEXT = "odt";

    public static final String EXTENSION_ODT = "odt";

    public static final String DOC_ODF_TEXT = "application/vnd.oasis.opendocument.text";

    public static final String EXTENSION_ODS = "ods";

    public static final String DOC_ODF_SPREADSHEET = "application/vnd.oasis.opendocument.spreadsheet";

    public static final String EXTENSION_ODP = "odp";

    public static final String DOC_ODF_PRESENTATION = "application/vnd.oasis.opendocument.presentation";

    public static final String EXTENSION_ODG = "odg";

    public static final String DOC_ODF_GRAPHIC = "application/vnd.oasis.opendocument.graphics";

    public static final String EXTENSION_DOCX = "docx";

    public static final String DOC_DOCX_TEXT = "application/vnd.openxmlformats-officedocument.wordprocessingml.document";

    public static final String EXTENSION_XLSX = "xlsx";

    public static final String DOC_DOCX_SPREADSHEET = "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet";

    public static final String EXTENSION_PPTX = "pptx";

    public static final String DOC_DOCX_PRESENTATION = "application/vnd.openxmlformats-officedocument.presentationml.presentation";

    public static final String EXTENSION_XML = "xml";

    public static final String TEXT_XML = "text/xml";

    public static final String OCTET_STREAM = "application/octet-stream";

    public static final int FOLDER_PERMISSIONS_READ = 0;

    public static final int FOLDER_PERMISSIONS_WRITE = 1;

    public static final int FOLDER_PERMISSIONS_FOLDER = 2;

    public static final int FOLDER_PERMISSIONS_SIZE = 3;

    /**
     * Initializes a new {@link FileHelper}.
     */
    public FileHelper(ServiceLookup services, String folderId, String fileId) {
        super();

        m_services = services;
        m_folderId = folderId;
        m_fileId = fileId;
    }

    /**
     * Initializes a new {@link FileHelper}.
     */
    public FileHelper(ServiceLookup services, String folderId, String fileId, String version) {
        super();

        m_services = services;
        m_folderId = folderId;
        m_fileId = fileId;
        m_version = version;
    }

    /**
     * Retrieves the service lookup to create new services.
     *
     * @return
     *  The service lookup used by the file helper instance.
     */
    public ServiceLookup getServiceLookup() {
        return m_services;
    }

    /**
     * Retrieves the folder id of this file helper instance.
     *
     * @return
     *  The folder id used by the file helper instance.
     */
    public String getFolderId() {
        return m_folderId;
    }

    /**
     * Retrieves the file id of this file helper instance.
     *
     * @return
     *  The file id used by the file helper instance.
     */
    public String getFileId() {
        return m_fileId;
    }

    /**
     * Retrieves the document format.
     *
     * @return
     *  The document format of the file used by the file helper
     *  instance.
     */
    public DocumentFormat getDocumentFormat() {
        return m_documentFormat;
    }

    /**
     * Retrieves the document type.
     *
     * @return
     *  The document format of the file used by the file helper
     *  instance.
     */
    public DocumentType getDocumentType() {
        return m_documentType;
    }

    /**
     * Retrieves the size of the document.
     *
     * @return
     *  The size of the document in bytes.
     */
    public long getDocumentSize() {
        return m_documentSize;
    }

    /**
     * @return write protect state of the file
     */
    public boolean isWriteProtected() {
        return m_writeProtected;
    }

    /**
     * Retrieves the date when the lock of the file
     * is released.
     *
     * @return Date when the lock is released or
     *  null if no lock has been set.
     */
    public Date lockedUntil() {
        return m_lockedUntil;
    }

    /**
     * Retrieves the user who locked the file.
     *
     * @return
     *  The name of the user who locked file file. Can be null
     *  if there is no active lock.
     */
    public String getLockedByUser() {
        return m_lockedByUser;
    }

    /**
     * Retrieves the document stream from the file.
     *
     * @param session
     *  The session of the user who requested to retrieve the document
     *  stream.
     *
     * @return
     *  The document stream or null if the document stream could not
     *  be retrieved.
     */
    public InputStream getDocumentStream(Session session) {
        final boolean[] writeProteced = { false };
        final Date[] lockedUntil = { null };
        final String[] lockedByUser = { "" };
        final DocumentFormat[] documentFormat = { DocumentFormat.NONE };
        final long[] documentSize = { 0 };
        final InputStream resultStm = getDocumentStream(
            m_services,
            session,
            m_fileId,
            m_folderId,
            FileStorageFileAccess.CURRENT_VERSION,
            writeProteced,
            lockedUntil,
            lockedByUser,
            documentFormat,
            documentSize);

        if (null != resultStm) {
            m_writeProtected = writeProteced[0];
            m_lockedUntil = lockedUntil[0];
            m_lockedByUser = lockedByUser[0];
            m_documentSize = documentSize[0];

            // set DocumentFormat
            if (DocumentFormat.NONE == m_documentFormat) {
                m_documentFormat = documentFormat[0];
            }

            // set DocumentType
            if ((DocumentType.NONE == m_documentType) && (DocumentFormat.NONE != m_documentFormat)) {
                m_documentType = getDocumentTypeFromDocumentFormat(m_documentFormat);
            }
        }

        return resultStm;
    }

    /**
     * Retrieves the document stream from the provided path.
     *
     * @param path
     *  A full-qualified path to a file stored in local file system.
     *
     * @return
     *  The document data as an input stream or null if there was a problem
     *  accessing the file.
     */
    static public InputStream getDocumentStreamFromLocalFile(String path) {
        final java.io.File localFile = new java.io.File(path);
        InputStream inputStream = null;

        if (localFile.exists() && localFile.isFile()) {
            try {
                return new BufferedInputStream(new FileInputStream(localFile));
            } catch (FileNotFoundException e) {
                //TODO: handle exception
            }
        }

        return inputStream;
    }

    /**
     * Retrieves the document stream requested by the AJAX request data.
     *
     * @param services
     *  The service lookup to be used by the method to create necessary
     *  services.
     *
     * @param session
     *  The session of the user who requested to retrieve the document
     *  stream.
     *
     * @param requestData
     *  The AJAX request data which must contain the id of the file, folder_id and
     *  the version to be used to retrieve the document stream. If the version
     *  is omitted the method uses the current version of the file.
     *
     * @return
     *  The document stream or null if the document stream could not
     *  be retrieved.
     */
    static public InputStream getDocumentStream(ServiceLookup services, Session session, AJAXRequestData requestData) {
        final String fileId = requestData.getParameter("id");
        final String folderId = requestData.getParameter("folder_id");
        String version = requestData.getParameter("version");

        if ((null == version) || (0 == version.length()) || (version.equals("0"))) {
            version = FileStorageFileAccess.CURRENT_VERSION;
        }

        return getDocumentStream(services, session, fileId, folderId, version, null, null, null, null, null);
    }

    /**
     * Retrieves the document stream specified by the fileId, folderId and version.
     *
     * @param services
     *  The service lookup to be used by the method to create necessary
     *  services.
     *
     * @param session
     *  The session of the user who requested to retrieve the document
     *  stream.
     *
     * @param fileId
     *  The file id of the file to read the document stream from.
     *
     * @param folderId
     *  The folder id where the file resides.
     *
     * @param version
     *  The version of the file to be used to read the document stream from.
     *  If the version is null the method uses the current version.
     *
     * @param writeProtected
     *  Provides the write protected state of the file from the perspective
     *  of the user.
     *
     * @param lockedUntil
     *  Provides how long the file has exclusively been locked.
     *
     * @param lockedByUser
     *  Provides who has locked the file exclusively.
     *
     * @param documentFormat
     *  Provides the document format of the file.
     *
     * @param documentSize
     *  Provides the size of the file.
     *
     * @return
     *  The document stream or null if the stream couldn't be not be provided.
     */
    static protected InputStream getDocumentStream(ServiceLookup services, Session session, String fileId, String folderId, String version, boolean[] writeProtected, Date[] lockedUntil, String[] lockedByUser, DocumentFormat[] documentFormat, long[] documentSize) {
        InputStream documentStm = null;
        final IDBasedFileAccessFactory fileFactory = services.getService(IDBasedFileAccessFactory.class);
        final UserService userService = services.getService(UserService.class);
        final IDBasedFileAccess fileAccess = ((null != fileFactory) && (null != session)) ? fileFactory.createAccess(session) : null;

        if ((null != fileAccess) && !Strings.isEmpty(fileId)) {
            try {
                if ((null != writeProtected) && (writeProtected.length > 0)) {
                    writeProtected[0] = !folderHasWriteAccess(fileAccess, folderId);
                }

                final String accessId = getFileAccessId(fileId, folderId);
                documentStm = fileAccess.getDocument(accessId, version);

                if ((null != documentStm) && (null != documentFormat) && (documentFormat.length > 0)) {
                    com.openexchange.file.storage.File metadata = null;

                    try {
                        metadata = fileAccess.getFileMetadata(accessId, version);
                    } catch (OXException e) {
                        LOG.warn("Couldn't retrieve meta data of the file", e);
                    }

                    if (null != metadata) {
                        documentFormat[0] = getDocumentFormat(metadata.getFileName());

                        if ((null != documentSize) && (documentSize.length > 0)) {
                            documentSize[0] = metadata.getFileSize();
                        }

                        if ((null != lockedUntil[0]) && (lockedUntil.length > 0)) {
                            lockedUntil[0] = metadata.getLockedUntil();

                            if ((null != lockedByUser) && (lockedByUser.length > 0) && (null != userService)) {
                                try {
                                    final User user = userService.getUser(metadata.getModifiedBy(), userService.getContext(session.getContextId()));
                                    lockedByUser[0] = user.getDisplayName();
                                } catch (OXException e) {
                                    // set user name to empty for unkown user name
                                    lockedByUser[0] = "";
                                    if (LOG.isWarnEnabled()) {
                                        LOG.warn("Couldn't retrieve user name who locked the file", e);
                                    }
                                }
                            }
                        }
                    }
                }
            } catch (Exception e) {
                documentStm = null;
                LOG.error("Couldn't read document stream", e);
            } finally {
                try {
                    fileAccess.finish();
                } catch (OXException e) {
                    LOG.warn("Couldn't finish the file access", e);
                }
            }
        }

        return documentStm;
    }

    /**
     * Writes the provided document stream to the file associated with this
     * FileHelper instance.
     *
     * @param session
     *  The session of the user who wants to write the document stream to the
     *  file.
     *
     * @param documentStream
     *  The document stream that should be stored into the file.
     *
     * @param userExtension
     *  Specifies the extension to be used for the file.
     *
     * @param setUserExtension
     *  Specifies if the extension of the file should be set or
     *  removed. If TRUE the user extension will be set otherwise the user
     *  extension will be removed.
     *
     * @param errorCode
     *  Specifies what problem cause the fail of the method.
     *
     * @param revisionless
     *  Specifies if the stream should be written to the file using a new
     *  version or overwrite the latest version. TRUE means overwrite
     *  the latest version with the provided document stream. FALSE to
     *  create a new version.
     *
     * @return
     *  The new version of the file. Could be same as the old one if the
     *  stream has been written with revisionless = TRUE.
     */
    public String writeDocumentStream(Session session, InputStream documentStream, String userExtension, boolean setUserExtension, ErrorCode[] errorCode, boolean revisionless) {
        String newFileVersion = ""; // empty string means error, version can never be "".
        final IDBasedFileAccessFactory fileFactory = m_services.getService(IDBasedFileAccessFactory.class);
        final IDBasedFileAccess fileAccess = ((null != fileFactory) && (null != session)) ? fileFactory.createAccess(session) : null;

        if ((null != documentStream) && (null != fileAccess) && !Strings.isEmpty(m_fileId)) {
            final String accessId = getFileAccessId(m_fileId, m_folderId);
            final String version = FileStorageFileAccess.CURRENT_VERSION;
            boolean rollback = false;

            try {
                final com.openexchange.file.storage.File metadata = fileAccess.getFileMetadata(accessId, version);
                final com.openexchange.file.storage.DefaultFile newMetadata = new DefaultFile();
                final String fileName = metadata.getFileName();
                final ArrayList<com.openexchange.file.storage.File.Field> modifiedColumns = new ArrayList<com.openexchange.file.storage.File.Field>();

                newMetadata.setId(m_fileId);
                newMetadata.setFolderId(m_folderId);
                newMetadata.setFileMIMEType(metadata.getFileMIMEType());

                // ensure that the filename has the given user extension, if set as call parameter
                if (null != userExtension) {
                    final boolean hasUserExtension = fileName.endsWith(userExtension);

                    if (!hasUserExtension && setUserExtension) {
                        // add user extension
                        newMetadata.setFileName(fileName + userExtension);
                        modifiedColumns.add(com.openexchange.file.storage.File.Field.FILENAME);
                    } else if (hasUserExtension && !setUserExtension) {
                        // remove user extension
                        newMetadata.setFileName(fileName.substring(0, fileName.length() - userExtension.length()));
                        modifiedColumns.add(com.openexchange.file.storage.File.Field.FILENAME);
                    }
                }

                fileAccess.startTransaction();
                rollback = true;
                if (revisionless) {
                    // try to save this document using the non-revision save available at IDBasedIgnorableVersionFileAccess
                    if (fileAccess instanceof com.openexchange.file.storage.composition.IDBasedIgnorableVersionFileAccess) {
                        ((com.openexchange.file.storage.composition.IDBasedIgnorableVersionFileAccess)fileAccess).saveDocument(
                            newMetadata, documentStream, FileStorageFileAccess.DISTANT_FUTURE, modifiedColumns, true);
                    }
                } else {
                    fileAccess.saveDocument(newMetadata, documentStream, FileStorageFileAccess.DISTANT_FUTURE, modifiedColumns);
                }
                fileAccess.commit();
                rollback = false;

                newFileVersion = newMetadata.getVersion();
            } catch (final OXException e) {
                if (e.getPrefix().equalsIgnoreCase("FLS")) {
                    // file storage exceptions
                    if (e.getCode() == QuotaFileStorageExceptionCodes.STORE_FULL.getNumber()) {
                        errorCode[0] = ErrorCode.FILTER_QUOTA_REACHED_ERROR;
                    }
                }
                if (LOG.isErrorEnabled()) {
                    LOG.error("Couldn't write document stream, OX Exception caught: ", e);
                }
            } catch (final Exception e) {
                if (LOG.isErrorEnabled()) {
                    LOG.error("Couldn't write document stream, Exception caught: ", e);
                }
            }
            finally {
                // Roll-back (if needed) and finish
                if (rollback) {
                    TransactionAwares.rollbackSafe(fileAccess);
                }
                TransactionAwares.finishSafe(fileAccess);
            }
        }

        return newFileVersion;
    }

    /**
     * Renames the document file in the info store using the provided
     * new file name.
     *
     * @param session
     *  The session of the user who requested to rename the document file.
     *
     * @param newFileName
     *  The new file name of the document file.
     *
     * @param newFileVersion
     *  The file version where the file name has been changed.
     *
     * @param errorCode
     *  A possible error code if the rename failed.
     *
     * @return
     *  The new file name of the document file. Can be null if the rename
     *  operation failed.
     */
    public String renameDocument(Session session, String newFileName, String[] newFileVersion, ErrorCode[] errorCode) {
        String resultFileName = null;
        final IDBasedFileAccessFactory fileFactory = m_services.getService(IDBasedFileAccessFactory.class);
        final IDBasedFileAccess fileAccess = ((null != fileFactory) && (null != session)) ? fileFactory.createAccess(session) : null;
        final String currentVersion = FileStorageFileAccess.CURRENT_VERSION;

        ErrorCode result = ErrorCode.NO_ERROR;
        if (null != fileAccess) {
            try {
                if (!Strings.isEmpty(newFileName)) {
                    final ArrayList<com.openexchange.file.storage.File.Field> modifiedColumns = new ArrayList<com.openexchange.file.storage.File.Field>();
                    final DefaultFile metadata = new DefaultFile();

                    metadata.setId(m_fileId);
                    metadata.setFolderId(m_folderId);
                    metadata.setFileName(newFileName);
                    metadata.setVersion(currentVersion);

                    modifiedColumns.add(com.openexchange.file.storage.File.Field.FILENAME);
                    fileAccess.saveFileMetadata(metadata, FileStorageFileAccess.DISTANT_FUTURE, modifiedColumns);
                    fileAccess.commit();

                    // retrieve the real filename from the metadata, it may have
                    // been changed from the requested one by the file service;
                    // add a short sleep in order to avoid asynchronous errors while
                    // getting the updated metadata, which may currently happen with
                    // cascaded DB backends
                    // (see also https://bugs.open-xchange.com/show_bug.cgi?id=26080)
                    final String accessId = getFileAccessId(m_fileId, m_folderId);
                    final com.openexchange.file.storage.File newMetaData = fileAccess.getFileMetadata(accessId, currentVersion);

                    resultFileName = (null != newMetaData) ? newMetaData.getFileName() : newFileName;

                    if ((null != newFileVersion) && (newFileVersion.length > 0)) {
                        newFileVersion[0] = (null != newMetaData) ? newMetaData.getVersion() : metadata.getVersion();
                    }
                }
            } catch (Exception e) {
                result = ErrorCode.RENAMEDOCUMENT_FAILED_ERROR;
                // try to find out if there is a more specific error message
                if (e instanceof OXException) {
                    OXException ox = (OXException)e;
                    if (ox.isPrefix("IFO")) {
                        if (ox.getCode() == InfostoreExceptionCodes.VALIDATION_FAILED_CHARACTERS.getNumber()) {
                            result = ErrorCode.RENAMEDOCUMENT_VALIDATION_FAILED_CHARACTERS_ERROR;
                        }
                    }
                }
                LOG.error(result.getDescription(), e);
            }
        }

        errorCode[0] = result;

        return resultFileName;
    }

    /**
     * Provides the latest version string of the file
     *
     * @param session
     *  The session to be used to access the file store.
     *
     * @return
     *  The version string or null.
     *
     * @throws OXEception
     *  Throws an exception if the meta data of the document file
     *  couldn't be retrieved.
     */
    public String getVersionFromFile(Session session) throws OXException {
        return (String) getFieldValueFromFile(session, com.openexchange.file.storage.File.Field.VERSION);
    }

    /**
     * Provides the file name of the file
     *
     * @param session
     *  The session to be used to access the file store.
     *
     * @return
     *  The file name or null.
     *
     * @throws OXEception
     *  Throws an exception if the meta data of the document file
     *  couldn't be retrieved.
     */
    public String getFilenameFromFile(Session session) throws OXException {
        return (String) getFieldValueFromFile(session, com.openexchange.file.storage.File.Field.FILENAME);
    }

    /**
     * Changes the createdBy meta data entry of the document to the
     * user that is associated with the session.
     *
     * @param session
     *  The session of the user that requested to change the createdBy
     *  meta data file entry.
     *
     * @return
     *  TRUE if the createdBy entry has been changed, otherwise FALSE. A
     *  failure to set the created by meta data entry could be the result
     *  of missing permissions.
     */
    public boolean setCreatedByForFile(Session session) {
        final IDBasedFileAccessFactory fileFactory = m_services.getService(IDBasedFileAccessFactory.class);
        final IDBasedFileAccess fileAccess = ((null != fileFactory) && (null != session)) ? fileFactory.createAccess(session) : null;
        final String currentVersion = FileStorageFileAccess.CURRENT_VERSION;

        boolean result = false;

        try {
            if (null != fileAccess) {
                final ArrayList<com.openexchange.file.storage.File.Field> modifiedColumns = new ArrayList<com.openexchange.file.storage.File.Field>();
                final DefaultFile metadata = new DefaultFile();

                metadata.setId(m_fileId);
                metadata.setFolderId(m_folderId);
                metadata.setVersion(currentVersion);
                metadata.setCreatedBy(session.getUserId());
                metadata.setLastModified(new Date());

                modifiedColumns.add(com.openexchange.file.storage.File.Field.CREATED_BY);
                modifiedColumns.add(com.openexchange.file.storage.File.Field.LAST_MODIFIED);
                fileAccess.saveFileMetadata(metadata, FileStorageFileAccess.DISTANT_FUTURE, modifiedColumns);
                fileAccess.commit();
                result = true;
            }
        } catch (OXException e) {
            LOG.error("RT connection: Couldn't set new createdBy user for document", e);
        }

        return result;
    }

    /**
     * Retrieves the file meta data using the provided session.
     *
     * @param session
     *  The session to be used as the context to retrieve the file meta data.
     *
     * @param version
     *  A version of the file where the meta data should be retrieved from.
     *  Can be null where the method uses the latest version.
     *
     * @return
     *  The file meta data or null if necessary services are not available.
     *
     * @throws OXException
     *  Throws an exception if the meta data of the document file couldn't
     *  be retrieved.
     */
    public File getMetaDataFromFile(Session session, String version) throws OXException {
        final IDBasedFileAccessFactory fileFactory = m_services.getService(IDBasedFileAccessFactory.class);
        final IDBasedFileAccess fileAccess = ((null != fileFactory) && (null != session)) ? fileFactory.createAccess(session) : null;
        final String fileVersion = (null == version) ? FileStorageFileAccess.CURRENT_VERSION : version;
        com.openexchange.file.storage.File metaData = null;

        if (null != fileAccess) {
            final String accessId = Strings.isEmpty(m_folderId) ? m_fileId : (m_folderId + "/" + m_fileId);
            metaData = fileAccess.getFileMetadata(accessId, fileVersion);
        }

        return metaData;
    }

    /**
     * Provides the specified field value of the file
     *
     * @param session
     *  The session to be used to access the file store.
     *
     * @return
     *  The field value or null.
     *
     * @throws OXEception
     *  Throws an exception if the meta data of the document file
     *  couldn't be retrieved.
     */
    protected Object getFieldValueFromFile(Session session, com.openexchange.file.storage.File.Field field) throws OXException {

        Object result = null;

        try {
            final String currentVersion = FileStorageFileAccess.CURRENT_VERSION;
            com.openexchange.file.storage.File metaData = getMetaDataFromFile(session, currentVersion);

            if (null != metaData) {
                switch (field) {
                case FILENAME: {
                    result = metaData.getFileName();
                    break;
                }

                case FILE_SIZE: {
                    result = Long.valueOf(metaData.getFileSize());
                    break;
                }

                case FILE_MIMETYPE: {
                    result = metaData.getFileMIMEType();
                    break;
                }

                case VERSION: {
                    result = metaData.getVersion();
                    break;
                }

                default: {
                    break;
                }
                }
            }
        } catch (OXException e) {
            if (LOG.isWarnEnabled()) {
                LOG.warn("RT connection: Couldn't retrieve field value from document meta data", e);
            }
            throw e;
        }

        return result;
    }

    // - Statics ---------------------------------------------------------------

    /**
     * Retrieve user read permission for a certain folder.
     *
     * @param fileAccess
     *  The file access to be used to retrieve the write
     *  permissions of the user.
     *
     * @param folderId
     *  The folder id where want to know the write permissions
     *  of the user.
     *
     * @return
     *  TRUE if the user has read permission in the provided folder,
     *  other FALSE.
     */
    static public boolean folderHasReadAccess(IDBasedFileAccess fileAccess, String folderId) {
        boolean ret = false;

        int[] result = getFolderPermissions(fileAccess, folderId);
        if (null != result && result.length >= 3) {
            int readPermission = result[FOLDER_PERMISSIONS_READ];
            ret = (readPermission > FileStoragePermission.NO_PERMISSIONS);
        }
        return ret;
    }

    /**
     * Retrieve user write permission for a certain folder.
     *
     * @param fileAccess
     *  The file access to be used to retrieve the write
     *  permissions of the user.
     *
     * @param folderId
     *  The folder id where want to know the write permissions
     *  of the user.
     *
     * @return
     *  TRUE if the user has write permission in the provided folder,
     *  other FALSE.
     */
    static public boolean folderHasWriteAccess(IDBasedFileAccess fileAccess, String folderId) {
        boolean ret = false;

        int[] result = getFolderPermissions(fileAccess, folderId);
        if (null != result && result.length >= 3) {
            int writePermission = result[FOLDER_PERMISSIONS_WRITE];
            ret = (writePermission > FileStoragePermission.NO_PERMISSIONS);
        }
        return ret;
    }

    /**
     * Retrieve user create permission for a certain folder.
     *
     * @param fileAccess
     *  The file access to be used to retrieve the create
     *  permissions of the user.
     *
     * @param folderId
     *  The folder id where want to know the create permissions
     *  of the user.
     *
     * @return
     *  TRUE if the user has the create permission in the provided folder,
     *  other FALSE.
     */
    static public boolean folderHasCreateAccess(IDBasedFileAccess fileAccess, String folderId) {
        boolean ret = false;

        int[] result = getFolderPermissions(fileAccess, folderId);
        if (null != result && result.length >= 3) {
            int createPermission = result[FOLDER_PERMISSIONS_FOLDER];
            ret = (createPermission >= FileStoragePermission.CREATE_OBJECTS_IN_FOLDER);
        }
        return ret;
    }

    /**
     * Retrieve the user permissions for the provided folder id.
     * @param fileAccess
     *  The file access used to retrieve the user permissions.
     * @param folderId
     *  The folder id to retrieve the user permissions.
     *
     * @return
     *  An array with three entries for read, write and folder permissions or
     *  null if the method is unable to retrieve the permissions.
     */
    static public int[] getFolderPermissions(IDBasedFileAccess fileAccess, String folderId) {
        int[] result = null;

        if (fileAccess instanceof FolderAware) {
            try {
                // create the info store file URL to check for the folder id
                final String infoStoreFileURL = "com.openexchange.infostore://infostore/" + folderId + "/0";
                final FileStorageFolder folder = ((FolderAware) fileAccess).optFolder(infoStoreFileURL);

                if (folder != null) {
                    final FileStoragePermission permission = folder.getOwnPermission();

                    int readPermission = permission.getReadPermission();
                    int writePermission = permission.getWritePermission();
                    int folderPermission = permission.getFolderPermission();

                    // create array with user permissions for the folder
                    result = new int[FOLDER_PERMISSIONS_SIZE];
                    result[FOLDER_PERMISSIONS_READ] = readPermission;
                    result[FOLDER_PERMISSIONS_WRITE] = writePermission;
                    result[FOLDER_PERMISSIONS_FOLDER] = folderPermission;
                }
            } catch (final Exception e) {
                LOG.warn("GetFolderPermissions - retrieving user permissions for folder failed with exception", e);
            }
        }

        return result;
    }

    /**
     * Maps the user language provided by a User object to a ISO compliant
     * language code. E.g. en_US => en-US
     *
     * @param userLanguage
     *  The user language as provided by the User methods getPreferredLanguage
     *
     * @return
     *  An ISO compliant language code or null if language code couldn't be
     *  parsed correctly.
     */
    static public String mapUserLanguageToLangCode(String userLanguage) {
        return (null != userLanguage) ? (userLanguage.indexOf('_') >= 0 ? userLanguage.replace('_', '-') : userLanguage) : null;
    }

    /**
     * Gets the base name, minus the full path and extension, from a full
     * filename.
     *
     * @param fileName
     *  The file name where the base name should be retrieved from. The
     *  fileName can be null.
     *
     * @return
     *  The base name of the given filename or null if fileName is null.
     */
    static public String getBaseName(String fileName) {
        return FilenameUtils.getBaseName(fileName);
    }

    /**
     * Gets the file name including the extension, minus the full path, from
     * a full path filename.
     *
     * @param fullPathFileName
     *  The filename with  the full path including the extension. The
     *  fullPathFileName can be null.
     *
     * @return
     *  The file name of the given full path file name or null if the
     *  fullPathFileName is null.
     */
    static public String getFileName(String fullPathFileName) {
        return FilenameUtils.getName(fullPathFileName);
    }

    /**
     * Retrieves the extension of a file name.
     *
     * @param fileName
     *  The file name, the extension should be retrieved from.
     *
     * @return
     *  The extension of the given filename or an empty string in case of
     *  (null == fileName) or if no extension is set at all
     */
    static public String getExtension(String fileName) {
        String extension = "";

        if (null != fileName) {
            int beginIndex = fileName.lastIndexOf(".");
            int endIndex = beginIndex;

            if (beginIndex > -1) {
                while (++endIndex < fileName.length()) {
                    int c = fileName.codePointAt(endIndex);

                    if (!(((c >= 'a') && (c <= 'z')) || ((c >= 'A') && (c <= 'Z'))))
                        break;
                }
            }

            if ((beginIndex > -1) && (endIndex <= fileName.length()) && (beginIndex < (endIndex - 1))) {
                extension = fileName.substring(beginIndex + 1, endIndex);
            }
        }

        return extension.toLowerCase();
    }

    /**
     * Tries to determine the mime type from a given file name.
     *
     * @param fileName
     *  The file name that should be used to retrieve the mime type from.
     *
     * @return
     *  The mime type of the fileName or the empty string if the
     *  mime type could not be determined.
     */
    static public String getMimeType(String fileName) {
        final String extension = getExtension(fileName);
        String mimeType = "";

        if (extension.length() > 0) {
            switch (getDocumentFormat(fileName)) {
            case DOCX: {
                mimeType = DOC_DOCX_TEXT;
                break;
            }

            case PPTX: {
                mimeType = DOC_DOCX_PRESENTATION;
                break;
            }

            case XLSX: {
                mimeType = DOC_DOCX_SPREADSHEET;
                break;
            }

            case ODT: {
                mimeType = DOC_ODF_TEXT;
                break;
            }

            case ODS: {
                mimeType = DOC_ODF_SPREADSHEET;
                break;
            }

            case NONE:
            case ODG:
            case ODP:
            default: {
                if (extension.equals(EXTENSION_DOCX)) {
                    mimeType = DOC_DOCX_TEXT;
                } else if (extension.equals(EXTENSION_XLSX)) {
                    mimeType = DOC_DOCX_SPREADSHEET;
                } else if (extension.equals(EXTENSION_PPTX)) {
                    mimeType = DOC_DOCX_PRESENTATION;
                } else if (extension.equals(EXTENSION_ODT)) {
                    mimeType = DOC_ODF_TEXT;
                } else if (extension.equals(EXTENSION_ODS)) {
                    mimeType = DOC_ODF_SPREADSHEET;
                } else if (extension.equals(EXTENSION_ODP)) {
                    mimeType = DOC_ODF_PRESENTATION;
                } else if (extension.equals(EXTENSION_ODG)) {
                    mimeType = DOC_ODF_GRAPHIC;
                } else if (extension.equals(EXTENSION_BMP)) {
                    mimeType = IMAGE_BMP;
                } else if (extension.equals(EXTENSION_EMF)) {
                    mimeType = IMAGE_EMF;
                } else if (extension.equals(EXTENSION_GIF)) {
                    mimeType = IMAGE_GIF;
                } else if (extension.equals(EXTENSION_JPG_1)) {
                    mimeType = IMAGE_JPEG;
                } else if (extension.equals(EXTENSION_JPG_2)) {
                    mimeType = IMAGE_JPEG;
                } else if (extension.equals(EXTENSION_PICT)) {
                    mimeType = IMAGE_PICT;
                } else if (extension.equals(EXTENSION_PNG)) {
                    mimeType = IMAGE_PNG;
                } else if (extension.equals(EXTENSION_TIFF)) {
                    mimeType = IMAGE_TIFF;
                } else if (extension.equals(EXTENSION_TIFF2)) {
                    mimeType = IMAGE_TIFF;
                } else if (extension.equals(EXTENSION_WMF)) {
                    mimeType = IMAGE_WMF;
                } else if (extension.equals(EXTENSION_XML)) {
                    mimeType = TEXT_XML;
                }

                break;
            }
            }
        }

        return ((mimeType.length() < 1) ? OCTET_STREAM : mimeType);
    }

    /**
     * Provides a global map containing associations between extension
     * and general document format.
     *
     * @return
     *  The global map containing associations between extension
     *  and general document format.
     */
    static public HashMap<String, DocumentFormat> getExtensionMap() {
        // Java DCL pattern with volatile member
        HashMap<String, DocumentFormat> serviceMap = m_extensionMap;

        if (null == serviceMap) {
            synchronized (m_Synchronizer) {
                if (null == (serviceMap = m_extensionMap)) {
                    serviceMap = m_extensionMap = new HashMap<String, DocumentFormat>();

                    m_extensionMap.put("docx", DocumentFormat.DOCX);
                    m_extensionMap.put("docx_ox", DocumentFormat.DOCX);
                    m_extensionMap.put("docm", DocumentFormat.DOCX);
                    m_extensionMap.put("docm_ox", DocumentFormat.DOCX);
                    m_extensionMap.put("dotx", DocumentFormat.DOCX);
                    m_extensionMap.put("dotx_ox", DocumentFormat.DOCX);
                    m_extensionMap.put("dotm", DocumentFormat.DOCX);
                    m_extensionMap.put("dotm_ox", DocumentFormat.DOCX);
                    m_extensionMap.put("pptx", DocumentFormat.PPTX);
                    m_extensionMap.put("pptx_ox", DocumentFormat.PPTX);
                    m_extensionMap.put("xlsx", DocumentFormat.XLSX);
                    m_extensionMap.put("xlsx_ox", DocumentFormat.XLSX);
                    m_extensionMap.put("xltx", DocumentFormat.XLSX);
                    m_extensionMap.put("xltx_ox", DocumentFormat.XLSX);
                    m_extensionMap.put("xlsm", DocumentFormat.XLSX);
                    m_extensionMap.put("xlsm_ox", DocumentFormat.XLSX);
                    m_extensionMap.put("xltm", DocumentFormat.XLSX);
                    m_extensionMap.put("xltm_ox", DocumentFormat.XLSX);
                    m_extensionMap.put("odt", DocumentFormat.ODT);
                    m_extensionMap.put("odt_ox", DocumentFormat.ODT);
                    m_extensionMap.put("ott", DocumentFormat.ODT);
                    m_extensionMap.put("ott_ox", DocumentFormat.ODT);
                    m_extensionMap.put("ods", DocumentFormat.ODS);
                    m_extensionMap.put("ods_ox", DocumentFormat.ODS);
                    m_extensionMap.put("ots", DocumentFormat.ODS);
                    m_extensionMap.put("ots_ox", DocumentFormat.ODS);

                }
            }
        }

        return serviceMap;
    }

    /**
     * Tries to determine the general document format from a given file name.
     *
     * @param fileName
     *  The file name to be used to determine the document format.
     *
     * @return
     *  The document format or DocumentFormat.NONE if no known format could
     *  be found.
     */
    static public DocumentFormat getDocumentFormat(String fileName) {
        final String fileExtension = getExtension(fileName);
        final HashMap<String, DocumentFormat> extensionMap = getExtensionMap();
        DocumentFormat documentFormat = DocumentFormat.NONE;

        if ((null != fileExtension) && extensionMap.containsKey(fileExtension)) {
            documentFormat = extensionMap.get(fileExtension);
        }

        return documentFormat;
    }

    /**
     * Tries to determine the general document type from a given file name.
     *
     * @param fileName
     *  The file name to be used to determine the document format.
     *
     * @return
     *  The document type or DocumentType.NONE if no known type could
     *  be found.
     */
    static public DocumentType getDocumentType(String fileName) {
        return getDocumentTypeFromDocumentFormat(getDocumentFormat(fileName));
    }

    /**
     * Maps a document format to a document type.
     *
     * @param documentFormat
     *  The document format where the type should be determined.
     *
     * @return
     *  The document type or DocumentType.NONE if no known format has been
     *  provided.
     */
    static public DocumentType getDocumentTypeFromDocumentFormat(DocumentFormat documentFormat) {
        DocumentType documentType = DocumentType.NONE;

        if (null != documentFormat) {
            switch (documentFormat) {
            case DOCX:
            case ODT: {
                documentType = DocumentType.TEXT;
                break;
            }

            case XLSX:
            case ODS: {
                documentType = DocumentType.SPREADSHEET;
                break;
            }

            case PPTX:
            case ODP:
            case ODG: {
                documentType = DocumentType.PRESENTATION;
                break;
            }

            case NONE:
            default: {
                break;
            }
            }
        }

        return documentType;
    }

    /**
     * Provides the user's default folder id from the session.
     *
     * @param folderService
     *  The folder service to be used to retrieve the user folder id.
     *
     * @param session
     *  The session of the user.
     *
     * @return
     *  The folder id of the user or null if the folder id couldn't be
     *  retrieved.
     */
    static public String getUserFolderId(FolderService folderService, final Session session) {
        String folderId = null;

        if ((null != folderService) && (null != session) && (session instanceof ServerSession)) {
            final String treeId = com.openexchange.folderstorage.FolderStorage.REAL_TREE_ID;
            final ContentType contentType = InfostoreContentType.getInstance();

            try {
                UserizedFolder defaultFolder = folderService.getDefaultFolder(
                    ((ServerSession) session).getUser(),
                    treeId,
                    contentType,
                    session,
                    null);

                if (null != defaultFolder) {
                    folderId = defaultFolder.getID();
                }
            } catch (OXException e) {
                LOG.warn("GetUserFolderId - retrieving default folder of user failed with exception", e);
            }
        }

        return folderId;
    }

    /**
     * Provides the user's default folder id from the session.
     *
     * @param session
     *  The session of the user.
     *
     * @param services
     *  The service lookup instance to be used by this method.
     *
     * @return
     *  The folder id of the user or null if the folder id couldn't be
     *  retrieved.
     */
    static public String getUserFolderId(final Session session, ServiceLookup services) {
        String folderId = null;

        if (null != services) {
            FolderService folderService = services.getService(FolderService.class);
            folderId = FileHelper.getUserFolderId(folderService, session);
        }

        return folderId;
    }

    /**
     * @param fileId
     * @param folderId
     * @return The mangled access id in the form of 'folderId/fileId'
     */
    static public String getFileAccessId(String fileId, String folderId) {
        return (Strings.isEmpty(folderId) ? fileId : (folderId + "/" + fileId));
    }

    // - Static members --------------------------------------------------------

    static private HashMap<String, DocumentFormat> m_extensionMap = null;

    static private volatile Object m_Synchronizer = new Object();

    // - Members --------------------------------------------------------

    private ServiceLookup m_services = null;

    private String m_folderId = null;

    private String m_fileId = null;

    private DocumentFormat m_documentFormat = DocumentFormat.NONE;

    private DocumentType m_documentType = DocumentType.NONE;

    private long m_documentSize = -1;

    @SuppressWarnings("unused")
    private volatile String m_version = FileStorageFileAccess.CURRENT_VERSION;

    private boolean m_writeProtected = false;

    private Date m_lockedUntil = null;

    private String m_lockedByUser = "";
}
